//***************************************************************************
// "projectile.c"
// Code for enemy projectile objects.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include "ingame.h"
#include "level.h"
#include "objects.h"
#include "physics.h"
#include "platforms.h"
#include "projectile.h"
#include "sound.h"

// How long do some projectiles last. Note that this should match their
// animation!
#define TOXICGAS_LEN       28       // Toxic gas
#define EXPLOSION_LEN      24       // Explosion
#define BIGEXPLOSION_LEN   32       // Big explosion

//***************************************************************************
// init_mortar
// Initializes a mortar object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_mortar(Object *obj) {
   // Set the hitbox
   set_hitbox(obj, -6, 6, -6, 6);

   // Set animation
   set_object_anim(obj, retrieve_object_anim(OB_ANIM_MORTAR));
}

//***************************************************************************
// run_mortar
// Logic for toxic mortar objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_mortar(Object *obj) {
   // If the projectile is too far don't bother keeping it around
   // Needed in case the projectile manages to travel really far (e.g. thrown
   // into a pit or something), also destroys the projectile if it goes
   // outbounds and there's nothing to collide with
   if (is_too_far(obj)) {
      delete_object(obj);
      return;
   }

   // Apply physics
   apply_physics(obj);
   obj->gravity += 0x20;

   // Explode if we hit something
   if (obj->blocked || obj->hurt) {
      add_object(OBJ_EXPLOSION, obj->x, obj->y, 0);
      delete_object(obj);
      return;
   }
}

//***************************************************************************
// init_toxic_gas
// Initializes a toxic gas cloud object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_toxic_gas(Object *obj) {
   // Set the hitbox
   set_hitbox(obj, -8, 8, -8, 8);

   // Set animation
   set_object_anim(obj, retrieve_object_anim(OB_ANIM_TOXICGAS));
}

//***************************************************************************
// run_toxic_gas
// Logic for toxic gas cloud objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_toxic_gas(Object *obj) {
   // Just move where we were told...
   //obj->x += speed_to_int(obj->speed);
   //obj->y += speed_to_int(obj->gravity);
   apply_physics(obj);

   // Expire after a while
   obj->timer++;
   if (obj->timer == TOXICGAS_LEN) {
      delete_object(obj);
      return;
   }
}

//***************************************************************************
// init_fireball
// Initializes a fireball object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_fireball(Object *obj) {
   // Set the hitbox
   set_hitbox(obj, -8, 8, -8, 8);

   // Set animation
   set_object_anim(obj, retrieve_object_anim(OB_ANIM_FIREBALL));
}

//***************************************************************************
// init_explosion
// Initializes a small explosion object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_explosion(Object *obj) {
   // Set the hitbox
   set_hitbox(obj, -16, 16, -16, 16);

   // Set animation
   set_object_anim(obj, retrieve_object_anim(OB_ANIM_EXPLOSION));

   // Set how long this explosion takes
   obj->timer = EXPLOSION_LEN;

   // Break nearby stuff
   break_stuff(obj->x - 0x10, obj->y - 0x10);
   break_stuff(obj->x + 0x10, obj->y - 0x10);
   break_stuff(obj->x - 0x10, obj->y + 0x10);
   break_stuff(obj->x + 0x10, obj->y + 0x10);

   // Play sound effect
   // We don't do this if this is the explosions when beating the final boss
   // because they happen too often, the boss itself will handle it instead
   if (obj->type != OBJ_ENDEXPLOSION)
      play_2d_sfx(SFX_EXPLOSION, obj->x, obj->y);
}

//***************************************************************************
// init_big_explosion
// Initializes a big explosion object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_big_explosion(Object *obj) {
   // Set the hitbox
   set_hitbox(obj, -32, 32, -32, 32);

   // Set animation
   set_object_anim(obj, retrieve_object_anim(OB_ANIM_BIGEXPLOSION));

   // Set how long this explosion takes
   obj->timer = BIGEXPLOSION_LEN;

   // Break nearby stuff
   struct {
      int32_t x;
      int32_t y;
   } list[9] = {
      { -0x20, -0x20 }, { 0x00, -0x20 }, { 0x1F, -0x20 },
      { -0x20,  0x00 }, { 0x00,  0x00 }, { 0x1F,  0x00 },
      { -0x20,  0x1F }, { 0x00,  0x1F }, { 0x1F,  0x1F }
   };
   for (unsigned i = 0; i < 9; i++)
      break_stuff(obj->x + list[i].x, obj->y + list[i].y);

   // Play sound effect
   play_2d_sfx(SFX_BIGEXPLOSION, obj->x, obj->y);
}

//***************************************************************************
// run_explosion
// Logic for small and big explosion objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_explosion(Object *obj) {
   // Expire after a while
   obj->timer--;
   if (obj->timer == 0) {
      delete_object(obj);
      return;
   }
}

//***************************************************************************
// break_stuff
// Breaks stuff that can be broken (e.g. obstructions) and happens to be
// located at the specified coordinates.
//---------------------------------------------------------------------------
// param x: horizontal coordinate
// param y: vertical coordinate
//***************************************************************************

void break_stuff(int32_t x, int32_t y) {
   // Check if there's an obstruction here
   LevelTile *tile = get_tile_by_pixel(x, y);
   if (tile->obstruction)
      break_obstruction(tile, x, y);

   // Check if there's a bridge here
   if (tile->collision == TILE_BRIDGE) {
      // Make tile empty
      tile->collision = TILE_EMPTY;

      // Update affected tile and its surroundings
      int32_t xr = x >> TILE_SIZE_BIT;
      int32_t yr = y >> TILE_SIZE_BIT;

      update_tile_pieces(xr, yr);
      update_tile_pieces(xr - 1, yr);
      update_tile_pieces(xr + 1, yr);
      update_tile_pieces(xr, yr - 1);
      update_tile_pieces(xr, yr + 1);
      update_tile_pieces(xr - 1, yr - 1);
      update_tile_pieces(xr + 1, yr - 1);
      update_tile_pieces(xr - 1, yr + 1);
      update_tile_pieces(xr + 1, yr + 1);
   }
}
