//***************************************************************************
// "player.c"
// Code for player objects.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include "bosses.h"
#include "enemies.h"
#include "ingame.h"
#include "input.h"
#include "level.h"
#include "objects.h"
#include "physics.h"
#include "player.h"
#include "platforms.h"
#include "settings.h"
#include "sound.h"
#include "video.h"

// Different parameters that affect the player physics
// Note: all of these used to be constants, so...
#define MAX_SPEED       (settings.player_speed)  // Maximum running speed
#define FAST_SPEED      (settings.player_fast)   // When player is moving too fast
#define ACCEL_SPEED     (settings.player_accel)  // How fast to accelerate when running
#define FRICTION_SPEED  (settings.player_friction) // "   "  " deccelerate
#define RUN_SPEED       0x40        // Minimum speed to show run animation
#define SKID_SPEED      0x200       // Minimum speed to play skid sound
#define CLIMB_SPEED     (settings.player_climb)  // How fast to move when climbing
#define HAMMER_SPEED    0xC00       // Speed at which the hammer falls
#define WINGS_SPEED     0x200       // How fast can move while flying up
#define PARASOL_SPEED   0x80        // Maximum falling speed with the parasol
#define WEIGHT          (settings.player_weight) // How fast to accelerate when falling
#define WEIGHT_FAST     0x28        // How faster to go for variable jump
#define JUMP_FORCE      (settings.player_jump)   // How high can the player jump
#define CLIMB_HOP       0x340       // How high to hop when done climbing
#define HURT_SPEED      0x280       // How far the player bounces when hurt
#define HURT_RECOIL     0x400       // How high the player bounces when hurt
#define DEAD_RECOIL     0x500       // How high the player bounces when dead
#define INVULNERABILITY 120         // Invulnerability when getting hit
#define WINGS_TIME      60          // How long do wings let the player fly
#define ANGER_LIMIT     3           // How many times to lose to get angry
#define PLAYER_HEIGHT   (settings.player_height)   // Usual height
#define PLAYER_CROUCH   (settings.player_crouch)   // Height when crouching

// And for the sonars
#define HSONAR_DELAY    12          // Delay between horizontal beeps
#define VSONAR_DELAY    12          // Delay between vertical beeps
#define VSONAR_MAXDIST  0x200       // Maximum distance to worry about

// Possible animations for a player
typedef enum {
   PL_ANIM_IDLE,        // Idle
   PL_ANIM_ANGRY,       // Idle (upset)
   PL_ANIM_RUN,         // Running (normal)
   PL_ANIM_RUNFAST,     // Running (fast)
   PL_ANIM_SKID,        // Skidding
   PL_ANIM_JUMP,        // Jumping
   PL_ANIM_FALL,        // Falling
   PL_ANIM_CROUCH,      // Crouching
   PL_ANIM_LOOKUP,      // Looking up
   PL_ANIM_BALANCE,     // Balancing (forwards)
   PL_ANIM_BALANCE2,    // Balancing (backwards)
   PL_ANIM_HURT,        // Got hurt
   PL_ANIM_DEAD,        // Died
   PL_ANIM_CLIMBIDLE,   // Climbing (idle)
   PL_ANIM_CLIMBUP,     // Climbing (moving up)
   PL_ANIM_CLIMBDOWN,   // Climbing (moving down)
   PL_ANIM_FLY,         // Flying
   PL_ANIM_HAMMER,      // Dropping hammer
   PL_ANIM_PARASOL,     // Falling with parasol
   PL_ANIM_PINBALL,     // Pinballing
   NUM_PL_ANIM          // Number of animations
} PlayerAnim;

// Where player graphics are stored
static GraphicsSet *gfxset_player = NULL;
static AnimFrame *anim_player[NUM_PL_ANIM];

// Used to make the player look angry when losing too often
static unsigned anger_count;

//***************************************************************************
// load_player
// Loads the player assets.
//***************************************************************************

void load_player(void) {
   // Load graphics
   gfxset_player = load_graphics_set("graphics/sol");

   // Get a list of all the animations we need
#define ANIM(id, name) anim_player[id] = get_anim(gfxset_player, name);
   ANIM(PL_ANIM_IDLE, "idle_ingame");
   ANIM(PL_ANIM_ANGRY, "angry_ingame");
   ANIM(PL_ANIM_RUN, "run");
   ANIM(PL_ANIM_RUNFAST, "run_fast");
   ANIM(PL_ANIM_SKID, "skid");
   ANIM(PL_ANIM_JUMP, "jump");
   ANIM(PL_ANIM_FALL, "fall");
   ANIM(PL_ANIM_CROUCH, "crouch");
   ANIM(PL_ANIM_LOOKUP, "look_up");
   ANIM(PL_ANIM_BALANCE, "balance");
   ANIM(PL_ANIM_BALANCE2, "balance2");
   ANIM(PL_ANIM_HURT, "hurt");
   ANIM(PL_ANIM_DEAD, "dead");
   ANIM(PL_ANIM_CLIMBIDLE, "climb_idle");
   ANIM(PL_ANIM_CLIMBUP, "climb_up");
   ANIM(PL_ANIM_CLIMBDOWN, "climb_down");
   ANIM(PL_ANIM_FLY, "fly");
   ANIM(PL_ANIM_HAMMER, "hammer");
   ANIM(PL_ANIM_PARASOL, "parasol");
   ANIM(PL_ANIM_PINBALL, "pinball");
#undef ANIM
}

//***************************************************************************
// unload_player
// Frees up the resources taken up by player assets.
//***************************************************************************

void unload_player(void) {
   // Unload graphics
   if (gfxset_player) {
      destroy_graphics_set(gfxset_player);
      gfxset_player = NULL;
   }
}

//***************************************************************************
// init_player
// Initializes a player object. Needed to set the player's hitbox or the
// physics at the beginning will screw up.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_player(Object *obj) {
   // Don't be angry if we're just starting, otherwise get angrier
   if (prev_game_mode != GAMEMODE_INGAME)
      reset_anger();
   else if (anger_count < ANGER_LIMIT)
      anger_count++;

   // Set physics
   set_hitbox(obj, -7, 7, -PLAYER_HEIGHT + 0x10, 15);

   // Fill health
   if (!settings.collect_health)
      obj->health = settings.extra_health ? 5 : 3;
   else
      obj->health = 0;

   // Create shield object for this player
   Object *ptr = add_object(OBJ_SHIELD, obj->x, obj->y, 0);
   ptr->otherobj = obj;

   // Create power-up attachment object for this player (shows which power-up
   // you have at the moment)
   if (settings.show_power) {
      ptr = add_object(OBJ_POWERUP, obj->x, obj->y, 0);
      obj->otherobj = ptr;
   }

   // Reset direction in one-switch mode
   input.oneswitch.dir = 0;
   input.oneswitch.toggle = 0;

   // Spawn the sonars in audiovideo mode
   if (settings.audiovideo) {
      add_object(OBJ_HSONAR, obj->x, obj->y, 0);
      add_object(OBJ_VSONAR, obj->x, obj->y, 0);
   }
}

//***************************************************************************
// run_player
// Game logic for a player object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//---------------------------------------------------------------------------
// How this object uses some the not-so-obvious object properties... (the
// player object is the one object to make use to of the object structure to
// its full extent, and many properties were designed for it originally)
//
// obj->timer: depends on power-up
// obj->invincibility: whether invincibility is in use or not
// obj->invulnerability: whether we're flashing or not
// obj->health: how many hearts do we have left
// obj->dir: which direction we're facing
// obj->jumping: whether we're jumping or just falling
// obj->crouch: whether we're crouching or not
// obj->active: whether power-up ability is in use
// obj->hurt: whether we're recoiling or not
// obj->dead: whether we're in death animation or not
// obj->shield: whether we have a shield or not
// obj->otherobj: object showing the power-up attachment
//---------------------------------------------------------------------------
// To-do: check the above list to ensure it's up to date
//***************************************************************************

void run_player(Object *obj) {
   // Enforce a power-up?
   if (settings.force_power)
      obj->type = OBJ_PLAYER + settings.force_power;

   // Free movement mode?
   if (settings.free_move) {
      // No more interaction with the world!
      obj->has_hitbox = 0;

      // Set movement speed
      int32_t speed = input.player.hold[PL_INPUT_ACTION] ? 4 : 2;

      // Use arrow keys to move around
      if (input.player.hold[PL_INPUT_UP])
         obj->y -= speed;
      if (input.player.hold[PL_INPUT_DOWN])
         obj->y += speed;
      if (input.player.hold[PL_INPUT_LEFT])
         obj->x -= speed;
      if (input.player.hold[PL_INPUT_RIGHT])
         obj->x += speed;

      // No more interaction with the world!
      return;
   }

   // Keeping track of the old state is useful later
   int32_t old_speed = obj->speed;
   int32_t old_gravity = obj->gravity;
   int old_on_ground = obj->on_ground;

   // Mouse-switch controls?
   if (settings.mouse_switch) {
      // Climbing as a spider? (this needs different mappings due to moving
      // with up/down instead of left/right)
      if (obj->type == OBJ_PLAYERSPIDER && obj->active) {
         input.player.hold[PL_INPUT_UP] = input.mouseswitch.up;
         input.player.hold[PL_INPUT_DOWN] = input.mouseswitch.down;
         input.player.hold[PL_INPUT_ACTION] = input.mouseswitch.left;
         input.player.press[PL_INPUT_ACTION] = input.mouseswitch.left & 2;
         input.player.hold[PL_INPUT_ACTION] |= input.mouseswitch.right;
         input.player.press[PL_INPUT_ACTION] |= input.mouseswitch.right & 2;
      }

      // Normal control mapping?
      else {
         input.player.hold[PL_INPUT_UP] = 0;
         input.player.hold[PL_INPUT_LEFT] = input.mouseswitch.left;
         input.player.hold[PL_INPUT_RIGHT] = input.mouseswitch.right;
         input.player.hold[PL_INPUT_DOWN] = input.mouseswitch.down;
         input.player.hold[PL_INPUT_ACTION] = input.mouseswitch.up;
         input.player.press[PL_INPUT_ACTION] = input.mouseswitch.up & 2;
      }
   }

   // Get current user input
   int action_up = 0;
   int action_down = 0;
   int action_left = 0;
   int action_right = 0;
   int action_jump = 0;
   int action_jumpstart = 0;

   // Normal controls?
   if (!settings.one_switch) {
      action_up = input.player.hold[PL_INPUT_UP];
      action_down = input.player.hold[PL_INPUT_DOWN];
      action_left = input.player.hold[PL_INPUT_LEFT];
      action_right = input.player.hold[PL_INPUT_RIGHT];
      action_jump = input.player.hold[PL_INPUT_ACTION];
      action_jumpstart = input.player.press[PL_INPUT_ACTION];
   }

   // One-switch controls?
   else {
      // Change direction when tapping
      if (input.oneswitch.tap) {
         input.oneswitch.dir++;
         input.oneswitch.dir &= 3;
      }

      // Er, OK...
      if (obj->type == OBJ_PLAYERHAMMER && obj->active)
         input.oneswitch.dir = 0;

      // Handle jumping
      if ((obj->type == OBJ_PLAYERWINGS || obj->type == OBJ_PLAYERPARASOL)
      && (obj->jumping || obj->active)) {
         input.oneswitch.toggle ^= input.oneswitch.tap2;
      } else {
         input.oneswitch.toggle = 0;
      }
      action_jump = input.oneswitch.hold || input.oneswitch.toggle;
      action_jumpstart = input.oneswitch.tap2;

      // Set movement actions
      if (obj->type == OBJ_PLAYERSPIDER && obj->active) {
         action_up = input.oneswitch.dir == 1;
         action_down = input.oneswitch.dir == 3;
      } else {
         action_left = input.oneswitch.dir == 3;
         action_right = input.oneswitch.dir == 1;
      }
   }

   // Get rid of impossible user input
   if (action_up && action_down)
      action_up = action_down = 0;
   if (action_left && action_right)
      action_left = action_right = 0;

   // If the main player died, restart the level when it goes off-screen
   if (obj->dead && obj->gravity > 0 && obj->y > camera.y + 0x100 &&
   obj == get_first_object(OBJGROUP_PLAYER))
      fade_off_and_switch(GAMEMODE_INGAME);

   // Can't control ourselves if hurt!
   if (!obj->dead && !obj->hurt) {
      // Climbing?
      if (obj->type == OBJ_PLAYERSPIDER && obj->active) {
         // We don't move horizontally!
         obj->speed = 0;

         // Use up and down to climb around
         // We also put a cap to prevent the player from going off-screen by
         // climbing up (which could lead to serious gameplay issues...)
         if (action_up && obj->y - speed_to_int(CLIMB_SPEED) > 0)
            obj->gravity = -CLIMB_SPEED;
         else if (action_down)
            obj->gravity = CLIMB_SPEED;
         else
            obj->gravity = 0;
      }

      // Normal controls apply
      else {
         // Determine at what speed we accelerate/decelerate. When we aren't
         // on the ground our ability to change our momentum is reduced since
         // there isn't friction.
         int32_t accel_speed = obj->on_ground ?
            ACCEL_SPEED : ACCEL_SPEED >> 1;
         int32_t friction_speed = obj->on_ground ?
            FRICTION_SPEED : FRICTION_SPEED >> 1;

         // Run around
         if (action_right) {
            if (obj->dir && obj->on_ground &&
            (obj->speed <= -SKID_SPEED || obj->speed <= -settings.player_fast))
               play_sfx(SFX_SKID);
            obj->dir = 0;
            accelerate_object(obj, obj->speed >= 0 ? accel_speed :
               friction_speed, MAX_SPEED);
         } else if (action_left) {
            if (!obj->dir && obj->on_ground &&
            (obj->speed >= SKID_SPEED || obj->speed >= settings.player_fast))
               play_sfx(SFX_SKID);
            obj->dir = 1;
            accelerate_object(obj, obj->speed <= 0 ? -accel_speed :
               -friction_speed, MAX_SPEED);
         } else if (!settings.no_friction/* || abs(obj->speed) <= RUN_SPEED*/
         || (!obj->dir && obj->speed < 0) || (obj->dir && obj->speed > 0))
            decelerate_object(obj, friction_speed);
      }
   }

   // Falling?
   if (!obj->on_ground) {
      // Climbing?
      if (obj->active && obj->type == OBJ_PLAYERSPIDER) {
         // Press jump button to hop away from the wall (and stop climbing)
         if (action_jumpstart) {
            // Face away from the wall
            obj->dir = ~obj->dir;

            // Set up hop physics
            obj->speed = obj->dir ? -MAX_SPEED : MAX_SPEED;
            obj->gravity = -JUMP_FORCE/2;
            obj->jumping = 1;

            // Play sound effect as if we were jumping
            play_sfx(SFX_JUMP);

            // Not climbing anymore!
            obj->active = 0;

            // One-switch shenanigans...
            input.oneswitch.dir = 0;
         }

         // If we reached the end of the wall then we can't proceed going
         // further. Hop so we can reach the floor and stop climbing here.
         if (!can_climb(obj)) {
            obj->gravity = obj->gravity < 0 ? -CLIMB_HOP : 0;
            obj->active = 0;
         }
      }

      // Hammering?
      else if (obj->active && obj->type == OBJ_PLAYERHAMMER) {
         // Override momentum
         obj->speed = 0;
         obj->gravity = HAMMER_SPEED;
      }

      // Flying?
      else if (obj->active && obj->type == OBJ_PLAYERWINGS) {
         // Go *up* instead of down
         obj->gravity -= WEIGHT;
         if (obj->gravity < -WINGS_SPEED)
            obj->gravity = -WINGS_SPEED;

         // We can only fly for a limited time
         obj->timer--;
         if (obj->timer == 0)
            obj->active = 0;

         // The player must hold down the jump button to keep flying
         if (!action_jump)
            obj->active = 0;
      }

      // Normal in-air semantics or parasol
      else {
         // Go down!
         obj->gravity += WEIGHT;

         // Using the parasol?
         if (obj->type == OBJ_PLAYERPARASOL && obj->active) {
            // Impose a limit on how fast we can fall
            if (obj->gravity > PARASOL_SPEED)
               obj->gravity = PARASOL_SPEED;

            // The player must hold down the jump button to keep using the
            // parasol (otherwise Sol starts falling fast again)
            if (!action_jump)
               obj->active = 0;
         }

         // Variable jump
         // Basically, if you don't hold down the jump button while going up,
         // you will fall faster than usual. Note that this only matters when
         // jumping, not if something else sends you flying around.
         if (obj->jumping && obj->gravity < 0 &&
         !action_jump)
            obj->gravity += WEIGHT_FAST;

         // If we have the spider power-up, we can double jump next to walls
         // to start climbing
         if (obj->type == OBJ_PLAYERSPIDER && obj->jumping &&
         action_jumpstart && can_climb(obj)) {
            obj->active = 1;
            obj->jumping = 0;
            input.oneswitch.dir = 0;
         }

         // If we have the hammer power-up, double jumping will result in
         // Sol using the hammer and dash downwards immediately
         if (obj->type == OBJ_PLAYERHAMMER && obj->jumping &&
         action_jumpstart) {
            obj->active = 1;
            obj->jumping = 0;
            obj->speed = 0;
            obj->gravity = HAMMER_SPEED;
            input.oneswitch.dir &= 2;
            play_sfx(SFX_HAMMER);
         }

         // If we have the wings power-up, double jumping will make Sol fly
         // for a bit (enough to reach a bit higher places)
         if (obj->type == OBJ_PLAYERWINGS && obj->jumping &&
         action_jumpstart) {
            // Set physics
            obj->active = 1;
            obj->jumping = 0;
            obj->timer = WINGS_TIME;

            // Play sound effect
            play_sfx(SFX_FLY);
         }

         // If we have the parasol power-up, double jumping will make Sol
         // use the parasol, which lets her slow down her fall
         if (obj->type == OBJ_PLAYERPARASOL && obj->jumping &&
         action_jumpstart) {
            // Set physics
            obj->active = 1;
            obj->jumping = 0;
         }
      }
   }

   // Landed?
   else {
      // If we were climbing, flying or using the parasol then stop
      if (obj->active && (obj->type == OBJ_PLAYERSPIDER ||
      obj->type == OBJ_PLAYERWINGS || obj->type == OBJ_PLAYERPARASOL)) {
         obj->active = 0;
         if (obj->type == OBJ_PLAYERSPIDER)
            input.oneswitch.dir = 0;
      }

      // If we were recoiling from being hurt, restore normal physics
      if (obj->hurt) {
         obj->hurt = 0;
         obj->speed = 0;
         obj->invulnerability = INVULNERABILITY;
      }

      // If we were using the hammer then bounce
      if (obj->active && obj->type == OBJ_PLAYERHAMMER) {
         do_hammer_bounce(obj);
      }

      // Jump?
      else if (action_jumpstart) {
         // Set physics accordingly
         obj->gravity = -JUMP_FORCE;
         obj->on_ground = 0;
         obj->jumping = 1;
         obj->pinball = settings.spin_jump;

         // Play sound effect
         play_sfx(SFX_JUMP);
      }

      // Just stay here
      else {
         obj->jumping = 0;
      }
   }

   // Crouch?
   obj->crouching = action_down && obj->on_ground && !obj->speed;

   // Set object hitbox
   if (obj->dead)
      obj->has_hitbox = 0;
   else if (obj->type == OBJ_PLAYERPARASOL && obj->active)
      set_hitbox(obj, -7, 7, -16, 15);
   else if (obj->pinball)
      set_hitbox(obj, -7, 7, -12, 15);
   else if (obj->crouching)
      set_hitbox(obj, -7, 7, -PLAYER_CROUCH + 0x10, 15);
   else
      set_hitbox(obj, -7, 7, -PLAYER_HEIGHT + 0x10, 15);

   // Interact with objects that can hurt us only if we didn't get just hurt
   // yet (prevents objects from exploding and such while we recoil)
   if (!obj->dead && !obj->hurt) {
      // Check if we have been hit by an enemy projectile
      for (Object *other = get_first_object(OBJGROUP_PROJECTILE);
      other != NULL; other = other->next) {
         if (collision(obj, other)) {
            hurt_player(obj, other->x);
            other->hurt = 1;
            break;
         }
      }

      // Check if we have hit an enemy
      for (Object *enemy = get_first_object(OBJGROUP_ENEMY); enemy != NULL;
      enemy = enemy->next) {
         // Did we collide with this enemy?
         if (!collision(obj, enemy))
            continue;

         // Did we land on the enemy? If so, attack it and bounce!
         // (also attack enemies when invincible)
         if ((!obj->on_ground && obj->gravity > 0 && obj->y < enemy->y) ||
         obj->invincibility || obj->pinball) {
            if (obj->type == OBJ_PLAYERHAMMER && obj->active)
               do_hammer_bounce(obj);
            else
               obj->gravity = -abs(obj->gravity);
            destroy_enemy(enemy);
         }

         // Nope, got hurt instead...
         else {
            hurt_player(obj, enemy->x);
         }

         // There, interacted with an enemy, done
         // Yeah, only one enemy at a time...
         break;
      }

      // Check if we have hit an boss
      for (Object *boss = get_first_object(OBJGROUP_BOSS); boss != NULL;
      boss = boss->next) {
         // Did we collide with this boss?
         if (!boss->health || boss->invulnerability || !collision(obj, boss))
            continue;

         // Did we land on the boss? If so, attack it and bounce!
         // (also attack bosses when invincible)
         if ((!obj->on_ground && obj->gravity > 0 && obj->y < boss->y) ||
         obj->invincibility || obj->pinball) {
            if (obj->type == OBJ_PLAYERHAMMER && obj->active)
               do_hammer_bounce(obj);
            else
               obj->gravity = -abs(obj->gravity);
            hurt_boss(boss);
         }

         // Nope, got hurt instead...
         else {
            hurt_player(obj, boss->x);
         }

         // There, interacted with an boss, done
         // Yeah, only one boss at a time...
         break;
      }
   }

   // Did we fall into a bottomless pit?
   if (obj->y > camera.limit_bottom)
      kill_player(obj);

   // Run physics (momentum, map collision, etc.)
   apply_physics(obj);

   // Don't let the player go off-screen!
   if (obj->x <= camera.limit_left + 8) {
      obj->x = camera.limit_left + 8;
      if (obj->speed < 0)
         obj->speed = 0;
      obj->on_wall = 1;
   }
   if (obj->x >= camera.limit_right - 8) {
      obj->x = camera.limit_right - 8;
      if (obj->speed > 0)
         obj->speed = 0;
      obj->on_wall = 1;
   }

   // Determine which animation to use based on our current status
   // Warning: massive if block ahead!
   PlayerAnim anim = (anger_count >= ANGER_LIMIT) ?
      PL_ANIM_ANGRY : PL_ANIM_IDLE;

   // Power-up specific animation?
   if (obj->active) {
      switch (obj->type) {
         // Spider power-up (animation depends on whether the player is
         // moving or not, so this one is more complex than the rest)
         case OBJ_PLAYERSPIDER:
            if (obj->gravity > 0)
               anim = PL_ANIM_CLIMBDOWN;
            else if (obj->gravity < 0)
               anim = PL_ANIM_CLIMBUP;
            else
               anim = PL_ANIM_CLIMBIDLE;
            break;

         // Power-ups with just a single animation
         case OBJ_PLAYERHAMMER: anim = PL_ANIM_HAMMER; break;
         case OBJ_PLAYERWINGS: anim = PL_ANIM_FLY; break;
         case OBJ_PLAYERPARASOL: anim = PL_ANIM_PARASOL; break;
         default: break;
      }
   }

   // Normal animation?
   else if (obj->dead)
      anim = PL_ANIM_DEAD;
   else if (obj->hurt)
      anim = PL_ANIM_HURT;
   else if (obj->pinball)
      anim = PL_ANIM_PINBALL;
   else if (!obj->on_ground)
      anim = obj->gravity < 0 ? PL_ANIM_JUMP : PL_ANIM_FALL;
   else if (obj->speed >= FAST_SPEED)
      anim = obj->dir ? PL_ANIM_SKID : PL_ANIM_RUNFAST;
   else if (obj->speed <= -FAST_SPEED)
      anim = obj->dir ? PL_ANIM_RUNFAST : PL_ANIM_SKID;
   else if (obj->speed >= RUN_SPEED)
      anim = obj->dir ? PL_ANIM_SKID : PL_ANIM_RUN;
   else if (obj->speed <= -RUN_SPEED)
      anim = obj->dir ? PL_ANIM_RUN : PL_ANIM_SKID;
   else if (obj->crouching)
      anim = PL_ANIM_CROUCH;
   else if (action_up)
      anim = PL_ANIM_LOOKUP;
   else if (obj->on_cliff_l)
      anim = obj->dir ? PL_ANIM_BALANCE : PL_ANIM_BALANCE2;
   else if (obj->on_cliff_r)
      anim = obj->dir ? PL_ANIM_BALANCE2 : PL_ANIM_BALANCE;

   // Set new object animation
   set_object_anim(obj, anim_player[anim]);

   // Extra clues for audiovideo mode
   if (settings.audiovideo) {
      // Used to determine the pitch of the footsteps
      // Upwards slopes have higher pitch footsteps
      // Downwards slopes have lower pitch footsteps
      uint32_t flags = SFLAG_REVERBOK;

      TileType type = get_tile_by_pixel(obj->x, obj->abs_hitbox.y2)->collision;
      switch (type) {
         case TILE_NESW_1: case TILE_NESW_1_BG:
         case TILE_NESW_2: case TILE_NESW_2_BG:
            flags |= obj->dir ? SFLAG_LOPITCH : SFLAG_HIPITCH;
            break;
         case TILE_NWSE_1: case TILE_NWSE_1_BG:
         case TILE_NWSE_2: case TILE_NWSE_2_BG:
            flags |= obj->dir ? SFLAG_HIPITCH : SFLAG_LOPITCH;
            break;
         default:
            break;
      }

      // Footsteps
      if ((anim == PL_ANIM_RUN && !(game_anim % 12)) ||
      (anim == PL_ANIM_RUNFAST && !(game_anim % 6)))
         play_sfx_ex(SFX_FOOTSTEP, obj->x, obj->y, 0x100, 0, flags);

      // Bumping into the level
      if (obj->on_ground && !old_on_ground)
         play_sfx(SFX_LAND);
      if (obj->on_wall && abs(old_speed) >= 0x100)
         play_sfx(SFX_WALL);
      if (obj->on_ceiling)
         play_sfx(SFX_CEILING);

      // Falling
      if (old_gravity <= 0x100 && obj->gravity > 0x100)
         play_sfx(SFX_FALL);
   }

   // Invincibles yet?
   if (obj->invincibility) {
      // Spawn a sparkle every so often
      if ((obj->invincibility & 0x03) == 0) {
         // If we place the sparkles truly randomly, the end result is ugly,
         // because they will tend to get grouped into some part instead of
         // being evenly distributed (yes, this is what happens with true
         // randomness), so calculate some offsets depending on the timing to
         // spread them a bit more evenly.
         int32_t base_x = obj->x - 0x0E;
         int32_t base_y = obj->y - 0x18;
         if (obj->invincibility & 0x04) base_x += 0x0E;
         if (obj->invincibility & 0x08) base_y += 0x12;

         // Spawn the sparkle
         add_object(OBJ_SPARKLE,
            base_x + get_random(0x0C),
            base_y + get_random(0x13),
            0);
      }

      // Reduce invincibility time left
      obj->invincibility--;

      // Over?
      if (obj->invincibility == 0)
         play_ingame_bgm(0);
   }

   // If we have a power up, make the power-up attachment object show the
   // proper sprite. Otherwise, make it invisible (so it looks like there's
   // nothing attached to us)
   // Also, if the power-up is active then the player sprite will be showing
   // the power-up attachment itself (since it has to do more complex
   // animations), so make the attachment object invisible there as well.
   if (obj->otherobj != NULL) {
      if (obj->active || obj->pinball) {
         set_object_anim(obj->otherobj, NULL);
      } else {
         int anim;
         switch (obj->type) {
            case OBJ_PLAYERWINGS: anim = OB_ANIM_WINGS; break;
            case OBJ_PLAYERSPIDER: anim = OB_ANIM_SPIDER; break;
            case OBJ_PLAYERHAMMER: anim = OB_ANIM_HAMMER; break;
            case OBJ_PLAYERPARASOL: anim = OB_ANIM_PARASOL; break;
            default: anim = -1; break;
         }

         set_object_anim(obj->otherobj, anim != -1 ?
            retrieve_object_anim(anim) : NULL);
      }

      // Keep the power-up attachment in sync with us
      obj->otherobj->x = obj->x;
      obj->otherobj->y = obj->y;
      obj->otherobj->dir = obj->dir;
      obj->otherobj->invulnerability = obj->invulnerability;

      // Determine where the attachment should appear
      // Each player sprite has an "attachment offset" which is used to move
      // the attachment object. This is done so it appears to move along with
      // the player's animation instead of being stuck to its coordinates.
      if (obj->frame) {
         if (obj->frame->flags & SPR_HFLIP)
            obj->otherobj->dir = ~obj->otherobj->dir;
         obj->otherobj->x += obj->otherobj->dir ?
             -obj->frame->x_attachment : obj->frame->x_attachment;
         obj->otherobj->y += obj->frame->y_attachment;
      }
   }
}

//***************************************************************************
// hurt_player
// Attempts to hurt a player object. Usual gameplay rules apply (so e.g. if
// the player is invulnerable, the player won't get hurt).
//---------------------------------------------------------------------------
// param obj: pointer to player object
// param x: X coordinate of damage source
//---------------------------------------------------------------------------
// Tip: if you don't have a damage source (e.g. the player wasn't damaged by
// collision against an object), you can pass the player's X coordinate as
// the second parameter.
//***************************************************************************

void hurt_player(Object *obj, int32_t x) {
   // Do nothing in free movement mode
   if (settings.free_move)
      return;

   // Can't get hurt?
   if (obj->dead || obj->hurt || obj->invulnerability || obj->invincibility)
      return;

   // Reduce health and kill player if the health is depleted
   // Shields prevent health from being lost as well as the power-up
   // The "no damage" cheat prevents health from depleting (though it's still
   // possible to die in other ways), while the "no health" cheat behaves as
   // if the player always had only one point of health left.
   // It's all a horrible mess as you can see :D

   // Lose shield if we had one
   if (obj->shield)
      obj->shield = 0;

   // Get hurt then
   else if (!settings.no_damage) {
      // Reduce health
      if (settings.no_health) {
         obj->health = 0;
         obj->dead = 1;
      } else if (obj->type != OBJ_PLAYER &&
      (settings.power_shield && settings.force_power == POWER_NONE)) {
         /* Lose power-up, see below */
      } else if (settings.collect_health) {
         obj->dead = obj->health ? 0 : 1;
         obj->health = 0;
      } else {
         obj->health--;
         obj->dead = obj->health ? 0 : 1;
      }

      // Get rid of power-up
      obj->type = OBJ_PLAYER;

      // Ran out of health? D:
      if (obj->dead) {
         obj->dead = 0;       // Trust me...
         kill_player(obj);
         return;
      }
   }

   // Set player's direction. The player should be facing whatever damaged
   // it. If the position is the same, just don't bother with it then.
   if (obj->x < x) obj->dir = 0;
   if (obj->x > x) obj->dir = 1;

   // Push player back
   obj->speed = obj->dir ? HURT_SPEED : -HURT_SPEED;
   obj->gravity = -HURT_RECOIL;
   obj->on_ground = 0;
   obj->jumping = 0;
   obj->active = 0;

   // Set player as being hurt
   obj->hurt = 1;

   // Play sound effect
   play_sfx(SFX_HURT);
}

//***************************************************************************
// kill_player
// Instantaneously kills a player, regardless of its health status.
//---------------------------------------------------------------------------
// param obj: pointer to player object
//***************************************************************************

void kill_player(Object *obj) {
   // Do nothing in free movement mode
   if (settings.free_move)
      return;

   // Already dead?
   if (obj->dead)
      return;

   // Make player bounce up
   obj->speed = 0;
   obj->gravity = -DEAD_RECOIL;
   obj->on_ground = 0;
   obj->jumping = 0;
   obj->has_hitbox = 0;

   // Get rid of power-up
   obj->type = OBJ_PLAYER;
   obj->active = 0;

   // Quick hack so the player stops flashing if it just got hurt before
   // dying (can this really be considered a hack?)
   obj->invincibility = 0;
   obj->invulnerability = 0;

   // Mark player as dead
   obj->shield = 0;
   obj->health = 0;
   obj->dead = 1;

   // Play sound effect
   play_sfx(SFX_HURT);
}

//***************************************************************************
// do_hammer_bounce
// Does the physics for when a player bounces when using the hammer.
//---------------------------------------------------------------------------
// param obj: pointer to player object
//***************************************************************************

void do_hammer_bounce(Object *obj) {
   // Used to know if we broke something
   // (this affects what sound it makes)
   int broke = 0;

   // Check if there was an obstruction under us, and break it if so
   LevelTile *tile = get_tile_by_pixel(obj->x, obj->y + 0x10);
   if (tile->obstruction) {
      break_obstruction(tile, obj->x, obj->y + 0x10);
      broke = 1;
   } else {
      // The original check will only detect obstructions right under the
      // player, but can still miss if the player is on the border of one.
      // These checks handle those cases - note that if there *was* an
      // obstruction right under the player then that one will get destroyed,
      // not the adjacent ones (ensures only one can be destroyed at any
      // given time).

      // Check to the left
      tile = get_tile_by_pixel(obj->abs_hitbox.x1, obj->y + 0x10);
      if (tile->obstruction) {
         break_obstruction(tile, obj->abs_hitbox.x1, obj->y + 0x10);
         broke = 1;
      }

      // Check to the right
      tile = get_tile_by_pixel(obj->abs_hitbox.x2, obj->y + 0x10);
      if (tile->obstruction) {
         break_obstruction(tile, obj->abs_hitbox.x2, obj->y + 0x10);
         broke = 1;
      }
   }

   // Play sound effect
   play_sfx(broke ? SFX_BREAK : SFX_CRUSH);

   // Bounce!
   obj->active = 0;
   obj->gravity = -HAMMER_SPEED / 2;
   obj->jumping = 1;
}

//***************************************************************************
// reset_anger
// Makes the player stop being angry!
//***************************************************************************

void reset_anger(void) {
   anger_count = 0;
}

//***************************************************************************
// init_hsonar
// Initializes the horizontal sonar object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_hsonar(Object *obj) {
   // Set up hitbox
   set_hitbox(obj, -0x20, 0x20, -0x10, 0x0F);
}

//***************************************************************************
// run_hsonar
// Horizontal sonar behavior, scans around the level to give extra clues
// about the floor.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_hsonar(Object *obj) {
   // Update timer
   if (obj->timer) {
      obj->timer--;
      return;
   }

   // We're going to be tracking the main player
   Object *player = get_first_object(OBJGROUP_PLAYER);
   if (player == NULL || player->dead)
      return;

   // Let the game physics handle sonar detection
   obj->x = player->x;
   obj->y = player->y;
   obj->speed = player->speed;
   obj->gravity = player->gravity;
   obj->dir = player->dir;
   refresh_hitbox(obj);
   apply_physics(obj);

   // Check for gaps
   if (player->on_ground) {
      if (obj->on_cliff_l && obj->dir) {
         play_sfx(SFX_SONARLEFT);
         obj->timer = HSONAR_DELAY;
      }
      if (obj->on_cliff_r && !obj->dir) {
         play_sfx(SFX_SONARRIGHT);
         obj->timer = HSONAR_DELAY;
      }
   }
}

//***************************************************************************
// init_vsonar
// Initializes the vertical sonar object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_vsonar(Object *obj) {
   (void)(obj);
   // Set up hitbox
   //set_hitbox(obj, -0x10, 0x10, -0x30, 0x2F);
}

//***************************************************************************
// run_vsonar
// Vertical sonar behavior, scans around the level to give extra clues about
// what's over and under the player.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_vsonar(Object *obj) {
   // Update timer
   if (obj->timer) {
      obj->timer--;
      return;
   }

   // We're going to be tracking the main player
   Object *player = get_first_object(OBJGROUP_PLAYER);
   if (player == NULL || player->dead)
      return;

   // Maximum distance detected for the nearest floor or platform
   // This is so we only beep for the nearest one
   int32_t max_dist = VSONAR_MAXDIST;
   int32_t dist_x = 0;
   int32_t dist_y = 0;

   // Scan upwards?
   if (player->on_ground || player->gravity < 0) {
      // Check all platforms for the closest one, if any
      for (Object *platform = get_first_object(OBJGROUP_PLATFORM);
      platform != NULL; platform = platform->next) {
         if (platform->y > player->y) continue;
         if (!platform->has_hitbox) continue;

         int32_t x = platform->x - player->x;
         int32_t y = platform->y - player->y;
         int32_t dist = abs(x) + abs(y);

         if (dist < max_dist) {
            max_dist = dist;
            dist_x = x;
            dist_y = y;
         }
      }

      // Beep!
      if (max_dist < VSONAR_MAXDIST) {
         play_sfx_ex(SFX_SONARUP, dist_x, dist_y, 0x100, 0,
            SFLAG_ABSOLUTE || SFLAG_NOADJUST);
         obj->timer = VSONAR_DELAY;
      }
   }

   // Scan downwards?
   else {
      // Check all platforms for the closest one, if any
      for (Object *platform = get_first_object(OBJGROUP_PLATFORM);
      platform != NULL; platform = platform->next) {
         if (platform->y < player->y) continue;
         if (!platform->has_hitbox) continue;

         int32_t x = platform->x - player->x;
         int32_t y = platform->y - player->y;
         int32_t dist = abs(x) + abs(y);

         if (dist < max_dist) {
            max_dist = dist;
            dist_x = x;
            dist_y = y;
         }
      }

      // Beep!
      if (max_dist < VSONAR_MAXDIST) {
         play_sfx_ex(SFX_SONARDOWN, dist_x, dist_y, 0x100, 0,
            SFLAG_ABSOLUTE || SFLAG_NOADJUST);
         obj->timer = VSONAR_DELAY;
      }
   }
}
