//***************************************************************************
// "platforms.c"
// Code for collidable objects (platforms, pushables, etc.).
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include "ingame.h"
#include "level.h"
#include "objects.h"
#include "player.h"
#include "platforms.h"
#include "physics.h"
#include "sound.h"
#include "tables.h"

// Pushable behavior parameters
#define PUSHABLE_WEIGHT 0x50     // Speed at which the pushables fall
#define PUSHABLE_SPEED 0x80      // Speed at which it can be pushed

// Platform behavior parameters
#define PLATFORM_SWING 2         // Controls how far the platforms move
#define PLATFORM_BREAK 90        // How long before a platform can break
#define PLATFORM_RESPAWN 300     // How long before said platform respawns

//***************************************************************************
// init_pushable
// Initializes a pushable object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_pushable(Object *obj) {
   // Set hitbox
   set_hitbox(obj, -16, 15, -16, 15);
}

//***************************************************************************
// run_pushable
// Logic for pushable objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_pushable(Object *obj) {
   // Too far to bother?
   if (is_too_far(obj))
      return;

   // Keep track of whether we were on something already
   // We need this later to know if we just fell
   int was_onground = obj->on_ground;
   int32_t old_gravity = obj->gravity;

   // Apply momentum
   obj->gravity += PUSHABLE_WEIGHT;
   apply_physics(obj);

   // Kill all horizontal momentum by default
   obj->speed = 0;

   // Did we fall on something?
   if (obj->on_ground && !was_onground && old_gravity >= 0x100)
      play_2d_sfx(SFX_CRUSH, obj->x, obj->y);

   // Set sprite
   set_object_anim(obj, retrieve_level_anim(LV_ANIM_PUSHABLE));
}

//***************************************************************************
// init_platform
// Initializes a generic platform object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_platform(Object *obj) {
   // Set hitbox
   set_hitbox(obj, -32, 31, -16, -16);

   // If the flipping flag is set, start swinging the other way (for
   // moving platforms)
   if (obj->dir) {
      obj->dir = 0;
      obj->timer = 0x80;
   }
}

//***************************************************************************
// run_platform
// Logic for idle platform objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_platform(Object *obj) {
   // Set sprite
   set_object_anim(obj, retrieve_level_anim(LV_ANIM_PLATFORMSTD));
}

//***************************************************************************
// run_horizontal_platform
// Logic for horizontal moving platform objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_horizontal_platform(Object *obj) {
   // Calculate new horizontal position (swing horizontally)
   // We put the difference in our speed variable so the physics engine can
   // handle carrying objects properly (the speed is used to calculate how
   // much we moved from our new position, essentially).
   int new_x = obj->base_x + (sines[(game_anim + obj->timer) & 0xFF]
      >> PLATFORM_SWING);
   obj->speed = (new_x - obj->x) << 8;
   obj->x += speed_to_int(obj->speed);

   // Set sprite
   set_object_anim(obj, retrieve_level_anim(LV_ANIM_PLATFORMSTD));
}

//***************************************************************************
// run_vertical_platform
// Logic for vertical moving platform objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_vertical_platform(Object *obj) {
   // Calculate new vertical position (swing vertically)
   // We put the difference in our gravity variable so the physics engine can
   // handle carrying objects properly (the gravity is used to calculate how
   // much we moved from our new position, essentially).
   int new_y = obj->base_y + (sines[(game_anim + obj->timer) & 0xFF]
      >> PLATFORM_SWING);
   obj->gravity = (new_y - obj->y) << 8;
   obj->y += speed_to_int(obj->gravity);

   // Set sprite
   set_object_anim(obj, retrieve_level_anim(LV_ANIM_PLATFORMSTD));
}

//***************************************************************************
// run_breakable_platform
// Logic for breakable platform objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_breakable_platform(Object *obj) {
   // Still collidable?
   if (obj->timer < PLATFORM_BREAK) {
      set_hitbox(obj, -32, 31, -16, -16);
      set_object_anim(obj, retrieve_level_anim(obj->timer ?
         LV_ANIM_PLATFORMBRK2 : LV_ANIM_PLATFORMBRK));
   }

   // Broke up already?
   else {
      obj->has_hitbox = 0;
      set_object_anim(obj, NULL);
   }

   // If we're about to break, update the timer
   // This also does the respawn once the platform is already broken
   if (obj->timer > 0) {
      obj->timer++;

      // Respawn?
      if (obj->timer == PLATFORM_RESPAWN)
         obj->timer = 0;

      // Just broke up?
      else if (obj->timer == PLATFORM_BREAK) {
         // Information about each particle that gets spawned
         struct {
            int32_t offset_x;       // X distance from platform
            int32_t offset_y;       // Y distance from platform
         } data[] = {
            { -0x18, -0x08 },
            { -0x08, -0x08 },
            {  0x08, -0x08 },
            {  0x18, -0x08 },
            { -0x10,  0x00 },
            {  0x00,  0x00 },
            {  0x10,  0x00 },
         };

         // Spawn all particles
         for (unsigned i = 0; i < 7; i++) {
            Object *other = add_object(OBJ_PLATFORMLEFT,
               obj->x + data[i].offset_x,
               obj->y + data[i].offset_y,
               get_random(1));
            other->gravity = get_random(0xFF);
         }

         // Make noise
         play_2d_sfx(SFX_CRACK, obj->x, obj->y);
      }
   }
}

//***************************************************************************
// init_platform_leftover
// Initializes a platform leftover object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_platform_leftover(Object *obj) {
   // Set animation
   set_object_anim(obj, retrieve_level_anim(LV_ANIM_PLATFORMLEFT));
}

//***************************************************************************
// init_obstruction
// Initializes an obstruction object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_obstruction(Object *obj) {
   // Make the tile occupied by this obstruction solid
   LevelTile *tile = get_tile_by_pixel(obj->x, obj->y);
   tile->force_solid = 1;
   tile->obstruction = 1;
}

//***************************************************************************
// break_obstruction
// Causes an obstruction object to break. Note the object gets deleted, so if
// you need any data from it make sure to retrieve it beforehand.
//---------------------------------------------------------------------------
// param tile: pointer to tile data
// param x: horizontal position of tile (in pixels)
// param y: vertical position of tile (in pixels)
//***************************************************************************

void break_obstruction(LevelTile *tile, int32_t x, int32_t y) {
   // Retrieve center of the obstruction
   x &= ~TILE_SIZE_MASK;
   x += TILE_SIZE/2;
   y &= ~TILE_SIZE_MASK;
   y += TILE_SIZE/2;

   // Tile not solid anymore
   tile->force_solid = 0;
   tile->obstruction = 0;

   // Play sound effect
   play_2d_sfx(SFX_BREAK, x, y);

   // Create leftovers
   struct {
      int32_t speed_x;
      int32_t speed_y;
   } leftover_data[] = {
      { 0x080, -0x300 }, { -0x080, -0x300 },
      { 0x100, -0x380 }, { -0x100, -0x380 },
      { 0x180, -0x200 }, { -0x180, -0x200 },
      { 0x200, -0x280 }, { -0x200, -0x280 },
      { 0x000, -0x400 }
   };

   for (unsigned i = 0; i < 9; i++) {
      Object *ptr = add_object(OBJ_OBSTRUCTLEFT, x, y, 0);
      ptr->speed = leftover_data[i].speed_x
         - 0x80 + get_random(0x100);
      ptr->gravity = leftover_data[i].speed_y
         - 0x80 + get_random(0x100);
      ptr->dir = ptr->speed < 0 ? 1 : 0;
   }
}

//***************************************************************************
// init_obstruction_leftover
// Initializes an obstruction leftover object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_obstruction_leftover(Object *obj) {
   // Set animation
   set_object_anim(obj, retrieve_level_anim(LV_ANIM_OBSTRUCTLEFT));
}

//***************************************************************************
// init_door
// Initializes a door object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_door(Object *obj) {
   // Set hitbox
   set_hitbox(obj, -8, 7, -48, 15);

   // If the direction flag is set, instead of flipping the sprite set the
   // door so it goes downwards instead of upwards
   if (obj->dir) {
      obj->dir = 0;
      obj->crouching = 1;
   }
}

//***************************************************************************
// run_door
// Logic for door objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_door(Object *obj) {
   // Reset gravity so platform physics don't spazz out
   obj->gravity = 0;

   // Should be opened?
   if (get_map_switch(obj->type - OBJ_DOOR_1)) {
      // Still not fully opened?
      if (obj->timer < 0x20) {
         // Make door go upwards
         obj->gravity = obj->crouching ? 0x200 : -0x200;
         obj->y += speed_to_int(obj->gravity);
         refresh_hitbox(obj);

         // Adjust timer
         obj->timer++;
      }
   }

   // Set animation
   set_object_anim(obj, retrieve_level_anim(LV_ANIM_DOOR));
}

//***************************************************************************
// init_generator
// Initializes a generator object
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_generator(Object *obj) {
   // Set animation
   set_object_anim(obj, retrieve_object_anim(obj->type == OBJ_GENERATOR ?
      OB_ANIM_GENERATOR : OB_ANIM_MAINGENERATOR));

   // Set hitbox
   set_hitbox(obj, -48, 47, -48, 15);

   // Make tiles solid
   LevelTile *tile;
   tile = get_tile_by_pixel(obj->x, obj->y);
   tile->force_solid = 1;
   tile = get_tile_by_pixel(obj->x - TILE_SIZE, obj->y);
   tile->force_solid = 1;
   tile = get_tile_by_pixel(obj->x + TILE_SIZE, obj->y);
   tile->force_solid = 1;
   tile = get_tile_by_pixel(obj->x, obj->y - TILE_SIZE);
   tile->force_solid = 1;
   tile = get_tile_by_pixel(obj->x - TILE_SIZE, obj->y - TILE_SIZE);
   tile->force_solid = 1;
   tile = get_tile_by_pixel(obj->x + TILE_SIZE, obj->y - TILE_SIZE);
   tile->force_solid = 1;
}

//***************************************************************************
// run_generator
// Logic for generator objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_generator(Object *obj) {
   // Still working?
   if (!obj->dead)
      return;

   // Explode!
   switch (obj->timer) {
      case 0x00:
         // Vanish
         set_object_anim(obj, NULL);
         {
            LevelTile *tile;
            tile = get_tile_by_pixel(obj->x, obj->y);
            tile->force_solid = 0;
            tile = get_tile_by_pixel(obj->x - TILE_SIZE, obj->y);
            tile->force_solid = 0;
            tile = get_tile_by_pixel(obj->x + TILE_SIZE, obj->y);
            tile->force_solid = 0;
            tile = get_tile_by_pixel(obj->x, obj->y - TILE_SIZE);
            tile->force_solid = 0;
            tile = get_tile_by_pixel(obj->x - TILE_SIZE, obj->y - TILE_SIZE);
            tile->force_solid = 0;
            tile = get_tile_by_pixel(obj->x + TILE_SIZE, obj->y - TILE_SIZE);
            tile->force_solid = 0;
         };

         // First explosion
         obj->y -= 0x10;
         add_object(OBJ_BIGEXPLOSION, obj->x, obj->y, 0);
         apply_quake(0x20);
         break;

      // Second set
      case 0x10:
         apply_quake(0x20);
         add_object(OBJ_BIGEXPLOSION, obj->x - 0x10, obj->y - 0x10, 0);
         add_object(OBJ_BIGEXPLOSION, obj->x + 0x10, obj->y - 0x10, 0);
         add_object(OBJ_BIGEXPLOSION, obj->x - 0x10, obj->y + 0x10, 0);
         add_object(OBJ_BIGEXPLOSION, obj->x + 0x10, obj->y + 0x10, 0);
         break;

      // Third set
      case 0x20:
         apply_quake(0x20);
         add_object(OBJ_BIGEXPLOSION, obj->x - 0x20, obj->y, 0);
         add_object(OBJ_BIGEXPLOSION, obj->x + 0x20, obj->y, 0);
         break;

      // Fourth set
      case 0x30:
         apply_quake(0x20);
         add_object(OBJ_BIGEXPLOSION, obj->x - 0x30, obj->y - 0x10, 0);
         add_object(OBJ_BIGEXPLOSION, obj->x + 0x30, obj->y - 0x10, 0);
         add_object(OBJ_BIGEXPLOSION, obj->x - 0x30, obj->y + 0x10, 0);
         add_object(OBJ_BIGEXPLOSION, obj->x + 0x30, obj->y + 0x10, 0);
         delete_object(obj);
         return;
   }

   // Advance timer
   obj->timer++;
}
