//***************************************************************************
// "physics.c"
// Physics for objects. Map collision is handled here, as well as some
// physics interaction between objects.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include <stdint.h>
#include "enemies.h"
#include "ingame.h"
#include "level.h"
#include "objects.h"
#include "player.h"
#include "physics.h"

// Private function prototypes
static int map_collision_down(Object *);
static int map_collision_up(Object *);
static int map_collision_right(Object *);
static int map_collision_left(Object *);
static int is_collision_down(uint8_t, int, int);
static int is_collision_up(uint8_t, int, int);
static int is_collision_right(uint8_t, int, int);
static int is_collision_left(uint8_t, int, int);
static void response_down(Object *, int32_t);

// We have to do this so many times it isn't funny anymore
// Turning this into a macro for legibility purposes
// NOTE: this used to call another function which was more verbose, thereby
// why this macro was here originally... but even then it's still quite a
// long name so the macro sticks around.
#define TILE(x,y) (get_map_collision((x), (y)))

// Quick inline functions to determine if a given collision type is a slope
// or not. If we add new slope types then we'll have less places to modify.
static inline int is_slope(uint8_t type) {
   return (type >= TILE_NWSE_1 && type <= TILE_NESW_2) ||
          (type >= TILE_NWSE_1_BG && type <= TILE_NESW_2_BG);
}
static inline int is_nwse(uint8_t type) {
   return type == TILE_NWSE_1 || type == TILE_NWSE_1_BG ||
          type == TILE_NWSE_2 || type == TILE_NWSE_2_BG;
}
static inline int is_nesw(uint8_t type) {
   return type == TILE_NESW_1 || type == TILE_NESW_1_BG ||
          type == TILE_NESW_2 || type == TILE_NESW_2_BG;
}

// Pinball mode stuff
static void play_pinball(const Object *obj, int condition) {
   if ((obj->group == OBJGROUP_PLAYER || obj->group == OBJGROUP_ENEMY)
   && condition)
      play_sfx(SFX_PINBALL);
}

//***************************************************************************
// apply_physics
// Applies the game physics onto an object. This includes updating the
// object's position based on its current momentum and ensuring proper
// collision response against the map and platform-like objects.
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void apply_physics(Object *obj) {
   // If the object has no hitbox, then it can't collide with the map, so
   // just apply momentum to it and behave like there's no collision at all.
   if (!obj->has_hitbox) {
      obj->on_ground = 0;
      obj->on_wall = 0;
      obj->on_ceiling = 0;
      obj->blocked = 0;
      obj->x += speed_to_int(obj->speed);
      obj->y += speed_to_int(obj->gravity);
      return;
   }

   // Check if the object is stepping on a conveyor belt
   int belt_left = 0;
   int belt_right = 0;
   if (obj->on_ground) {
      // Check for belt to the left
      if (get_tile_by_pixel(obj->abs_hitbox.x1, obj->abs_hitbox.y2 + 1)
      ->collision == TILE_BELT_LEFT ||
      get_tile_by_pixel(obj->abs_hitbox.x2, obj->abs_hitbox.y2 + 1)
      ->collision == TILE_BELT_LEFT)
         belt_left = 1;

      // Check for belt to the right
      if (get_tile_by_pixel(obj->abs_hitbox.x1, obj->abs_hitbox.y2 + 1)
      ->collision == TILE_BELT_RIGHT ||
      get_tile_by_pixel(obj->abs_hitbox.x2, obj->abs_hitbox.y2 + 1)
      ->collision == TILE_BELT_RIGHT)
         belt_right = 1;
   }

   // We need to keep this in case we get crushed
   int was_onground = obj->on_ground;

   // Reset collision flags
   obj->on_ground = 0;
   obj->on_cliff = 0;
   obj->on_cliff_l = 0;
   obj->on_cliff_r = 0;
   obj->on_wall = 0;
   obj->on_ceiling = 0;
   obj->blocked = 0;

   // Determine how many pixels to move around
   int dist_x = speed_to_int(obj->speed);
   int dist_y = speed_to_int(obj->gravity);

   // Apply conveyor belt physics if relevant
   if (belt_left)
      dist_x -= speed_to_int(BELT_SPEED);
   if (belt_right)
      dist_x += speed_to_int(BELT_SPEED);

   // Check collision against platforms
   // This lets objects to stand on top of them
   {
      // By default assume the object is on a cliff
      // This is unintuitive, but we have to do it this way to be able to
      // handle when the object steps on multiple platforms at the same time
      // (e.g. between the borders of two platforms). We'll reset them if we
      // don't find any collision with platforms.
      obj->on_cliff_l = 1;
      obj->on_cliff_r = 1;

      // The hitbox coordinates used for checking collisions against the
      // top of the platforms. Done mainly for readability purposes, though
      // it may help with optimization too.
      int x1 = obj->x + obj->rel_hitbox.x1;
      int x2 = obj->x + obj->rel_hitbox.x2;
      int y  = obj->y + obj->rel_hitbox.y2 + 1;

      // Where we store how much the object should move if a platform carries
      // it horizontally. We use an accumulator because if we pushed the
      // object directly then it'd get moved twice if it turned out to be
      // standing on top of two platforms at the same time.
      int32_t push_x = 0;

      // Check against all platforms
      for (Object *other = get_first_object(OBJGROUP_PLATFORM);
      other != NULL; other = other->next) {
         // Don't check against ourselves!
         if (obj == other)
            continue;

         // Make sure it even has collision
         // (breakable platforms work by not having collision when they
         // break, then eventually "respawning" when their timer expires by
         // restoring their collision)
         if (!other->has_hitbox)
            continue;

         // Calculate how much the platform moved
         // We need this to determine its previous position
         // Putting this into variables both to increase legibility and so
         // it doesn't get calculated multitple times for no reason
         int x_offset = speed_to_int(other->speed);
         int y_offset = speed_to_int(other->gravity);

         // Is this platform tangible only from the top?
         if (other->rel_hitbox.y1 == other->rel_hitbox.y2) {
            // Only collide when the object is falling
            if (dist_y < 0)
               continue;

            // Can we step on this object?
            if (other->abs_hitbox.x2 - x_offset >= x1 &&
                other->abs_hitbox.x1 - x_offset <= x2 &&
                other->abs_hitbox.y1 - y_offset >= y &&
                other->abs_hitbox.y1 - y_offset <= y + dist_y)
            {
               // Set the maximum distance to travel down
               // If we don't hit the map earlier, this means we're stepping
               // on the platform and thereby we should stop on top of it.
               dist_y = other->abs_hitbox.y1 - y;

               // Mark the object as being on the ground already - if it
               // doesn't collide with the map then this will set the object
               // as stepping on the platform.
               response_down(obj, dist_y);

               // Check if we have our feet on the platform instead of just
               // hanging around. We check for this and not the opposite so
               // we can handle when the object is stepping on two platforms
               // at the same time.
               if (obj->abs_hitbox.x1 >= other->abs_hitbox.x1)
                  obj->on_cliff_l = 0;
               if (obj->abs_hitbox.x2 <= other->abs_hitbox.x2)
                  obj->on_cliff_r = 0;

               // Store the momentum of this platform
               // To-do: find out why this doesn't work well with pushables
               // <Sik> Let's just give up on that for now? ._.'
               // To-do: check if the above to-do is still relevant...
               if (other->speed < 0 && push_x > x_offset)
                  push_x = x_offset;
               else if (other->speed > 0 && push_x < x_offset)
                  push_x = x_offset;

               // Is the platform breakable? If so, make sure it breaks
               if (other->type == OBJ_PLATFORM_B && other->timer == 0 &&
               obj->group != OBJGROUP_SONAR)
                  other->timer = 1;
            }
         }

         // Is this a solid platform?
         else {
            // Check if the two objects collide
            // The hitbox offsetting is to allow objects to safely step on
            // platforms without wobbling or anything like that (similar to
            // how top-only platforms are handled)
            int32_t coll_offset = y_offset > 0 ? y_offset+1 : 1;
            other->abs_hitbox.y1 -= coll_offset;
            if (!collision(obj, other)) {
               other->abs_hitbox.y1 += coll_offset;
               continue;
            }
            other->abs_hitbox.y1 += coll_offset;

            // Quick hack to prevent the player from getting stuck in case
            // of jumping right at the border of a platform
            if (dist_y < 0 && obj->y < other->abs_hitbox.y1)
               continue;

            // At the top of the platform?
            if (obj->abs_hitbox.y2 < other->abs_hitbox.y1 +
            coll_offset + dist_y) {
               // Was moving upwards already?
               if (dist_y < 0)
                  continue;

               // Make object stop moving
               response_down(obj, dist_y);

               // Push object away from the platform
               /*dist_y = (other->abs_hitbox.y1 - obj->rel_hitbox.y2 - 1)
                  - obj->y;*/
               obj->y = other->abs_hitbox.y1 - obj->rel_hitbox.y2 - 1;
               dist_y = 0;

               // Check if we have our feet on the platform instead of just
               // hanging around. We check for this and not the opposite so
               // we can handle when the object is stepping on two platforms
               // at the same time.
               if (obj->abs_hitbox.x1 >= other->abs_hitbox.x1)
                  obj->on_cliff_l = 0;
               if (obj->abs_hitbox.x2 <= other->abs_hitbox.x2)
                  obj->on_cliff_r = 0;

               // Store the momentum of this platform
               // To-do: find out why this doesn't work well with pushables
               // <Sik> Let's just give up on that for now? ._.'
               if (other->speed < 0 && push_x > x_offset)
                  push_x = x_offset;
               else if (other->speed > 0 && push_x < x_offset)
                  push_x = x_offset;
            }

            // At the left side of the platform?
            else if (obj->x < other->abs_hitbox.x1) {
               // Make object stop moving
               obj->speed = 0;
               obj->on_wall = 1;
               obj->blocked = 1;

               // Push object away from the platform
               dist_x = (other->abs_hitbox.x1 - obj->rel_hitbox.x2 - 1)
                  - obj->x;

               // If the platform is a pushable then push it
               if (other->type == OBJ_PUSHABLE &&
               obj->group != OBJGROUP_SONAR) {
                  other->speed = 0x100;
                  obj->speed = 0x40;
               }
            }

            // At the right side of the platform?
            else if (obj->x > other->abs_hitbox.x2) {
               // Make object stop moving
               obj->speed = 0;
               obj->on_wall = 1;
               obj->blocked = 1;

               // Push object away from the platform
               dist_x = (other->abs_hitbox.x2 - obj->rel_hitbox.x1 + 1)
                  - obj->x;

               // If the platform is a pushable then push it
               if (other->type == OBJ_PUSHABLE &&
               obj->group != OBJGROUP_SONAR) {
                  other->speed = -0x100;
                  obj->speed = -0x40;
               }
            }

            // At the bottom of the platform?
            else if (obj->y >= other->abs_hitbox.y2 /*&&
            y_offset - dist_y >= 0*/) {
               // Got crushed?
               if (was_onground || obj->on_ground) {
                  // If the object can be destroyed, do it
                  if (obj->group == OBJGROUP_PLAYER)
                     kill_player(obj);
                  else if (obj->group == OBJGROUP_ENEMY)
                     destroy_enemy(obj);

                  // Don't bother trying to mess with the object's position
                  // If it didn't get crushed then it'll just stay where it
                  // is (as if it couldn't move).
                  return;
               }

               // Was moving downwards already?
               if (y_offset - dist_y < 0)
                  continue;

               // Make object stop moving
               obj->gravity = 0;
               obj->blocked = 1;
               obj->on_ceiling = 1;

               // Push object away from the platform
               /*dist_y = (other->abs_hitbox.y2 - obj->rel_hitbox.y1 + 1)
                  - obj->y;*/
               obj->y = other->abs_hitbox.y2 - obj->rel_hitbox.y1 + 1;
               dist_y = 0;
            }
         }
      }

      // If we didn't step on a platform then restore reset the cliff flags
      // since we only care about the map then
      if (!obj->on_ground) {
         obj->on_cliff_l = 0;
         obj->on_cliff_r = 0;
      }

      // Apply the platform momentum
      // To-do: find out why this doesn't work properly with pushables
      // <Sik> Let's just give up on that for now? ._.'
      dist_x += push_x;

      if (obj->type == OBJ_PUSHABLE)
         obj->speed += push_x << 8;
   }

   // Enemies may be blocked by pushables
   // Doing this here because trying to get this to work in the pushable
   // object code seems to cause a ridiculous amount of bugs...
   if (obj->group == OBJGROUP_ENEMY) {
      // Check against all platforms
      for (Object *other = get_first_object(OBJGROUP_PLATFORM);
      other != NULL; other = other->next) {
         // Skip non-pushables
         if (other->type != OBJ_PUSHABLE)
            continue;

         // Contact?
         if (!collision(obj, other))
            continue;

         // Collision from the left?
         if (obj->x + dist_x < other->x) {
            int temp = obj->abs_hitbox.x2 -
                       other->abs_hitbox.x1 - 1;
            if (dist_x > temp) {
               if (dist_x > 0)
                  obj->on_wall = 1;
               dist_x = temp;
               obj->speed = 0;
            }
         }

         // Collision from the right?
         else {
            int temp = other->abs_hitbox.x2 -
                       obj->abs_hitbox.x1 + 1;
            if (dist_x < temp) {
               if (dist_x < 0)
                  obj->on_wall = 1;
               dist_x = temp;
               obj->speed = 0;
            }
         }
      }
   }

   // Perform collision response against the map
   while (dist_x || dist_y) {
      // Move down?
      if (dist_y > 0) {
         if (map_collision_down(obj)) {
            response_down(obj, dist_y);
            dist_y = 0;
         } else {
            obj->y++;
            dist_y--;
         }
      }

      // Move up?
      else if (dist_y < 0) {
         if (map_collision_up(obj)) {
            if (!settings.pinball) {
               obj->gravity = 0;
            } else {
               play_pinball(obj, 1);
               obj->gravity = -obj->gravity;
               obj->gravity += 0x100;
               obj->pinball = 1;
            }
            obj->on_ceiling = 1;
            obj->blocked = 1;
            dist_y = 0;
         } else {
            obj->y--;
            dist_y++;
         }
      }

      // Move right?
      if (dist_x > 0) {
         if (map_collision_right(obj)) {
            if (obj->speed > 0) {
               if (!settings.pinball) {
                  obj->speed = 0;
               } else {
                  play_pinball(obj, 1);
                  obj->speed = -obj->speed;
                  obj->speed -= 0x100;
                  obj->gravity -= 0x100;
                  obj->pinball = 1;
               }
            }
            obj->on_wall = 1;
            obj->blocked = 1;
            dist_x = 0;
         } else {
            obj->x++;
            dist_x--;
         }
      }

      // Move left?
      else if (dist_x < 0) {
         if (map_collision_left(obj)) {
            if (obj->speed < 0) {
               if (!settings.pinball) {
                  obj->speed = 0;
               } else {
                  play_pinball(obj, 1);
                  obj->speed = -obj->speed;
                  obj->speed += 0x100;
                  obj->gravity -= 0x100;
                  obj->pinball = 1;
               }
            }
            obj->on_wall = 1;
            obj->blocked = 1;
            dist_x = 0;
         } else {
            obj->x--;
            dist_x++;
         }
      }
   }

   // When we're stepping on something, we should set the relevant collision
   // flags to tell the object this.
   if (obj->gravity >= 0 && map_collision_down(obj)) {
      obj->on_ground = 1;

      // Check for cliffs
      // Don't bother doing this with all objects though, since it could be
      // a waste of time (seriously, look how complex this code is already!).
      // Only bother with the kind of objects that need it.
      if (obj->group == OBJGROUP_PLAYER || obj->group == OBJGROUP_ENEMY ||
      obj->group == OBJGROUP_SONAR) {
         // Coordinates we have to look for
         int x1 = obj->x + obj->rel_hitbox.x1;
         int x2 = obj->x + obj->rel_hitbox.x2;
         int y = obj->y + obj->rel_hitbox.y2 + 1;

         // Retrieve tiles at those positions
         // Account for slopes (they need to be handled differently since
         // they don't follow the hitbox)
         if (is_slope(TILE(obj->x, y-1))) {
            x1 = obj->x;
            x2 = obj->x;
         }
         uint8_t type1 = TILE(x1, y);
         if (is_slope(type1)) {
            x1 = obj->x;
            type1 = TILE(x1, y);
         }
         uint8_t type2 = TILE(x2, y);
         if (is_slope(type2)) {
            x2 = obj->x;
            type2 = TILE(x2, y);
         }

         // Check if there's collision on both points. If either one fails,
         // then it means we're on a cliff.
         if (!is_collision_down(type1, x1, y))
            obj->on_cliff_l = 1;
         if (!is_collision_down(type2, x2, y))
            obj->on_cliff_r = 1;
         if (obj->on_cliff_l || obj->on_cliff_r)
            obj->on_cliff = 1;
      }
   }

   // When we're pushing against a wall, we should set the relevant collision
   // flag to tell the object this.
   if (obj->speed > 0 && map_collision_right(obj)) {
      obj->on_wall = 1;
      if (!settings.pinball)
         obj->speed = 0;
   }
   else if (obj->speed < 0 && map_collision_left(obj)) {
      obj->on_wall = 1;
      if (!settings.pinball)
         obj->speed = 0;
   }

   // Set the blocked flag if the object got blocked in any way. If the
   // object moved absolutely freely then leave it cleared instead.
   if (obj->on_ground || obj->on_wall)
      obj->blocked = 1;

   // Update hitbox to reflect the new position
   refresh_hitbox(obj);
}

//***************************************************************************
// map_collision_down [internal]
// Checks if the object would collide with the map when going down.
//---------------------------------------------------------------------------
// param obj: pointer to object
// return: non-zero if collides, zero otherwise
//***************************************************************************

static int map_collision_down(Object *obj) {
   // Y coordinate to look for
   int y = obj->y + obj->rel_hitbox.y2 + 1;

   // Determine collision with the tile immediately below us
   uint8_t type = TILE(obj->x, y);
   if (is_collision_down(type, obj->x, y)) return 1;

   // If the object is on a slope, stop here. Otherwise, the following checks
   // will interfere with the slope physics due to surrounding tiles.
   type = TILE(obj->x, y + 1);
   if (is_slope(type))
      return 0;

   // Determine collision with the tiles to the left of us, if any
   int min = obj->x + obj->rel_hitbox.x1;
   if ((min & ~TILE_SIZE_MASK) != (obj->x & ~TILE_SIZE_MASK)) {
      for (int xr = obj->x - TILE_SIZE; xr > min; xr -= TILE_SIZE) {
         type = TILE(xr, y);
         if (is_slope(type)) return 0;
         if (is_collision_down(type, xr, y)) return 1;
      }
      type = TILE(min, y);
      if (is_slope(type)) return 0;
      if (is_collision_down(type, min, y)) return 1;
   }

   // Determine collision with the tiles to the right of us, if any
   int max = obj->x + obj->rel_hitbox.x2;
   if ((max & ~TILE_SIZE_MASK) != (obj->x & ~TILE_SIZE_MASK)) {
      for (int xr = obj->x + TILE_SIZE; xr < max; xr += TILE_SIZE) {
         type = TILE(xr, y);
         if (is_slope(type)) return 0;
         if (is_collision_down(type, xr, y)) return 1;
      }
      type = TILE(max, y);
      if (is_slope(type)) return 0;
      if (is_collision_down(type, max, y)) return 1;
   }

   // No collision
   return 0;
}

//***************************************************************************
// map_collision_up [internal]
// Checks if the object would collide with the map when going up.
//---------------------------------------------------------------------------
// param obj: pointer to object
// return: non-zero if collides, zero otherwise
//***************************************************************************

static int map_collision_up(Object *obj) {
   // Y coordinate to look for
   int y = obj->y + obj->rel_hitbox.y1 - 1;

   // Determine collision with the tile immediately above us
   uint8_t type = TILE(obj->x, y);
   if (is_collision_up(type, obj->x, y)) return 1;

   // Determine collision with the tiles to the left of us, if any
   int min = obj->x + obj->rel_hitbox.x1;
   if ((min & ~TILE_SIZE_MASK) != (obj->x & ~TILE_SIZE_MASK)) {
      for (int xr = obj->x - TILE_SIZE; xr > min; xr -= TILE_SIZE) {
         type = TILE(xr, y);
         if (is_collision_up(type, xr, y)) return 1;
      }
      type = TILE(min, y);
      if (is_collision_up(type, min, y)) return 1;
   }

   // Determine collision with the tiles to the right of us, if any
   int max = obj->x + obj->rel_hitbox.x2;
   if ((max & ~TILE_SIZE_MASK) != (obj->x & ~TILE_SIZE_MASK)) {
      for (int xr = obj->x + TILE_SIZE; xr < max; xr += TILE_SIZE) {
         type = TILE(xr, y);
         if (is_collision_up(type, xr, y)) return 1;
      }
      type = TILE(max, y);
      if (is_collision_up(type, max, y)) return 1;
   }

   // No collision
   return 0;
}

//***************************************************************************
// map_collision_right [internal]
// Checks if the object would collide with the map when going right.
//---------------------------------------------------------------------------
// param obj: pointer to object
// return: non-zero if collides, zero otherwise
//***************************************************************************

static int map_collision_right(Object *obj) {
   // X coordinate to look for
   int x = obj->x + obj->rel_hitbox.x2 + 1;

   // Determine collision with the tile immediately next to us
   uint8_t type = TILE(x, obj->y);
   if (is_collision_right(type, x, obj->y)) return 1;

   // Determine collision with the tiles above us, if any
   int min = obj->y + obj->rel_hitbox.y1;
   if ((min & ~TILE_SIZE_MASK) != (obj->y & ~TILE_SIZE_MASK)) {
      for (int yr = obj->y - TILE_SIZE; yr > min; yr -= TILE_SIZE) {
         type = TILE(x, yr);
         if (is_collision_right(type, x, yr)) return 1;
      }
      type = TILE(x, min);
      if (is_collision_right(type, x, min)) return 1;
   }

   // If the object is on an uphill slope, break here, otherwise the object
   // may get stuck against solid tiles next to the slope tiles. Yes, this is
   // a hack. Make the object run nicely over the slope while we're at it
   // (ensure it goes up as it moves).
   int y = obj->y + obj->rel_hitbox.y2;
   type = TILE(obj->x, y);
   if (is_nesw(type)) {
      if (is_collision_down(type, obj->x, y)) {
         int y2 = obj->y + obj->rel_hitbox.y1 - 1;
         if (is_collision_up(TILE(obj->x, y2), obj->x, y2))
            return 1;
         else
            obj->y--;
      }
      return 0;
   }

   // If the object is on a downhill slope, again, break here and fix its
   // position accordingly.
   type = TILE(obj->x, y + 1);
   if (is_nwse(type)) {
      if (is_collision_down(type, obj->x, y + 1) &&
      !is_collision_down(TILE(obj->x + 1, y + 1), obj->x + 1, y + 1))
         obj->y++;
      return 0;
   }

   // Determine collision with the tiles below us, if any
   int max = obj->y + obj->rel_hitbox.y2;
   if ((max & ~TILE_SIZE_MASK) != (obj->y & ~TILE_SIZE_MASK)) {
      for (int yr = obj->y + TILE_SIZE; yr < max; yr += TILE_SIZE) {
         type = TILE(x, yr);
         if (is_collision_right(type, x, yr)) return 1;
      }
      type = TILE(x, max);
      if (is_collision_right(type, x, max)) return 1;
   }

   // No collision
   return 0;
}

//***************************************************************************
// map_collision_left [internal]
// Checks if the object would collide with the map when going left.
//---------------------------------------------------------------------------
// param obj: pointer to object
// return: non-zero if collides, zero otherwise
//***************************************************************************

static int map_collision_left(Object *obj) {
   // X coordinate to look for
   int x = obj->x + obj->rel_hitbox.x1 - 1;

   // Determine collision with the tile immediately next to us
   uint8_t type = TILE(x, obj->y);
   if (is_collision_left(type, x, obj->y)) return 1;

   // Determine collision with the tiles above us, if any
   int min = obj->y + obj->rel_hitbox.y1;
   if ((min & ~TILE_SIZE_MASK) != (obj->y & ~TILE_SIZE_MASK)) {
      for (int yr = obj->y - TILE_SIZE; yr > min; yr -= TILE_SIZE) {
         type = TILE(x, yr);
         if (is_collision_left(type, x, yr)) return 1;
      }
      type = TILE(x, min);
      if (is_collision_left(type, x, min)) return 1;
   }

   // If the object is on an uphill slope, break here, otherwise the object
   // may get stuck against solid tiles next to the slope tiles. Yes, this is
   // a hack. Make the object run nicely over the slope while we're at it
   // (ensure it goes up as it moves).
   int y = obj->y + obj->rel_hitbox.y2;
   type = TILE(obj->x, y);
   if (is_nwse(type)) {
      if (is_collision_down(type, obj->x, y)) {
         int y2 = obj->y + obj->rel_hitbox.y1 - 1;
         if (is_collision_up(TILE(obj->x, y2), obj->x, y2))
            return 1;
         else
            obj->y--;
      }
      return 0;
   }

   // If the object is on a downhill slope, again, break here and fix its
   // position accordingly.
   type = TILE(obj->x, y + 1);
   if (is_nesw(type)) {
      if (is_collision_down(type, obj->x, y + 1) &&
      !is_collision_down(TILE(obj->x - 1, y + 1), obj->x - 1, y + 1))
         obj->y++;
      return 0;
   }

   // Determine collision with the tiles below us, if any
   int max = obj->y + obj->rel_hitbox.y2;
   if ((max & ~TILE_SIZE_MASK) != (obj->y & ~TILE_SIZE_MASK)) {
      for (int yr = obj->y + TILE_SIZE; yr < max; yr += TILE_SIZE) {
         type = TILE(x, yr);
         if (is_collision_left(type, x, yr)) return 1;
      }
      type = TILE(x, max);
      if (is_collision_left(type, x, max)) return 1;
   }

   // No collision
   return 0;
}

//***************************************************************************
// is_collision_down [internal]
// Checks if when going down, trying to get to the specified coordinates
// results in a collision.
//---------------------------------------------------------------------------
// param type: collision type of tile
// param x: target X coordinate
// param y: target Y coordinate
// return: non-zero if collides, zero otherwise
//***************************************************************************

static int is_collision_down(uint8_t type, int x, int y) {
   // Put coordinates in tile space
   x &= TILE_SIZE_MASK;
   y &= TILE_SIZE_MASK;

   // Determine what kind of collision is it
   switch (type) {
      // Thin floor
      case TILE_FLOOR_BG:
      case TILE_BRIDGE:
         return y == 0;

      // Slopes
      case TILE_NWSE_1: case TILE_NWSE_1_BG:
         return x / 2 <= y;
      case TILE_NWSE_2: case TILE_NWSE_2_BG:
         return x / 2 <= y - TILE_SIZE/2;
      case TILE_NESW_1: case TILE_NESW_1_BG:
         return ((TILE_SIZE-1 - x) >> 1) <= y;
      case TILE_NESW_2: case TILE_NESW_2_BG:
         return ((TILE_SIZE-1 - x) >> 1) <= y - TILE_SIZE/2;

      // Simple collision types
      case TILE_SOLID:
      case TILE_BELT_LEFT:
      case TILE_BELT_RIGHT:
         return 1;
      default:
         return 0;
   }
}

//***************************************************************************
// is_collision_up [internal]
// Checks if when going up, trying to get to the specified coordinates
// results in a collision.
//---------------------------------------------------------------------------
// param type: collision type of tile
// param x: target X coordinate
// param y: target Y coordinate
// return: non-zero if collides, zero otherwise
//***************************************************************************

static int is_collision_up(uint8_t type, int x, int y) {
   // Put coordinates in tile space
   x &= TILE_SIZE_MASK;
   y &= TILE_SIZE_MASK;

   // Determine what kind of collision is it
   switch (type) {
      // Slopes
      case TILE_NWSE_1: case TILE_NWSE_1_BG:
      case TILE_NWSE_2: case TILE_NWSE_2_BG:
      case TILE_NESW_1: case TILE_NESW_1_BG:
      case TILE_NESW_2: case TILE_NESW_2_BG:
         return y == TILE_SIZE-1;

      // Simple collision types
      case TILE_SOLID:
      case TILE_BELT_LEFT:
      case TILE_BELT_RIGHT:
         return 1;
      default:
         return 0;
   }
}

//***************************************************************************
// is_collision_right [internal]
// Checks if when going right, trying to get to the specified coordinates
// results in a collision.
//---------------------------------------------------------------------------
// param type: collision type of tile
// param x: target X coordinate
// param y: target Y coordinate
// return: non-zero if collides, zero otherwise
//***************************************************************************

static int is_collision_right(uint8_t type, int x, int y) {
   // Put coordinates in tile space
   x &= TILE_SIZE_MASK;
   y &= TILE_SIZE_MASK;

   // Determine what kind of collision is it
   switch (type) {
      // Slope sides
      // We gotta be careful with these or we'll break slope physics...
      // Just in case we err on the side of caution (i.e. this may be off by
      // one pixel as non-collision)
      case TILE_NWSE_1: case TILE_NWSE_1_BG:
         return x == 0 && y > 0;
      case TILE_NWSE_2: case TILE_NWSE_2_BG:
         return x == 0 && y > TILE_SIZE/2;
      /*case TILE_NESW_1: case TILE_NESW_1_BG:
         return y > TILE_SIZE/2;*/

      // Simple collision types
      case TILE_SOLID:
      case TILE_BELT_LEFT:
      case TILE_BELT_RIGHT:
         return 1;
      default:
         return 0;
   }
}

//***************************************************************************
// is_collision_left [internal]
// Checks if when going left, trying to get to the specified coordinates
// results in a collision.
//---------------------------------------------------------------------------
// param type: collision type of tile
// param x: target X coordinate
// param y: target Y coordinate
// return: non-zero if collides, zero otherwise
//***************************************************************************

static int is_collision_left(uint8_t type, int x, int y) {
   // Put coordinates in tile space
   x &= TILE_SIZE_MASK;
   y &= TILE_SIZE_MASK;

   // Determine what kind of collision is it
   switch (type) {
      // Slope sides
      // We gotta be careful with these or we'll break slope physics...
      // Just in case we err on the side of caution (i.e. this may be off by
      // one pixel as non-collision)
      case TILE_NESW_1: case TILE_NESW_1_BG:
         return x == TILE_SIZE-1 && y > 0;
      case TILE_NESW_2: case TILE_NESW_2_BG:
         return x == TILE_SIZE-1 && y > TILE_SIZE/2;
      /*case TILE_NWSE_1: case TILE_NWSE_1_BG:
         return y > TILE_SIZE/2;*/

      // Simple collision types
      case TILE_SOLID:
      case TILE_BELT_LEFT:
      case TILE_BELT_RIGHT:
         return 1;
      default:
         return 0;
   }
}

//***************************************************************************
// response_down [internal]
// Response when the object has hit the ground
//***************************************************************************

static void response_down(Object *obj, int32_t dist) {
   // Normal physics
   if (!settings.pinball) {
      obj->gravity = 0;
      obj->pinball = 0;
   }

   // Bounce the object
   else {
      play_pinball(obj, dist > 1);
      obj->pinball = dist > 1 ? 1 : 0;
      obj->gravity = -obj->gravity;
      obj->gravity += 0x100;
   }

   // Tell the object it hit the floor
   obj->on_ground = 1;
   obj->blocked = 1;
}

//***************************************************************************
// accelerate_object
// Accelerates an object to the left or the right by a given amount of speed.
// A speed cap is in place that will be respected if the object tries to go
// too fast (though it won't kill the speed if it was already too fast before
// calling this function).
//---------------------------------------------------------------------------
// param obj: pointer to object
// param speed: how much to accelerate (positive = right, negative = left)
// param cap: speed cap (should be a positive value)
//***************************************************************************

void accelerate_object(Object *obj, int32_t speed, int32_t cap) {
   // Going right?
   if (speed > 0 && obj->speed < cap) {
      // Increase speed
      obj->speed += speed;

      // Hit the speed cap?
      if (obj->speed > cap)
         obj->speed = cap;
   }

   // Going left?
   else if (speed < 0 && obj->speed > -cap) {
      // Increase speed
      obj->speed += speed;

      // Hit the speed cap?
      if (obj->speed < -cap)
         obj->speed = -cap;
   }
}

//***************************************************************************
// decelerate_object
// Decelerates an object horizontally by the given amount of friction. If the
// object goes too slow (if it was even moving), it's immediately stopped.
//---------------------------------------------------------------------------
// param obj: pointer to object
// param friction: how much to decelerate the object
//***************************************************************************

void decelerate_object(Object *obj, int32_t friction) {
   // Going right?
   if (obj->speed > 0) {
      // Decelerate object
      obj->speed -= friction;

      // Whoops, stopped?
      if (obj->speed < 0)
         obj->speed = 0;
   }

   // Going left?
   else if (obj->speed < 0) {
      // Deccelerate object
      obj->speed += friction;

      // Whoops, stopped?
      if (obj->speed > 0)
         obj->speed = 0;
   }
}

//***************************************************************************
// set_hitbox
// Sets the hitbox of an object. The hitbox boundaries are relative to the
// object coordinates. The world-coordinates hitbox is updated automatically.
//---------------------------------------------------------------------------
// param obj: pointer to object
// param x1: left boundary
// param x2: right boundary
// param y1: top boundary
// param y2: bottom boundary
//***************************************************************************

void set_hitbox(Object *obj, int32_t x1, int32_t x2, int32_t y1, int32_t y2)
{
   // Object now has a hitbox
   obj->has_hitbox = 1;

   // Update the relative hitbox coordinates
   obj->rel_hitbox.x1 = x1;
   obj->rel_hitbox.x2 = x2;
   obj->rel_hitbox.y1 = y1;
   obj->rel_hitbox.y2 = y2;

   // Update the absolute hitbox coordinates
   obj->abs_hitbox.x1 = x1 + obj->x;
   obj->abs_hitbox.x2 = x2 + obj->x;
   obj->abs_hitbox.y1 = y1 + obj->y;
   obj->abs_hitbox.y2 = y2 + obj->y;
}

//***************************************************************************
// refresh_hitbox
// Refreshes the absolute hitbox of an object to match its current position.
// Call this if the object's position changed and you need to perform any
// physics on it.
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void refresh_hitbox(Object *obj) {
   obj->abs_hitbox.x1 = obj->rel_hitbox.x1 + obj->x;
   obj->abs_hitbox.x2 = obj->rel_hitbox.x2 + obj->x;
   obj->abs_hitbox.y1 = obj->rel_hitbox.y1 + obj->y;
   obj->abs_hitbox.y2 = obj->rel_hitbox.y2 + obj->y;
}

//***************************************************************************
// collision
// Checks if two objects are colliding with each other.
//---------------------------------------------------------------------------
// param obj1: pointer to first object
// param obj2: pointer to second object
// return: zero if there isn't a collision, non-zero if there's collision
//***************************************************************************

int collision(const Object *obj1, const Object *obj2) {
   // First of all both objects need to have a hitbox...
   if (!obj1->has_hitbox || !obj2->has_hitbox)
      return 0;

   // Check if the hitboxes overlap, and return the result
   // This is just a standard AABB check :P
   return obj1->abs_hitbox.x1 <= obj2->abs_hitbox.x2 &&
          obj1->abs_hitbox.x2 >= obj2->abs_hitbox.x1 &&
          obj1->abs_hitbox.y1 <= obj2->abs_hitbox.y2 &&
          obj1->abs_hitbox.y2 >= obj2->abs_hitbox.y1;
}

//***************************************************************************
// can_climb
// Checks whether an object can climb onto a wall or not. Used by the player
// with a spider power-up, to tell when it's next to a climbable wall.
//---------------------------------------------------------------------------
// param obj: object to check
// return: non-zero if can climb, zero otherwise
//***************************************************************************

int can_climb(const Object *obj) {
   // Don't let the player climb off-screen!
   if (obj->y < 0)
      return 0;

   // Facing right?
   if (!obj->dir) {
      // Look at the point right next to the object
      uint8_t type = TILE(obj->abs_hitbox.x2+1, obj->y);
      int bottom_half = (obj->y & TILE_SIZE_MASK) >= TILE_SIZE/2;

      // Check if there's a wall we can climb onto here
      if (type == TILE_SOLID ||
      type == TILE_BELT_LEFT || type == TILE_BELT_RIGHT ||
      type == TILE_NWSE_1 || type == TILE_NWSE_1_BG ||
      (bottom_half && type == TILE_NESW_1) ||
      (bottom_half && type == TILE_NESW_1_BG) ||
      (bottom_half && type == TILE_NWSE_2) ||
      (bottom_half && type == TILE_NWSE_2_BG))
         return 1;

      // Nothing to climb onto!
      return 0;
   }

   // Facing left?
   else {
      //Look at the point right next to the object
      uint8_t type = TILE(obj->abs_hitbox.x1-1, obj->y);
      int bottom_half = (obj->y & TILE_SIZE_MASK) >= TILE_SIZE/2;

      // Check if there's a wall we can climb onto here
      if (type == TILE_SOLID ||
      type == TILE_NESW_1 || type == TILE_NESW_1_BG ||
      (bottom_half && type == TILE_NWSE_1) ||
      (bottom_half && type == TILE_NWSE_1_BG) ||
      (bottom_half && type == TILE_NESW_2) ||
      (bottom_half && type == TILE_NESW_2_BG))
         return 1;

      // Nothing to climb onto!
      return 0;
   }

   // If we get here something went wrong...
   // Return "can't climb" just in case
   return 0;
}
