//***************************************************************************
// "parser.c"
// Text file parser. Used to read the several scripts such as the animation
// lists or the level background information.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "main.h"
#include "parser.h"

// Maximum lengths for ID names
#define MAX_ID_LEN 0x1F

//***************************************************************************
// parse_args
// Parses a string looking for arguments and generates an argument list out
// of it. Returns that argument list.
//---------------------------------------------------------------------------
// param str: string to parse
// return: argument list, or NULL if there was a parsing error
//---------------------------------------------------------------------------
// To-do: audit code for issues and such (this code was taken from mdtiler
// and adapted as needed). This code *should* work, but better to be sure.
// To-do: change it to return error details when parsing fails.
//***************************************************************************

Args *parse_args(const char *str) {
   // Initialize structure
   Args *args = (Args *) malloc(sizeof(Args));
   if (args == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   args->count = 0;
   args->list = NULL;

   // Go through entire line
   while (*str) {
      // Skip whitespace
      if (isspace(*str)) {
         str++;
         continue;
      }

      // Comment? End of line?
      if (*str == '#' || *str == '\0')
         break;

      // Unquoted token?
      if (*str != '\"') {
         // Make room for a new token in the list
         char **temp_list = (char **) realloc(args->list,
            sizeof(char *) * (args->count + 1));
         if (temp_list == NULL)
            abort_program(ERR_NOMEMORY, NULL);
         args->list = temp_list;

         // Token starts here
         const char *start = str;

         // Try to find the end of the token
         // Also make sure there aren't any quotes inside...
         do {
            str++;
            if (*str == '\"') {
               free_args(args);
               return NULL;
            }
         } while (!(isspace(*str) || *str == '\0'));

         // Get length of the token
         size_t length = str - start;

         // Allocate memory for this token
         char *buffer = (char *) malloc(length + 1);
         if (buffer == NULL)
            abort_program(ERR_NOMEMORY, NULL);
         memcpy(buffer, start, length);
         buffer[length] = '\0';

         // Put token inside the token list
         args->list[args->count] = buffer;
         args->count++;
      }

      // Quoted token?
      else {
         // Make room for a new token in the list
         char **temp_list = (char **) realloc(args->list,
            sizeof(char *) * (args->count + 1));
         if (temp_list == NULL)
            abort_program(ERR_NOMEMORY, NULL);
         args->list = temp_list;

         // Skip quote
         str++;

         // Mark where the token starts
         // Also we need to count this character by character, since escaped
         // quotes take up two bytes in the string but they will be kept as
         // a single one.
         const char *start = str;
         size_t length = 0;

         // Look for the ending quote
         for (;;) {
            // Uh oh, missing quote...
            if (*str == '\0') {
               free_args(args);
               return NULL;
            }

            // Quote? Check if it's an escaped quote or the end of the token
            else if (*str == '\"') {
               str++;
               if (*str == '\"') {
                  length++;
                  str++;
               } else
                  break;
            }

            // Normal character, count it as usual
            else {
               length++;
               str++;
            }
         }

         // Allocate memory for this token
         char *buffer = (char *) malloc(length + 1);
         if (buffer == NULL)
            abort_program(ERR_NOMEMORY, NULL);

         // Copy token into buffer, taking into account escaped quotes (which
         // will be stored as a single quote)
         char *ptr = buffer;
         for (size_t i = 0; i < length; i++) {
            if (*start == '\"') {
               *ptr++ = '\"';
               start += 2;
            } else
               *ptr++ = *start++;
         }
         *ptr = '\0';

         // Put token inside the token list
         args->list[args->count] = buffer;
         args->count++;
      }
   }

   // Done!
   return args;
}

//***************************************************************************
// add_arg
// Adds an argument to an argument list. Used by the launcher which generates
// an argument list on the fly.
//---------------------------------------------------------------------------
// param args: pointer to argument list
// param str: pointer to string to add
//***************************************************************************

void add_arg(Args *args, const char *str) {
   // We need a copy of the string to store
   char *newstr = (char *) malloc(strlen(str) + 1);
   if (newstr == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   strcpy(newstr, str);

   // Make room for the new string in the arguments list
   args->count++;
   args->list = (char **) realloc(args->list, sizeof(char **) * args->count);
   if (args->list == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Store new string into the list
   args->list[args->count-1] = newstr;
}

//***************************************************************************
// free_args
// Destroys an argument list.
//---------------------------------------------------------------------------
// param args: pointer to argument list (becomes invalid after call)
//***************************************************************************

void free_args(Args *args) {
   // Get rid of the strings
   for (size_t i = 0; i < args->count; i++)
      free(args->list[i]);

   // Get rid of the structure
   if (args->list)
      free(args->list);
   free(args);
}

//***************************************************************************
// is_integer
// Checks if a string contains a signed integer or not.
//---------------------------------------------------------------------------
// param str: string to check
// return: non-zero if signed integer, zero otherwise
//***************************************************************************

int is_integer(const char *str) {
   // Skip minus sign if found
   if (*str == '-')
      str++;

   // Check if it's composed entirely of numbers
   do {
      if (!isdigit(*str)) return 0;
   } while (*(++str));

   // Signed integer!
   return 1;
}

//***************************************************************************
// is_uinteger
// Checks if a string contains an unsigned integer or not.
//---------------------------------------------------------------------------
// param str: string to check
// return: non-zero if unsigned integer, zero otherwise
//***************************************************************************

int is_uinteger(const char *str) {
   // Check if it's composed entirely of numbers
   do {
      if (!isdigit(*str)) return 0;
   } while (*(++str));

   // Unsigned integer!
   return 1;
}

//***************************************************************************
// is_valid_id
// Checks if a string is a valid script name. The string is turned into
// lowercase if it's valid, it's casing is undefined if it isn't valid
// (symbols should remain the same though).
//---------------------------------------------------------------------------
// param name: pointer to string to check
// return: non-zero if valid, zero if invalid
//***************************************************************************

int is_valid_id(char *name) {
   // OK, for starters it MUST have at least one character
   if (*name == '\0')
      return 0;

   // Look for all characters to see if we find any invalid one
   // While we're at it, also make sure the name is short enough
   for (unsigned len = 0; *name; len++, name++) {
      if (len > MAX_ID_LEN) return 0;
      *name = tolower(*name);
      if (isalnum(*name)) continue;
      if (*name == '_') continue;
      return 0;
   }

   // String is valid!
   return 1;
}

//***************************************************************************
// is_color
// Checks if a string contains a RGB color or not. Colors are specified as a
// 6-digit hexadecimal number, with rrggbb format (00 being darkest and FF
// being brightest).
//---------------------------------------------------------------------------
// param str: string to check
// return: non-zero if color, zero otherwise
//***************************************************************************

int is_color(const char *str) {
   // Colors are always 6 characters
   if (strlen(str) != 6)
      return 0;

   // Ensure it's a hexadecimal number
   do {
      if (!isxdigit(*str)) return 0;
   } while (*(++str));

   // String is valid!
   return 1;
}
