//***************************************************************************
// "options_mouse.c"
// Code for the options mouse menu.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include "background.h"
#include "menu.h"
#include "reader.h"
#include "settings.h"
#include "text.h"
#include "video.h"

// UI positions
#define BUTTON_X1 (screen_cx - 0x70)      // Left column
#define BUTTON_X2 (screen_cx - 0x20)      // Middle column
#define BUTTON_X3 (screen_cx + 0x30)      // Right column
#define BUTTON_Y1 (screen_cy - 0x50)      // Top row
#define BUTTON_Y2 (screen_cy - 0x18)      // Middle row
#define BUTTON_Y3 (screen_cy + 0x20)      // Bottom row

#define TITLE_X (screen_cx)               // Title horizontal position
#define TITLE_Y (screen_cy - 0x58)        // Title vertical position
#define CAPTION_X (screen_cx)             // Caption horizontal position
#define CAPTION_Y (screen_cy + 0x58)      // Caption vertical position

// Position of the back button in test mode
#define ENDTEST_X1   (screen_cx - 0x10)
#define ENDTEST_Y1   (screen_h - 0x20)
#define ENDTEST_X2   (screen_cx + 0x0F)
#define ENDTEST_Y2   (screen_h - 0x09)

// List of buttons
enum {
   BUTTON_ENABLE,
   BUTTON_GRID,
   BUTTON_TEST,
   BUTTON_INC_X,
   BUTTON_DEC_X,
   BUTTON_INC_Y,
   BUTTON_DEC_Y,
   BUTTON_BACK,
   NUM_BUTTONS
};

// Where graphics are stored
static GraphicsSet *gfxset = NULL;
static Sprite *spr_button[NUM_BUTTONS][8];
static Sprite *spr_end_test[2];

// Used to animate stuff
static unsigned anim;
static unsigned button_anim;

// Names for each button
static const char *button_names[] = {
   "enable_off",
   "grid_off",
   "test",
   "inc_x",
   "dec_x",
   "inc_y",
   "dec_y",
   "back"
};

// Set when inside the test mode, clear otherwise
static unsigned testing;

// Private function prototypes
static void init_options_mouse_menu(void);
static void button_enable(void);
static void button_grid(void);
static void button_test(void);
static void button_inc_x(void);
static void button_inc_y(void);
static void button_dec_x(void);
static void button_dec_y(void);
static void button_back(void);

//***************************************************************************
// load_options_mouse
// Loads the assets for the options mouse menu. Called while the main menu is
// loading (so everything loads in a bulk).
//***************************************************************************

void load_options_mouse(void) {
   // Load graphics
   gfxset = load_graphics_set("graphics/options_mouse");

#define SPR(name) get_sprite(gfxset, name)
   // Load sprites
   for (unsigned i = 0; i < NUM_BUTTONS; i++) {
      char buffer[0x40];
      sprintf(buffer, "button_%s_dim", button_names[i]);
      spr_button[i][0] = SPR(buffer);
      sprintf(buffer, "button_%s_lit_1", button_names[i]);
      spr_button[i][1] = SPR(buffer);
      sprintf(buffer, "button_%s_lit_2", button_names[i]);
      spr_button[i][2] = SPR(buffer);
      sprintf(buffer, "button_%s_lit_3", button_names[i]);
      spr_button[i][3] = SPR(buffer);
   }

   // Yeah, I'm a lazy bastard lol
   spr_button[BUTTON_ENABLE][4] = SPR("button_enable_on_dim");
   spr_button[BUTTON_ENABLE][5] = SPR("button_enable_on_lit_1");
   spr_button[BUTTON_ENABLE][6] = SPR("button_enable_on_lit_2");
   spr_button[BUTTON_ENABLE][7] = SPR("button_enable_on_lit_3");
   spr_button[BUTTON_GRID][4] = SPR("button_grid_on_dim");
   spr_button[BUTTON_GRID][5] = SPR("button_grid_on_lit_1");
   spr_button[BUTTON_GRID][6] = SPR("button_grid_on_lit_2");
   spr_button[BUTTON_GRID][7] = SPR("button_grid_on_lit_3");

   // For the test mode
   spr_end_test[0] = SPR("end_test_dim");
   spr_end_test[1] = SPR("end_test_lit");
#undef SPR
}

//***************************************************************************
// init_options_mouse
// Initializes the options mouse menu.
//***************************************************************************

void init_options_mouse(void) {
   // Reset animations
   anim = 0;

   // No, not testing yet
   testing = 0;

   // Initialize menu
   init_options_mouse_menu();

   // Make the cursor visible
   set_cursor(CURSOR_ARROW);

   // Make screen visible
   fade_on();
}

//***************************************************************************
// init_options_mouse_menu [internal]
// Sets up the menu in this screen.
//***************************************************************************

static void init_options_mouse_menu(void) {
   init_menu();
   set_reinit_menu_func(init_options_mouse_menu);

   menu.cancel = button_back;
   menu.defoption.up = BUTTON_TEST;
   menu.defoption.down = BUTTON_ENABLE;
   menu.defoption.left = BUTTON_GRID;
   menu.defoption.right = BUTTON_ENABLE;

   menu.options[BUTTON_ENABLE].box.x1 = BUTTON_X1;
   menu.options[BUTTON_ENABLE].box.y1 = BUTTON_Y1;
   menu.options[BUTTON_ENABLE].box.x2 = BUTTON_X1 + 0x3F;
   menu.options[BUTTON_ENABLE].box.y2 = BUTTON_Y1 + 0x2F;
   menu.options[BUTTON_ENABLE].move.up = BUTTON_TEST;
   menu.options[BUTTON_ENABLE].move.down = BUTTON_DEC_X;
   menu.options[BUTTON_ENABLE].move.left = BUTTON_GRID;
   menu.options[BUTTON_ENABLE].move.right = BUTTON_INC_Y;
   menu.options[BUTTON_ENABLE].move.oneswitch = BUTTON_INC_Y;
   menu.options[BUTTON_ENABLE].action.accept = button_enable;
   menu.options[BUTTON_ENABLE].caption = text.options_mouse.enable;

   menu.options[BUTTON_INC_Y].box.x1 = BUTTON_X2;
   menu.options[BUTTON_INC_Y].box.y1 = BUTTON_Y1;
   menu.options[BUTTON_INC_Y].box.x2 = BUTTON_X2 + 0x3F;
   menu.options[BUTTON_INC_Y].box.y2 = BUTTON_Y1 + 0x2F;
   menu.options[BUTTON_INC_Y].move.up = BUTTON_DEC_Y;
   menu.options[BUTTON_INC_Y].move.down = BUTTON_DEC_Y;
   menu.options[BUTTON_INC_Y].move.left = BUTTON_ENABLE;
   menu.options[BUTTON_INC_Y].move.right = BUTTON_GRID;
   menu.options[BUTTON_INC_Y].move.oneswitch = BUTTON_GRID;
   menu.options[BUTTON_INC_Y].action.accept = button_inc_y;
   menu.options[BUTTON_INC_Y].caption = text.options_mouse.inc_y;

   menu.options[BUTTON_GRID].box.x1 = BUTTON_X3;
   menu.options[BUTTON_GRID].box.y1 = BUTTON_Y1;
   menu.options[BUTTON_GRID].box.x2 = BUTTON_X3 + 0x3F;
   menu.options[BUTTON_GRID].box.y2 = BUTTON_Y1 + 0x2F;
   menu.options[BUTTON_GRID].move.up = BUTTON_BACK;
   menu.options[BUTTON_GRID].move.down = BUTTON_INC_X;
   menu.options[BUTTON_GRID].move.left = BUTTON_INC_Y;
   menu.options[BUTTON_GRID].move.right = BUTTON_ENABLE;
   menu.options[BUTTON_GRID].move.oneswitch = BUTTON_INC_X;
   menu.options[BUTTON_GRID].action.accept = button_grid;
   menu.options[BUTTON_GRID].caption = text.options_mouse.grid;

   menu.options[BUTTON_DEC_X].box.x1 = BUTTON_X1;
   menu.options[BUTTON_DEC_X].box.y1 = BUTTON_Y2;
   menu.options[BUTTON_DEC_X].box.x2 = BUTTON_X1 + 0x3F;
   menu.options[BUTTON_DEC_X].box.y2 = BUTTON_Y2 + 0x2F;
   menu.options[BUTTON_DEC_X].move.up = BUTTON_ENABLE;
   menu.options[BUTTON_DEC_X].move.down = BUTTON_TEST;
   menu.options[BUTTON_DEC_X].move.left = BUTTON_INC_X;
   menu.options[BUTTON_DEC_X].move.right = BUTTON_INC_X;
   menu.options[BUTTON_DEC_X].move.oneswitch = BUTTON_ENABLE;
   menu.options[BUTTON_DEC_X].action.accept = button_dec_x;
   menu.options[BUTTON_DEC_X].caption = text.options_mouse.dec_x;

   menu.options[BUTTON_INC_X].box.x1 = BUTTON_X3;
   menu.options[BUTTON_INC_X].box.y1 = BUTTON_Y2;
   menu.options[BUTTON_INC_X].box.x2 = BUTTON_X3 + 0x3F;
   menu.options[BUTTON_INC_X].box.y2 = BUTTON_Y2 + 0x2F;
   menu.options[BUTTON_INC_X].move.up = BUTTON_GRID;
   menu.options[BUTTON_INC_X].move.down = BUTTON_BACK;
   menu.options[BUTTON_INC_X].move.left = BUTTON_DEC_X;
   menu.options[BUTTON_INC_X].move.right = BUTTON_DEC_X;
   menu.options[BUTTON_INC_X].move.oneswitch = BUTTON_BACK;
   menu.options[BUTTON_INC_X].action.accept = button_inc_x;
   menu.options[BUTTON_INC_X].caption = text.options_mouse.inc_x;

   menu.options[BUTTON_TEST].box.x1 = BUTTON_X1;
   menu.options[BUTTON_TEST].box.y1 = BUTTON_Y3;
   menu.options[BUTTON_TEST].box.x2 = BUTTON_X1 + 0x3F;
   menu.options[BUTTON_TEST].box.y2 = BUTTON_Y3 + 0x2F;
   menu.options[BUTTON_TEST].move.up = BUTTON_DEC_X;
   menu.options[BUTTON_TEST].move.down = BUTTON_ENABLE;
   menu.options[BUTTON_TEST].move.left = BUTTON_BACK;
   menu.options[BUTTON_TEST].move.right = BUTTON_DEC_Y;
   menu.options[BUTTON_TEST].move.oneswitch = BUTTON_DEC_X;
   menu.options[BUTTON_TEST].action.accept = button_test;
   menu.options[BUTTON_TEST].caption = text.options_mouse.test;

   menu.options[BUTTON_DEC_Y].box.x1 = BUTTON_X2;
   menu.options[BUTTON_DEC_Y].box.y1 = BUTTON_Y3;
   menu.options[BUTTON_DEC_Y].box.x2 = BUTTON_X2 + 0x3F;
   menu.options[BUTTON_DEC_Y].box.y2 = BUTTON_Y3 + 0x2F;
   menu.options[BUTTON_DEC_Y].move.up = BUTTON_INC_Y;
   menu.options[BUTTON_DEC_Y].move.down = BUTTON_INC_Y;
   menu.options[BUTTON_DEC_Y].move.left = BUTTON_TEST;
   menu.options[BUTTON_DEC_Y].move.right = BUTTON_BACK;
   menu.options[BUTTON_DEC_Y].move.oneswitch = BUTTON_TEST;
   menu.options[BUTTON_DEC_Y].action.accept = button_dec_y;
   menu.options[BUTTON_DEC_Y].caption = text.options_mouse.dec_y;

   menu.options[BUTTON_BACK].box.x1 = BUTTON_X3;
   menu.options[BUTTON_BACK].box.y1 = BUTTON_Y3;
   menu.options[BUTTON_BACK].box.x2 = BUTTON_X3 + 0x3F;
   menu.options[BUTTON_BACK].box.y2 = BUTTON_Y3 + 0x2F;
   menu.options[BUTTON_BACK].move.up = BUTTON_INC_X;
   menu.options[BUTTON_BACK].move.down = BUTTON_GRID;
   menu.options[BUTTON_BACK].move.left = BUTTON_DEC_Y;
   menu.options[BUTTON_BACK].move.right = BUTTON_TEST;
   menu.options[BUTTON_BACK].move.oneswitch = BUTTON_DEC_Y;
   menu.options[BUTTON_BACK].action.accept = button_back;
   menu.options[BUTTON_BACK].caption = text.options.back;
   menu.options[BUTTON_BACK].sfx = SFX_CANCEL;
}

//***************************************************************************
// run_options_mouse
// Processes the logic for the options mouse menu.
//***************************************************************************

void run_options_mouse(void) {
   // Inside test mode?
   if (testing) {
      // Determine if the mouse is hovering the button
      int inside = input.cursor.x >= ENDTEST_X1 &&
                   input.cursor.y >= ENDTEST_Y1 &&
                   input.cursor.x <= ENDTEST_X2 &&
                   input.cursor.y <= ENDTEST_Y2
                   ? 1 : 0;

      // Quit test mode?
      if ((inside && input.cursor.click) ||
      input.menu.cancel || input.oneswitch.tap || input.oneswitch.tap2) {
         play_sfx(SFX_CANCEL);
         testing = 0;
      }

      // Set cursor adequately
      set_cursor(CURSOR_CROSS);
   }

   // Inside the menu?
   else {
      update_menu();
   }

   // Update background animation
   update_background();

   // Update animations
   anim++;
   switch (anim >> 3 & 3) {
      case 0: button_anim = 1; break;
      case 1: button_anim = 2; break;
      case 2: button_anim = 3; break;
      case 3: button_anim = 2; break;
   }
}

//***************************************************************************
// draw_options_mouse
// Draws the options mouse menu.
//***************************************************************************

void draw_options_mouse(void) {
   // Draw background
   draw_background();

   // Determine where the grid divisions are
   int32_t grid_x1 = screen_cx - settings.mousesw_x;
   int32_t grid_y1 = screen_cy - settings.mousesw_y;
   int32_t grid_x2 = screen_cx + settings.mousesw_x;
   int32_t grid_y2 = screen_cy + settings.mousesw_y;

   // Highlight piece of the grid the mouse is on
   int32_t x1, y1, x2, y2;

   if (input.cursor.x <= grid_x1) {
      x1 = 0;
      x2 = grid_x1;
   } else if (input.cursor.x >= grid_x2) {
      x1 = grid_x2;
      x2 = screen_w;
   } else {
      x1 = grid_x1;
      x2 = grid_x2;
   }

   if (input.cursor.y <= grid_y1) {
      y1 = 0;
      y2 = grid_y1;
   } else if (input.cursor.y >= grid_y2) {
      y1 = grid_y2;
      y2 = screen_h;
   } else {
      y1 = grid_y1;
      y2 = grid_y2;
   }

   if (input.cursor.inside)
      fill_rectangle(x1, y1, x2, y2,
      settings.cga_mode ? 0x800000 : 0x800080);

   // Draw grid that will be used for the mouse controls in-game
   if (settings.mousesw_grid) {
      draw_vline(grid_x1, 0, screen_h - 1, 0xFFFFFF);
      draw_vline(grid_x2, 0, screen_h - 1, 0xFFFFFF);
      draw_hline(0, grid_y1, screen_w - 1, 0xFFFFFF);
      draw_hline(0, grid_y2, screen_w - 1, 0xFFFFFF);
   }

   // Testing?
   if (testing) {
      // Determine if the mouse is hovering the button
      int inside = input.cursor.x >= ENDTEST_X1 &&
                   input.cursor.y >= ENDTEST_Y1 &&
                   input.cursor.x <= ENDTEST_X2 &&
                   input.cursor.y <= ENDTEST_Y2
                   ? 1 : 0;

      // Draw button
      draw_sprite(spr_end_test[inside], ENDTEST_X1, ENDTEST_Y1, SPR_NOFLIP);
   }

   // Nope, show menu
   else {
      // Draw all buttons
      for (int i = 0; i < NUM_BUTTONS; i++) {
         // Determine anim to use
         // Hackish way to account for pressed buttons (lazy etc.)
         unsigned which = (menu.selected == i) ? button_anim : 0;

         if (i == BUTTON_ENABLE && settings.mouse_switch)
            which += 4;
         if (i == BUTTON_GRID && settings.mousesw_grid)
            which += 4;

         // Draw sprite
         draw_sprite(spr_button[i][which],
            menu.options[i].box.x1, menu.options[i].box.y1, SPR_NOFLIP);
      }

      // If an option is selected, show a border around it as well as its
      // caption above all the buttons
      if (menu.selected != -1) {
         // Where the data for the selected option is stored
         MenuOption *ptr = &menu.options[menu.selected];

         // Draw the border on top of the button
         if ((anim & 0x08) /*|| oneswitch_edit*/) {
            draw_rectangle(ptr->box.x1+1, ptr->box.y1+1,
               ptr->box.x2-1, ptr->box.y2-1, settings.box_lit[1]);
            draw_rectangle(ptr->box.x1, ptr->box.y1,
               ptr->box.x2, ptr->box.y2, settings.box_lit[0]);
            draw_rectangle(ptr->box.x1-1, ptr->box.y1-1,
               ptr->box.x2+1, ptr->box.y2+1, settings.box_lit[0]);
            draw_rectangle(ptr->box.x1-2, ptr->box.y1-2,
               ptr->box.x2+2, ptr->box.y2+2, settings.box_lit[1]);
            draw_hline(ptr->box.x1-2, ptr->box.y2+3,
               ptr->box.x2+2, settings.box_lit[2]);
         }

         // Determine "param" to use for the caption
         int int_param = 0;
         const char *str_param = NULL;

         switch (menu.selected) {
            case BUTTON_ENABLE:
               str_param = settings.mouse_switch ?
                  text.options_mouse.yes : text.options_mouse.no;
               break;

            case BUTTON_GRID:
               str_param = settings.mousesw_grid ?
                  text.options_mouse.show : text.options_mouse.hide;
               break;

            case BUTTON_INC_X:
            case BUTTON_DEC_X:
               int_param = settings.mousesw_x * 2;
               break;

            case BUTTON_INC_Y:
            case BUTTON_DEC_Y:
               int_param = settings.mousesw_y * 2;
               break;

            default:
               break;
         }

         // Draw the caption
         if (str_param != NULL)
            draw_text_str(ptr->caption, str_param,
            CAPTION_X, CAPTION_Y, FONT_LIT, ALIGN_TOP);
         else
            draw_text_int(ptr->caption, int_param,
            CAPTION_X, CAPTION_Y, FONT_LIT, ALIGN_TOP);

         // Output caption to screen reader
         if (str_param != NULL)
            set_reader_text_str(ptr->caption, str_param);
         else
            set_reader_text_int(ptr->caption, int_param);
      }
   }

   // Draw title
   draw_text(text.options.mouse, TITLE_X, TITLE_Y, FONT_LIT, ALIGN_BOTTOM);
}

//***************************************************************************
// unload_options_mouse
// Unloads the assets for the options mouse menu. Called while the main menu
// is unloading (as everything was loaded in a bulk).
//***************************************************************************

void unload_options_mouse(void) {
   // Unload graphics
   destroy_graphics_set(gfxset);
}

//***************************************************************************
// button_enable [internal]
// Button for toggling whether mouse switch mode is being used or not.
//***************************************************************************

static void button_enable(void) {
   settings.mouse_switch ^= 1;
}

//***************************************************************************
// button_grid [internal]
// Button for toggling whether the grid is visible or not.
//***************************************************************************

static void button_grid(void) {
   settings.mousesw_grid ^= 1;
}

//***************************************************************************
// button_test [internal]
// Button for going into the test mode.
//***************************************************************************

static void button_test(void) {
   testing = 1;
}

//***************************************************************************
// button_inc_x [internal]
// button_inc_y [internal]
// button_dec_x [internal]
// button_dec_y [internal]
// Buttons for altering the distances in the grid divisions.
//***************************************************************************

static void button_inc_x(void) {
   settings.mousesw_x += 5;
}

static void button_inc_y(void) {
   settings.mousesw_y += 5;
}

static void button_dec_x(void) {
   if (settings.mousesw_x > 5)
      settings.mousesw_x -= 5;
}

static void button_dec_y(void) {
   if (settings.mousesw_y > 5)
      settings.mousesw_y -= 5;
}

//***************************************************************************
// button_back [internal]
// Button for going back. Returns to the options main menu.
//***************************************************************************

static void button_back(void) {
   fade_off_and_switch(GAMEMODE_OPTIONS);
}
