//***************************************************************************
// "options_gameplay.c"
// Code for the options gameplay menu.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include <stdio.h>
#include "background.h"
#include "menu.h"
#include "reader.h"
#include "settings.h"
#include "text.h"
#include "video.h"

// UI positions
#define DIFF_X (screen_cx - 112)          // X: difficulty button
#define DIFF_Y (SPEED_Y - 0x34)           // Y: difficulty button
#define DIFF_BX1 (DIFF_X + 0x40)          // X: difficulty "easy"
#define DIFF_BX2 (DIFF_BX1 + 0x30)        // X: difficulty "normal"
#define DIFF_BX3 (DIFF_BX2 + 0x30)        // X: difficulty "hard"
#define DIFF_BY (DIFF_Y + 0x08)           // Y: difficulty values

#define SPEED_X (screen_cx - 112)         // X: speed button
#define SPEED_Y (screen_cy - 0x10)        // Y: speed button
#define SPEED_BX (screen_cx - 82)         // X: speed bar
#define SPEED_BY (SPEED_Y + 0x08)         // Y: speed bar

//#define ONESW_X (screen_cx - 0x68)        // X: one-switch
//#define ONESW_Y (SPEED_Y + 0x24)          // Y: one-switch

//#define LANG_X (screen_cx - 0x20)         // X: language
#define LANG_X (screen_cx - 0x44)         // X: language
#define LANG_Y (SPEED_Y + 0x24)           // Y: language

//#define BACK_X (screen_cx + 0x28)         // X: back
#define BACK_X (screen_cx + 0x04)         // X: back
#define BACK_Y (LANG_Y)                   // Y: back

#define WARN_BX1 (screen_cx - 0x68)       // X: Warning "Cancel" button
#define WARN_BX2 (screen_cx + 0x08)       // X: Warning "OK" button
#define WARN_BCX1 (WARN_BX1 + 56)         // X: Warning "Cancel" label
#define WARN_BCX2 (WARN_BX2 + 56)         // X: Warning "OK" label
#define WARN_Y (screen_cy - 0x3C)         // Y: Warning text
#define WARN_TY (screen_cy - 0x58)        // Y: Warning title
#define WARN_BY (screen_cy + 0x48)        // Y: Warning buttons
#define WARN_BCY (WARN_BY + 12)           // Y: Warning labels

// List of buttons
enum {
   BUTTON_DIFFICULTY,
   BUTTON_SPEED,
   //BUTTON_ONESWITCH,
   BUTTON_LANGUAGE,
   BUTTON_BACK,
   BUTTON_OK,
   BUTTON_CANCEL,
   NUM_BUTTONS
};

// Where graphics are stored
static GraphicsSet *gfxset = NULL;
static Sprite *spr_button[NUM_BUTTONS][4];
static Sprite *spr_diff[3][2];
//static Sprite *spr_oneswitch[2][4];

// Used to animate stuff
static unsigned anim;
static unsigned button_anim;

// Names for each button
static const char *button_names[] = {
   "difficulty",
   "speed",
   //"--",
   "language",
   "back",
   "ok",
   "cancel"
};

// Used to modify the difficulty and game speed in one-switch mode
static unsigned oneswitch_edit;     // Set if modifying a setting
static unsigned oneswitch_timer;    // Delay before next value

// Used to know whether to show the one-switch mode warning
static int warning;

// Private function prototypes
static void button_dec_difficulty(void);
static void button_inc_difficulty(void);
static void button_dec_speed(void);
static void button_inc_speed(void);
//static void button_oneswitch(void);
static void button_language(void);
static void button_back(void);
static void button_one_switch(void); // <-- WTF I should rename this o.o
static void set_real_menu(void);
//static void set_warning_menu(void);
//static void draw_warning(void);
//static void button_oneswitch_no(void);
//static void button_oneswitch_yes(void);

//***************************************************************************
// load_options_gameplay
// Loads the assets for the options gameplay menu. Called while the main menu
// is loading (so everything loads in a bulk).
//***************************************************************************

void load_options_gameplay(void) {
   // Load graphics
   gfxset = load_graphics_set("graphics/options_gameplay");

   // Load sprites
#define SPR(name) get_sprite(gfxset, name)
   for (unsigned i = 0; i < NUM_BUTTONS; i++) {
      char buffer[0x40];
      sprintf(buffer, "button_%s_dim", button_names[i]);
      spr_button[i][0] = SPR(buffer);
      sprintf(buffer, "button_%s_lit_1", button_names[i]);
      spr_button[i][1] = SPR(buffer);
      sprintf(buffer, "button_%s_lit_2", button_names[i]);
      spr_button[i][2] = SPR(buffer);
      sprintf(buffer, "button_%s_lit_3", button_names[i]);
      spr_button[i][3] = SPR(buffer);
   }

   spr_diff[DIFF_EASY][0] = SPR("button_easy_off");
   spr_diff[DIFF_EASY][1] = SPR("button_easy_on");
   spr_diff[DIFF_NORMAL][0] = SPR("button_normal_off");
   spr_diff[DIFF_NORMAL][1] = SPR("button_normal_on");
   spr_diff[DIFF_HARD][0] = SPR("button_hard_off");
   spr_diff[DIFF_HARD][1] = SPR("button_hard_on");

   /*
   spr_oneswitch[0][0] = SPR("button_oneswitch_off_dim");
   spr_oneswitch[0][1] = SPR("button_oneswitch_off_lit_1");
   spr_oneswitch[0][2] = SPR("button_oneswitch_off_lit_2");
   spr_oneswitch[0][3] = SPR("button_oneswitch_off_lit_3");
   spr_oneswitch[1][0] = SPR("button_oneswitch_on_dim");
   spr_oneswitch[1][1] = SPR("button_oneswitch_on_lit_1");
   spr_oneswitch[1][2] = SPR("button_oneswitch_on_lit_2");
   spr_oneswitch[1][3] = SPR("button_oneswitch_on_lit_3");
   */
#undef SPR
}

//***************************************************************************
// init_options_gameplay
// Initializes the options gameplay menu.
//***************************************************************************

void init_options_gameplay(void) {
   // Not in warning mode!
   warning = 0;

   // Reset animations
   anim = 0;

   // Initialize menu
   set_real_menu();
   oneswitch_edit = 0;
   oneswitch_timer = 0;

   // Make the cursor visible
   set_cursor(CURSOR_ARROW);

   // Make screen visible
   fade_on();
}

//***************************************************************************
// run_options_gameplay
// Processes the logic for the options gameplay menu.
//***************************************************************************

void run_options_gameplay(void) {
   // Changing a value? (in one-switch mode)
   if (oneswitch_edit) {
      // Delay between changes
      oneswitch_timer--;
      if (oneswitch_timer == 0) {
         oneswitch_timer = settings.menu_delay;

         // Change value as needed
         switch (menu.selected) {
            case BUTTON_DIFFICULTY: button_inc_difficulty(); break;
            case BUTTON_SPEED: button_dec_speed(); break;
            default: break;
         }
      }

      // Tap to be done
      if (input.oneswitch.tap || input.oneswitch.tap2) {
         oneswitch_edit = 0;
         play_sfx(SFX_OK);
      }
   }

   // Update the menu
   else {
      update_menu();
   }

   // Update background animation
   // Make its speed reflect the selected game speed
   {
      // Determine how many times to update the background animation
      unsigned updates;
      switch (settings.game_speed) {
         case 0: updates = (anim & 3) ? 0 : 1; break;
         case 1: updates = (anim & 7) % 3 ? 0 : 1; break;
         case 2: updates = (anim & 1) ? 1 : 0; break;
         case 3: updates = (anim & 3) ? 1 : 0; break;
         case 4: updates = 1; break;
         case 5: updates = (anim & 3) ? 1 : 2; break;
         case 6: updates = (anim & 1) ? 2 : 1; break;
         case 7: updates = (anim & 3) ? 2 : 1; break;
         case 8: updates = 2; break;
         default: updates = 1; break;
      }

      // Make the background move
      while (updates-- > 0)
         update_background();
   }

   // Update animations
   anim++;
   switch (anim >> 3 & 3) {
      case 0: button_anim = 1; break;
      case 1: button_anim = 2; break;
      case 2: button_anim = 3; break;
      case 3: button_anim = 2; break;
   }

   // Nothing more to do if we're showing the one-switch warning
   if (warning)
      return;

   // Click on the difficulty buttons to change the difficulty
   for (int i = DIFF_EASY; i <= DIFF_HARD; i++) {
      if (input.cursor.click &&
      input.cursor.x >= DIFF_BX1+(i*0x30) &&
      input.cursor.x <= DIFF_BX1+(i*0x30) + 39 &&
      input.cursor.y >= DIFF_BY &&
      input.cursor.y <= DIFF_BY + 31) {
         settings.difficulty = i;
         play_sfx(SFX_OK);
         break;
      }
   }

   // Click on the speed bar to change the game speed
   if (input.cursor.click &&
   input.cursor.x >= SPEED_BX && input.cursor.x <= SPEED_BX + 159 &&
   input.cursor.y >= SPEED_BY && input.cursor.y <= SPEED_BY + 15) {
      //settings.game_speed = (input.cursor.x - SPEED_BX) / 0x20;
      settings.game_speed = (input.cursor.x - SPEED_BX) / 0x12;
      play_sfx(SFX_OK);
   }
}

//***************************************************************************
// draw_options_gameplay
// Draws the options gameplay menu.
//***************************************************************************

void draw_options_gameplay(void) {
   // Get the correct sprites fot the one-switch button
   /*
   for (unsigned i = 0; i < 4; i++)
      spr_button[BUTTON_ONESWITCH][i]=spr_oneswitch[settings.one_switch][i];
   */

   // Draw background
   draw_background();

   // Draw all buttons
   for (int i = 0; i < NUM_BUTTONS; i++) {
      // Make sure that the button is set
      if (menu.options[i].box.x1 < 0 ||
      menu.options[i].box.y1 < 0)
         continue;

      // Button exists, draw it
      draw_sprite(menu.selected == i ?
         spr_button[i][button_anim] : spr_button[i][0],
         menu.options[i].box.x1, menu.options[i].box.y1, SPR_NOFLIP);
   }

   // Draw warning?
   /*
   if (warning)
      draw_warning();
   */

   // If an option is selected, show a border around it as well as its
   // caption above all the buttons
   if (menu.selected != -1) {
      // Where the data for the selected option is stored
      MenuOption *ptr = &menu.options[menu.selected];

      // Draw the border on top of the button
      if ((anim & 0x08) || oneswitch_edit) {
         draw_rectangle(ptr->box.x1+1, ptr->box.y1+1,
            ptr->box.x2-1, ptr->box.y2-1, settings.box_lit[1]);
         draw_rectangle(ptr->box.x1, ptr->box.y1,
            ptr->box.x2, ptr->box.y2, settings.box_lit[0]);
         draw_rectangle(ptr->box.x1-1, ptr->box.y1-1,
            ptr->box.x2+1, ptr->box.y2+1, settings.box_lit[0]);
         draw_rectangle(ptr->box.x1-2, ptr->box.y1-2,
            ptr->box.x2+2, ptr->box.y2+2, settings.box_lit[1]);
         draw_hline(ptr->box.x1-2, ptr->box.y2+3,
            ptr->box.x2+2, settings.box_lit[2]);
      }

      // Don't draw the caption during the warning
      if (warning)
         return;

      // Determine "param" to use for the caption
      const char *param;
      switch (menu.selected) {
         case BUTTON_DIFFICULTY:
            param = text.options_gameplay.diff_names[settings.difficulty];
            break;

         case BUTTON_SPEED:
            switch (settings.game_speed) {
               case 0: param = "25"; break;
               case 1: param = "37.5"; break;
               case 2: param = "50"; break;
               case 3: param = "75"; break;
               case 4: param = "100"; break;
               case 5: param = "125"; break;
               case 6: param = "150"; break;
               case 7: param = "175"; break;
               case 8: param = "200"; break;
               default: param = "???"; break;
            }
            break;

         /*
         case BUTTON_ONESWITCH:
            param = settings.one_switch ?
               text.options_gameplay.enabled :
               text.options_gameplay.disabled;
            break;
         */

         default:
            param = "???";
            break;
      }

      // Draw the caption
      draw_text_str(ptr->caption, param,
         screen_cx, screen_h - 0x10, FONT_LIT, ALIGN_BOTTOM);
      set_reader_text_str(ptr->caption, param);
   }

   // No more to draw if showing the warning
   if (warning)
      return;

   // Draw the difficulty values
   draw_sprite(spr_diff[DIFF_EASY][settings.difficulty == DIFF_EASY ?
      1 : 0], DIFF_BX1, DIFF_BY, SPR_NOFLIP);
   draw_sprite(spr_diff[DIFF_NORMAL][settings.difficulty == DIFF_NORMAL ?
      1 : 0], DIFF_BX2, DIFF_BY, SPR_NOFLIP);
   draw_sprite(spr_diff[DIFF_HARD][settings.difficulty == DIFF_HARD ?
      1 : 0], DIFF_BX3, DIFF_BY, SPR_NOFLIP);

   // Draw the game speed bar
   for (unsigned i = 0; i < 9; i++) {
      // Colors for each segment when lit
      static const uint32_t colors[] = {
         0x00C000,
         0x40D000,
         0x80E000,
         0xC0F000,
         0xFFFF00,
         0xFFD000,
         0xFFA000,
         0xFF7000,
         0xFF4000
      };

      // Get base coordinates for this segment
      //int32_t x = SPEED_BX + i * 0x20;
      int32_t x = SPEED_BX + i * 0x12;
      int32_t y = SPEED_BY;

      // Draw inside of segment
      fill_rectangle(x+3, y+3, x+14, y+12,
         i <= settings.game_speed ? colors[i] : 0x303030);

      // Draw spot light
      draw_hline(x+5, y+5, x+7, 0xFFFFFF);
      draw_hline(x+5, y+6, x+7, 0xFFFFFF);

      // Draw border
      draw_rectangle(x+2, y+2, x+15, y+13, 0x000000);
   }

   // Draw title
   draw_text(text.options.gameplay, screen_cx, 0x10, FONT_LIT, ALIGN_TOP);
}

//***************************************************************************
// unload_options_gameplay
// Unloads the assets for the options gameplay menu menu. Called while the
// main menu is unloading (as everything was loaded in a bulk).
//***************************************************************************

void unload_options_gameplay(void) {
   // Unload graphics
   destroy_graphics_set(gfxset);
}

//***************************************************************************
// button_dec_difficulty [internal]
// Callback when pressing left when difficulty is selected. Decreases the
// difficulty.
//***************************************************************************

static void button_dec_difficulty(void) {
   if (settings.difficulty == DIFF_EASY)
      settings.difficulty = DIFF_HARD;
   else
      settings.difficulty--;
}

//***************************************************************************
// button_inc_difficulty [internal]
// Callback when pressing right when difficulty is selected. Increases the
// difficulty.
//***************************************************************************

static void button_inc_difficulty(void) {
   if (settings.difficulty == DIFF_HARD)
      settings.difficulty = DIFF_EASY;
   else
      settings.difficulty++;
}

//***************************************************************************
// button_dec_speed [internal]
// Callback when pressing left when game speed is selected. Decreases the
// game speed.
//***************************************************************************

static void button_dec_speed(void) {
   if (settings.game_speed == 0)
      //settings.game_speed = 4;
      settings.game_speed = 8;
   else
      settings.game_speed--;
}

//***************************************************************************
// button_inc_speed [internal]
// Callback when pressing right when game speed is selected. Increases the
// game speed.
//***************************************************************************

static void button_inc_speed(void) {
   //if (settings.game_speed == 4)
   if (settings.game_speed == 8)
      settings.game_speed = 0;
   else
      settings.game_speed++;
}

//***************************************************************************
// button_oneswitch [internal]
// Button for toggling one-switch. Shows a warning instead, since it's very
// easy to get it wrong (it's here for accessibility reasons).
//***************************************************************************

/*
static void button_oneswitch(void) {
   // Turn on warning
   warning = 1;
   set_warning_menu();
}
*/

//***************************************************************************
// button_language [internal]
// Button for going to the language select.
//***************************************************************************

static void button_language(void) {
   // Switch to the language select
   fade_off_and_switch(GAMEMODE_LANGSELECT);
}

//***************************************************************************
// button_back [internal]
// Button for going back. Returns to the options main menu.
//***************************************************************************

static void button_back(void) {
   fade_off_and_switch(GAMEMODE_OPTIONS);
}

//***************************************************************************
// button_one_switch [internal]
// Starts the one-switch editing mode for the difficulty and game speed.
//***************************************************************************

static void button_one_switch(void) {
   oneswitch_edit = 1;
   oneswitch_timer = settings.menu_delay;
}

//***************************************************************************
// set_real_menu [internal]
// Initializes the menu normally used.
//***************************************************************************

static void set_real_menu(void) {
   // Reset menu
   init_menu();
   set_reinit_menu_func(set_real_menu);
   menu.cancel = button_back;
   menu.defoption.up = BUTTON_LANGUAGE;
   menu.defoption.down = BUTTON_DIFFICULTY;
   menu.defoption.left = BUTTON_DIFFICULTY;
   menu.defoption.right = BUTTON_DIFFICULTY;

   // Difficulty slider
   menu.options[BUTTON_DIFFICULTY].box.x1 = DIFF_X;
   menu.options[BUTTON_DIFFICULTY].box.y1 = DIFF_Y;
   menu.options[BUTTON_DIFFICULTY].box.x2 = DIFF_X + 223;
   menu.options[BUTTON_DIFFICULTY].box.y2 = DIFF_Y + 0x2F;
   menu.options[BUTTON_DIFFICULTY].move.up = BUTTON_LANGUAGE;
   menu.options[BUTTON_DIFFICULTY].move.down = BUTTON_SPEED;
   menu.options[BUTTON_DIFFICULTY].move.oneswitch = BUTTON_SPEED;
   menu.options[BUTTON_DIFFICULTY].action.left = button_dec_difficulty;
   menu.options[BUTTON_DIFFICULTY].action.right = button_inc_difficulty;
   menu.options[BUTTON_DIFFICULTY].action.oneswitch = button_one_switch;
   menu.options[BUTTON_DIFFICULTY].caption=text.options_gameplay.difficulty;

   // Game speed slider
   menu.options[BUTTON_SPEED].box.x1 = SPEED_X;
   menu.options[BUTTON_SPEED].box.y1 = SPEED_Y;
   menu.options[BUTTON_SPEED].box.x2 = SPEED_X + 223;
   menu.options[BUTTON_SPEED].box.y2 = SPEED_Y + 0x1F;
   menu.options[BUTTON_SPEED].move.up = BUTTON_DIFFICULTY;
   menu.options[BUTTON_SPEED].move.down = BUTTON_LANGUAGE;
   //menu.options[BUTTON_SPEED].move.oneswitch = BUTTON_ONESWITCH;
   menu.options[BUTTON_SPEED].move.oneswitch = BUTTON_LANGUAGE;
   menu.options[BUTTON_SPEED].action.left = button_dec_speed;
   menu.options[BUTTON_SPEED].action.right = button_inc_speed;
   menu.options[BUTTON_SPEED].action.oneswitch = button_one_switch;
   menu.options[BUTTON_SPEED].caption = text.options_gameplay.game_speed;

   // One-switch button
   /*
   menu.options[BUTTON_ONESWITCH].box.x1 = ONESW_X;
   menu.options[BUTTON_ONESWITCH].box.y1 = ONESW_Y;
   menu.options[BUTTON_ONESWITCH].box.x2 = ONESW_X + 0x3F;
   menu.options[BUTTON_ONESWITCH].box.y2 = ONESW_Y + 0x2F;
   menu.options[BUTTON_ONESWITCH].move.up = BUTTON_SPEED;
   menu.options[BUTTON_ONESWITCH].move.left = BUTTON_BACK;
   menu.options[BUTTON_ONESWITCH].move.right = BUTTON_LANGUAGE;
   menu.options[BUTTON_ONESWITCH].move.down = BUTTON_DIFFICULTY;
   menu.options[BUTTON_ONESWITCH].move.oneswitch = BUTTON_LANGUAGE;
   menu.options[BUTTON_ONESWITCH].action.accept = button_oneswitch;
   menu.options[BUTTON_ONESWITCH].caption = text.options_gameplay.one_switch;
   */

   // Language button
   menu.options[BUTTON_LANGUAGE].box.x1 = LANG_X;
   menu.options[BUTTON_LANGUAGE].box.y1 = LANG_Y;
   menu.options[BUTTON_LANGUAGE].box.x2 = LANG_X + 0x3F;
   menu.options[BUTTON_LANGUAGE].box.y2 = LANG_Y + 0x2F;
   menu.options[BUTTON_LANGUAGE].move.up = BUTTON_SPEED;
   //menu.options[BUTTON_LANGUAGE].move.left = BUTTON_ONESWITCH;
   menu.options[BUTTON_LANGUAGE].move.left = BUTTON_BACK;
   menu.options[BUTTON_LANGUAGE].move.right = BUTTON_BACK;
   menu.options[BUTTON_LANGUAGE].move.down = BUTTON_DIFFICULTY;
   menu.options[BUTTON_LANGUAGE].move.oneswitch = BUTTON_BACK;
   menu.options[BUTTON_LANGUAGE].action.accept = button_language;
   menu.options[BUTTON_LANGUAGE].caption = text.options_gameplay.language;

   // Back button
   menu.options[BUTTON_BACK].box.x1 = BACK_X;
   menu.options[BUTTON_BACK].box.y1 = BACK_Y;
   menu.options[BUTTON_BACK].box.x2 = BACK_X + 0x3F;
   menu.options[BUTTON_BACK].box.y2 = BACK_Y + 0x2F;
   menu.options[BUTTON_BACK].move.up = BUTTON_SPEED;
   menu.options[BUTTON_BACK].move.down = BUTTON_DIFFICULTY;
   menu.options[BUTTON_BACK].move.left = BUTTON_LANGUAGE;
   //menu.options[BUTTON_BACK].move.right = BUTTON_ONESWITCH;
   menu.options[BUTTON_BACK].move.right = BUTTON_LANGUAGE;
   menu.options[BUTTON_BACK].move.oneswitch = BUTTON_DIFFICULTY;
   menu.options[BUTTON_BACK].action.accept = button_back;
   menu.options[BUTTON_BACK].caption = text.options.back;
   menu.options[BUTTON_BACK].sfx = SFX_CANCEL;
}

//***************************************************************************
// set_warning_menu [internal]
// Initializes the menu used during the one-switch warning.
//***************************************************************************

/*
static void set_warning_menu(void) {
   // Reset menu
   init_menu();
   set_reinit_menu_func(set_warning_menu);
   menu.cancel = button_oneswitch_no;
   menu.defoption.up = BUTTON_CANCEL;
   menu.defoption.down = BUTTON_CANCEL;
   menu.defoption.left = BUTTON_OK;
   menu.defoption.right = BUTTON_CANCEL;
   menu.selected = BUTTON_CANCEL;

   // Cancel button
   menu.options[BUTTON_CANCEL].box.x1 = WARN_BX1;
   menu.options[BUTTON_CANCEL].box.y1 = WARN_BY;
   menu.options[BUTTON_CANCEL].box.x2 = WARN_BX1 + 0x5F;
   menu.options[BUTTON_CANCEL].box.y2 = WARN_BY + 0x17;
   menu.options[BUTTON_CANCEL].move.left = BUTTON_OK;
   menu.options[BUTTON_CANCEL].move.right = BUTTON_OK;
   menu.options[BUTTON_CANCEL].move.oneswitch = BUTTON_OK;
   menu.options[BUTTON_CANCEL].action.accept = button_oneswitch_no;

   // OK button
   menu.options[BUTTON_OK].box.x1 = WARN_BX2;
   menu.options[BUTTON_OK].box.y1 = WARN_BY;
   menu.options[BUTTON_OK].box.x2 = WARN_BX2 + 0x5F;
   menu.options[BUTTON_OK].box.y2 = WARN_BY + 0x17;
   menu.options[BUTTON_OK].move.left = BUTTON_CANCEL;
   menu.options[BUTTON_OK].move.right = BUTTON_CANCEL;
   menu.options[BUTTON_OK].move.oneswitch = BUTTON_CANCEL;
   menu.options[BUTTON_OK].action.accept = button_oneswitch_yes;
}
*/

//***************************************************************************
// draw_warning
// Draws the warning when trying to toggle one-switch mode.
//***************************************************************************

/*
static void draw_warning(void) {
   // Determine which message to show
   const char **ptr = settings.one_switch ?
      text.options_gameplay.onesw_off :
      text.options_gameplay.onesw_on;

   // Draw title
   draw_text(text.options_gameplay.warning, screen_cx, WARN_TY,
      FONT_LIT, ALIGN_CENTER);

   // Show message
   for (unsigned i = 0; i < 8; i++)
      draw_text(ptr[i], screen_cx, WARN_Y + i*0x10, FONT_LIT, ALIGN_CENTER);

   // Draw labels
   draw_text(text.options_gameplay.cancel, WARN_BCX1, WARN_BCY,
      menu.selected == BUTTON_CANCEL ? FONT_LIT : FONT_DIM, ALIGN_CENTER);
   draw_text(text.options_gameplay.ok, WARN_BCX2, WARN_BCY,
      menu.selected == BUTTON_OK ? FONT_LIT : FONT_DIM, ALIGN_CENTER);
}
*/

//***************************************************************************
// button_oneswitch_no [internal]
// Button for cancelling toggling one-switch mode.
//***************************************************************************

/*
static void button_oneswitch_no(void) {
   // Turn off warning
   warning = 0;
   set_real_menu();
   menu.selected = BUTTON_ONESWITCH;
}
*/

//***************************************************************************
// button_oneswitch_yes [internal]
// Button for accepting toggling one-switch mode.
//***************************************************************************

/*
static void button_oneswitch_yes(void) {
   // Toggle one-switch mode
   settings.one_switch ^= 1;

   // Turn off warning
   warning = 0;
   set_real_menu();
   menu.selected = BUTTON_ONESWITCH;
}
*/
