//***************************************************************************
// "options.c"
// Code for the options main menu.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include "main.h"
#include "background.h"
#include "input.h"
#include "loading.h"
#include "menu.h"
#include "options_gameplay.h"
#include "options_input.h"
#include "options_mouse.h"
#include "options_sound.h"
#include "options_video.h"
#include "reader.h"
#include "settings.h"
#include "sound.h"
#include "text.h"
#include "video.h"

// First and last options submenu game modes
#define MIN_OPT_MODE    GAMEMODE_OPT_GAMEPLAY
#define MAX_OPT_MODE    GAMEMODE_OPT_MOUSE

// List of option screens to go
enum {
   BUTTON_GAMEPLAY,        // Change gameplay settings
   BUTTON_VIDEO,           // Change video settings
   BUTTON_SOUND,           // Change sound settings
   BUTTON_KEYBOARD,        // Change keyboard controls
   BUTTON_JOYSTICK,        // Change joystick controls
   BUTTON_MOUSE,           // Change mouse controls
   BUTTON_BACK,            // Return to title screen
   NUM_BUTTONS
};

// Button placement (based off the center)
#define BUTTON_X1A   (screen_cx - 108)       // Top left
#define BUTTON_X2A   (screen_cx - 32)        // Top middle
#define BUTTON_X3A   (screen_cx + 44)        // Top right

#define BUTTON_X1B   (screen_cx - 146)       // Bottom left
#define BUTTON_X2B   (screen_cx - 70)        // Bottom middle left
#define BUTTON_X3B   (screen_cx + 6)         // Bottom middle right
#define BUTTON_X4B   (screen_cx + 82)        // Bottom right

#define BUTTON_Y1    (screen_cy - 54)        // Top row
#define BUTTON_Y2    (screen_cy + 6)         // Bottom row

// Where graphics are stored
static GraphicsSet *gfxset_menu = NULL;
static GraphicsSet *gfxset_options = NULL;
static Sprite *spr_buttons[NUM_BUTTONS][4];

// Used to animate the buttons
static unsigned menu_anim;

// Private function prototypes
static void init_options_menu(void);
static void load_options(void);
static void button_gameplay(void);
static void button_video(void);
static void button_sound(void);
static void button_keyboard(void);
static void button_joystick(void);
static void button_mouse(void);
static void button_back(void);

//***************************************************************************
// init_options
// Initializes the options main menu.
//***************************************************************************

void init_options(void) {
   // Load assets
   if (prev_game_mode < MIN_OPT_MODE ||
   prev_game_mode > MAX_OPT_MODE) {
      loading_screen(load_options);

      // Start playing music
      play_bgm(BGM_MENU);
   }

#define SPR(button, name) \
spr_buttons[button][0] = get_sprite(gfxset_options, "button_"name"_dim"); \
spr_buttons[button][1] = get_sprite(gfxset_options, "button_"name"_lit_1"); \
spr_buttons[button][2] = get_sprite(gfxset_options, "button_"name"_lit_2"); \
spr_buttons[button][3] = get_sprite(gfxset_options, "button_"name"_lit_3");
   SPR(BUTTON_GAMEPLAY, "gameplay");
   SPR(BUTTON_VIDEO, "video");
   SPR(BUTTON_SOUND, "sound");
   SPR(BUTTON_KEYBOARD, "keyboard");
   SPR(BUTTON_JOYSTICK, "joystick");
   SPR(BUTTON_MOUSE, "mouse");
   SPR(BUTTON_BACK, "back");
#undef SPR

   // Initialize menu data
   init_options_menu();

   // Make the cursor visible
   set_cursor(CURSOR_ARROW);

   // Make screen visible
   fade_on();
}

//***************************************************************************
// init_options_menu [internal]
// Sets up the menu in this screen.
//***************************************************************************

static void init_options_menu(void) {
   init_menu();
   set_reinit_menu_func(init_options_menu);
   menu_anim = 0;

   menu.defoption.up = BUTTON_KEYBOARD;
   menu.defoption.down = BUTTON_GAMEPLAY;
   menu.defoption.left = BUTTON_SOUND;
   menu.defoption.right = BUTTON_GAMEPLAY;
   menu.cancel = button_back;

   menu.options[BUTTON_GAMEPLAY].box.x1 = BUTTON_X1A;
   menu.options[BUTTON_GAMEPLAY].box.y1 = BUTTON_Y1;
   menu.options[BUTTON_GAMEPLAY].box.x2 = BUTTON_X1A + 0x3F;
   menu.options[BUTTON_GAMEPLAY].box.y2 = BUTTON_Y1 + 0x2F;
   menu.options[BUTTON_GAMEPLAY].move.up = BUTTON_KEYBOARD;
   menu.options[BUTTON_GAMEPLAY].move.down = BUTTON_KEYBOARD;
   menu.options[BUTTON_GAMEPLAY].move.left = BUTTON_SOUND;
   menu.options[BUTTON_GAMEPLAY].move.right = BUTTON_VIDEO;
   menu.options[BUTTON_GAMEPLAY].move.oneswitch = BUTTON_VIDEO;
   menu.options[BUTTON_GAMEPLAY].action.accept = button_gameplay;
   menu.options[BUTTON_GAMEPLAY].caption = text.options.gameplay;

   //if (!settings.audiovideo) {
      menu.options[BUTTON_VIDEO].box.x1 = BUTTON_X2A;
      menu.options[BUTTON_VIDEO].box.y1 = BUTTON_Y1;
      menu.options[BUTTON_VIDEO].box.x2 = BUTTON_X2A + 0x3F;
      menu.options[BUTTON_VIDEO].box.y2 = BUTTON_Y1 + 0x2F;
      menu.options[BUTTON_VIDEO].move.up = BUTTON_JOYSTICK;
      menu.options[BUTTON_VIDEO].move.down = BUTTON_JOYSTICK;
      menu.options[BUTTON_VIDEO].move.left = BUTTON_GAMEPLAY;
      menu.options[BUTTON_VIDEO].move.right = BUTTON_SOUND;
      menu.options[BUTTON_VIDEO].move.oneswitch = BUTTON_SOUND;
      menu.options[BUTTON_VIDEO].action.accept = button_video;
      menu.options[BUTTON_VIDEO].caption = text.options.video;
   //}

   menu.options[BUTTON_SOUND].box.x1 = BUTTON_X3A;
   menu.options[BUTTON_SOUND].box.y1 = BUTTON_Y1;
   menu.options[BUTTON_SOUND].box.x2 = BUTTON_X3A + 0x3F;
   menu.options[BUTTON_SOUND].box.y2 = BUTTON_Y1 + 0x2F;
   menu.options[BUTTON_SOUND].move.up = BUTTON_MOUSE;
   menu.options[BUTTON_SOUND].move.down = BUTTON_MOUSE;
   menu.options[BUTTON_SOUND].move.left = BUTTON_VIDEO;
   menu.options[BUTTON_SOUND].move.right = BUTTON_GAMEPLAY;
   menu.options[BUTTON_SOUND].move.oneswitch = BUTTON_KEYBOARD;
   menu.options[BUTTON_SOUND].action.accept = button_sound;
   menu.options[BUTTON_SOUND].caption = text.options.sound;

   menu.options[BUTTON_KEYBOARD].box.x1 = BUTTON_X1B;
   menu.options[BUTTON_KEYBOARD].box.y1 = BUTTON_Y2;
   menu.options[BUTTON_KEYBOARD].box.x2 = BUTTON_X1B + 0x3F;
   menu.options[BUTTON_KEYBOARD].box.y2 = BUTTON_Y2 + 0x2F;
   menu.options[BUTTON_KEYBOARD].move.up = BUTTON_GAMEPLAY;
   menu.options[BUTTON_KEYBOARD].move.down = BUTTON_GAMEPLAY;
   menu.options[BUTTON_KEYBOARD].move.left = BUTTON_BACK;
   menu.options[BUTTON_KEYBOARD].move.right = BUTTON_JOYSTICK;
   menu.options[BUTTON_KEYBOARD].move.oneswitch = BUTTON_JOYSTICK;
   menu.options[BUTTON_KEYBOARD].action.accept = button_keyboard;
   menu.options[BUTTON_KEYBOARD].caption = text.options.keyboard;

   menu.options[BUTTON_JOYSTICK].box.x1 = BUTTON_X2B;
   menu.options[BUTTON_JOYSTICK].box.y1 = BUTTON_Y2;
   menu.options[BUTTON_JOYSTICK].box.x2 = BUTTON_X2B + 0x3F;
   menu.options[BUTTON_JOYSTICK].box.y2 = BUTTON_Y2 + 0x2F;
   menu.options[BUTTON_JOYSTICK].move.up = BUTTON_VIDEO;
   menu.options[BUTTON_JOYSTICK].move.down = BUTTON_VIDEO;
   menu.options[BUTTON_JOYSTICK].move.left = BUTTON_KEYBOARD;
   menu.options[BUTTON_JOYSTICK].move.right = BUTTON_MOUSE;
   menu.options[BUTTON_JOYSTICK].move.oneswitch = BUTTON_MOUSE;
   menu.options[BUTTON_JOYSTICK].action.accept = button_joystick;
   menu.options[BUTTON_JOYSTICK].caption = text.options.joystick;

   menu.options[BUTTON_MOUSE].box.x1 = BUTTON_X3B;
   menu.options[BUTTON_MOUSE].box.y1 = BUTTON_Y2;
   menu.options[BUTTON_MOUSE].box.x2 = BUTTON_X3B + 0x3F;
   menu.options[BUTTON_MOUSE].box.y2 = BUTTON_Y2 + 0x2F;
   menu.options[BUTTON_MOUSE].move.up = BUTTON_SOUND;
   menu.options[BUTTON_MOUSE].move.down = BUTTON_SOUND;
   menu.options[BUTTON_MOUSE].move.left = BUTTON_JOYSTICK;
   menu.options[BUTTON_MOUSE].move.right = BUTTON_BACK;
   menu.options[BUTTON_MOUSE].move.oneswitch = BUTTON_BACK;
   menu.options[BUTTON_MOUSE].action.accept = button_mouse;
   menu.options[BUTTON_MOUSE].caption = text.options.mouse;

   menu.options[BUTTON_BACK].box.x1 = BUTTON_X4B;
   menu.options[BUTTON_BACK].box.y1 = BUTTON_Y2;
   menu.options[BUTTON_BACK].box.x2 = BUTTON_X4B + 0x3F;
   menu.options[BUTTON_BACK].box.y2 = BUTTON_Y2 + 0x2F;
   menu.options[BUTTON_BACK].move.up = BUTTON_SOUND;
   menu.options[BUTTON_BACK].move.down = BUTTON_SOUND;
   menu.options[BUTTON_BACK].move.left = BUTTON_MOUSE;
   menu.options[BUTTON_BACK].move.right = BUTTON_KEYBOARD;
   menu.options[BUTTON_BACK].move.oneswitch = BUTTON_GAMEPLAY;
   menu.options[BUTTON_BACK].action.accept = button_back;
   menu.options[BUTTON_BACK].caption = text.options.back;
   menu.options[BUTTON_BACK].sfx = SFX_CANCEL;
}

//***************************************************************************
// load_options [internal]
// Loads the assets for the options main menu. Run during the loading screen.
//***************************************************************************

#define NUM_LOAD 9
static void load_options(void) {
   gfxset_menu = load_graphics_set("graphics/menus");
   set_loading_total(1, NUM_LOAD);
   gfxset_options = load_graphics_set("graphics/options");
   set_loading_total(2, NUM_LOAD);
   load_background("graphics/menus/background", gfxset_menu);
   set_loading_total(3, NUM_LOAD);
   load_bgm(BGM_MENU);
   set_loading_total(4, NUM_LOAD);
   load_options_gameplay();
   set_loading_total(5, NUM_LOAD);
   load_options_video();
   set_loading_total(6, NUM_LOAD);
   load_options_sound();
   set_loading_total(7, NUM_LOAD);
   load_options_input();
   set_loading_total(8, NUM_LOAD);
   load_options_mouse();
   set_loading_total(9, NUM_LOAD);
}
#undef NUM_LOAD

//***************************************************************************
// run_options
// Processes the logic for the options main menu.
//***************************************************************************

void run_options(void) {
   // Update the menu
   update_menu();
   menu_anim++;

   // Update background animation
   update_background();
}

//***************************************************************************
// draw_options
// Draws the options main menu.
//***************************************************************************

void draw_options(void) {
   // Draw background
   draw_background();

   // Draw all buttons
   for (unsigned i = 0; i < NUM_BUTTONS; i++) {
      // Determine sprite to use
      const Sprite *sprite;
      if (menu.selected != (signed) i)
         sprite = spr_buttons[i][0];
      else switch (menu_anim >> 3 & 0x03) {
         case 0x00: sprite = spr_buttons[i][1]; break;
         case 0x01: sprite = spr_buttons[i][2]; break;
         case 0x02: sprite = spr_buttons[i][3]; break;
         case 0x03: sprite = spr_buttons[i][2]; break;
      }

      // Draw button
      draw_sprite(sprite, menu.options[i].box.x1,
         menu.options[i].box.y1, SPR_NOFLIP);
   }

   // If an option is selected, show a border around it as well as its
   // caption above all the buttons
   if (menu.selected != -1) {
      // Where the data for the selected option is stored
      MenuOption *ptr = &menu.options[menu.selected];

      // Draw the border on top of the button
      if (menu_anim & 0x08) {
         draw_rectangle(ptr->box.x1+1, ptr->box.y1+1,
            ptr->box.x2-1, ptr->box.y2-1, settings.box_lit[1]);
         draw_rectangle(ptr->box.x1, ptr->box.y1,
            ptr->box.x2, ptr->box.y2, settings.box_lit[0]);
         draw_rectangle(ptr->box.x1-1, ptr->box.y1-1,
            ptr->box.x2+1, ptr->box.y2+1, settings.box_lit[0]);
         draw_rectangle(ptr->box.x1-2, ptr->box.y1-2,
            ptr->box.x2+2, ptr->box.y2+2, settings.box_lit[1]);
         draw_hline(ptr->box.x1-2, ptr->box.y2+3,
            ptr->box.x2+2, settings.box_lit[2]);
      }

      // Draw the caption
      draw_text(ptr->caption, screen_cx, screen_h - 0x10, FONT_LIT,
         ALIGN_BOTTOM);
      set_reader_text(ptr->caption);
   }

   // Draw title
   draw_text(text.options.title, screen_cx, 0x10, FONT_LIT, ALIGN_TOP);
}

//***************************************************************************
// deinit_options
// Deinitializes the options main menu.
//***************************************************************************

void deinit_options(void) {
   // Unload stuff ONLY if we're not going to one of the submenus
   if (next_game_mode < MIN_OPT_MODE ||
   next_game_mode > MAX_OPT_MODE) {
      // Unload music
      unload_all_bgm();

      // Unload data for submenus
      unload_options_mouse();
      unload_options_input();
      unload_options_sound();
      unload_options_video();
      unload_options_gameplay();

      // Unload graphics
      unload_background();
      destroy_graphics_set(gfxset_menu);
   }

   // Save settings
   save_config();
}

//***************************************************************************
// button_gameplay [internal]
// Function for the "Gameplay" button. Goes to the gameplay menu.
//***************************************************************************

static void button_gameplay(void) {
   // Go to the gameplay menu
   fade_off_and_switch(GAMEMODE_OPT_GAMEPLAY);
}

//***************************************************************************
// button_video [internal]
// Function for the "Video" button. Goes to the video menu.
//***************************************************************************

static void button_video(void) {
   // Go to the video menu
   fade_off_and_switch(GAMEMODE_OPT_VIDEO);
}

//***************************************************************************
// button_sound [internal]
// Function for the "Sound" button. Goes to the sound menu.
//***************************************************************************

static void button_sound(void) {
   // Go to the sound menu
   fade_off_and_switch(GAMEMODE_OPT_SOUND);
}

//***************************************************************************
// button_keyboard [internal]
// Function for the "Keyboard" button. Goes to the keyboard menu.
//***************************************************************************

static void button_keyboard(void) {
   // Go to the keyboard menu
   fade_off_and_switch(GAMEMODE_OPT_KEYBOARD);
}

//***************************************************************************
// button_joystick [internal]
// Function for the "Joystick" button. Goes to the joystick menu.
//***************************************************************************

static void button_joystick(void) {
   // Go to the joystick menu
   fade_off_and_switch(GAMEMODE_OPT_JOYSTICK);
}

//***************************************************************************
// button_mouse [internal]
// Function for the "Mouse" button. Goes to the mouse menu.
//***************************************************************************

static void button_mouse(void) {
   // Go to the mouse menu
   fade_off_and_switch(GAMEMODE_OPT_MOUSE);
}

//***************************************************************************
// button_back [internal]
// Function for the "Back" button. Returns back to the title screen.
//***************************************************************************

static void button_back(void) {
   // Return back to the title screen
   fade_off_and_switch(GAMEMODE_TITLE);
}
