//***************************************************************************
// "objects.h"
// Definitions for objects.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

#ifndef OBJECTS_H
#define OBJECTS_H

// Required headers
#include <stdint.h>
#include "video.h"

// List of possible object groups
// The order of the groups is important since it determines how they're drawn
// on screen. Groups later in the list will be drawn above groups earlier in
// the list.
typedef enum {
   OBJGROUP_BOSS6_L1,         // Mr. Evil, layer 1 (machine)
   OBJGROUP_BOSS6_L2,         // Mr. Evil, layer 2 (incoming missiles)
   OBJGROUP_SCENERY,          // Standard scenery
   OBJGROUP_PLATFORM,         // Platforms, pushables, etc.
   OBJGROUP_HAZARD,           // Hazard (behind other objects)
   OBJGROUP_ITEM,             // Miscellaneous items
   OBJGROUP_PROJECTILE,       // Enemy projectiles
   OBJGROUP_BOSS,             // Bosses (active)
   OBJGROUP_ENEMY,            // Enemies
   OBJGROUP_PARTICLE,         // Dust, scrap, etc.
   OBJGROUP_POWERUP,          // Players' power-up attachments
   OBJGROUP_SONAR,            // Player sonar (audiovideo only)
   OBJGROUP_PLAYER,           // Players
   OBJGROUP_SHIELD,           // Players' shields
   OBJGROUP_HIPARTICLE,       // Same as PARTICLE but on a higher layer
   OBJGROUP_HIHAZARD,         // Hazard (in front of other objects)
   OBJGROUP_BOSS6_L3,         // Mr. Evil, layer 3 (outgoing missiles)
   NUM_OBJGROUPS              // Number of object groups
} ObjGroup;

// List of possible object types
typedef enum {
   OBJ_DUMMY,                 // Dummy object

   OBJ_PLAYER,                // Player object (no power-up)
   OBJ_PLAYERSPIDER,          // Player object (spider power-up)
   OBJ_PLAYERHAMMER,          // Player object (hammer power-up)
   OBJ_PLAYERWINGS,           // Player object (wings power-up)
   OBJ_PLAYERPARASOL,         // Player object (parasol power-up)

   OBJ_POWERUP,               // Power-up attachment (for players)

   OBJ_FLAMER,                // Flamer enemy
   OBJ_SPRAYER,               // Sprayer enemy
   OBJ_TURRET,                // Turret enemy
   OBJ_ROLLER,                // Roller enemy
   OBJ_GRABBER,               // Grabber enemy
   OBJ_SPIDER,                // Spider enemy
   OBJ_HEATER,                // Heater enemy
   OBJ_BOMB,                  // Bomb enemy

   OBJ_BOSS1,                 // Bulldozer boss
   OBJ_BOSS2,                 // Trash truck boss
   OBJ_BOSS3,                 // Helicopter boss
   OBJ_BOSS4,                 // Oil truck boss
   OBJ_BOSS5,                 // Driller boss
   OBJ_BOSS6,                 // Mr. Evil machine

   OBJ_BOSS1_SPAWN,           // Bulldozer boss spawn point
   OBJ_BOSS2_SPAWN,           // Trash truck boss spawn point
   OBJ_BOSS3_SPAWN,           // Helicopter boss spawn point
   OBJ_BOSS4_SPAWN,           // Oil truck boss spawn point
   OBJ_BOSS5_SPAWN,           // Driller boss spawn point
   OBJ_BOSS6_SPAWN,           // Mr. Evil spawn point

   OBJ_DRIVER,                // Driver fleeing when a boss explodes

   OBJ_SHIP,                  // Ship itself
   OBJ_SHIPLOCHUNK,           // Ship boss chunk (behind stuff)
   OBJ_SHIPHICHUNK,           // Ship boss chunk (in front of stuff)
   OBJ_GRENADE,               // Helicopter grenade
   OBJ_TRAILER,               // Oil truck boss trailer

   OBJ_B6_BASE,               // Base of Mr. Evil's machine
   OBJ_B6_FLAME,              // Flame coming from Mr. Evil's machine
   OBJ_B6_LAUNCHER,           // Missile launchers of Mr. Evil's machine
   OBJ_B6_HAND,               // Hands of Mr. Evil's machine
   OBJ_B6_EVIL,               // Mr. Evil himself
   OBJ_MISSILE_IN,            // Incoming missile
   OBJ_MISSILE_OUT,           // Outgoing missile

   OBJ_MORTAR,                // Generic mortar
   OBJ_TOXICGAS,              // Toxic gas cloud
   OBJ_FIREBALL,              // Fireball
   OBJ_EXPLOSION,             // Small explosion
   OBJ_BIGEXPLOSION,          // Big explosion

   OBJ_SCRAPGEAR,             // Scrap (gear)
   OBJ_SCRAPSPRING,           // Scrap (spring)
   OBJ_SMOKE,                 // Smoke
   OBJ_ENDEXPLOSION,          // Factory explosion

   OBJ_SPRING,                // Spring
   OBJ_SSPRING,               // Superspring
   OBJ_BALLOON,               // Balloon
   OBJ_PUSHABLE,              // Pushable
   OBJ_PLATFORM,              // Idle platform
   OBJ_PLATFORM_H,            // Horizontal moving platform
   OBJ_PLATFORM_V,            // Vertical moving platform
   OBJ_PLATFORM_B,            // Breakable platform
   OBJ_PLATFORMLEFT,          // Platform leftover
   OBJ_OBSTRUCTION,           // Obstruction
   OBJ_OBSTRUCTLEFT,          // Obstruction leftover
   OBJ_DOOR_1,                // Door #1
   OBJ_DOOR_2,                // Door #2
   OBJ_DOOR_3,                // Door #3
   OBJ_DOOR_4,                // Door #4
   OBJ_DOOR_5,                // Door #5
   OBJ_DOOR_6,                // Door #6
   OBJ_DOOR_7,                // Door #7
   OBJ_DOOR_8,                // Door #8
   OBJ_GENERATOR,             // Generator
   OBJ_MAINGENERATOR,         // Main generator

   OBJ_HEART,                 // Heart
   OBJ_INVINCIBILITY,         // Invincibility
   OBJ_SPARKLE,               // Invincibility sparkle
   OBJ_FALLSPARKLE,           // Falling sparkle
   OBJ_CHECKPOINT,            // Checkpoint
   OBJ_GOAL,                  // Goal
   OBJ_BLUESTAR,              // Blue star
   OBJ_SHIELD,                // Shield
   OBJ_SHIELD_ITEM,           // Shield (item)
   OBJ_WINGS_ITEM,            // Wings power-up (item)
   OBJ_SPIDER_ITEM,           // Spider power-up (item)
   OBJ_HAMMER_ITEM,           // Hammer power-up (item)
   OBJ_PARASOL_ITEM,          // Parasol power-up (item)

   OBJ_SWITCH_1,              // Switch #1
   OBJ_SWITCH_2,              // Switch #2
   OBJ_SWITCH_3,              // Switch #3
   OBJ_SWITCH_4,              // Switch #4
   OBJ_SWITCH_5,              // Switch #5
   OBJ_SWITCH_6,              // Switch #6
   OBJ_SWITCH_7,              // Switch #7
   OBJ_SWITCH_8,              // Switch #8

   OBJ_SPIKES_U,              // Spikes pointing upwards
   OBJ_SPIKES_D,              // Spikes pointing downwards
   OBJ_SPIKES_L,              // Spikes pointing left
   OBJ_SPIKES_R,              // Spikes pointing right
   OBJ_CRUSHER,               // Crushers
   OBJ_LIQUIDHAZARD,          // Liquid hazard
   OBJ_BOUNCINGHAZARD,        // Bouncing hazard
   OBJ_FIRE,                  // Fire
   OBJ_STALACTITE,            // Stalactite
   OBJ_COIL_F,                // Coil (floor)
   OBJ_COIL_C,                // Coil (ceiling)
   OBJ_BUZZSAW,               // Buzzsaw (static)
   OBJ_BUZZSAW_C,             // Buzzsaw (ceiling)
   OBJ_BUZZSAW_F,             // Buzzsaw (floor)

   OBJ_SCENERYSMALL,          // Scenery (small)
   OBJ_SCENERYBIG,            // Scenery (big)
   OBJ_DANCER,                // Dancer :P

   OBJ_WARNING,               // Warning sign
   OBJ_HSONAR,                // Player horizontal sonar (audiovideo only)
   OBJ_VSONAR,                // Player vertical sonar (audiovideo only)

   NUM_OBJTYPES               // Number of object types
} ObjType;

// Definition of a hitbox
typedef struct {
   int32_t x1;                // Left boundary
   int32_t x2;                // Right boundary
   int32_t y1;                // Top boundary
   int32_t y2;                // Bottom boundary
} HitBox;

// Definition of an object
typedef struct Object {
   ObjType type;              // What kind of object is this
   ObjGroup group;            // Object group to which it belongs
   struct Object *next;       // Next object in the list
   struct Object *prev;       // Previous object in the list
   struct Object *otherobj;   // Pointer to another object (custom use)

   int32_t x;                 // Horizontal coordinate
   int32_t y;                 // Vertical coordinate
   int32_t speed;             // Horizontal momentum
   int32_t gravity;           // Vertical momentum
   int32_t base_x;            // Base horizontal coordinate for some actions
   int32_t base_y;            // Base vertical coordinate for some actions
   uint16_t timer;            // Timer for behavior control and such
   uint16_t invincibility;    // How long is the object invincible
   uint16_t invulnerability;  // How long is the object invulnerable (flash)
   unsigned health;           // Remaining health points

   unsigned dir: 1;           // 0 = facing right, 1 = facing left
   unsigned crouching: 1;     // Set when crouching
   unsigned jumping: 1;       // Set when jumping
   unsigned active: 1;        // Some generic active flag for objects
   unsigned hurt: 1;          // Set when getting hurt
   unsigned dead: 1;          // Set when object is dying
   unsigned shield: 1;        // Set if object has a shield

   unsigned pinball: 1;       // Set when pinballing

   unsigned blocked: 1;       // Set when colliding with the tilemap
   unsigned on_ground: 1;     // Set when stepping on something
   unsigned on_cliff: 1;      // Set when stepping on a cliff
   unsigned on_cliff_l: 1;    // Set only for cliffs on the left
   unsigned on_cliff_r: 1;    // Set only for cliffs on the right
   unsigned on_wall: 1;       // Set when just hit a wall
   unsigned on_ceiling: 1;    // Set when just hit the ceiling
   unsigned has_hitbox: 1;    // Set if it has a hit box set
   unsigned current: 1;       // Set for the object being currently processed

   const AnimFrame *frame;    // Current animation frame
   unsigned frame_time;       // Time remaining until next frame
   const AnimFrame *last_anim;// Last animation used by the object

   HitBox rel_hitbox;         // Hitbox relative to object coordinates
   HitBox abs_hitbox;         // Hitbox in absolute world coordinates
} Object;

// Possible animations for miscellaneous objects
typedef enum {
   OB_ANIM_SCRAPGEAR,      // Scrap (gear)
   OB_ANIM_SCRAPSPRING,    // Scrap (spring)
   OB_ANIM_MORTAR,         // Mortar
   OB_ANIM_TOXICGAS,       // Toxic gas
   OB_ANIM_FIREBALL,       // Fireball
   OB_ANIM_EXPLOSION,      // Small explosion
   OB_ANIM_BIGEXPLOSION,   // Big explosion
   OB_ANIM_SMOKE,          // Smoke

   OB_ANIM_SPRINGIDLE,     // Spring (idle)
   OB_ANIM_SPRINGWORK,     // Spring (working)
   OB_ANIM_SSPRINGIDLE,    // Superspring (idle)
   OB_ANIM_SSPRINGWORK,    // Superspring (working)

   OB_ANIM_HEARTIDLE,      // Heart (idle)
   OB_ANIM_HEARTTAKEN,     // Heart (taken)
   OB_ANIM_INVINCIBILITY,  // Invincibility
   OB_ANIM_SPARKLE,        // Invincibility sparkle
   OB_ANIM_CHECKOFF,       // Checkpoint (off)
   OB_ANIM_CHECKON,        // Checkpoint (on)
   OB_ANIM_GOALOFF,        // Goal (off)
   OB_ANIM_GOALON,         // Goal (on)
   OB_ANIM_BLUESTAR,       // Blue star
   OB_ANIM_SHIELD,         // Shield
   OB_ANIM_SHIELD_ITEM,    // Shield (item)

   OB_ANIM_WINGS,          // Wings power-up
   OB_ANIM_WINGS_ITEM,     // Wings power-up (item)
   OB_ANIM_SPIDER,         // Spider power-up
   OB_ANIM_SPIDER_ITEM,    // Spider power-up (item)
   OB_ANIM_HAMMER,         // Hammer power-up
   OB_ANIM_HAMMER_ITEM,    // Hammer power-up (item)
   OB_ANIM_PARASOL,        // Parasol power-up
   OB_ANIM_PARASOL_ITEM,   // Parasol power-up (item)

   OB_ANIM_SWITCH_OFF,     // Switch turned off
   OB_ANIM_SWITCH_ON,      // Switch turned on

   OB_ANIM_GENERATOR,      // Generator
   OB_ANIM_MAINGENERATOR,  // Main generator

   OB_ANIM_SPIKES_U,       // Spikes pointing upwards
   OB_ANIM_SPIKES_D,       // Spikes pointing downwards
   OB_ANIM_SPIKES_L,       // Spikes pointing left
   OB_ANIM_SPIKES_R,       // Spikes pointing right
   OB_ANIM_COIL_F_IDLE,    // Coil idle (floor)
   OB_ANIM_COIL_F_READY,   // Coil ready (floor)
   OB_ANIM_COIL_F_ACTIVE,  // Coil active (floor)
   OB_ANIM_COIL_C_IDLE,    // Coil idle (ceiling)
   OB_ANIM_COIL_C_READY,   // Coil ready (ceiling)
   OB_ANIM_COIL_C_ACTIVE,  // Coil active (ceiling)
   OB_ANIM_BUZZSAW,        // Buzzsaw (static)
   OB_ANIM_BUZZSAW_C,      // Buzzsaw (ceiling)
   OB_ANIM_BUZZSAW_F,      // Buzzsaw (floor)

   OB_ANIM_DANCERWAIT,     // Dancer (waiting)
   OB_ANIM_DANCERDANCE,    // Dancer (dancing :P)

   OB_ANIM_WARNING,        // Warning sign

   NUM_OB_ANIM             // Number of animations
} ObjectAnim;

// Function prototypes
void load_objects(void);
void unload_objects(void);
void init_objects(void);
Object *add_object(ObjType, int32_t, int32_t, int);
Object *get_first_object(ObjGroup);
void run_objects(void);
void draw_objects(void);
void draw_far_objects(void);
void draw_near_objects(void);
void draw_single_object(const Object *);
void delete_object(Object *);
void deinit_objects(void);
void set_object_anim(Object *, const AnimFrame *);
const AnimFrame *retrieve_object_anim(ObjectAnim);
int is_too_far(const Object *);

#endif
