//***************************************************************************
// "menu.h"
// Code for handling all menu interfaces
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

#ifndef MENU_H
#define MENU_H

// Required headers
#include <stdint.h>
#include "sound.h"

// Data for a menu option
typedef struct {
   // Collision box
   struct {
      int32_t x1, y1;      // Top-left coordinates
      int32_t x2, y2;      // Bottom-right coordinates
   } box;

   // What option to select when using the cursors
   // (if a function is assigned, that overrides this)
   struct {
      int8_t up;           // Option to select when pressing up
      int8_t down;         // Option to select when pressing down
      int8_t left;         // Option to select when pressing left
      int8_t right;        // Option to select when pressing right
      int8_t oneswitch;    // Next option in one-switch mode
   } move;

   // If we want to call a function when certain button is pressed, then
   // assign it here.
   struct {
      void (*up)(void);          // Action for pressing up
      void (*down)(void);        // Action for pressing down
      void (*left)(void);        // Action for pressing left
      void (*right)(void);       // Action for pressing right
      void (*accept)(void);      // Action for accepting
      void (*oneswitch)(void);   // Action for accepting (one-switch mode)
   } action;

   // Some text attached to the option (usually the option name)
   const char *caption;

   // Sound effect they play when triggered
   // Can be SFX_NONE to do nothing instead
   SfxID sfx;

   // Some other properties
   unsigned disabled: 1;         // Option can't be selected
} MenuOption;

// Where the menu data is stored
#define MAX_MENU 32
typedef struct {
   // Description about each option
   MenuOption options[MAX_MENU];

   // Action to take when the player presses the cancel button
   void (*cancel)(void);

   // Currently selected option
   int8_t selected;

   // What option to select when pressing a cursor but nothing is currently
   // selected yet
   struct {
      int8_t up;           // Default option when pressing up
      int8_t down;         // Default option when pressing down
      int8_t left;         // Default option when pressing left
      int8_t right;        // Default option when pressing right
   } defoption;
} Menu;
extern Menu menu;

// Function prototypes
void init_menu(void);
void set_reinit_menu_func(void (*func)(void));
void reinit_menu(void);
void update_menu(void);

#endif
