//***************************************************************************
// "menu.c"
// Code for handling all menu interfaces
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include "input.h"
#include "menu.h"
#include "settings.h"
#include "sound.h"

// This should go somewhere :P
Menu menu;

// Function called when the menu needs to be reinitialized
// If this is NULL, then no function is called
static void (*reinit_func)(void) = NULL;

// Used to wait between options in one-switch mode
static unsigned oneswitch_timer;

//***************************************************************************
// init_menu
// Initializes (resets) the menu, leaving it empty and ready to build on.
// Call this then proceed to specify the data for each menu option.
//***************************************************************************

void init_menu(void) {
   // Reset all menu options
   for (unsigned i = 0; i < MAX_MENU; i++) {
      // To make our life easier :P
      MenuOption *ptr = &menu.options[i];

      // Make collision box unreachable to the cursor
      ptr->box.x1 = -1;
      ptr->box.y1 = -1;
      ptr->box.x2 = -1;
      ptr->box.y2 = -1;

      // Go nowhere when selecting using buttons
      ptr->move.up = -1;
      ptr->move.down = -1;
      ptr->move.left = -1;
      ptr->move.right = -1;
      ptr->move.oneswitch = -1;

      // Do nothing when using the buttons
      ptr->action.up = NULL;
      ptr->action.down = NULL;
      ptr->action.left = NULL;
      ptr->action.right = NULL;
      ptr->action.accept = NULL;
      ptr->action.oneswitch = NULL;

      // No text assigned
      ptr->caption = NULL;

      // Default sound effect
      ptr->sfx = SFX_OK;

      // Option can be selected
      ptr->disabled = 0;
   }

   // No cancel action by default
   menu.cancel = NULL;

   // Select the first option by default
   menu.selected = 0;

   // No default option for each cursor
   menu.defoption.up = -1;
   menu.defoption.down = -1;
   menu.defoption.left = -1;
   menu.defoption.right = -1;

   // Reset one-switch mode timer
   oneswitch_timer = settings.menu_delay;
}

//***************************************************************************
// set_reinit_menu_func, reinit_menu
// Used to reinitialize the menu when the resolution changes without warning
// (this can happen when pressing Alt+Enter, if this results in choosing a
// different ratio).
//***************************************************************************

void set_reinit_menu_func(void (*func)(void)) {
   reinit_func = func;
}

void reinit_menu(void) {
   if (reinit_func != NULL)
      reinit_func();
}

//***************************************************************************
// update_menu
// Updates the menu logic (checks user input to trigger menu actions, etc.).
//---------------------------------------------------------------------------
// Can't we get rid of some of the nesting somehow? x_x;
//***************************************************************************

void update_menu(void) {
   // One-switch controls?
   if (settings.one_switch) {
      // No cursor in one-switch mode
      set_cursor(CURSOR_NONE);

      // Change option every so often
      oneswitch_timer--;
      if (oneswitch_timer == 0) {
         // Reset timer
         oneswitch_timer = settings.menu_delay;

         // Move onto the next option
         if (menu.selected != -1)
            menu.selected = menu.options[menu.selected].move.oneswitch;
         else
            menu.selected = 0;
      }

      // Selecting the current option?
      if ((input.oneswitch.tap || input.oneswitch.tap2) &&
      menu.selected != -1) {
         const MenuOption *option = &menu.options[menu.selected];
         if (option->action.oneswitch != NULL) {
            play_sfx(option->sfx);
            option->action.oneswitch();
         } else if (option->action.accept != NULL) {
            play_sfx(option->sfx);
            option->action.accept();
         }

         // Reset timer again
         oneswitch_timer = settings.menu_delay;
      }

      // Done here...
      return;
   }

   // Audiovideo mode?
   if (settings.audiovideo) {
      // Select next option?
      if (input.menu.down) {
         play_sfx(SFX_BEEP);
         do {
            menu.selected++;
            if (menu.selected >= MAX_MENU)
               menu.selected = 0;
         } while (menu.options[menu.selected].box.x1 < 0);
      }

      // Select previous option?
      if (input.menu.up) {
         play_sfx(SFX_BEEP);
         do {
            if (menu.selected <= 0)
               menu.selected = MAX_MENU;
            menu.selected--;
         } while (menu.options[menu.selected].box.x1 < 0);
      }

      // Select previous value?
      if (input.menu.left && menu.selected != -1) {
         if (menu.options[menu.selected].action.left) {
            play_sfx(SFX_BEEP);
            menu.options[menu.selected].action.left();
         }
      }

      // Select next value?
      if (input.menu.right && menu.selected != -1) {
         if (menu.options[menu.selected].action.right) {
            play_sfx(SFX_BEEP);
            menu.options[menu.selected].action.right();
         }
      }

      // Trigger option?
      if (input.menu.accept && menu.selected != -1) {
         if (menu.options[menu.selected].action.accept) {
            play_sfx(menu.options[menu.selected].sfx);
            menu.options[menu.selected].action.accept();
         }
      }

      // Cancelled?
      if (input.menu.cancel && menu.cancel) {
         play_sfx(SFX_CANCEL);
         menu.cancel();
      }

      // Done here...
      return;
   }

   // Keep track of what option was selected before
   int old_selected = menu.selected;

   // Select option above?
   if (input.menu.up) {
      if (menu.selected != -1) {
         if (menu.options[menu.selected].action.up != NULL) {
            play_sfx(SFX_BEEP);
            menu.options[menu.selected].action.up();
         } else if (menu.options[menu.selected].move.up != -1) {
            if (!menu.options[menu.options[menu.selected].move.up].disabled)
               menu.selected = menu.options[menu.selected].move.up;
         }
      } else if (menu.defoption.up != -1) {
         menu.selected = menu.defoption.up;
      }
   }

   // Select option below?
   if (input.menu.down) {
      if (menu.selected != -1) {
         if (menu.options[menu.selected].action.down != NULL) {
            play_sfx(SFX_BEEP);
            menu.options[menu.selected].action.down();
         } else if (menu.options[menu.selected].move.down != -1)
            if (!menu.options[menu.options[menu.selected].move.down].disabled)
               menu.selected = menu.options[menu.selected].move.down;
      } else if (menu.defoption.down != -1) {
         menu.selected = menu.defoption.down;
      }
   }

   // Select option to the left?
   if (input.menu.left) {
      if (menu.selected != -1) {
         if (menu.options[menu.selected].action.left != NULL) {
            play_sfx(SFX_BEEP);
            menu.options[menu.selected].action.left();
         } else if (menu.options[menu.selected].move.left != -1)
            if (!menu.options[menu.options[menu.selected].move.left].disabled)
               menu.selected = menu.options[menu.selected].move.left;
      } else if (menu.defoption.left != -1) {
         menu.selected = menu.defoption.left;
      }
   }

   // Select option to the right?
   if (input.menu.right) {
      if (menu.selected != -1) {
         if (menu.options[menu.selected].action.right != NULL) {
            play_sfx(SFX_BEEP);
            menu.options[menu.selected].action.right();
         } else if (menu.options[menu.selected].move.right != -1)
            if (!menu.options[menu.options[menu.selected].move.right].disabled)
               menu.selected = menu.options[menu.selected].move.right;
      } else if (menu.defoption.right != -1) {
         menu.selected = menu.defoption.right;
      }
   }

   // Check if the cursor is hovering over something
   int hovering = -1;
   for (int i = 0; i < MAX_MENU; i++) {
      if (!menu.options[i].disabled &&
      input.cursor.x >= menu.options[i].box.x1 &&
      input.cursor.x <= menu.options[i].box.x2 &&
      input.cursor.y >= menu.options[i].box.y1 &&
      input.cursor.y <= menu.options[i].box.y2) {
         hovering = i;
         break;
      }
   }

   // If the cursor moved, make it select an option (if it's hovering over
   // any, otherwise select nothing)
   if (input.cursor.moved || input.cursor.click)
      menu.selected = input.cursor.inside ? hovering : -1;

   // Changed the option?
   if (old_selected != menu.selected && menu.selected != -1)
      play_sfx(SFX_BEEP);

   // Accepted an option?
   if ((input.menu.accept || input.cursor.click) &&
   menu.selected != -1 && menu.options[menu.selected].action.accept) {
      play_sfx(menu.options[menu.selected].sfx);
      menu.options[menu.selected].action.accept();
   }

   // Cancelled?
   if (input.menu.cancel && menu.cancel) {
      play_sfx(SFX_CANCEL);
      menu.cancel();
   }

   // Set cursor sprite accordingily depending on whether it's hovering over
   // a selectable option or not
   set_cursor(hovering != -1 ? CURSOR_HAND : CURSOR_ARROW);
}
