//***************************************************************************
// "main.h"
// Header with the stuff for the core functionality for the program.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

#ifndef MAIN_H
#define MAIN_H

// Don't use vanilla MinGW, seriously!
#ifdef __MINGW32__
#include <_mingw_mac.h>
#ifndef __MINGW64_VERSION_MAJOR
#error "Don't use vanilla MinGW, use MinGW-w64 instead!"
#endif
#endif

// Some game-specific information
#define GAME_NAME "Sol"       // Name shown in the titlebar
#define VER_MAJOR 1           // Major version
#define VER_MINOR 30          // Minor version (0..99)
#define VER_SUFFIX ""         // Version suffix ("" for no suffix)

// Define this to 0 to remove the launcher
// Use this if you don't want to use IUP
#define HAS_LAUNCHER 1

// Possible error codes
typedef enum {
   ERR_NONE,            // No error :)

   // These errors can happen before the language file is loaded, so we need
   // English translations built-in just in case they happen that early.
   ERR_NOMEMORY,        // Ran out of memory
   ERR_NOASSETS,        // Couldn't load assets
   ERR_LOADMOD,         // Couldn't load mod
   ERR_INITFS,          // Couldn't initialize file subsystem
   ERR_NOLANG,          // No languages found
   ERR_LOADLANG,        // Couldn't load language file

   // The language files should have been loaded by this point, so we don't
   // need built-in translations for these.
   ERR_INITSDL,         // Couldn't initialize SDL
   ERR_INITVIDEO,       // Couldn't initialize video subsystem
   ERR_UPDATEVIDEO,     // Couldn't update screen contents
   ERR_INITREADER,      // Couldn't initialize screen reader
   ERR_BADSAVEGAME,     // Invalidated savegame
   ERR_LOADFONT,        // Couldn't load font
   ERR_LOADSCENES,      // Couldn't load scene list
   ERR_NUMSCENES,       // Too many scenes
   ERR_LOADLEVEL,       // Couldn't load level (normal playthrough)
   ERR_LOADLEVELED,     // Couldn't load level (from editor)
   ERR_LOADCUTSCENE,    // Couldn't load cutscene
   ERR_LOADBG,          // Couldn't load background
   ERR_BADCOMMAND,      // Invalid command
   ERR_NUMARGS,         // Wrong number of arguments
   ERR_GFXSETID,        // Invalid graphics set ID
   ERR_ANIMID,          // Invalid animation ID
   ERR_ELEMENTID,       // Invalid element ID
   ERR_BGMID,           // Invalid background music ID
   ERR_SFXID,           // Invalid sound effect ID
   ERR_COORDVAL,        // Invalid coordinate value
   ERR_SPEEDVAL,        // Invalid speed value
   ERR_TIMERVAL,        // Invalid timer value
   ERR_VOICEID,         // Invalid voice ID
   ERR_FILTERTYPE,      // Invalid filter type
   ERR_FILELIST,        // Couldn't retrieve file list
   ERR_MEDIALIST,       // Couldn't retrieve media list

   // Huh...
   ERR_UNKNOWN,         // Unknown error
   NUM_ERRCODES         // Number of error codes
} ErrCode;

// Possible game modes
typedef enum {
   GAMEMODE_QUIT = -1,        // Quit program (NOT A REAL MODE)
   GAMEMODE_INGAME,           // In-game
   GAMEMODE_CUTSCENE,         // Cutscene
   GAMEMODE_EDITOR,           // Level editor
   GAMEMODE_TITLE,            // Title screen
   GAMEMODE_SAVESLOT,         // Select save slot screen
   GAMEMODE_OPTIONS,          // Options main menu
   GAMEMODE_OPT_GAMEPLAY,     // Options gameplay menu
   GAMEMODE_OPT_VIDEO,        // Options video menu
   GAMEMODE_OPT_SOUND,        // Options sound menu
   GAMEMODE_OPT_KEYBOARD,     // Options keyboard menu
   GAMEMODE_OPT_JOYSTICK,     // Options joystick menu
   GAMEMODE_OPT_MOUSE,        // Options mouse menu
   GAMEMODE_LOGO,             // Azura Sun logo
   GAMEMODE_LANGSELECT,       // Language select menu
   NUM_GAMEMODES              // Number of game modes
} GameMode;

// Information used to keep track of the game modes
extern GameMode game_mode;          // Currently running game mode
extern GameMode prev_game_mode;     // Game mode run before this one
extern GameMode next_game_mode;     // Game mode to switch to

// Function prototypes
unsigned update_program(void);
void switch_game_mode(GameMode);
void quit_program(void);
int run_program(const char *);
int run_program_with_param(const char *, const char *);

// Call this function if something is seriously wrong.
// This function does *not* return (it shows an error message if possible and
// then proceeds to quit the program immediately). When using GCC it will be
// labeled with the appropriate attribute so the compiler doesn't waste time
// generating return code that will never be used.
#ifdef __GNUC__
void __attribute__((__noreturn__))
abort_program(ErrCode, const char *);
#else
void abort_program(ErrCode, const char *);
#endif

#endif
