//***************************************************************************
// "logo.c"
// Skeleton code for the Azura Sun logo screen.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stdio.h>
#include "input.h"
#include "loading.h"
#include "reader.h"
#include "settings.h"
#include "sound.h"
#include "text.h"
#include "video.h"

// Where the graphics we use are stored
static GraphicsSet *gfxset_logo = NULL;
static const Sprite *spr_name;
static const Sprite *spr_sol[5];

// Positions for each sprite (X coordinate)
static int pos_name;
static int pos_sol;

// Used to animate the head
static int head_anim;

// Private function prototypes
static void load_logo(void);

//***************************************************************************
// init_logo
// Initializes the logo screen.
//***************************************************************************

void init_logo(void) {
   // Load assets
   loading_screen(load_logo);

   // Load logo
   char logo_name[0x40];
   sprintf(logo_name, "logo_%s", get_language_id());
   spr_name = get_sprite(gfxset_logo, logo_name);
   if (spr_name->width == 0 || spr_name->height == 0)
      spr_name = get_sprite(gfxset_logo, "logo");

   // Load Sol's head
#define SPR(x) get_sprite(gfxset_logo, x)
   spr_sol[0] = SPR("sol_1");
   spr_sol[1] = SPR("sol_2");
   spr_sol[2] = SPR("sol_3");
   spr_sol[3] = SPR("sol_4");
   spr_sol[4] = SPR("sol_5");
#undef SPR

   // Determine X coordinates for each of the sprites to show
   pos_sol = screen_cx - (spr_name->width + spr_sol[0]->width + 8) / 2;
   pos_name = pos_sol + spr_sol[0]->width + 8;

   // Reset head animation
   head_anim = -90;

   // Hide cursor
   set_cursor(CURSOR_NONE);

   // Make screen visible
   fade_on();
}

//***************************************************************************
// load_logo [internal]
// Loads the assets for the logo screen. Run during the loading screen.
//***************************************************************************

static void load_logo(void) {
   gfxset_logo = load_graphics_set("graphics/logo");
   set_loading_total(1, 1);
}

//***************************************************************************
// run_logo
// Processes the logic for the logo screen.
//***************************************************************************

void run_logo(void) {
   // Animate head
   head_anim++;

   // Ding!
   if (head_anim == -4)
      play_sfx(SFX_LOGO);

   // Quit the logo if we spent too much time here
   if (head_anim == 90)
      fade_off_and_switch(GAMEMODE_TITLE);

   // Quit logo early?
   if (input.menu.accept || input.menu.cancel ||
   input.oneswitch.tap || input.oneswitch.tap2)
      fade_off_and_switch(GAMEMODE_TITLE);
}

//***************************************************************************
// draw_logo
// Draws the logo screen.
//***************************************************************************

void draw_logo(void) {
   // Determine which sprite to use for Sol's head
   int which = (head_anim + 10) >> 2;
   if (which < 0) which = 0;
   if (which > 4) which = 4;
   const Sprite *head_spr = spr_sol[which];

   // Clear background
   clear_screen(0x000000);

   // Draw sprites
   draw_sprite(head_spr, pos_sol, screen_cy - 20, SPR_NOFLIP);
   draw_sprite(spr_name, pos_name, screen_cy - 16, SPR_NOFLIP);

   // Show company name
   set_reader_text(text.logo.name);
}

//***************************************************************************
// deinit_logo
// Deinitializes the logo screen.
//***************************************************************************

void deinit_logo(void) {
   // Unload graphics
   if (gfxset_logo) {
      destroy_graphics_set(gfxset_logo);
      gfxset_logo = NULL;
   }
}
