//***************************************************************************
// "loading.c"
// Code that shows the loading screen. Warning: multithreading ahead, here be
// dragons, stay out of here if you don't know what you're doing.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <SDL2/SDL.h>
#include "input.h"
#include "loading.h"
#include "settings.h"
#include "text.h"
#include "video.h"

// Progress bar proportions
#define BAR_X1 (screen_cx-0x80)
#define BAR_X2 (screen_cx+0x80)
#define BAR_Y1 (screen_h-0x20)
#define BAR_Y2 (screen_h-0x10)

// Different loading statuses
enum {
   STATUS_NOTLOADING,
   STATUS_LOADING,
   STATUS_ABORTING,
   STATUS_ABORTED
};
static SDL_atomic_t status = { STATUS_NOTLOADING };

// Private function prototypes
static int loading_thread(void *);

// Flag used to tell when the loading thread is done
// It's 0 when loading starts and 1 when loading is over
static SDL_atomic_t ready;

// Used to tell how much of the progress bar is used
static SDL_atomic_t loaded;

//***************************************************************************
// loading_screen
// Shows a loading screen while the given function is running
//---------------------------------------------------------------------------
// param func: function to execute
//***************************************************************************

void loading_screen(void (*func)(void)) {
   // We're loading now
   SDL_AtomicSet(&status, STATUS_LOADING);

   // Reset ready flag
   SDL_AtomicSet(&ready, 0);

   // Reset amount to load
   SDL_AtomicSet(&loaded, 0);

   // Create loading thread
   SDL_Thread *thread = SDL_CreateThread(loading_thread, "loading", func);

   // Don't show the bar until at least a minimum amount of time ellapsed
   // This is done to prevent it from flashing in cases where loading ends
   // up being instantaneous (which ideally would be every time, but oh well)
   // In case you wonder: 6 frames = 0.1 seconds;
   int dont_show = settings.attract ? 60 : 6;

   // No threading? (stick to single thread then, don't complain about not
   // handling events...)
   if (thread == NULL) {
      // OK, not multithreading, reset this...
      SDL_AtomicSet(&status, STATUS_NOTLOADING);

      // Draw screen
      // To-do: implement progress bar
      clear_screen(0x000000);
      draw_text("Loading...", screen_w - 0x20, screen_h - 0x10,
         FONT_LIT, ALIGN_BOTTOMRIGHT);
      update_program();

      // Call function as-is...
      func();
   }

   // Created thread, wait until it's done
   else {
      // Hide the cursor (well, in fullscreen, in windowed it'll still be an
      // arrow, which is why we still need to update the input)
      set_cursor(CURSOR_NONE);

      // Wait until thread is done (which means the function is done)
      while (!SDL_AtomicGet(&ready)) {
         // Uh oh...
         if (SDL_AtomicGet(&status) == STATUS_ABORTING) {
            SDL_AtomicSet(&status, STATUS_ABORTED);
            for (;;);
         }

         // Update events
         update_program();
         update_input();

         // Check how much was loaded so far
         int filled = SDL_AtomicGet(&loaded);

         // Draw screen
         clear_screen(0x000000);
         if (dont_show > 0)
            dont_show--;
         else {
            draw_rectangle(BAR_X1-1, BAR_Y1-1, BAR_X2+1, BAR_Y2+1,
               settings.box_lit[1]);
            draw_rectangle(BAR_X1, BAR_Y1, BAR_X2, BAR_Y2,
               settings.box_lit[0]);
            draw_rectangle(BAR_X1+1, BAR_Y1+1, BAR_X2-1, BAR_Y2-1,
               settings.box_lit[1]);
            fill_rectangle(BAR_X1, BAR_Y1, BAR_X1+filled, BAR_Y2,
               settings.box_lit[0]);
            draw_vline(BAR_X1+filled+1, BAR_Y1+1, BAR_Y2-1,
               settings.box_lit[1]);
            draw_cursor();
         }
      }

      // Done loading
      SDL_AtomicSet(&status, STATUS_NOTLOADING);
   }
}

//***************************************************************************
// loading_thread [internal]
// Wrapper used to call the function that was passed to loading_screen. It
// also tells it when the function has finished so it can stop idling.
//---------------------------------------------------------------------------
// param func: function to execute
//***************************************************************************

static int loading_thread(void *func) {
   // Call function
   // Wow, that's a mess...
   ((void (*)(void))func)();

   // We're done
   SDL_AtomicSet(&ready, 1);
   return 0;
}

//***************************************************************************
// set_loading_total
// Sets how much of the progress bar is filled
//***************************************************************************

void set_loading_total(int amount, int total) {
   // Update progress bar
   SDL_AtomicSet(&loaded, amount * (BAR_X2-BAR_X1) / total);
}

//***************************************************************************
// abort_loading
// Helper for abort_program, used to halt the main thread in case of aborting
// because otherwise it'll try to access resources which aren't available
// anymore.
//***************************************************************************

void abort_loading(void) {
   // Are we loading in the first place?
   if (SDL_AtomicGet(&status) == STATUS_NOTLOADING)
      return;

   // Tell main thread to halt
   SDL_AtomicSet(&status, STATUS_ABORTING);
   while (SDL_AtomicGet(&status) != STATUS_ABORTED);

   // Now we can safely continue...
}
