//***************************************************************************
// "level.h"
// Level-related definitions
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

#ifndef LEVEL_H
#define LEVEL_H

// Required headers
#include <stddef.h>
#include <stdint.h>
#include "video.h"

// Size of a tile. Must be a power of two.
#define TILE_SIZE_BIT 5                // For bit shifting
#define TILE_SIZE (1<<TILE_SIZE_BIT)   // Size in pixels
#define TILE_SIZE_MASK (TILE_SIZE-1)   // For bit masking

// Limits for the level dimensions
// The minimum size is the minimum that fits in a 384×240 screen (the largest
// dimensions for the virtual resolution). The maximum size is what fits
// inside a 16-bit value.
#define MIN_MAP_WIDTH 12
#define MIN_MAP_HEIGHT 8
#define MAX_MAP_WIDTH 0xFFFF
#define MAX_MAP_HEIGHT 0xFFFF

// Speed of conveyor belts
#define BELT_SPEED 0x180

// Level themes
typedef enum {
   THEME_VIRTUAL,          // Virtual
   THEME_PARK,             // Park
   THEME_SEWER,            // Sewer
   THEME_HARBOR,           // Harbor
   THEME_DESERT,           // Desert
   THEME_CAVE,             // Cave
   THEME_FACTORY,          // Factory
   THEME_CARNIVAL,         // Carnival
   NUM_THEMES              // Number of level themes
} Theme;

// Available tile types for the collision map
typedef enum {
   TILE_EMPTY,             // Empty
   TILE_SOLID,             // Solid
   TILE_NWSE_1,            // 2:1 NWSE slope (left half)
   TILE_NWSE_2,            // 2:1 NWSE slope (right half)
   TILE_NESW_1,            // 2:1 NESW slope (right half)
   TILE_NESW_2,            // 2:1 NESW slope (left half)

   TILE_EMPTY_BG,          // Empty w/background
   TILE_FLOOR_BG,          // Thin floor w/background
   TILE_NWSE_1_BG,         // 2:1 NWSE slope (left half) w/background
   TILE_NWSE_2_BG,         // 2:1 NWSE slope (right half) w/background
   TILE_NESW_1_BG,         // 2:1 NESW slope (right half) w/background
   TILE_NESW_2_BG,         // 2:1 NESW slope (left half) w/background

   TILE_HIDDEN,            // Hidden passage
   TILE_BRIDGE,            // Bridge
   TILE_BELT_LEFT,         // Conveyor belt (going left)
   TILE_BELT_RIGHT,        // Conveyor belt (going right)

   NUM_TILETYPES           // Number of tile types
} TileType;

// Object IDs for the level format
// Note only placeable objects are here. Also, object IDs must fit within a
// byte. In other words, they can't be larger than 0xFF.
typedef enum {
   LVOBJ_NONE,             // No object
   LVOBJ_FLAMER,           // Enemy: flamer
   LVOBJ_SPRAYER,          // Enemy: sprayer
   LVOBJ_TURRET,           // Enemy: turret
   LVOBJ_ROLLER,           // Enemy: roller
   LVOBJ_GRABBER,          // Enemy: grabber
   LVOBJ_SPIDER,           // Enemy: spider
   LVOBJ_HEATER,           // Enemy: heater
   LVOBJ_BOMB,             // Enemy: bomb
   LVOBJ_SPRING,           // Item: spring
   LVOBJ_PUSHABLE,         // Platform: pushable
   LVOBJ_PLATFORM,         // Platform: idle platform
   LVOBJ_PLATFORM_H,       // Platform: horizontal moving platform
   LVOBJ_PLATFORM_V,       // Platform: vertical moving platform
   LVOBJ_PLATFORM_B,       // Platform: breakable platform
   LVOBJ_HEART,            // Item: heart
   LVOBJ_INVINCIBILITY,    // Item: invincibility
   LVOBJ_CHECKPOINT,       // Item: checkpoint
   LVOBJ_SPIKES_U,         // Hazard: spikes pointing upwards
   LVOBJ_SPIKES_D,         // Hazard: spikes pointing downwards
   LVOBJ_SCENERYSMALL,     // Scenery: small
   LVOBJ_SCENERYBIG,       // Scenery: big
   LVOBJ_SSPRING,          // Item: superspring
   LVOBJ_SPIKES_L,         // Hazard: spikes pointing left
   LVOBJ_SPIKES_R,         // Hazard: spikes pointing right
   LVOBJ_OBSTRUCTION,      // Platform: obstruction
   LVOBJ_SHIELD,           // Item: shield
   LVOBJ_CRUSHER,          // Hazard: crusher
   LVOBJ_LIQUIDHAZARD,     // Hazard: liquid
   LVOBJ_GOAL,             // Item: goal
   LVOBJ_WINGS,            // Item: wings power-up
   LVOBJ_HAMMER,           // Item: hammer power-up
   LVOBJ_DANCER,           // Item: dancer
   LVOBJ_FIRE,             // Hazard: fire
   LVOBJ_PARASOL,          // Item: parasol power-up
   LVOBJ_SPIDERPOW,        // Item: spider power-up
   LVOBJ_STALACTITE,       // Hazard: stalactite
   LVOBJ_SWITCH_1,         // Item: switch #1
   LVOBJ_SWITCH_2,         // Item: switch #2
   LVOBJ_SWITCH_3,         // Item: switch #3
   LVOBJ_SWITCH_4,         // Item: switch #4
   LVOBJ_SWITCH_5,         // Item: switch #5
   LVOBJ_SWITCH_6,         // Item: switch #6
   LVOBJ_SWITCH_7,         // Item: switch #7
   LVOBJ_SWITCH_8,         // Item: switch #8
   LVOBJ_DOOR_1,           // Platform: door
   LVOBJ_DOOR_2,           // Platform: door
   LVOBJ_DOOR_3,           // Platform: door
   LVOBJ_DOOR_4,           // Platform: door
   LVOBJ_DOOR_5,           // Platform: door
   LVOBJ_DOOR_6,           // Platform: door
   LVOBJ_DOOR_7,           // Platform: door
   LVOBJ_DOOR_8,           // Platform: door
   LVOBJ_BOUNCINGHAZARD,   // Hazard: bouncing
   LVOBJ_COIL_F,           // Hazard: coil (floor)
   LVOBJ_COIL_C,           // Hazard: coil (ceiling)
   LVOBJ_BUZZSAW,          // Hazard: buzzsaw (static)
   LVOBJ_BUZZSAW_C,        // Hazard: buzzsaw (ceiling)
   LVOBJ_BUZZSAW_F,        // Hazard: buzzsaw (floor)
   LVOBJ_BOSS1,            // Bulldozer boss
   LVOBJ_BOSS5,            // Driller boss
   LVOBJ_BOSS2,            // Trash truck boss
   LVOBJ_BOSS4,            // Oil truck boss
   LVOBJ_BOSS3,            // Ship boss
   LVOBJ_BOSS6,            // Mr. Evil
   LVOBJ_BLUESTAR,         // Blue star
   LVOBJ_BALLOON,          // Balloon
   LVOBJ_GENERATOR,        // Generator
   LVOBJ_MAINGENERATOR,    // Main generator
   NUM_LVOBJIDS            // Number of level object IDs
} LevelObjID;

// Information stored in a tile
typedef struct {
   uint8_t collision;

   uint8_t piece_tl;       // ID of top-left piece (high layer)
   uint8_t piece_tr;       // ID of top-right piece (high layer)
   uint8_t piece_bl;       // ID of bottom-left piece (high layer)
   uint8_t piece_br;       // ID of bottom-right piece (high layer)

   uint8_t bgpiece_tl;     // ID of top-left piece (low layer)
   uint8_t bgpiece_tr;     // ID of top-right piece (low layer)
   uint8_t bgpiece_bl;     // ID of bottom-left piece (low layer)
   uint8_t bgpiece_br;     // ID of bottom-right piece (low layer)

   unsigned force_solid:1; // Force solid collision? (used by e.g. spikes)
   unsigned obstruction:1; // Is there an obstruction in this tile?
} LevelTile;

// Possible animations for level-specific objects
typedef enum {
   LV_ANIM_PUSHABLE,       // Pushable object
   LV_ANIM_PLATFORMSTD,    // Standard platform
   LV_ANIM_PLATFORMBRK,    // Breakable platform
   LV_ANIM_PLATFORMBRK2,   // Breaking platform
   LV_ANIM_PLATFORMLEFT,   // Platform leftover
   LV_ANIM_OBSTRUCTION,    // Obstruction
   LV_ANIM_OBSTRUCTLEFT,   // Obstruction leftover
   LV_ANIM_SMALLSCENERY,   // Small scenery
   LV_ANIM_BIGSCENERY,     // Big scenery
   LV_ANIM_CRUSHER,        // Crusher
   LV_ANIM_LIQUIDHAZARD,   // Liquid hazard
   LV_ANIM_BOUNCINGHAZARD, // Bouncing hazard
   LV_ANIM_FIRE,           // Fire hazard
   LV_ANIM_STALACTITE,     // Stalactite
   LV_ANIM_DOOR,           // Door
   LV_ANIM_BALLOON_IDLE,   // Balloon (idle)
   LV_ANIM_BALLOON_POP,    // Balloon (popped)
   NUM_LV_ANIM             // Number of animations
} LevelAnim;

// Function prototypes
void load_level(void);
void unload_level(void);
void update_level(void);
void draw_level_high(void);
void draw_level_low(void);
void draw_obstructions(void);
LevelTile *get_tile(int32_t, int32_t);
LevelTile *get_tile_by_pixel(int32_t, int32_t);
uint8_t get_map_collision(int32_t x, int32_t y);
void update_tile_pieces(uint16_t, uint16_t);
void update_tile_clues(uint16_t, uint16_t);
const AnimFrame *retrieve_level_anim(LevelAnim);
const AnimFrame *retrieve_liquid_anim(void);
void set_spawn_point(int32_t, int32_t);
void reset_spawn_point(void);
Theme get_level_theme(void);
void get_level_size(uint16_t *, uint16_t *);
size_t get_num_blue_stars(void);
size_t get_total_blue_stars(void);
void inc_blue_stars(void);

#endif
