//***************************************************************************
// "langselect.c"
// Code for the language select screen.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include "main.h"
#include "background.h"
#include "menu.h"
#include "reader.h"
#include "scene.h"
#include "settings.h"
#include "text.h"
#include "video.h"

// Private function prototypes
static void init_language_select_menu(void);
static void button_select(void);
static void button_cancel(void);

//***************************************************************************
// init_language_select
// Initializes the language select screen.
//***************************************************************************

void init_language_select(void) {
   init_language_select_menu();
   fade_on();
}

//***************************************************************************
// init_language_select_menu [internal]
// Sets up the language list menu.
//***************************************************************************

static void init_language_select_menu(void) {
   // Get number of languages
   unsigned num_languages = get_num_languages();

   // Set up menu
   init_menu();
   set_reinit_menu_func(init_language_select_menu);
   menu.defoption.up = num_languages-1;
   menu.defoption.down = 0;

   if (prev_game_mode == GAMEMODE_OPT_GAMEPLAY)
      menu.cancel = button_cancel;

   // Calculate where the first language would appear vertically
   int32_t y_offset = screen_cy - num_languages * 0x08;

   // Go through all languages
   for (unsigned i = 0; i < num_languages; i++, y_offset += 0x10) {
      // Get language name
      const char *name = get_language_name(i);
      menu.options[i].caption = name;

      // Set up coordinates
      int32_t x_offset = calc_text_len(name) / 2 + 4;
      menu.options[i].box.x1 = screen_cx - x_offset;
      menu.options[i].box.x2 = screen_cx + x_offset;
      menu.options[i].box.y1 = y_offset;
      menu.options[i].box.y2 = y_offset + 0x0F;

      // Set up the next and previous options (for keyboard and joystick)
      menu.options[i].move.up = (i == 0) ? num_languages-1 : i-1;
      menu.options[i].move.down = (i == num_languages-1) ? 0 : i+1;
      menu.options[i].move.oneswitch = menu.options[i].move.down;

      // Finally make the option select the language
      menu.options[i].action.accept = button_select;
   }
}

//***************************************************************************
// run_language_select
// Processes the logic for the language select screen.
//***************************************************************************

void run_language_select(void) {
   // Update menu
   update_menu();

   // Update background
   update_background();
}

//***************************************************************************
// draw_language_select
// Draws the language select screen.
//***************************************************************************

void draw_language_select(void) {
   // Clear background
   clear_screen(0x000000);

   // Draw background
   draw_background();

   // Draw menu
   for (int i = 0; i < MAX_MENU; i++) {
      // Not an option?
      if (menu.options[i].caption == NULL)
         break;

      // Draw option label
      draw_text(menu.options[i].caption,
         (menu.options[i].box.x1 + menu.options[i].box.x2) / 2,
         menu.options[i].box.y1,
         menu.selected == i ? FONT_LIT : FONT_DIM,
         ALIGN_TOP);
   }

   // Show selected option in the screen reader
   if (menu.selected != -1)
      set_reader_text(menu.options[menu.selected].caption);
}

//***************************************************************************
// deinit_language_select
// Deinitializes the language select screen.
//***************************************************************************

void deinit_language_select(void) {
   // Force a language reload
   unload_language();
   load_language();

   // Flush scene list since it relies on the language
   unload_scene_list();
   load_scene_list();
}

//***************************************************************************
// button_select [internal]
// Function called when a language is selected
//***************************************************************************

static void button_select(void) {
   // Switch to the selected language
   settings.language = menu.selected;

   // Start game
   fade_off_and_switch(prev_game_mode == GAMEMODE_OPT_GAMEPLAY ?
      GAMEMODE_OPT_GAMEPLAY : GAMEMODE_LOGO);
}

//***************************************************************************
// button_cancel [internal]
// Function called when cancelling a language choice (only from the options
// gameplay menu).
//***************************************************************************

static void button_cancel(void) {
   // Return to menu
   fade_off_and_switch(GAMEMODE_OPT_GAMEPLAY);
}
