//***************************************************************************
// "items.c"
// Code for miscellaneous items.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include "editor.h"
#include "ingame.h"
#include "level.h"
#include "objects.h"
#include "player.h"
#include "physics.h"
#include "savegame.h"
#include "scene.h"
#include "settings.h"
#include "sound.h"
#include "tables.h"
#include "tally.h"

// Spring behavior parameters
#define SPRING_IMPULSE  0x7C0    // How far to push the player upwards
#define SPRING_DELAY    16       // How long it takes to spring

// Superspring behavior parameters
#define SSPRING_IMPULSE 0xA00    // How far to push the player upwards
#define SSPRING_DELAY   24       // How long it takes to spring

// Heart behavior parameters
#define HEART_VANISH    60       // How long it takes to vanish
#define HEART_BOUNCE    480      // Impulse upwards when being taken
#define HEART_WEIGHT    8        // Weight applied when vanishing

//***************************************************************************
// run_dummy_obj
// Dummy object, just commits suicide, used in exceptional cases when we need
// to create an object but for some reason we shouldn't (e.g. spawning
// enemies when the "no enemies" cheat is turned on).
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_dummy_obj(Object *obj) {
   delete_object(obj);
}

//***************************************************************************
// init_spring
// Initializes a spring object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_spring(Object *obj) {
   // Set hitbox
   set_hitbox(obj, -15, 15, 0, 15);
}

//***************************************************************************
// run_spring
// Logic for spring objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_spring(Object *obj) {
   // Too far to bother?
   if (is_too_far(obj))
      return;

   // Already triggered? (wait until we're done with the animation)
   if (obj->timer > 0)
      obj->timer--;

   // Idle?
   else {
      // See if a player is stepping on us
      // Impulse the player upwards if that happens
      for (Object *other = get_first_object(OBJGROUP_PLAYER); other != NULL;
      other = other->next) {
         // Collision? Also make sure the player is not going upwards
         if (collision(obj, other) && other->gravity >= 0) {
            // Make the player go upwards
            other->gravity = (obj->type == OBJ_SSPRING) ?
               -SSPRING_IMPULSE : -SPRING_IMPULSE;
            other->jumping = 0;
            other->hurt = 0;

            // Stop any spinning jump
            if (!settings.pinball)
               other->pinball = 0;

            // Stop any power-up that was active
            other->active = 0;

            // Get spring to work
            obj->timer = obj->type == OBJ_SSPRING ?
               SSPRING_DELAY : SPRING_DELAY;

            // Play sound effect
            play_sfx(SFX_SPRING);

            // We can only impulse one player at a time
            // So, stop here since we already did it
            break;
         }
      }
   }

   // Set animation
   if (obj->type == OBJ_SPRING)
      set_object_anim(obj, retrieve_object_anim(obj->timer ?
      OB_ANIM_SPRINGWORK : OB_ANIM_SPRINGIDLE));
   else
      set_object_anim(obj, retrieve_object_anim(obj->timer ?
      OB_ANIM_SSPRINGWORK : OB_ANIM_SSPRINGIDLE));
}

//***************************************************************************
// init_balloon
// Initializes a balloon object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_balloon(Object *obj) {
   // Set hitbox
   set_hitbox(obj, -16, 15, -16, 15);
}

//***************************************************************************
// run_balloon
// Logic for balloon objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_balloon(Object *obj) {
   // Not popped?
   if (obj->timer == 0) {
      // Check for collision against players
      for (Object *player = get_first_object(OBJGROUP_PLAYER);
      player != NULL; player = player->next) {
         // Make sure there's a collision first
         if (!collision(obj, player))
            continue;

         // Player stomping on us?
         if (player->gravity > 0 && player->y <= obj->y) {
            play_sfx(SFX_BALLOON);
            player->gravity = -0x580;
            obj->timer = 120;
            break;
         }
      }
   }

   // Wait for respawning
   else
      obj->timer--;

   // Set animation
   set_object_anim(obj, retrieve_level_anim(obj->timer ?
      LV_ANIM_BALLOON_POP : LV_ANIM_BALLOON_IDLE));
}

//***************************************************************************
// init_heart
// Initializes a heart object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_heart(Object *obj) {
   // Set hitbox
   set_hitbox(obj, -8, 7, -8, 7);
}

//***************************************************************************
// run_heart
// Logic for heart objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_heart(Object *obj) {
   // Already taken? (make a vanish animation then)
   if (obj->dead) {
      // Rise
      obj->y += speed_to_int(obj->gravity);
      obj->gravity += HEART_WEIGHT;

      // Check if we should vanish for real
      obj->timer++;
      if (obj->timer == HEART_VANISH)
         delete_object(obj);

      // Show proper animation
      set_object_anim(obj, retrieve_object_anim(OB_ANIM_HEARTTAKEN));
      return;
   }

   // Too far to bother?
   if (is_too_far(obj))
      return;

   // Taken by the player?
   Object *player = get_first_object(OBJGROUP_PLAYER);
   if (player != NULL && collision(obj, player)) {
      // Start the animation
      obj->dead = 1;
      obj->gravity = -HEART_BOUNCE;

      // Award health (if possible)
      if (player->health < (settings.extra_health ? 5 : 3) ||
      settings.collect_health)
         player->health++;

      // Add bonus
      add_item_bonus(1);

      // Play sound effect
      play_sfx(SFX_HEART);

      // Get rid of the hitbox since it interferes in audiovideo mode
      // (not as bad as the situation with the checkpoint, but still)
      obj->has_hitbox = 0;
   }

   // Set animation
   set_object_anim(obj, retrieve_object_anim(OB_ANIM_HEARTIDLE));
}

//***************************************************************************
// init_invincibility
// Initializes an invincibility object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_invincibility(Object *obj) {
   // Set hitbox
   set_hitbox(obj, -10, 10, -10, 10);

   // Set animation
   set_object_anim(obj, retrieve_object_anim(OB_ANIM_INVINCIBILITY));
}

//***************************************************************************
// run_invincibility
// Logic for invincibility objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_invincibility(Object *obj) {
   // Too far to bother?
   if (is_too_far(obj))
      return;

   // Taken by the player?
   Object *player = get_first_object(OBJGROUP_PLAYER);
   if (player != NULL && collision(obj, player)) {
      play_ingame_bgm(BGM_INVINCIBILITY);
      player->invincibility = settings.invincibility_time;
      add_item_bonus(2);
      delete_object(obj);
      return;
   }
}

//***************************************************************************
// init_checkpoint
// Initializes an checkpoint object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_checkpoint(Object *obj) {
   // Set animation
   set_object_anim(obj, retrieve_object_anim(OB_ANIM_CHECKOFF));
}

//***************************************************************************
// run_checkpoint
// Logic for checkpoint objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_checkpoint(Object *obj) {
   // Too far to bother?
   if (is_too_far(obj))
      return;

   // Triggered by the player?
   Object *player = get_first_object(OBJGROUP_PLAYER);
   if (player != NULL && collision(obj, player)) {
      // Make ourselves active
      obj->active = 1;
      set_object_anim(obj, retrieve_object_anim(OB_ANIM_CHECKON));

      // Already set?
      if (obj->x == player->base_x && obj->y == player->base_y)
         return;

      // Set the new spawn point
      set_spawn_point(obj->x, obj->y);
      player->base_x = obj->x;
      player->base_y = obj->y;

      // Make player not angry :)
      reset_anger();

      // Refill the player's health
      if (settings.refill_health && !settings.no_health)
         player->health = settings.extra_health ? 5 : 3;

      // Update savegame (if relevant)
      if (get_editor_map() == NULL) {
         SaveGame savedata;
         get_savegame(&savedata);
         savedata.checkpoint_x = obj->x / TILE_SIZE;
         savedata.checkpoint_y = obj->y / TILE_SIZE;
         save_savegame(&savedata);
      }

      // Play sound
      play_sfx(SFX_CHECKPOINT);

      // Force pause on in one-switch mode
      // TO-DO: make this a setting too?
      if (settings.one_switch)
         toggle_pause();
   }

   // Reset animation if not active...
   if (player != NULL && (player->base_x != obj->x ||
   player->base_y != obj->y)) {
      obj->active = 0;
      set_object_anim(obj, retrieve_object_anim(OB_ANIM_CHECKOFF));
   }

   // Set hitbox depending on whether we're active or not
   // Not really an issue most of the time, but it becomes a problem in
   // audiovideo mode since it keeps rendering non-stop otherwise and it
   // confuses players when respawning
   if (!obj->active)
      set_hitbox(obj, -16, 15, -32, 15);
   else
      obj->has_hitbox = 0;
}

//***************************************************************************
// init_goal
// Initializes a goal object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_goal(Object *obj) {
   // Set hitbox
   set_hitbox(obj, -16, 15, -16, 15);

   // Set animation
   set_object_anim(obj, retrieve_object_anim(OB_ANIM_GOALOFF));
}

//***************************************************************************
// run_goal
// Logic for goal objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_goal(Object *obj) {
   // Too far to bother?
   // Only skip if the goal wasn't triggered though, we want to make sure the
   // level finishes even if the player ends up going very far from the goal
   if (is_too_far(obj) && !obj->active)
      return;

   // Already triggered?
   if (obj->active) {
      // Fly off-screen while making some fancy moves around
      obj->timer++;
      obj->x = obj->base_x;
      obj->y = obj->base_y;
      obj->x += cosines[obj->timer << 2 & 0xFF] * obj->timer >> 8;
      obj->x -= sines[obj->timer << 2 & 0xFF] * obj->timer >> 8;
      obj->y -= sines[obj->timer << 2 & 0xFF] * obj->timer >> 8;

      // Generate sparkles
      Object *sparkle = add_object(OBJ_FALLSPARKLE,
         obj->x - 0x10 + get_random(0x20),
         obj->y - 0x10 + get_random(0x20),
         0);
      sparkle->speed = -0x100 + get_random(0x200);
      sparkle->gravity = -0x100 + get_random(0x200);

      // Vanish after a while...
      if (obj->timer > 5 * 60) {
         // Finish the level
         finish_level();

         // OK vanish for real now...
         delete_object(obj);
         return;
      }
   } else {
      // Every so often generate a sparkle
      if ((game_anim & 0x07) == 0x00) {
         // Generate sparkles
         add_object(OBJ_SPARKLE,
            obj->x - 0x10 + get_random(0x20),
            obj->y - 0x10 + get_random(0x20),
            0);
      }

      // Triggered by the player?
      Object *player = get_first_object(OBJGROUP_PLAYER);
      if (player != NULL && collision(obj, player)) {
         // Make ourselves active
         obj->active = 1;
         set_object_anim(obj, retrieve_object_anim(OB_ANIM_GOALON));

         // No need to be angry anymore :)
         reset_anger();

         // Play sound effect
         play_sfx(SFX_GOAL);

         // Update save if needed
         if (get_editor_map() == NULL)
            advance_savegame_scene();

         // Get rid of the player if goalvanish is enabled
         if (settings.goal_vanish) {
            // Spawn stars :D :D :D
            for (unsigned i = 0; i < 0x20; i++) {
               int32_t x = player->x + get_random(0x20) - 0x10;
               int32_t y = player->y + get_random(0x28) - 0x18;
               Object *sparkle = add_object(OBJ_FALLSPARKLE, x, y, 0);
               sparkle->speed = (x - player->x) << (4 + (i & 0x01));
               sparkle->gravity = (y - player->y) << (4 + (i & 0x01));
            }

            // Get rid of objects attached to the player
            if (player->otherobj != NULL)
               delete_object(player->otherobj);
            for (Object *ptr = get_first_object(OBJGROUP_SHIELD);
            ptr != NULL; ) {
               Object *old = ptr;
               ptr = ptr->next;
               delete_object(old);
            }

            // Get rid of object
            delete_object(player);
         }
      }
   }
}

//***************************************************************************
// init_blue_star
// Initializes a blue star object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_blue_star(Object *obj) {
   // Set hitbox
   set_hitbox(obj, -16, 15, -16, 15);

   // Set animation
   set_object_anim(obj, retrieve_object_anim(OB_ANIM_BLUESTAR));
}

//***************************************************************************
// run_blue_star
// Logic for blue star objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_blue_star(Object *obj) {
   // Too far to bother?
   if (is_too_far(obj))
      return;

   // Every so often generate a sparkle
   if ((game_anim & 0x07) == 0x00) {
      // Generate sparkles
      add_object(OBJ_SPARKLE,
         obj->x - 0x10 + get_random(0x20),
         obj->y - 0x10 + get_random(0x20),
         0);
   }

   // Taken by the player?
   Object *player = get_first_object(OBJGROUP_PLAYER);
   if (player != NULL && collision(obj, player)) {
      // Play sound effect
      play_sfx(SFX_BLUESTAR);

      // Sparkle a lot when taken
      for (unsigned i = 0; i < 0x20; i++) {
         Object *sparkle = add_object(OBJ_FALLSPARKLE,
            obj->x - 0x10 + get_random(0x20),
            obj->y - 0x10 + get_random(0x20),
            0);
         sparkle->speed = -0x100 + get_random(0x200);
         sparkle->gravity = -0x200 + get_random(0x200);
      }

      // Open door when all blue stars are collected
      inc_blue_stars();
      if (get_num_blue_stars() >= get_total_blue_stars())
         set_map_switch(7);

      // Vanish!
      delete_object(obj);
      return;
   }
}

//***************************************************************************
// run_shield
// Logic for shield objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_shield(Object *obj) {
   // Track player
   obj->x = obj->otherobj->x;
   obj->y = obj->otherobj->y;

   // Set animation depending if the player has a shield or not
   set_object_anim(obj, obj->otherobj->shield ?
      retrieve_object_anim(OB_ANIM_SHIELD) : NULL);
}

//***************************************************************************
// init_shield_item
// Initializes a shield item object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_shield_item(Object *obj) {
   // Set hitbox
   set_hitbox(obj, -16, 15, -16, 15);

   // Set animation
   set_object_anim(obj, retrieve_object_anim(OB_ANIM_SHIELD_ITEM));
}

//***************************************************************************
// run_shield_item
// Logic for shield item objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_shield_item(Object *obj) {
   // Too far to bother?
   if (is_too_far(obj))
      return;

   // Taken by the player?
   Object *player = get_first_object(OBJGROUP_PLAYER);
   if (player != NULL && collision(obj, player)) {
      // Give shield to the player
      player->shield = 1;

      // Play sound effect
      play_sfx(SFX_SHIELD);

      // Add bonus
      add_item_bonus(2);

      // Get rid of this object
      delete_object(obj);
      return;
   }
}

//***************************************************************************
// init_powerup_item
// Initializes a power-up item object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_powerup_item(Object *obj) {
   // Set hitbox
   set_hitbox(obj, -16, 15, -16, 15);

   // Set animation
   int anim;
   switch (obj->type) {
      case OBJ_WINGS_ITEM: anim = OB_ANIM_WINGS_ITEM; break;
      case OBJ_SPIDER_ITEM: anim = OB_ANIM_SPIDER_ITEM; break;
      case OBJ_HAMMER_ITEM: anim = OB_ANIM_HAMMER_ITEM; break;
      case OBJ_PARASOL_ITEM: anim = OB_ANIM_PARASOL_ITEM; break;
      default: return;
   }
   set_object_anim(obj, retrieve_object_anim(anim));
}

//***************************************************************************
// run_powerup_item
// Logic for power-up item objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_powerup_item(Object *obj) {
   // Too far to bother?
   if (is_too_far(obj))
      return;

   // Taken by the player?
   Object *player = get_first_object(OBJGROUP_PLAYER);
   if (player != NULL && collision(obj, player)) {
      // Award player the relevant power-up
      switch (obj->type) {
         case OBJ_WINGS_ITEM: player->type = OBJ_PLAYERWINGS; break;
         case OBJ_SPIDER_ITEM: player->type = OBJ_PLAYERSPIDER; break;
         case OBJ_HAMMER_ITEM: player->type = OBJ_PLAYERHAMMER; break;
         case OBJ_PARASOL_ITEM: player->type = OBJ_PLAYERPARASOL; break;
         default: break;
      }

      // Play sound
      play_sfx(SFX_POWERUP);

      // Add bonus
      add_item_bonus(5);

      // Get rid of this object
      delete_object(obj);
      return;
   }
}

//***************************************************************************
// init_switch
// Initializes a switch object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_switch(Object *obj) {
   // Set hitbox
   set_hitbox(obj, -12, 11, 0, 15);

   // Set animation
   set_object_anim(obj, retrieve_object_anim(OB_ANIM_SWITCH_OFF));
}

//***************************************************************************
// run_switch
// Logic for switch objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_switch(Object *obj) {
   // Get which switch ID is this
   unsigned which = obj->type - OBJ_SWITCH_1;

   // Needs to be triggered yet?
   if (!get_map_switch(which)) {
      // Check if any player touched us
      for (Object *other = get_first_object(OBJGROUP_PLAYER); other != NULL;
      other = other->next) {
         if (collision(obj, other)) {
            // Trigger switch
            set_map_switch(which);

            // Play sound
            play_sfx(SFX_SWITCH);

            // One player is enough
            // Don't bother looking for more
            break;
         }
      }
   }

   // Set animation
   set_object_anim(obj, retrieve_object_anim(get_map_switch(which) ?
      OB_ANIM_SWITCH_ON : OB_ANIM_SWITCH_OFF));
}

//***************************************************************************
// init_dancer
// Initializes a dancer object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_dancer(Object *obj) {
   // Set hitbox
   set_hitbox(obj, -16, 15, -24, 15);
}

//***************************************************************************
// run_dancer
// Logic for dancer objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_dancer(Object *obj) {
   // Not triggered yet?
   if (!obj->active) {
      // See if some player triggered us
      for (Object *player = get_first_object(OBJGROUP_PLAYER);
      player != NULL; player = player->next) {
         // Nope?
         if (!collision(obj, player))
            continue;

         // Give bonus
         add_special_bonus();

         // Get rid of the collision box due to audiovideo mode
         obj->has_hitbox = 0;

         // Trigger!
         play_sfx(SFX_YUPEE);
         obj->active = 1;
         break;
      }
   }

   // Set animation
   set_object_anim(obj, retrieve_object_anim(obj->active ?
      OB_ANIM_DANCERDANCE : OB_ANIM_DANCERWAIT));
}
