//***************************************************************************
// "input.h"
// User input subsystem definitions.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

#ifndef INPUT_H
#define INPUT_H

// Required headers
#include <stdint.h>
#include <SDL2/SDL.h>

// Type that can hold a keyboard scan code
typedef SDL_Scancode KeyCode;
#define NO_KEY SDL_SCANCODE_UNKNOWN

// Different input types
#define JOYIN_NONE 0x0000           // (nothing)
#define JOYIN_BUTTON 0x0100         // Button
#define JOYIN_AXISPOS 0x0200        // Axis (positive)
#define JOYIN_AXISNEG 0x0300        // Axis (negative)
#define JOYIN_HATUP 0x0400          // Hat (up)
#define JOYIN_HATDOWN 0x0500        // Hat (down)
#define JOYIN_HATLEFT 0x0600        // Hat (left)
#define JOYIN_HATRIGHT 0x0700       // Hat (right)
#define JOYIN_UNDEF 0xFF00          // (undefined)

// Possible actions a player may carry
typedef enum {
   PL_INPUT_LEFT,          // Run left
   PL_INPUT_RIGHT,         // Run right
   PL_INPUT_DOWN,          // Crouch
   PL_INPUT_UP,            // Look up
   PL_INPUT_ACTION,        // Jump
   PL_INPUT_PAUSE,         // Pause game
   NUM_PL_INPUT            // Number of actions
} PlayerInputID;

// Structure where the current user input status is kept
// This structure is updated every frame
typedef uint8_t PlayerInput[NUM_PL_INPUT];

typedef struct {
   // Player controls
   struct {
      PlayerInput hold;       // Buttons being currently held down
      PlayerInput press;      // Buttons that just got pressed this frame
   } player;

   // One-switch controls
   struct {
      unsigned tap: 1;        // Tap button (short)
      unsigned tap2: 1;       // Tap button (long)
      unsigned hold: 1;       // Hold button
      unsigned fivetap: 1;    // Five-tap

      unsigned toggle: 1;     // Helper for toggle actions
      unsigned dir: 3;        // Helper for multiway actions
   } oneswitch;

   // Mouse-switch controls
   // Bit 0 is hold, bit 1 is press
   struct {
      unsigned up: 2;         // Cursor is in the up region
      unsigned down: 2;       // Cursor is in the down region
      unsigned left: 2;       // Cursor is in the left region
      unsigned right: 2;      // Cursor is in the right region
      unsigned pause: 2;      // Pause input
   } mouseswitch;

   // Editor controls
   struct {
      unsigned up: 1;         // Scroll up
      unsigned down: 1;       // Scroll down
      unsigned left: 1;       // Scroll left
      unsigned right: 1;      // Scroll right
      unsigned fast: 1;       // Fast scrolling

      unsigned place: 1;      // Place tile
      unsigned remove: 1;     // Remove tile
      unsigned flip: 1;       // Flip object
      unsigned spawn: 1;      // Set spawn point
      unsigned pick: 1;       // Pick tile type
      unsigned next: 1;       // Next tile
      unsigned prev: 1;       // Previous tile

      unsigned b_new: 1;      // "New" button
      unsigned b_info: 1;     // "Info" button
      unsigned b_load: 1;     // "Load" button
      unsigned b_save: 1;     // "Save" button
      unsigned b_play: 1;     // "Play" button
      unsigned b_tilemap: 1;  // "Tilemap" button
      unsigned b_objects: 1;  // "Objects" button
      unsigned b_undo: 1;     // "Undo" button
      unsigned b_redo: 1;     // "Redo" button
      unsigned b_help: 1;     // "Help" button
      unsigned b_quit: 1;     // "Quit" button

      unsigned ui_up: 1;      // UI: up
      unsigned ui_down: 1;    // UI: down
      unsigned ui_xup: 1;     // UI: page up
      unsigned ui_xdown: 1;   // UI: page down
      unsigned ui_home: 1;    // UI: home
      unsigned ui_end: 1;     // UI: end
      unsigned ui_next: 1;    // UI: tab
      unsigned ui_prev: 1;    // UI: shift+tab
   } editor;

   // Cursor
   struct {
      int x;                  // X coordinate
      int y;                  // Y coordinate
      unsigned inside: 1;     // Set when cursor is inside the screen
      unsigned left: 1;       // Set when the main button is pressed
      unsigned right: 1;      // Set when the secondary button is pressed
      unsigned middle: 1;     // Set when the middle button is pressed
      unsigned click: 1;      // Set when the user clicked
      unsigned wheel_up: 1;   // Set when the wheel goes up
      unsigned wheel_down: 1; // Set when the wheel goes down
      unsigned moved: 1;      // Set if cursor moved

      int old_x;              // Used to check if the cursor was moved by
      int old_y;                // the code (there'd be a mismatch if so)
   } cursor;

   // Menu controls
   struct {
      unsigned up: 1;         // Select up
      unsigned down: 1;       // Select down
      unsigned left: 1;       // Select left
      unsigned right: 1;      // Select right
      unsigned accept: 1;     // Accept option
      unsigned cancel: 1;     // Quit menu
   } menu;

   // Debug mode
   struct {
      unsigned wings: 1;      // Give wings power-up
      unsigned spider: 1;     // Give spider power-up
      unsigned hammer: 1;     // Give hammer power-up
      unsigned parasol: 1;    // Give parasol power-up
   } debug;

   // Raw input, used by the options menu
   struct {
      int keyboard;           // Keyboard input
      uint16_t joystick;      // Joystick input
   } rawpress;

   // IME input, used by the file select dialog
   struct {
      char *entered;          // Text that was just entered
      char *entering;         // Text being entered

      unsigned left: 1;       // Previous character
      unsigned right: 1;      // Next character
      unsigned back: 1;       // Delete previous character
      unsigned del: 1;        // Delete next character
      unsigned start: 1;      // Go to start of line
      unsigned end: 1;        // Go to end of line
   } ime;

   // Flag used to tell if the input is idle (i.e. the user is not
   // interacting with the game). Mostly intended to trigger demo mode in
   // the title screen.
   unsigned idle: 1;
} Input;
extern Input input;

// Available cursors
typedef enum {
   CURSOR_NONE,            // No cursor
   CURSOR_ARROW,           // Arrow
   CURSOR_CROSS,           // Cross
   CURSOR_IBEAM,           // I-beam
   CURSOR_HAND,            // Hand

   NUM_CURSORS             // Number of cursors
} Cursor;

// Function prototypes
void init_input(void);
void reset_input(void);
void update_input(void);
void add_joystick(int);
void handle_key_event(int, int);
void handle_joy_axis(int, int, int);
void handle_joy_hat(int, int, int);
void handle_joy_button(int, int, int);
void handle_controller_axis(int, int, int);
void handle_controller_button(int, int);
void handle_mouse_motion(int, int, int);
void handle_mouse_button(int, int);
void handle_ime_entered(const char *);
void remove_joystick(int);
void set_joystick_defaults(void);
void deinit_input(void);
int id_to_scancode(const char *);
const char *scancode_to_id(int);
const char *get_key_name(int);
void set_cursor(Cursor);
void draw_cursor(void);

#endif
