//***************************************************************************
// "input.c"
// User input subsystem. Handles keyboard, mouse and joystick input.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <SDL2/SDL.h>
#include "settings.h"
#include "input.h"
#include "reader.h"
#include "recording.h"
#include "video.h"

// List of key names
// Note letters and numbers (the only single-character names) are handled
// separately since it's faster
static struct {
   const char *name;       // Name of key
   const char *label;      // Label to show in options menu
   int scancode;           // Scan code of key
} key_list[] = {
   { "none",         "N/A",         NO_KEY, },
   { "space",        "Space",       SDL_SCANCODE_SPACE },
   { "enter",        "Enter",       SDL_SCANCODE_RETURN },
   { "tab",          "Tab",         SDL_SCANCODE_TAB },
   { "backspace",    "Backspace",   SDL_SCANCODE_BACKSPACE },
   { "up",           "↑",           SDL_SCANCODE_UP },
   { "down",         "↓",           SDL_SCANCODE_DOWN },
   { "left",         "←",           SDL_SCANCODE_LEFT },
   { "right",        "→",           SDL_SCANCODE_RIGHT },
   { "ins",          "Ins",         SDL_SCANCODE_INSERT },
   { "del",          "Del",         SDL_SCANCODE_DELETE },
   { "home",         "Home",        SDL_SCANCODE_HOME },
   { "end",          "End",         SDL_SCANCODE_END },
   { "pgup",         "PgUp",        SDL_SCANCODE_PAGEUP },
   { "pgdn",         "PgDn",        SDL_SCANCODE_PAGEDOWN },
   { "lshift",       "Shift(←)",    SDL_SCANCODE_LSHIFT },
   { "rshift",       "Shift(→)",    SDL_SCANCODE_RSHIFT },
   { "lctrl",        "Ctrl(←)",     SDL_SCANCODE_LCTRL },
   { "rctrl",        "Ctrl(→)",     SDL_SCANCODE_RCTRL },
   { "lalt",         "Alt(←)",      SDL_SCANCODE_LALT },
   { "ralt",         "Alt(→)",      SDL_SCANCODE_RALT },
   { "lgui",         "GUI(←)",      SDL_SCANCODE_LGUI },
   { "rgui",         "GUI(→)",      SDL_SCANCODE_RGUI },
   { "menu",         "Menu",        SDL_SCANCODE_APPLICATION },
   { "f1",           "F1",          SDL_SCANCODE_F1 },
   { "f2",           "F2",          SDL_SCANCODE_F2 },
   { "f3",           "F3",          SDL_SCANCODE_F3 },
   { "f4",           "F4",          SDL_SCANCODE_F4 },
   { "f5",           "F5",          SDL_SCANCODE_F5 },
   { "f6",           "F6",          SDL_SCANCODE_F6 },
   { "f7",           "F7",          SDL_SCANCODE_F7 },
   { "f8",           "F8",          SDL_SCANCODE_F8 },
   { "f9",           "F9",          SDL_SCANCODE_F9 },
   { "f10",          "F10",         SDL_SCANCODE_F10 },
   { "f11",          "F11",         SDL_SCANCODE_F11 },
   { "f12",          "F12",         SDL_SCANCODE_F12 },
   { "prnscr",       "PrnScr",      SDL_SCANCODE_PRINTSCREEN },
   { "pause",        "Pause",       SDL_SCANCODE_PAUSE },
   { NULL, NULL, NO_KEY }
};

// Possible actions in a menu
typedef enum {
   MN_INPUT_UP,         // Move up
   MN_INPUT_DOWN,       // Move down
   MN_INPUT_LEFT,       // Move left
   MN_INPUT_RIGHT,      // Move right
   MN_INPUT_OK,         // Accept current option
   MN_INPUT_CANCEL,     // Cancel and go back
   NUM_MN_INPUT         // Number of inputs
} MenuInputID;

// Where the current user input status is stored
// This structure is updated every frame (i.e. remains stable within a frame,
// so it's safe to access it directly and assume it won't change until the
// next frame is processed).
Input input;

// Bitfield containing the status of a "key" (or button, for that matter)
typedef struct {
   unsigned pressed: 1;    // Key just went down
   unsigned typed: 1;      // As above but with repeats
   unsigned held: 1;       // Key is being held down
} KeyStatus;

// Where we keep the current status for all the keys in the keyboard
// To-do: modify this to allow for multiple keyboards?
static KeyStatus keys[SDL_NUM_SCANCODES];

// Structure used to keep track of the mouse status
static struct {
   int x;                  // Horizontal coordinate on screen
   int y;                  // Vertical coordinate on screen
   unsigned left: 1;       // Left button
   unsigned right: 1;      // Right button
   unsigned middle: 1;     // Middle button
   unsigned wheel_up: 1;   // Wheel going up
   unsigned wheel_down: 1; // Wheel going down
} mouse;

// Structure used to keep track of the joysticks
#define MAX_AXES 0x100
#define MAX_HATS 0x100
#define MAX_BUTTONS 0x100
typedef struct {
   SDL_Joystick *joystick;          // Joystick instance
   SDL_GameController *controller;  // Game controller instance (can be NULL)
   int instance;                    // Instance ID of this joystick
   /*int8_t axis[MAX_AXES];           // Position of each axis (-1, 0, 1)
   int8_t hats[MAX_HATS];           // Position of each hat
   uint8_t buttons[MAX_BUTTONS];    // Whether each button is pressed*/
   int8_t axes[SDL_CONTROLLER_AXIS_MAX]; // Status of each axis

   // Game controller mappings
   struct {
      uint16_t axes[SDL_CONTROLLER_AXIS_MAX];
      uint16_t buttons[SDL_CONTROLLER_BUTTON_MAX];
   } mappings;
} Joystick;

// Data needed for the joysticks
static unsigned num_joysticks = 0;
static Joystick *joysticks = NULL;

// To keep track of the joystick status
// Remember all joysticks can affect this!
static struct {
   int8_t axis[MAX_AXES];           // Position of each axis (-1, 0, 1)
   int8_t hats[MAX_HATS];           // Position of each hat
   uint8_t buttons[MAX_BUTTONS];    // Whether each button is pressed
} joystatus;

// Which cursor to use (use set_cursor to change it)
static Cursor cursor = CURSOR_NONE;

// Cursor graphics
static GraphicsSet *gfxset_cursor = NULL;
static Sprite *cursor_spr[NUM_CURSORS] = { NULL };

// Locations for the hotspot for each cursor
static const struct {
   int8_t x;         // Hotspot X coordinate
   int8_t y;         // Hotspot Y coordinate
} hotspots[NUM_CURSORS] = {
   { 0, 0 },         // CURSOR_NONE
   { 0, 0 },         // CURSOR_ARROW
   { 7, 7 },         // CURSOR_CROSS
   { 7, 7 },         // CURSOR_IBEAM
   { 7, 1 },         // CURSOR_HAND
};

// Timer used to differentiate between tap and hold in one-switch mode
static unsigned oneswitch_timer;

// Used to be able to trigger the five-tap gesture
static unsigned fivetap_count;   // How many times was tapped (5 = trigger)
static unsigned fivetap_timer;   // How many frames left before reset

// Flags for the modifier keys
#define MOD_SHIFT    0x01     // Shift keys
#define MOD_CTRL     0x02     // Ctrl keys
#define MOD_ALT      0x04     // Alt keys

// Private function prototypes
static int check_joyin(uint16_t);
static unsigned get_mod_flags(void);

//***************************************************************************
// init_input
// Initializes the user input subsystem.
//***************************************************************************

void init_input(void) {
#ifdef DEBUG
   fputs("Initializing input\n", stderr);
#endif

   // By default this is enabled?!
   SDL_StopTextInput();

   // Enable relative mode
   //SDL_SetRelativeMouseMode(SDL_TRUE);

   // Initialize the joystick subsystem if possible
   if (!settings.no_joystick) {
      SDL_InitSubSystem(SDL_INIT_JOYSTICK);
      SDL_InitSubSystem(SDL_INIT_GAMECONTROLLER);
      if (!SDL_WasInit(SDL_INIT_JOYSTICK))
         settings.no_joystick = 1;
   }

   // Reset input status
   reset_input();

   // Check if the joystick subsystem works
   if (SDL_NumJoysticks() < 0) {
      settings.no_joystick = 1;
      SDL_QuitSubSystem(SDL_INIT_JOYSTICK);
   }

   // Load cursor graphics
   gfxset_cursor = load_graphics_set("graphics/cursor");
#define SPR(name) get_sprite(gfxset_cursor, name);
   cursor_spr[CURSOR_ARROW] = SPR("arrow");
   cursor_spr[CURSOR_CROSS] = SPR("cross");
   cursor_spr[CURSOR_IBEAM] = SPR("ibeam");
   cursor_spr[CURSOR_HAND] = SPR("hand");
#undef SPR

   // Don't show any cursor yet
   cursor = CURSOR_NONE;
}

//***************************************************************************
// reset_input
// Resets the user input status.
//***************************************************************************

void reset_input(void) {
   // Reset game input
   memset(&input, 0, sizeof(input));

   // Reset keyboard status
   for (unsigned i = 0; i < SDL_NUM_SCANCODES; i++) {
      keys[i].pressed = 0;
      keys[i].typed = 0;
      keys[i].held = 0;
   }

   // Reset joystick status
   for (unsigned i = 0; i < MAX_AXES; i++)
      joystatus.axis[i] = 0;
   for (unsigned i = 0; i < MAX_HATS; i++)
      joystatus.hats[i] = 0;

   // Reset mouse status
   if (settings.one_switch) {
      input.cursor.x = screen_cx;
      input.cursor.y = screen_cy;
   } else {
      SDL_Window *window = SDL_GetMouseFocus();
      if (window == NULL) {
         input.cursor.x = -1;
         input.cursor.y = -1;
      } else {
         int window_width;
         int window_height;
         SDL_GetMouseState(&mouse.x, &mouse.y);
         SDL_GetWindowSize(window, &window_width, &window_height);
         input.cursor.x = mouse.x / (window_width / screen_w);
         input.cursor.y = mouse.y / (window_height / screen_h);
      }
   }
   input.cursor.old_x = input.cursor.x;
   input.cursor.old_y = input.cursor.y;

   // Reset raw input status
   input.rawpress.keyboard = NO_KEY;
   input.rawpress.joystick = JOYIN_NONE;

   // Reset one-switch status
   input.oneswitch.tap = 0;
   input.oneswitch.tap2 = 0;
   input.oneswitch.hold = 0;
   input.oneswitch.fivetap = 0;
   input.oneswitch.dir = 0;
   input.oneswitch.toggle = 0;
   oneswitch_timer = 0;
   fivetap_count = 0;
   fivetap_timer = 0;

   // Reset mouse-switch status
   input.mouseswitch.up = 0;
   input.mouseswitch.down = 0;
   input.mouseswitch.left = 0;
   input.mouseswitch.right = 0;
   input.mouseswitch.pause = 0;

   // Reset IME status
   if (input.ime.entered != NULL)
      free(input.ime.entered);
   input.ime.entered = NULL;
   if (input.ime.entering != NULL)
      free(input.ime.entering);
   input.ime.entering = NULL;
}

//***************************************************************************
// update_input
// Queries the input devices and updates the user input status.
//***************************************************************************

void update_input(void) {
   // Update the status of all joysticks (i.e. poll them)
   SDL_JoystickUpdate();

   // Determine which modifiers are pressed
   unsigned flags = get_mod_flags();

   // Is one-switch mode enabled?
   if (settings.one_switch) {
      // Get status of the switch
      int button =
         keys[settings.keyboard[PL_INPUT_ACTION][0]].held ||
         keys[settings.keyboard[PL_INPUT_ACTION][1]].held ||
         check_joyin(settings.joystick[PL_INPUT_ACTION][0]) ||
         check_joyin(settings.joystick[PL_INPUT_ACTION][1]) ||
         (SDL_GetMouseState(NULL, NULL) & SDL_BUTTON_LEFT);

      // Update one-switch input
      if (button) {
         oneswitch_timer++;
         input.oneswitch.tap = 0;
         input.oneswitch.tap2 = (oneswitch_timer==settings.tap_delay) ? 1:0;
         input.oneswitch.hold = (oneswitch_timer>=settings.tap_delay) ? 1:0;
         if (oneswitch_timer > settings.tap_delay)
            oneswitch_timer = settings.tap_delay;
      } else {
         input.oneswitch.tap = (oneswitch_timer &&
            oneswitch_timer < settings.tap_delay) ? 1 : 0;
         input.oneswitch.tap2 = 0;
         input.oneswitch.hold = 0;
         oneswitch_timer = 0;
      }

      // Handle five-tap gesture
      input.oneswitch.fivetap = 0;
      if (input.oneswitch.tap) {
         fivetap_count++;
         fivetap_timer = settings.fivetap_delay;
      }
      if (input.oneswitch.tap2) {
         fivetap_count = 0;
         fivetap_timer = 0;
      }
      if (fivetap_count == 5) {
         fivetap_count = 0;
         fivetap_timer = 0;
         input.oneswitch.fivetap = 1;
      }
      if (fivetap_timer > 0) {
         fivetap_timer--;
         if (fivetap_timer == 0)
            fivetap_count = 0;
      }
      if (input.oneswitch.fivetap) {
         input.oneswitch.tap = 0;
         input.oneswitch.tap2 = 0;
         input.oneswitch.hold = 0;
      }

      // Let pause menu work as usual
      input.player.press[PL_INPUT_PAUSE] = keys[SDL_SCANCODE_ESCAPE].pressed;
      input.player.hold[PL_INPUT_PAUSE] = keys[SDL_SCANCODE_ESCAPE].held;

      // And make it also respond to the five-tap gesture
      input.player.press[PL_INPUT_PAUSE] |= input.oneswitch.fivetap;
   }

   // Update all inputs for player actions in normal mode
   else for (unsigned action = 0; action < NUM_PL_INPUT; action++) {
      // Flag that gets set if *any* of the possible ways to enter an action
      // is triggered.
      int pressed = 0;

      // Check all keyboard assignments for this action
      for (unsigned i = 0; i < KEYS_PER_ACTION; i++) {
         // Retrieve next key we can use for this action
         // If no key was assigned for this slot, skip it
         KeyCode key_code = settings.keyboard[action][i];
         if (key_code == NO_KEY) continue;

         // Check if the key was pressed
         if (keys[key_code].held)
            pressed = 1;
      }

      // Check all joystick assignments for this action
      for (unsigned i = 0; i < KEYS_PER_ACTION; i++)
         pressed |= check_joyin(settings.joystick[action][i]);

      // Update the status for this action
      if (!settings.replay) {
         input.player.press[action] = pressed && !input.player.hold[action];
         input.player.hold[action] = pressed;
      }
   }

   // Menu keys
   input.menu.up = keys[SDL_SCANCODE_UP].typed;
   input.menu.down = keys[SDL_SCANCODE_DOWN].typed;
   input.menu.left = keys[SDL_SCANCODE_LEFT].typed;
   input.menu.right = keys[SDL_SCANCODE_RIGHT].typed;
   input.menu.accept = keys[SDL_SCANCODE_RETURN].typed ||
      keys[SDL_SCANCODE_KP_ENTER].typed || keys[SDL_SCANCODE_SPACE].typed;
   input.menu.cancel = keys[SDL_SCANCODE_ESCAPE].typed;

   // Editor movement keys
   if (flags == 0 || flags == MOD_SHIFT) {
      input.editor.up = keys[SDL_SCANCODE_UP].held ? 1 : 0;
      input.editor.down = keys[SDL_SCANCODE_DOWN].held ? 1 : 0;
      input.editor.left = keys[SDL_SCANCODE_LEFT].held ? 1 : 0;
      input.editor.right = keys[SDL_SCANCODE_RIGHT].held ? 1 : 0;
      input.editor.fast = (flags == MOD_SHIFT) ? 1 : 0;
   }

   input.editor.up |= check_joyin(JOYIN_AXISNEG|1);
   input.editor.down |= check_joyin(JOYIN_AXISPOS|1);
   input.editor.left |= check_joyin(JOYIN_AXISNEG|0);
   input.editor.right |= check_joyin(JOYIN_AXISPOS|0);
   input.editor.up |= check_joyin(JOYIN_HATUP|0);
   input.editor.down |= check_joyin(JOYIN_HATDOWN|0);
   input.editor.left |= check_joyin(JOYIN_HATLEFT|0);
   input.editor.right |= check_joyin(JOYIN_HATRIGHT|0);

   input.editor.fast |= check_joyin(JOYIN_BUTTON|2);

   // Switching objects
   input.editor.next = keys[SDL_SCANCODE_UP].typed && flags == MOD_CTRL;
   input.editor.prev = keys[SDL_SCANCODE_DOWN].typed && flags == MOD_CTRL;

   input.editor.next |=
      input.rawpress.joystick == (JOYIN_BUTTON|6) ? 1 : 0;
   input.editor.prev |=
      input.rawpress.joystick == (JOYIN_BUTTON|7) ? 1 : 0;

   // Editor shortcuts
   input.editor.b_new = keys[SDL_GetScancodeFromKey(SDLK_n)]
      .typed && flags == MOD_CTRL;
   input.editor.b_load = keys[SDL_GetScancodeFromKey(SDLK_l)]
      .typed && flags == MOD_CTRL;
   input.editor.b_save = keys[SDL_GetScancodeFromKey(SDLK_s)]
      .typed && flags == MOD_CTRL;
   input.editor.b_info = keys[SDL_GetScancodeFromKey(SDLK_i)]
      .typed && flags == MOD_CTRL;
   input.editor.b_play = keys[SDL_GetScancodeFromKey(SDLK_F10)]
      .typed && flags == 0;
   input.editor.b_tilemap = keys[SDL_GetScancodeFromKey(SDLK_t)]
      .typed && flags == 0;
   input.editor.b_objects = keys[SDL_GetScancodeFromKey(SDLK_o)]
      .typed && flags == 0;
   input.editor.b_undo = keys[SDL_GetScancodeFromKey(SDLK_z)]
      .typed && flags == MOD_CTRL;
   input.editor.b_redo = keys[SDL_GetScancodeFromKey(SDLK_y)]
      .typed && flags == MOD_CTRL;
   input.editor.b_help = keys[SDL_GetScancodeFromKey(SDLK_F1)]
      .typed && flags == 0;
   input.editor.b_quit = keys[SDL_GetScancodeFromKey(SDLK_ESCAPE)]
      .typed && flags == 0;

   input.editor.b_tilemap |=
      input.rawpress.joystick == (JOYIN_BUTTON|4) ? 1 : 0;
   input.editor.b_objects |=
      input.rawpress.joystick == (JOYIN_BUTTON|4) ? 1 : 0;

   // IME keys
   input.ime.left = keys[SDL_SCANCODE_LEFT].typed;
   input.ime.right = keys[SDL_SCANCODE_RIGHT].typed;
   input.ime.back = keys[SDL_SCANCODE_BACKSPACE].typed;
   input.ime.del = keys[SDL_SCANCODE_DELETE].typed;
   input.ime.start = keys[SDL_SCANCODE_HOME].typed;
   input.ime.end = keys[SDL_SCANCODE_END].typed;

   // Take a screenshot?
   // This used to be a debug key (see below) but decided to make it
   // available without debug mode because it's too useful
   if (keys[SDL_SCANCODE_F12].typed && flags == MOD_SHIFT)
      take_screenshot();

   // Toggle fullscreen by pressing Alt+Enter
   if (keys[SDL_SCANCODE_RETURN].typed && flags == MOD_ALT)
      toggle_fullscreen();

   // When using the screen reader, press Shift to force the screen reader
   // to output the textbuffer again
   if (keys[SDL_SCANCODE_LSHIFT].typed ||
   keys[SDL_SCANCODE_RSHIFT].typed)
      repeat_reader_text();

   // Debug keys
   if (settings.debug) {
      // Pause game
      if ((keys[SDL_SCANCODE_PAUSE].typed && flags == 0) ||
      (keys[SDL_SCANCODE_P].typed && flags == MOD_CTRL))
         settings.pause ^= 1;

      // Record game
      if (keys[SDL_SCANCODE_F12].typed && flags == (MOD_SHIFT | MOD_CTRL)) {
         if (is_recording())
            stop_recording();
         else
            start_recording();
      }

      // Power-up giveaway
      input.debug.wings = keys[SDL_SCANCODE_F1].typed && flags == 0;
      input.debug.spider = keys[SDL_SCANCODE_F2].typed && flags == 0;
      input.debug.hammer = keys[SDL_SCANCODE_F3].typed && flags == 0;
      input.debug.parasol = keys[SDL_SCANCODE_F4].typed && flags == 0;

      // Toogle debug features
      if (keys[SDL_SCANCODE_F11].typed && flags == 0)
         settings.show_coord ^= 1;
      if (keys[SDL_SCANCODE_F10].typed && flags == 0)
         settings.free_move ^= 1;
      if (keys[SDL_SCANCODE_F9].typed && flags == 0)
         settings.show_hitbox ^= 1;
      if (keys[SDL_SCANCODE_F8].typed && flags == 0)
         settings.show_time ^= 1;
   }

   // Check if the cursor is inside our window
   // There's only one window so we don't need to check if it's exactly our
   // window (SDL won't know about stuff from other programs), just if it's
   // NULL or not
   SDL_Window *window = SDL_GetMouseFocus();
   input.cursor.inside = window != NULL ? 1 : 0;

   // Keep track of the previous cursor coordinates
   // We need this to know if the cursor moved
   int old_mouse_x = input.cursor.x;
   int old_mouse_y = input.cursor.y;

   // Only mess with the cursor when not in one-switch mode
   //if (!settings.one_switch) {
      // Program tried to move the mouse? :o
      if (input.cursor.old_x != input.cursor.x ||
      input.cursor.old_y != input.cursor.y) {
         int window_width;
         int window_height;
         SDL_GetWindowSize(window, &window_width, &window_height);

         int x = input.cursor.x * (window_width / screen_w);
         int y = input.cursor.y * (window_height / screen_h);
         SDL_WarpMouseInWindow(window, x, y);

         if (input.cursor.x >= 0 && input.cursor.x < screen_w &&
         input.cursor.y >= 0 && input.cursor.y < screen_h)
            input.cursor.inside = 1;
         else
            input.cursor.inside = 0;
      }

      // Cursor inside the window? Retrieve the cursor data then!
      if (input.cursor.inside) {
         // Calculate position of the cursor in the virtual resolution
         // Note: borders taken care of in handle_mouse_motion
         int window_width;
         int window_height;
         SDL_GetWindowSize(window, &window_width, &window_height);
         input.cursor.x = mouse.x / (window_width / screen_w);
         input.cursor.y = mouse.y / (window_height / screen_h);

         // Set the status of the buttons
         input.cursor.click = mouse.left && !input.cursor.left ? 1 : 0;
         input.cursor.left = mouse.left;
         input.cursor.middle = mouse.middle;
         input.cursor.right = mouse.right;

         // Set the status of the wheel
         input.cursor.wheel_up = mouse.wheel_up;
         input.cursor.wheel_down = mouse.wheel_down;
      }

      // Cursor outside the window? Fill in bogus data then
      else {
         input.cursor.x = -1;
         input.cursor.y = -1;
         input.cursor.left = 0;
         input.cursor.middle = 0;
         input.cursor.right = 0;
         input.cursor.click = 0;
         input.cursor.wheel_up = 0;
         input.cursor.wheel_down = 0;
      }

      // Check if the cursor moved
      input.cursor.moved = (input.cursor.x != old_mouse_x ||
         input.cursor.y != old_mouse_y) ? 1 : 0;
   //}

   // Check mouse-switch input
   {
      // Get raw input
      int up = (input.cursor.y <= (int)(screen_cy - settings.mousesw_y));
      int down = (input.cursor.y >= (int)(screen_cy + settings.mousesw_y));
      int left = (input.cursor.x <= (int)(screen_cx - settings.mousesw_x));
      int right = (input.cursor.x >= (int)(screen_cx + settings.mousesw_x));
      int pause = input.cursor.right;

      // Set press status
      input.mouseswitch.up = (!input.mouseswitch.up && up) ? 2 : 0;
      input.mouseswitch.down = (!input.mouseswitch.down && down) ? 2 : 0;
      input.mouseswitch.left = (!input.mouseswitch.left && left) ? 2 : 0;
      input.mouseswitch.right = (!input.mouseswitch.right && right) ? 2 : 0;
      input.mouseswitch.pause = (!input.mouseswitch.pause && pause) ? 2 : 0;

      // Set hold status
      input.mouseswitch.up |= (up ? 1 : 0);
      input.mouseswitch.down |= (down ? 1 : 0);
      input.mouseswitch.left |= (left ? 1 : 0);
      input.mouseswitch.right |= (right ? 1 : 0);
      input.mouseswitch.pause |= (pause ? 1 : 0);
   }

   // Editor cursor controls, handled here as we need the cursor information
   // to process these (as they make use of the mouse). Use the keyboard
   // modifiers to get different actions for each button.
   input.editor.place = (input.cursor.left && !flags) ? 1 : 0;
   input.editor.remove = (input.cursor.right && !flags) ? 1 : 0;
   input.editor.flip = (input.cursor.click && !flags) ? 1 : 0;
   input.editor.spawn = (input.cursor.left && flags == MOD_SHIFT) ? 1 : 0;
   input.editor.pick = (input.cursor.middle && !flags) ? 1 : 0;
   input.editor.pick |= (input.cursor.left && flags == MOD_CTRL) ? 1 : 0;

   input.editor.place |=
      (keys[SDL_SCANCODE_SPACE].held && !flags) ? 1 : 0;
   input.editor.remove |=
      (keys[SDL_SCANCODE_DELETE].held && !flags) ? 1 : 0;
   input.editor.flip |=
      (keys[SDL_SCANCODE_SPACE].typed && !flags) ? 1 : 0;
   input.editor.pick |=
      (keys[SDL_SCANCODE_SPACE].held && flags == MOD_CTRL) ? 1 : 0;

   input.editor.place |= check_joyin(JOYIN_BUTTON|0);
   input.editor.flip |= (input.rawpress.joystick == (JOYIN_BUTTON|0)) ? 1 : 0;
   input.editor.remove |= check_joyin(JOYIN_BUTTON|1);
   input.editor.pick |= check_joyin(JOYIN_BUTTON|3);

   // Editor GUI controls
   input.editor.ui_up = keys[SDL_SCANCODE_UP].typed && !flags;
   input.editor.ui_down = keys[SDL_SCANCODE_DOWN].typed && !flags;
   input.editor.ui_xup = keys[SDL_SCANCODE_PAGEUP].typed && !flags;
   input.editor.ui_xdown = keys[SDL_SCANCODE_PAGEDOWN].typed && !flags;
   input.editor.ui_home = keys[SDL_SCANCODE_HOME].typed && !flags;
   input.editor.ui_end = keys[SDL_SCANCODE_END].typed && !flags;
   input.editor.ui_next = keys[SDL_SCANCODE_TAB].typed && !flags;
   input.editor.ui_prev = keys[SDL_SCANCODE_TAB].typed && (flags == MOD_SHIFT);

   input.editor.ui_up |= (input.rawpress.joystick == (JOYIN_AXISNEG|1));
   input.editor.ui_down |= (input.rawpress.joystick == (JOYIN_AXISPOS|1));
   input.editor.ui_up |= (input.rawpress.joystick == (JOYIN_HATUP|0));
   input.editor.ui_down |= (input.rawpress.joystick == (JOYIN_HATDOWN|0));
   input.editor.ui_next |= (input.rawpress.joystick == (JOYIN_BUTTON|5));
   input.editor.ui_prev |= (input.rawpress.joystick == (JOYIN_BUTTON|4));

   // Reset one-shot statuses for all keys now that we handled them
   for (unsigned i = 0; i < SDL_NUM_SCANCODES; i++) {
      keys[i].pressed = 0;
      keys[i].typed = 0;
   }

   // Same goes for raw presses :P
   input.rawpress.keyboard = NO_KEY;
   input.rawpress.joystick = JOYIN_NONE;

   // And IME input
   if (input.ime.entered != NULL) {
      free(input.ime.entered);
      input.ime.entered = NULL;
   }

   // Pause key behaves nothing like the others, only having a press event
   // and not a release event. We need to explicitly release it now for that
   // reason :/
   keys[SDL_SCANCODE_PAUSE].held = 0;

   // Check if input is idle
   if (settings.one_switch) {
      input.idle = 0;
   } else {
      input.idle = 1;

      for (unsigned i = 0; i < SDL_NUM_SCANCODES; i++)
         if (keys[i].held) { input.idle = 0; break; }

      for (unsigned i = 0; i < MAX_AXES; i++)
         if (joystatus.axis[i] != 0) { input.idle = 0; break; }
      for (unsigned i = 0; i < MAX_HATS; i++)
         if (joystatus.hats[i] != 0) { input.idle = 0; break; }
      for (unsigned i = 0; i < MAX_BUTTONS; i++)
         if (joystatus.buttons[i] != 0) { input.idle = 0; break; }

      if (input.cursor.moved)
         input.idle = 0;
   }

   // Set these two so later we can check for a mismatch
   // A mismatch means the program tried to move the cursor on its own
   input.cursor.old_x = input.cursor.x;
   input.cursor.old_y = input.cursor.y;
}

//***************************************************************************
// add_joystick
// Adds a joystick to the joystick list.
//---------------------------------------------------------------------------
// param index: joystick index
//***************************************************************************

void add_joystick(int index) {
   // Allocate more memory for joystick
   num_joysticks++;
   joysticks = (Joystick *) realloc(joysticks,
      sizeof(Joystick) * num_joysticks);
   if (joysticks == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   unsigned id = num_joysticks-1;

   // Open joystick
   joysticks[id].joystick = SDL_JoystickOpen(index);
   if (joysticks[id].joystick == NULL) {
      num_joysticks--;
      return;
   }

#ifdef DEBUG
   // Get the joystick GUID
   SDL_JoystickGUID guid = SDL_JoystickGetGUID(joysticks[id].joystick);
   fputs("Joystick GUID:", stdout);
   for (unsigned i = 0; i < 0x10; i++) {
      if (i % 4 == 0) putchar(' ');
      printf("%02X", guid.data[i]);
   }
   printf(" \"%s\"", SDL_JoystickName(joysticks[id].joystick));
   putchar('\n');
#endif

   // Get joystick instance
   joysticks[id].instance = SDL_JoystickInstanceID(joysticks[id].joystick);
   if (joysticks[id].instance == -1) {
      fputs("Error: can't open joystick!\n", stderr);
      SDL_JoystickClose(joysticks[id].joystick);
      num_joysticks--;
      return;
   }

   // Get controller mappings, if they're available
   joysticks[id].controller = SDL_GameControllerOpen(index);
   if (joysticks[id].controller != NULL) {
      // Get the mappings for all axes
      for (int i = 0; i < SDL_CONTROLLER_AXIS_MAX; i++) {
         SDL_GameControllerButtonBind bind =
            SDL_GameControllerGetBindForAxis(joysticks[id].controller, i);

         // At this point I gave up on commenting because I don't even
         // understand how this is supposed to be used. The joystick API in
         // SDL2 is so broken I'm surprised they didn't just throw it away
         // and replace it with the game controller API altogether. They'll
         // probably do that in an update...
         switch (bind.bindType) {
            case SDL_CONTROLLER_BINDTYPE_BUTTON:
               joysticks[id].mappings.axes[i] = JOYIN_BUTTON |
                  bind.value.button;
               break;

            case SDL_CONTROLLER_BINDTYPE_AXIS:
               joysticks[id].mappings.axes[i] = JOYIN_AXISPOS |
                  bind.value.axis;
               break;

            case SDL_CONTROLLER_BINDTYPE_HAT:
               joysticks[id].mappings.axes[i] = bind.value.hat.hat;
               switch (bind.value.hat.hat_mask) {
                  case SDL_HAT_UP:
                     joysticks[id].mappings.axes[i] |= JOYIN_HATUP;
                     break;
                  case SDL_HAT_DOWN:
                     joysticks[id].mappings.axes[i] |= JOYIN_HATDOWN;
                     break;
                  case SDL_HAT_LEFT:
                     joysticks[id].mappings.axes[i] |= JOYIN_HATLEFT;
                     break;
                  case SDL_HAT_RIGHT:
                     joysticks[id].mappings.axes[i] |= JOYIN_HATRIGHT;
                     break;
                  default:
                     joysticks[id].mappings.axes[i] = 0;
                     break;
               }
               break;

            default:
               joysticks[id].mappings.axes[i] = JOYIN_NONE;
               break;
         }
      }

      // Get the mappings for all buttons
      for (int i = 0; i < SDL_CONTROLLER_BUTTON_MAX; i++) {
         SDL_GameControllerButtonBind bind =
            SDL_GameControllerGetBindForButton(joysticks[id].controller, i);

         // At this point I gave up on commenting because I don't even
         // understand how this is supposed to be used. The joystick API in
         // SDL2 is so broken I'm surprised they didn't just throw it away
         // and replace it with the game controller API altogether. They'll
         // probably do that in an update...
         switch (bind.bindType) {
            case SDL_CONTROLLER_BINDTYPE_BUTTON:
               joysticks[id].mappings.buttons[i] = JOYIN_BUTTON |
                  bind.value.button;
               break;

            case SDL_CONTROLLER_BINDTYPE_AXIS:
               joysticks[id].mappings.buttons[i] = JOYIN_AXISPOS |
                  bind.value.axis;
               break;

            case SDL_CONTROLLER_BINDTYPE_HAT:
               joysticks[id].mappings.buttons[i] = bind.value.hat.hat;
               switch (bind.value.hat.hat_mask) {
                  case SDL_HAT_UP:
                     joysticks[id].mappings.buttons[i] |= JOYIN_HATUP;
                     break;
                  case SDL_HAT_DOWN:
                     joysticks[id].mappings.buttons[i] |= JOYIN_HATDOWN;
                     break;
                  case SDL_HAT_LEFT:
                     joysticks[id].mappings.buttons[i] |= JOYIN_HATLEFT;
                     break;
                  case SDL_HAT_RIGHT:
                     joysticks[id].mappings.buttons[i] |= JOYIN_HATRIGHT;
                     break;
                  default:
                     joysticks[id].mappings.buttons[i] = 0;
                     break;
               }
               break;

            default:
               joysticks[id].mappings.buttons[i] = JOYIN_NONE;
               break;
         }
      }
   }
}

//***************************************************************************
// handle_key_event
// Handles a key event (e.g. key goes down, goes up, gets repeated).
//---------------------------------------------------------------------------
// param scancode: scancode of affected key
// param way: non-zero for key-down, zero for key-up
//***************************************************************************

void handle_key_event(int scancode, int way) {
   // Huh...
   if (scancode < 0 || scancode >= SDL_NUM_SCANCODES)
      return;

   // Key down?
   if (way) {
      keys[scancode].pressed = !keys[scancode].held;
      keys[scancode].typed = 1;
      keys[scancode].held = 1;
   }

   // Key up?
   else {
      keys[scancode].pressed = 0;
      keys[scancode].typed = 0;
      keys[scancode].held = 0;
   }

   // Update raw input
   if (keys[scancode].pressed)
      input.rawpress.keyboard = scancode;

#ifdef _WIN32
   // Workaround for Alt+F4 on Windows *sigh*
   if (keys[scancode].pressed && scancode == SDL_SCANCODE_F4 &&
   (get_mod_flags() & MOD_ALT)) {
      SDL_Event event;
      event.type = SDL_QUIT;
      SDL_PushEvent(&event);
   }
#endif
}

//***************************************************************************
// handle_joy_axis
// Handles a joystick axis event (i.e. axis moved).
//---------------------------------------------------------------------------
// param axis: axis which moved
// param value: current position (-32768..32767)
// param instance: joystick instance
//***************************************************************************

void handle_joy_axis(int axis, int value, int instance) {
   // Disable axes?
   if (settings.no_joystick_axis)
      return;

   // Huh...
   if (axis < 0 || axis >= MAX_AXES)
      return;

   // Convert value to a digital position
   if (value < -10000)
      value = -1;
   else if (value > 10000)
      value = 1;
   else
      value = 0;

   // Determine if this instance belongs to a known joystick
   unsigned id;
   for (id = 0; id < num_joysticks; id++)
      if (joysticks[id].instance == instance)
         break;
   if (id == num_joysticks)
      return;

   // Handle menu movement assuming a reasonable mapping when not using a
   // game controller
   if ((id >= num_joysticks || joysticks[id].controller == NULL) &&
   axis >= 0 && axis <= 3) {
      int which = axis & 1;
      if (which == 1 && value == -1 && joystatus.axis[axis] != -1)
         input.menu.up = 1;
      if (which == 1 && value == 1 && joystatus.axis[axis] != 1)
         input.menu.down = 1;
      if (which == 0 && value == -1 && joystatus.axis[axis] != -1)
         input.menu.left = 1;
      if (which == 0 && value == 1 && joystatus.axis[axis] != 1)
         input.menu.right = 1;
   }

   // Update raw input
   if (value != 0 && value != joystatus.axis[axis]) {
      if (value > 0)
         input.rawpress.joystick = JOYIN_AXISPOS|axis;
      else
         input.rawpress.joystick = JOYIN_AXISNEG|axis;
   }

   // Store new value
   joystatus.axis[axis] = value;
}

//***************************************************************************
// handle_joy_hat
// Handles a joystick hat event (i.e. hat moved).
//---------------------------------------------------------------------------
// param hat: hat which moved
// param value: see SDL_HAT_*
// param instance: joystick instance
//***************************************************************************

void handle_joy_hat(int hat, int value, int instance) {
   // Huh...
   if (hat < 0 || hat >= MAX_HATS)
      return;

   // Determine if this instance belongs to a known joystick
   unsigned id;
   for (id = 0; id < num_joysticks; id++)
      if (joysticks[id].instance == instance)
         break;
   if (id == num_joysticks)
      return;

   // Handle menu movement assuming a reasonable mapping when not using a
   // game controller
   if ((id >= num_joysticks || joysticks[id].controller == NULL) &&
   (hat == 0 || hat == 1)) {
      if ((value & SDL_HAT_UP) && !(joystatus.hats[hat] & SDL_HAT_UP))
         input.menu.up = 1;
      if ((value & SDL_HAT_DOWN) && !(joystatus.hats[hat] & SDL_HAT_DOWN))
         input.menu.down = 1;
      if ((value & SDL_HAT_LEFT) && !(joystatus.hats[hat] & SDL_HAT_LEFT))
         input.menu.left = 1;
      if ((value & SDL_HAT_RIGHT) && !(joystatus.hats[hat] & SDL_HAT_RIGHT))
         input.menu.right = 1;
   }

   // Update raw input
   int changed = value;
   changed ^= joystatus.hats[hat];
   changed &= ~joystatus.hats[hat];
   if (changed) {
      if (changed & SDL_HAT_UP)
         input.rawpress.joystick = JOYIN_HATUP|hat;
      else if (changed & SDL_HAT_DOWN)
         input.rawpress.joystick = JOYIN_HATDOWN|hat;
      else if (changed & SDL_HAT_LEFT)
         input.rawpress.joystick = JOYIN_HATLEFT|hat;
      else if (changed & SDL_HAT_RIGHT)
         input.rawpress.joystick = JOYIN_HATRIGHT|hat;
   }

   // Store new value
   joystatus.hats[hat] = value;
}

//***************************************************************************
// handle_joy_button
// Handles a joystick button event (e.g. button goes down, goes up).
//---------------------------------------------------------------------------
// param button: button which got pressed/released
// param way: non-zero for pressed, zero for released
// param instance: joystick instance
//***************************************************************************

void handle_joy_button(int button, int way, int instance) {
   // Huh...
   if (button < 0 || button >= MAX_BUTTONS)
      return;

   // Determine if this instance belongs to a known joystick
   unsigned id;
   for (id = 0; id < num_joysticks; id++)
      if (joysticks[id].instance == instance)
         break;
   if (id == num_joysticks)
      return;

   // Handle menu input assuming a reasonable mapping when not using a
   // game controller
   if ((id >= num_joysticks || joysticks[id].controller == NULL) &&
   button >= 0 && button <= 3) {
      int which = button & 1;
      if (!which && way && !joystatus.buttons[button])
         input.menu.accept = 1;
      if (which && way && !joystatus.buttons[button])
         input.menu.cancel = 1;
   }

   // Update raw input
   if (way && !joystatus.buttons[button])
      input.rawpress.joystick = JOYIN_BUTTON|button;

   // Store new value
   joystatus.buttons[button] = way;
}

//***************************************************************************
// handle_controller_axis
// Handles a game controller axis event (i.e. axis moved).
//---------------------------------------------------------------------------
// param axis: axis which moved
// param value: current position (-32768..32767)
// param instance: joystick instance
//***************************************************************************

void handle_controller_axis(int axis, int value, int instance) {
   // Disable axes?
   if (settings.no_joystick_axis)
      return;

   // Determine if this instance belongs to a known game controller
   unsigned id;
   for (id = 0; id < num_joysticks; id++)
      if (joysticks[id].instance == instance)
         break;
   if (id == num_joysticks)
      return;
   if (joysticks[id].controller == NULL)
      return;

   // Turn value into a digital one
   if (value < -10000)
      value = -1;
   else if (value > 10000)
      value = 1;
   else
      value = 0;

   // Handle menu input
   switch (axis) {
      // Horizontal
      case SDL_CONTROLLER_AXIS_LEFTX:
      case SDL_CONTROLLER_AXIS_RIGHTX:
         if (value != joysticks[id].axes[axis]) {
            if (value < 0)
               input.menu.left = 1;
            if (value > 0)
               input.menu.right = 1;
         }
         break;

      // Vertical
      case SDL_CONTROLLER_AXIS_LEFTY:
      case SDL_CONTROLLER_AXIS_RIGHTY:
         if (value != joysticks[id].axes[axis]) {
            if (value < 0)
               input.menu.up = 1;
            if (value > 0)
               input.menu.down = 1;
         }
         break;

      // Ignore the rest
      default:
         break;
   }

   // Update its axis value
   joysticks[id].axes[axis] = value;
}

//***************************************************************************
// handle_controller_button
// Handles a game controller button
//---------------------------------------------------------------------------
// param button: button which got pressed/released
// param way: non-zero for pressed, zero for released
//***************************************************************************

void handle_controller_button(int button, int way) {
   // Handle menu input
   if (way) {
      switch (button) {
         // Movement
         case SDL_CONTROLLER_BUTTON_DPAD_UP:
            input.menu.up = 1;
            break;
         case SDL_CONTROLLER_BUTTON_DPAD_DOWN:
            input.menu.down = 1;
            break;
         case SDL_CONTROLLER_BUTTON_DPAD_LEFT:
            input.menu.left = 1;
            break;
         case SDL_CONTROLLER_BUTTON_DPAD_RIGHT:
            input.menu.right = 1;
            break;

         // Accept
         case SDL_CONTROLLER_BUTTON_A:
         case SDL_CONTROLLER_BUTTON_Y:
            input.menu.accept = 1;
            break;

         // Cancel
         case SDL_CONTROLLER_BUTTON_B:
         case SDL_CONTROLLER_BUTTON_X:
            input.menu.cancel = 1;
            break;

         // Ignore the rest
         default:
            break;
      }
   }
}

//***************************************************************************
// handle_mouse_motion
// Handles a mouse motion event (i.e. the cursor moved).
//---------------------------------------------------------------------------
// param x: new X coordinate
// param y: new Y coordinate
// param inside: non-zero if inside window, zero if outside window
//***************************************************************************

void handle_mouse_motion(int x, int y, int inside) {
   // Relative mouse mode hack? (used due to SDL fullscreen support being
   // broken under X11)
   // <Sik> Wait a second, is this hack implemented anymore? I thought it was
   // removed when the fullscreen method was changed
   if (SDL_GetRelativeMouseMode()) {
      // Add coordinates to the previous ones
      // Assume the cursor is always inside the window, since this hack is
      // only applied in fullscreen and you shouldn't be interacting with
      // other windows if this one has your focus.
      mouse.x += x;
      mouse.y += y;

      // Make sure the cursor stays within the screen
      if (mouse.x < 0)
         mouse.x = 0;
      if (mouse.y < 0)
         mouse.y = 0;
      if (mouse.x >= settings.fullscreen_width)
         mouse.x = settings.fullscreen_width - 1;
      if (mouse.y >= settings.fullscreen_height)
         mouse.y = settings.fullscreen_height - 1;
   }

   // Normal mouse handling
   else if (inside) {
      // Inside the window, store coordinates as-is
      mouse.x = x;
      mouse.y = y;

      // In fullscreen it's possible for the cursor to end up in an area with
      // a border, so clamp the cursor coordinates to prevent it from going
      // there (to make it look like the screen ends in the border)
      // Note: we use is_fullscreen because we need to make sure this belongs
      // to the current status of the window and not what the settings say
      // (they can mismatch in the video menu)
      if (is_fullscreen()) {
         // Calculate maximum coordinates
         int max_x = settings.fullscreen_width / screen_w * screen_w - 1;
         int max_y = settings.fullscreen_height / screen_h * screen_h - 1;

         // Clamp coordinates if needed
         if (mouse.x > max_x) {
            mouse.x = max_x;
            SDL_WarpMouseInWindow(NULL, mouse.x, mouse.y);
         }
         if (mouse.y > max_y) {
            mouse.y = max_y;
            SDL_WarpMouseInWindow(NULL, mouse.x, mouse.y);
         }
      }
   } else {
      // Outside the window, store bogus coordinates
      mouse.x = -1;
      mouse.y = -1;
   }
}

//***************************************************************************
// handle_mouse_button
// Handles a mouse button event (e.g. a button was pressed).
//---------------------------------------------------------------------------
// param button: which button was pressed
// param way: non-zero for pressed, zero for released
//***************************************************************************

void handle_mouse_button(int button, int way) {
   // Just in case...
   way = way ? 1 : 0;

   // Update the affected button as needed
   switch (button) {
      case SDL_BUTTON_LEFT: mouse.left = way; break;
      case SDL_BUTTON_RIGHT: mouse.right = way; break;
      case SDL_BUTTON_MIDDLE: mouse.middle = way; break;
      case -1: mouse.wheel_up = way; break;
      case -2: mouse.wheel_down = way; break;
      default: break;
   }
}

//***************************************************************************
// handle_ime_entered
// Handles text being entered through IME
//---------------------------------------------------------------------------
// param text: text being entered (must be UTF-8!)
//***************************************************************************

void handle_ime_entered(const char *text) {
   // Get rid of old text if needed
   if (input.ime.entered != NULL)
      free(input.ime.entered);

   // Store new text
   input.ime.entered = malloc(strlen(text)+1);
   if (input.ime.entered == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   strcpy(input.ime.entered, text);
}

//***************************************************************************
// check_joyin [internal]
// Checks if a given joystick input (button, axis, hat) is active
//---------------------------------------------------------------------------
// param id: joystick mapping
// return: 1 if active, 0 if not
//***************************************************************************

static int check_joyin(uint16_t id) {
   // Get which input
   uint16_t type = id & 0xFF00;
   uint16_t which = id & 0x00FF;

   // Disable joystick axes?
   if (settings.no_joystick_axis &&
   (type == JOYIN_AXISPOS || type == JOYIN_AXISNEG))
      return 0;

   // Figure out how to process it
   switch (type) {
      // Button
      case JOYIN_BUTTON:
         return joystatus.buttons[which] ? 1 : 0;

      // Axis
      case JOYIN_AXISPOS:
         return joystatus.axis[which] > 0 ? 1 : 0;
      case JOYIN_AXISNEG:
         return joystatus.axis[which] < 0 ? 1 : 0;

      // Hat
      case JOYIN_HATUP:
         return joystatus.hats[which] & SDL_HAT_UP ? 1 : 0;
      case JOYIN_HATDOWN:
         return joystatus.hats[which] & SDL_HAT_DOWN ? 1 : 0;
      case JOYIN_HATLEFT:
         return joystatus.hats[which] & SDL_HAT_LEFT ? 1 : 0;
      case JOYIN_HATRIGHT:
         return joystatus.hats[which] & SDL_HAT_RIGHT ? 1 : 0;

      // Errr...
      default:
         return 0;
   }
}

//***************************************************************************
// get_mod_flags [internal]
// Retrieves which modifier keys are being held down at the moment.
//---------------------------------------------------------------------------
// return: an OR of different MOD_* flags as relevant
//---------------------------------------------------------------------------
// notes: doing it this way because it seems SDL sometimes screws up with the
// modifier flags which causes shortcuts to malfunction and such. This is
// ugly, but does the job.
// to-do: check if keycodes should be used instead?
//***************************************************************************

static unsigned get_mod_flags(void) {
   unsigned flags = 0;

   if (keys[SDL_SCANCODE_LSHIFT].held || keys[SDL_SCANCODE_RSHIFT].held)
      flags |= MOD_SHIFT;
   if (keys[SDL_SCANCODE_LCTRL].held || keys[SDL_SCANCODE_RCTRL].held)
      flags |= MOD_CTRL;
   if (keys[SDL_SCANCODE_LALT].held || keys[SDL_SCANCODE_RALT].held)
      flags |= MOD_ALT;

   return flags;
}

//***************************************************************************
// remove_joystick
// Removes a joystick from the joystick list
//---------------------------------------------------------------------------
// param instance: instance of joystick
//***************************************************************************

void remove_joystick(int instance) {
   // Search for all joysticks
   for (unsigned i = 0; i < num_joysticks; i++) {
      // Found it?
      if (joysticks[i].instance == instance) {
         // Close controller
         if (joysticks[i].controller != NULL)
            SDL_GameControllerClose(joysticks[i].controller);

         // Close joystick
         if (joysticks[i].joystick != NULL)
            SDL_JoystickClose(joysticks[i].joystick);

         // Push all other joysticks back
         for (unsigned j = i+1; j < num_joysticks; j++)
            joysticks[i] = joysticks[j];

         // Done
         num_joysticks--;
         return;
      }
   }
}

//***************************************************************************
// set_joystick_defaults
// Assigns default mappings to the joystick input (unless the controls were
// already mapped). The mappings will depend on whatever is connected at the
// moment. Mostly used for the first-time init, but can be also used when the
// joystick mappings get whipped.
//***************************************************************************

void set_joystick_defaults(void) {
#ifdef DEBUG
   fputs("Updating joystick controls\n", stderr);
#endif

   // Check if there's a mapped controller to latch onto
   int controller = -1;
   for (unsigned i = 0; i < num_joysticks; i++)
   if (joysticks[i].controller != NULL) {
      controller = i;
      break;
   }

   // Do we have a controller with mappings?
   if (controller != -1)
   for (PlayerInputID i = 0; i < NUM_PL_INPUT; i++) {
      // If the primary input isn't mapped, give it a default
      if ((settings.joystick[i][0] & 0xFF00) == JOYIN_UNDEF) {
         // For the sake of readability
         const Joystick *j = &joysticks[controller];

         // Select a reasonable default
         uint16_t value;
         switch (i) {
            case PL_INPUT_UP: value =
               j->mappings.axes[SDL_CONTROLLER_AXIS_LEFTY]; break;
            case PL_INPUT_DOWN: value =
               j->mappings.axes[SDL_CONTROLLER_AXIS_LEFTY]; break;
            case PL_INPUT_LEFT: value =
               j->mappings.axes[SDL_CONTROLLER_AXIS_LEFTX]; break;
            case PL_INPUT_RIGHT: value =
               j->mappings.axes[SDL_CONTROLLER_AXIS_LEFTX]; break;
            case PL_INPUT_ACTION: value =
               j->mappings.buttons[SDL_CONTROLLER_BUTTON_A]; break;
            case PL_INPUT_PAUSE: value =
               j->mappings.buttons[SDL_CONTROLLER_BUTTON_START]; break;
            default: value =
               JOYIN_NONE; break;
         }

         // Flip axis in the case of up and left (those two need to be
         // negative, but the mappings can't distinguish that)
         if ((i == PL_INPUT_UP || i == PL_INPUT_LEFT) &&
         (value & 0xFF00) == JOYIN_AXISPOS)
            value += 0x100;

         // Store chosen value
         settings.joystick[i][0] = value;
      }

      // If the primary input isn't mapped, give it a default
      if ((settings.joystick[i][1] & 0xFF00) == JOYIN_UNDEF) {
         // For the sake of readability
         const Joystick *j = &joysticks[controller];

         // Select a reasonable default
         uint16_t value;
         switch (i) {
            case PL_INPUT_UP: value =
               j->mappings.buttons[SDL_CONTROLLER_BUTTON_DPAD_UP]; break;
            case PL_INPUT_DOWN: value =
               j->mappings.buttons[SDL_CONTROLLER_BUTTON_DPAD_DOWN]; break;
            case PL_INPUT_LEFT: value =
               j->mappings.buttons[SDL_CONTROLLER_BUTTON_DPAD_LEFT]; break;
            case PL_INPUT_RIGHT: value =
               j->mappings.buttons[SDL_CONTROLLER_BUTTON_DPAD_RIGHT]; break;
            case PL_INPUT_ACTION: value =
               j->mappings.buttons[SDL_CONTROLLER_BUTTON_B]; break;
            case PL_INPUT_PAUSE: value =
               j->mappings.buttons[SDL_CONTROLLER_BUTTON_BACK]; break;
            default: value =
               JOYIN_NONE; break;
         }

         // Flip axis in the case of up and left (those two need to be
         // negative, but the mappings can't distinguish that)
         if ((i == PL_INPUT_UP || i == PL_INPUT_LEFT) &&
         (value & 0xFF00) == JOYIN_AXISPOS)
            value += 0x100;

         // Store chosen value
         settings.joystick[i][1] = value;
      }
   }

   // Scan all inputs
   else for (PlayerInputID i = 0; i < NUM_PL_INPUT; i++) {
      // If the primary input isn't mapped, give it a default
      if ((settings.joystick[i][0] & 0xFF00) == JOYIN_UNDEF) {
         // Select a reasonable default
         uint16_t value;
         switch (i) {
            case PL_INPUT_UP:     value = JOYIN_AXISNEG|1; break;
            case PL_INPUT_DOWN:   value = JOYIN_AXISPOS|1; break;
            case PL_INPUT_LEFT:   value = JOYIN_AXISNEG|0; break;
            case PL_INPUT_RIGHT:  value = JOYIN_AXISPOS|0; break;
            case PL_INPUT_ACTION: value = JOYIN_BUTTON|0;  break;
            case PL_INPUT_PAUSE:  value = JOYIN_BUTTON|8;  break;
            default:              value = JOYIN_NONE;      break;
         }

         // Store chosen value
         settings.joystick[i][0] = value;
      }

      // If the primary input isn't mapped, give it a default
      if ((settings.joystick[i][1] & 0xFF00) == JOYIN_UNDEF) {
         // Select a reasonable default
         uint16_t value;
         switch (i) {
            case PL_INPUT_UP:     value = JOYIN_HATUP|0;    break;
            case PL_INPUT_DOWN:   value = JOYIN_HATDOWN|0;  break;
            case PL_INPUT_LEFT:   value = JOYIN_HATLEFT|0;  break;
            case PL_INPUT_RIGHT:  value = JOYIN_HATRIGHT|0; break;
            case PL_INPUT_ACTION: value = JOYIN_BUTTON|1;   break;
            case PL_INPUT_PAUSE:  value = JOYIN_BUTTON|9;   break;
            default:              value = JOYIN_NONE;       break;
         }

         // Store chosen value
         settings.joystick[i][1] = value;
      }
   }
}

//***************************************************************************
// deinit_input
// Deinitializes the user input subsystem.
//***************************************************************************

void deinit_input(void) {
   // Unload cursor graphics
   if (gfxset_cursor) {
      destroy_graphics_set(gfxset_cursor);
      gfxset_cursor = NULL;
   }

   // Close joysticks
   if (joysticks != NULL) {
      for (unsigned i = 0; i < num_joysticks; i++) {
         // Close controller
         if (joysticks[i].controller != NULL)
            SDL_GameControllerClose(joysticks[i].controller);

         // Close joystick
         if (joysticks[i].joystick != NULL)
            SDL_JoystickClose(joysticks[i].joystick);
      }

      // Deallocate list
      free (joysticks);
      num_joysticks = 0;
      joysticks = NULL;
   }

   // Deinitialize joystick subsystem
   SDL_QuitSubSystem(SDL_INIT_GAMECONTROLLER);
   SDL_QuitSubSystem(SDL_INIT_JOYSTICK);
}

//***************************************************************************
// id_to_scancode
// Converts a key ID into a scancode.
//---------------------------------------------------------------------------
// param key: key ID
// return: scancode
//***************************************************************************

int id_to_scancode(const char *key) {
   // Er...
   if (key == NULL || key[0] == '\0')
      return NO_KEY;

   // Letters or numbers? (handled separately)
   if (key[1] == '\0') {
      if (key[0] >= 'a' && key[0] <= 'z')
         return key[0] - 'a' + SDL_SCANCODE_A;
      if (key[0] >= '0' && key[0] <= '9')
         return key[0] - '0' + SDL_SCANCODE_0;
      return NO_KEY;
   }

   // Raw scancode?
   // To-do: check that it's a number
   if (key[0] == '?' && key[1] != '\0')
      return atoi(&key[1]);

   // Look for a match in the list
   for (unsigned i = 0; key_list[i].name != NULL; i++) {
      if (strcmp(key_list[i].name, key) == 0)
         return key_list[i].scancode;
   }

   // Not recognized...
   return NO_KEY;
}

//***************************************************************************
// scancode_to_id
// Converts a scancode into a key ID. The returned string is valid until the
// next call to this function.
//---------------------------------------------------------------------------
// param scancode: scancode
// return: key ID
//***************************************************************************

const char *scancode_to_id(int scancode) {
   // Where generated scancodes are stored
   static char buffer[0x10];

   // No key?
   if (scancode == NO_KEY)
      return "none";

   // Single-character key IDs?
   if (scancode >= SDL_SCANCODE_A && scancode <= SDL_SCANCODE_Z) {
      buffer[0] = scancode - SDL_SCANCODE_A + 'a';
      buffer[1] = '\0';
      return buffer;
   }
   if (scancode >= SDL_SCANCODE_0 && scancode <= SDL_SCANCODE_9) {
      buffer[0] = scancode - SDL_SCANCODE_0 + '0';
      buffer[1] = '\0';
      return buffer;
   }

   // Look for a match
   for (unsigned i = 0; key_list[i].name != NULL; i++) {
      if (key_list[i].scancode == scancode)
         return key_list[i].name;
   }

   // No match, return generic scancode
   sprintf(buffer, "?%d", scancode);
   return buffer;
}

//***************************************************************************
// get_key_name
// Gets the name of a key to show on screen. Note that the string may be
// valid only until the next call to get_key_name... Just use it to pass a
// name to draw_text_int directly! (pass the scancode as the integer)
//---------------------------------------------------------------------------
// param scancode: key scan code
// return: string to pass to draw_text_int
//***************************************************************************

const char *get_key_name(int scancode) {
   // Not assigned?
   if (scancode == NO_KEY)
      return "N/A";

   // See if it's a printable character
   // To-do: check that it adapts to the layout correctly
   // To-do: check if this works with symbols also
   int keycode = SDL_GetKeyFromScancode(scancode);
   const char *keyname = SDL_GetKeyName(keycode);
   if (keyname != NULL && keyname[0] != '\0' && keyname[1] == '\0')
      return keyname;

   // Look for key name from the list
   for (unsigned i = 0; key_list[i].name != NULL; i++) {
      if (scancode == key_list[i].scancode)
         return key_list[i].label;
   }

   // Unknown key
   return "?{param}";
}

//***************************************************************************
// set_cursor
// Sets which cursor to use. The cursor is drawn independently from whatever
// the game mode drawing function does and is always drawn where the mouse
// is.
//---------------------------------------------------------------------------
// param which: which cursor to use
//***************************************************************************

void set_cursor(Cursor which) {
   // If we aren't in fullscreen, then force the cursor to be shown
   if (!settings.fullscreen && !settings.one_switch && which == CURSOR_NONE)
      which = CURSOR_ARROW;

   // Is this cursor already set? (we check for this because changing the
   // OS cursor visibility could be slow and we don't want to kill the
   // framerate doing that all the time - yes, our code is sloppy and is
   // prone to call this function all the time *coughleveleditorcough*)
   if (cursor == which)
      return;

   // Store which type of cursor to show
   cursor = which;
}

//***************************************************************************
// draw_cursor
// Draws the cursor on screen, if set (and if the cursor is indeed inside the
// game window and out outside)
//***************************************************************************

void draw_cursor(void) {
#ifdef DEBUG
   // Debug crap indicating the state of each button
   // Should NOT be here, but it's the easiest place where to display this...
   if (get_mod_flags() & MOD_SHIFT)
      fill_rectangle(0x08, screen_h - 0x10, 0x18, screen_h - 8, 0x00FF00);
   if (get_mod_flags() & MOD_CTRL)
      fill_rectangle(0x20, screen_h - 0x10, 0x30, screen_h - 8, 0x0000FF);
   if (get_mod_flags() & MOD_ALT)
      fill_rectangle(0x38, screen_h - 0x10, 0x48, screen_h - 8, 0xFF0000);

   draw_rectangle(0x08, screen_h - 0x10, 0x18, screen_h - 8, 0x000000);
   draw_rectangle(0x20, screen_h - 0x10, 0x30, screen_h - 8, 0x000000);
   draw_rectangle(0x38, screen_h - 0x10, 0x48, screen_h - 8, 0x000000);
#endif

   // Cursor must be inside the screen...
   if (!settings.one_switch && !input.cursor.inside)
      return;

   // Draw cursor
   draw_sprite(cursor_spr[cursor],
      input.cursor.x - hotspots[cursor].x,
      input.cursor.y - hotspots[cursor].y,
      SPR_NOFLIP);
}
