//***************************************************************************
// "hazards.c"
// Code for non-enemy hazard objects.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include "enemies.h"
#include "ingame.h"
#include "level.h"
#include "objects.h"
#include "physics.h"
#include "player.h"
#include "settings.h"
#include "sound.h"
#include "tables.h"

// Bouncing hazard parameters
#define BOUNCING_SPEED 0x180        // Speed at which it moves
#define BOUNCING_WEIGHT 0x40        // Speed at which it falls
#define BOUNCING_BOUNCE 0x400       // How much it bounces

// Stalactite parameters
#define STALACTITE_RANGEX 0x20      // Horizontal range in which it triggers
#define STALACTITE_RANGEY 0x70      // Vertical range in which it triggers
#define STALACTITE_WEIGHT 0x30      // How fast it falls

// Coil parameters
#define COIL_WARMUP 0x20            // How long it takes to get ready
#define COIL_TIME_E 0x40            // Up to when electrocute in easy
#define COIL_TIME_N 0x50            // Up to when electrocute in normal
#define COIL_TIME_H 0x60            // Up to when electrocute in hard
#define COIL_CYCLE 0x80             // How long does the entire cycle take

//***************************************************************************
// init_spikes
// Initializes an spikes object pointing downwards.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_spikes(Object *obj) {
   // Each kind of spike has different properties depending on where they're
   // pointing (sprite is different, hitbox is located where the pointy part
   // of the spikes is).
   switch (obj->type) {
      // Spikes pointing up
      case OBJ_SPIKES_U:
         set_hitbox(obj, -16, 15, -17, -16);
         set_object_anim(obj, retrieve_object_anim(OB_ANIM_SPIKES_U));
         break;

      // Spikes pointing down
      case OBJ_SPIKES_D:
         set_hitbox(obj, -16, 15, 15, 16);
         set_object_anim(obj, retrieve_object_anim(OB_ANIM_SPIKES_D));
         break;

      // Spikes pointing left
      case OBJ_SPIKES_L:
         set_hitbox(obj, -17, -16, -16, 15);
         set_object_anim(obj, retrieve_object_anim(OB_ANIM_SPIKES_L));
         break;

      // Spikes pointing right
      case OBJ_SPIKES_R:
         set_hitbox(obj, 15, 16, -16, 15);
         set_object_anim(obj, retrieve_object_anim(OB_ANIM_SPIKES_R));
         break;

      // Huh...
      default:
         abort_program(ERR_UNKNOWN, NULL);
   }

   // Make the tile occupied by this spike solid
   // Easier than doing this through object interaction...
   get_tile_by_pixel(obj->x, obj->y)->force_solid = 1;
}

//***************************************************************************
// run_spikes
// Logic for spikes objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_spikes(Object *obj) {
   // Too far to bother?
   if (is_too_far(obj))
      return;

   // Check if some player has hit us
   for (Object *player = get_first_object(OBJGROUP_PLAYER);
   player != NULL; player = player->next) {
      // Hurt the player if it has touched our pointy part
      if (collision(obj, player))
         hurt_player(player, player->x);
   }
}

//***************************************************************************
// init_crusher
// Initializes a crusher object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_crusher(Object *obj) {
   // Create a warning sign below us to alert players
   // Not necessary under normal circumstances, EXTREMELY necessary when
   // playing the game zoomed in (especially if you can't hear the noise)
   if (settings.zoom)
      add_object(OBJ_WARNING, obj->x, obj->y + 0x80, 0);

   // Set hitbox
   set_hitbox(obj, -32, 31, -48, 47);

   // If the direction flag is set, instead of being flipped, make this
   // crusher be half a cycle off-sync with the default position
   // To-do: find a better explanation for this comment...
   if (obj->dir) {
      obj->dir = 0;
      obj->timer = 0x40;
   }
}

//***************************************************************************
// run_crusher
// Logic for crusher objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_crusher(Object *obj) {
   // Too far to bother?
   if (is_too_far(obj))
      return;

   // Used to keep track of how much we travelled
   uint32_t old_y = obj->y;

   // Determine our position
   // In order: going down, stuck down, going up, stuck up
   uint8_t place = (game_anim + obj->timer) & 0x7F;
   if (place < 0x10)
      obj->y = obj->base_y + place * 6;
   else if (place < 0x18)
      obj->y = obj->base_y + 0x60;
   else if (place < 0x78)
      obj->y = obj->base_y + (0x60 - (place - 0x18));
   else
      obj->y = obj->base_y;
   refresh_hitbox(obj);

   // Play sound effects
   if (place == 0x10)
      play_2d_sfx(SFX_CRUSH, obj->x, obj->y + 0x20);
   if (place == 0x78)
      play_2d_sfx(SFX_WARNINGALT, obj->x, obj->y + 0x80);

   // Store how much we moved, so as to let players sit on top of us
   obj->gravity = (obj->y - old_y) << 8;

   // Set sprite
   set_object_anim(obj, retrieve_level_anim(LV_ANIM_CRUSHER));
}

//***************************************************************************
// init_liquid_hazard
// Initializes a liquid hazard object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_liquid_hazard(Object *obj) {
   // Set hitbox
   set_hitbox(obj, -16, 15, -8, 15);
}

//***************************************************************************
// run_liquid_hazard
// Logic for liquid hazard objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_liquid_hazard(Object *obj) {
   // Set sprite
   //set_object_anim(obj, retrieve_level_anim(LV_ANIM_LIQUIDHAZARD));

   // Check if some player has been harmed by us
   for (Object *player = get_first_object(OBJGROUP_PLAYER);
   player != NULL; player = player->next) {
      // Hurt the player if it has touched us
      if (collision(obj, player))
         hurt_player(player, player->x);
   }

   // Used during the ship boss where water keeps spawning
   // Disappear when the water is too far to matter
   // (otherwise objects keep taking up memory for no reason)
   if (obj->timer > 0) {
      obj->timer--;
      if (obj->timer == 0) {
         delete_object(obj);
         return;
      }
   }

   // Force animation to be in sync
   obj->frame = retrieve_liquid_anim();
   obj->frame_time = 0xFFFF;

   // Because liquid tends to clutter in groups *and* its animation is
   // synchronized, using animation clues would be a mess, so we do this
   // from code instead and hope we don't overload too much
   // We shift the sound upwards a bit because usually the liquid will be
   // lower than the actual gap (this matters because otherwise the sound
   // can be barely heard, which is bad)
   if (settings.audiovideo) {
      unsigned tile_x = obj->x >> TILE_SIZE_BIT;
      if (((game_anim + (tile_x << 3)) & 0x1F) == 0x00)
         play_2d_sfx(SFX_CLUELIQUID, obj->x, obj->y - 0x30);
   }
}

//***************************************************************************
// init_fire_hazard
// Initializes a fire hazard object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_fire_hazard(Object *obj) {
   // Set hitbox
   set_hitbox(obj, -8, 7, -12, 11);
}

//***************************************************************************
// run_fire_hazard
// Logic for fire hazard objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_fire_hazard(Object *obj) {
   // Set sprite
   set_object_anim(obj, retrieve_level_anim(LV_ANIM_FIRE));

   // Check if some player has been harmed by us
   for (Object *player = get_first_object(OBJGROUP_PLAYER);
   player != NULL; player = player->next) {
      // Hurt the player if it has touched us
      if (collision(obj, player))
         hurt_player(player, obj->x);
   }
}

//***************************************************************************
// init_bouncing_hazard
// Initializes a bouncing hazard object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_bouncing_hazard(Object *obj) {
   // Set hitbox
   set_hitbox(obj, -12, 11, -12, 11);

   // Set initial speed
   obj->speed = obj->dir ? -BOUNCING_SPEED : BOUNCING_SPEED;
}

//***************************************************************************
// run_bouncing_hazard
// Logic for bouncing hazard objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_bouncing_hazard(Object *obj) {
   // Don't bother if too far...
   if (is_too_far(obj))
      return;

   // Set hitbox
   set_hitbox(obj, -12, 11, -12, 11);

   // Set sprite
   set_object_anim(obj, retrieve_level_anim(LV_ANIM_BOUNCINGHAZARD));

   // Check if some player has been harmed by us
   for (Object *player = get_first_object(OBJGROUP_PLAYER);
   player != NULL; player = player->next) {
      // Hurt the player if it has touched us
      if (collision(obj, player))
         hurt_player(player, obj->x);
   }

   // Check if some enemy got squashed by us too
   for (Object *enemy = get_first_object(OBJGROUP_ENEMY);
   enemy != NULL; enemy = enemy->next) {
      // Destroy the enemy if it has touched us
      if (collision(obj, enemy)) {
         destroy_enemy(enemy);
         break;
      }
   }

   // Move around
   int32_t old_speed = obj->speed;
   apply_physics(obj);

   // Make the hazard bounce around
   if (obj->on_ground) {
      obj->gravity = -BOUNCING_BOUNCE;
      play_2d_sfx(SFX_CRASH, obj->x, obj->y);
   } else
      obj->gravity += BOUNCING_WEIGHT;

   // Turn around if hit a wall
   if (obj->on_wall) {
      obj->speed = -old_speed;
      obj->dir = ~obj->dir;
      play_2d_sfx(SFX_CRASH, obj->x, obj->y);
   }
}

//***************************************************************************
// init_stalactite
// Initializes a stalactite object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_stalactite(Object *obj) {
   // Set hitbox
   set_hitbox(obj, -4, 3, -16, 7);
}

//***************************************************************************
// run_stalactite
// Logic for stalactite objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_stalactite(Object *obj) {
   // Set sprite
   set_object_anim(obj, retrieve_level_anim(LV_ANIM_STALACTITE));

   // Check if some player has been harmed by us
   for (Object *player = get_first_object(OBJGROUP_PLAYER);
   player != NULL; player = player->next) {
      // Is the player near enough to start falling?
      if (!obj->jumping &&
      player->x - obj->x <= STALACTITE_RANGEX &&
      player->x - obj->x >= -STALACTITE_RANGEX &&
      player->y - obj->y <= STALACTITE_RANGEY &&
      player->y > obj->y) {
         obj->jumping = 1;
         play_2d_sfx(SFX_DROP, obj->x, obj->y);
      }

      // Hurt the player if it has touched us
      if (collision(obj, player))
         hurt_player(player, obj->x);
   }

   // Falling already?
   if (obj->jumping) {
      // Go downwards
      obj->y += speed_to_int(obj->gravity);
      obj->gravity += STALACTITE_WEIGHT;

      // Are we gone yet?
      if (is_too_far(obj)) {
         delete_object(obj);
         return;
      }
   }
}

//***************************************************************************
// init_coil
// Initializes a coil object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_coil(Object *obj) {
   // Set hitbox
   // The hitbox covers the area that can hurt the player
   set_hitbox(obj, -48, 47,
      obj->type == OBJ_COIL_F ? -16 : 0,
      obj->type == OBJ_COIL_F ? 0 : 16);

   // Alternate timing?
   if (obj->dir) {
      obj->dir = 0;
      obj->timer = COIL_CYCLE/2;
   }
}

//***************************************************************************
// run_coil
// Logic for coil objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_coil(Object *obj) {
   // Don't bother if off-screen
   if (is_too_far(obj))
      return;

   // Determine at which step of the cycle we are
   unsigned timer = (game_anim + obj->timer) % COIL_CYCLE;

   // Determine for how long release the charge
   unsigned hurt_time;
   switch (get_difficulty()) {
      case DIFF_EASY: hurt_time = COIL_TIME_E; break;
      case DIFF_HARD: hurt_time = COIL_TIME_H; break;
      default:        hurt_time = COIL_TIME_N; break;
   }

   // Make noise if we started electrocuting
   if (timer == COIL_WARMUP)
      play_2d_sfx(SFX_SHOCK, obj->x, obj->y);

   // Electrocuting?
   if (timer >= COIL_WARMUP && timer < hurt_time) {
      // Check if some player got caught by the electricity
      for (Object *player = get_first_object(OBJGROUP_PLAYER);
      player != NULL; player = player->next) {
         // Hurt the player if it has touched us
         if (collision(obj, player))
            hurt_player(player, obj->x);
      }
   }

   // Determine what animation to use
   unsigned anim;
   if (timer < COIL_WARMUP)
      anim = OB_ANIM_COIL_F_READY;
   else if (timer < hurt_time)
      anim = OB_ANIM_COIL_F_ACTIVE;
   else
      anim = OB_ANIM_COIL_F_IDLE;
   if (obj->type != OBJ_COIL_F)
      anim += OB_ANIM_COIL_C_IDLE - OB_ANIM_COIL_F_IDLE;

   // Set sprite
   set_object_anim(obj, retrieve_object_anim(anim));
}

//***************************************************************************
// init_buzzsaw
// Initializes a buzzsaw object.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void init_buzzsaw(Object *obj) {
   // Set hitbox
   // The hitbox covers the area that can hurt the player
   set_hitbox(obj, -24, 23, -24, 23);

   // Alternate timing?
   if (obj->dir) {
      obj->dir = 0;
      obj->timer = 0x80;
   }

   // Determine animation to use
   int anim;
   switch (obj->type) {
      case OBJ_BUZZSAW_C: anim = OB_ANIM_BUZZSAW_C; break;
      case OBJ_BUZZSAW_F: anim = OB_ANIM_BUZZSAW_F; break;
      default:            anim = OB_ANIM_BUZZSAW;   break;
   }
   set_object_anim(obj, retrieve_object_anim(anim));
}

//***************************************************************************
// run_buzzsaw
// Logic for buzzsaw objects.
//---------------------------------------------------------------------------
// param obj: pointer to this object
//***************************************************************************

void run_buzzsaw(Object *obj) {
   // Don't bother if off-screen
   if (is_too_far(obj))
      return;

   // Moving buzzsaw?
   if (obj->type != OBJ_BUZZSAW) {
      // Determine at which position of the swing we are
      unsigned timer = game_anim + obj->timer;
      if (obj->type == OBJ_BUZZSAW_F)
         timer = 0xFF - timer;
      timer &= 0xFF;

      // Determine object position
      obj->y = obj->base_y + sines[timer] / 8;
      refresh_hitbox(obj);
   }

   // Check if some player got caught by the buzzsaw
   for (Object *player = get_first_object(OBJGROUP_PLAYER);
   player != NULL; player = player->next) {
      // Hurt the player if it has touched us
      if (collision(obj, player))
         hurt_player(player, obj->x);
   }
}
