//***************************************************************************
// "file.h"
// Definitions for the virtual filesystem.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

#ifndef FILE_H
#define FILE_H

// Required headers
#include <stdint.h>
#include <stddef.h>
#include <stdio.h>
#include <physfs.h>

// Which characters are considered separators in a script
// (basically all the standard whitespace characters)
#define SEPARATORS " \n\r\t\v\f"

// Different modes in which you can open a file
typedef enum {
   FILE_READ,              // Open for reading in the sandbox
   FILE_WRITE              // Open for writing in the sandbox
} FileMode;

// What kind of object is a file handle
typedef struct {
   FileMode mode;          // Mode in which file was opened
   PHYSFS_File *handle;    // PhysFS file handle
} File;

// Info about a directory
typedef struct {
   size_t num_dirs;        // Number of directories
   size_t num_files;       // Number of files
   char **dirs;            // List of directory names
   char **files;           // List of file names
} Dir;

// Data in an INI object
typedef struct {
   char *name;                // Name of the variable
   char *value;               // Value of the variable
} IniVar;
typedef struct {
   char *name;                // Name of the section
   size_t num_variables;      // Number of variables
   IniVar *variables;         // Array of variables
} IniSect;
typedef struct {
   size_t num_sections;       // Number of sections
   IniSect *sections;         // Array of sections
} Ini;

// Function prototypes
void init_filesystem(const char *);
void add_mod(const char *);
void add_local_mod(const char *);
void load_mods(void);
void deinit_filesystem(void);
void enable_user_filesystem(void);
void disable_user_filesystem(void);
File *open_file(const char *, FileMode);
File *open_save_file(const char *, FileMode);
File *open_user_file(const char *, FileMode);
int read_file(File *, void *, size_t);
size_t read_file_ex(File *, void *, size_t);
int write_file(File *, const void *, size_t);
int write_uint16_le(File *, uint16_t);
int write_uint32_le(File *, uint32_t);
void close_file(File *);
int end_of_file(File *);
char *get_default_dir(void);
char *get_home_dir(void);
char *get_pictures_dir(void);
char *get_config_dir(void);
char *get_manual_dir(void);
Dir *create_dir_list(void);
void add_dir_to_list(Dir *, const char *);
void add_file_to_list(Dir *, const char *);
Dir *get_dir_list(const char *);
Dir *get_media_list(void);
void filter_dir_list(Dir *, const char *);
void free_dir_list(Dir *);
const char *find_basename(const char *);
int is_root_dir(const char *);
char *get_parent_dir(const char *);
char *append_path(const char *, const char *);
char *read_line(File *);
uint64_t get_file_checksum(const char *);
Ini *create_ini(void);
Ini *load_ini(const char *);
const char *get_ini_var(const Ini*, const char*, const char*, const char*);
void set_ini_var(Ini *, const char *, const char *, const char *);
int save_ini(const char *, const Ini *);
void destroy_ini(Ini *);

#endif
