//***************************************************************************
// "file.c"
// File handling. Implements the virtual filesystem used by the program
// (which is what lets it access files inside archives and such).
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <errno.h>
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <SDL2/SDL.h>
#include <physfs.h>
#include "main.h"
#include "file.h"
#include "savegame.h"
#include "scene.h"
#include "settings.h"
#include "text.h"

// We use the Windows API for some system specific stuff
#ifdef _WIN32
#include <windows.h>
#include <shlobj.h>
#endif

// Used to keep track of checksums
typedef struct {
   uint64_t scenes;           // system/scenes
   uint64_t tweaks;           // system/tweaks.ini
   uint64_t levels[0x100];    // levels/*.sol
} Checksums;
static Checksums checksums;

// Private function prototypes
static void retrieve_checksums(Checksums *);
static int altered_checksums(const Checksums *, const Checksums *);
static File *do_open_file(const char *, FileMode);
static int sort_dir_list(const void *, const void *);

// List of mods being loaded
static const char **mods = NULL;
size_t num_mods = 0;

//***************************************************************************
// init_filesystem
// Initializes the file subsystem
//---------------------------------------------------------------------------
// param argv0: argv[0] passed to main(), because apparently on *nix systems
//              getting the executable path is impossible or something (and
//              even that has its own problems, but whatever...)
//***************************************************************************

void init_filesystem(const char *argv0) {
#ifdef DEBUG
   fputs("Initializing filesystem\n", stderr);
#endif

   // Initialize PhysFS
   if (!PHYSFS_init(argv0))
      abort_program(ERR_INITFS, NULL);

   // Used to keep track of whether the assets got mounted
   int mounted = 0;

   // Determine filename of the datafile
   const char *datafile = "soledad.dat";
   settings.demo = 0;

   if (argv0 != NULL) {
      // Find out where the filename starts
      const char *start = strrchr(argv0, '/');
#ifdef _WIN32
      const char *start2 = strrchr(argv0, '\\');
      if (start2 > start)
         start = start2;
#endif
      if (start == NULL)
         start = argv0;
      else
         start++;

      // Find out where the extension is, if any
      const char *end = strrchr(start, '.');
      if (end == NULL)
         end = start + strlen(start);

      // Check if the filename ends in "-demo"
      // If so, go into trial version
      if (end-5 >= start) {
         if (strncmp(end-5, "-demo", 5) == 0) {
            datafile = "soledad-demo.dat";
            settings.demo = 1;
         }
      }
   }

   // Get directory where the game data is stored
   // Note that on Linux this isn't necessarily the same path as where the
   // executable is stored, but to figure out that we'll need this anyway
   const char *basedir = PHYSFS_getBaseDir();

   // Allocate enough memory to generate the paths
   // Can't allocate statically since we don't know how big the base path
   // could be.
   char *buffer = (char *) malloc(strlen(basedir) + 0x40);
   if (buffer == NULL)
      abort_program(ERR_NOMEMORY, NULL);

#ifdef __linux__
   // On Linux things are a bit different, try to read the datafile from the
   // usual locations for when a program is installed
   if (!mounted &&
   (strcmp(basedir, "/usr/local/bin/") == 0 ||
   strcmp(basedir, "/usr/local/games/") == 0)) {
      char path[0x80];
      sprintf(path, "/usr/local/share/soledad/%s", datafile);
      mounted = PHYSFS_mount(path, "gamefs", 0);
   }

   if (!mounted &&
   (strcmp(basedir, "/usr/bin/") == 0 ||
   strcmp(basedir, "/usr/games/") == 0)) {
      char path[0x80];
      sprintf(path, "/usr/share/soledad/%s", datafile);
      mounted = PHYSFS_mount(path, "gamefs", 0);
   }
#endif

   // Check if we can load the datafile (this is what will happen in a
   // normal install, but not during development)
   if (!mounted) {
      sprintf(buffer, "%s/%s", basedir, datafile);
      mounted = PHYSFS_mount(buffer, "gamefs", 0);
   }

   // Could not load the datafile? try mounting the directories
   // directly instead. This is used during development.
   if (!mounted) {
      mounted = 1;

#define PATH(name) \
sprintf(buffer, "%s" name, basedir); \
mounted &= PHYSFS_mount(buffer, "gamefs/" name, 0);
      PATH("cutscenes");
      PATH("graphics");
      PATH("languages");
      PATH("levels");
      PATH("music");
      PATH("sounds");
      PATH("system");
#undef PATH
   }

   // STILL couldn't load the game assets? Hmph.
   if (!mounted)
      abort_program(ERR_NOASSETS, NULL);

   // Store savegames in the preferences directory for the program. This also
   // creates the directory if needed (first time boot will cause this). If
   // the directory can't be used, then the game will not save any data.
   {
      char *dir = SDL_GetPrefPath("azurasun", "soledad");
      if (dir != NULL) {
         PHYSFS_mount(dir, "/savefs/savedata", 0);
         SDL_free(dir);
      }
   }

   // Store savegames in the pictures directory... or whatever gets detected
   // as such (depending on the system it may go with home instead)
   // Note: if the screenshot directory can't be mounted, you become unable
   // to take screenshots and you'll get an error message if you try
   {
      char *dir = get_pictures_dir();
      PHYSFS_mount(dir, "/savefs/screenshots", 0);
      free(dir);
   }

#ifdef __linux__
   // Used specifically for the video recording feature. Yes, this expects
   // being mounted into a ramdrive. Enjoy.
   PHYSFS_mount("/media/RAMDRIVE", "savefs/videos", 0);
#endif

   // There, no need to mess with the paths anymore
   free(buffer);

   // Nice try...
   if (!settings.demo) {
      unsigned all_ok = 1;
      all_ok &= PHYSFS_exists("gamefs/graphics/level_sewer") ? 1 : 0;
      all_ok &= PHYSFS_exists("gamefs/graphics/level_harbor") ? 1 : 0;
      all_ok &= PHYSFS_exists("gamefs/graphics/level_desert") ? 1 : 0;
      all_ok &= PHYSFS_exists("gamefs/graphics/level_cave") ? 1 : 0;
      all_ok &= PHYSFS_exists("gamefs/graphics/level_factory") ? 1 : 0;
      all_ok &= PHYSFS_exists("gamefs/graphics/level_carnival") ? 1 : 0;

      // Aren't all level graphics present?
      // (they are in the full version, not in the demo version)
      if (!all_ok)
         abort_program(ERR_NOASSETS, NULL);
   }

   // Compute default checksums
   load_scene_list();
   retrieve_checksums(&checksums);

   // Load system stuff
   load_tweaks();
}

//***************************************************************************
// add_mod
// Adds a mod to the mod list. You must add all mods *before* calling
// load_mods. Also this function assumes the filenames come from argv (so it
// doesn't keep a copy of the strings).
//---------------------------------------------------------------------------
// param filename: name of mod file to add
//***************************************************************************

void add_mod(const char *filename) {
   // Make room for new filename
   num_mods++;
   mods = realloc(mods, sizeof(char**) * num_mods);
   if (mods == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Store filename
   mods[num_mods - 1] = filename;
}

//***************************************************************************
// add_local_mod
// Adds a mod from the executable's own directory.
//---------------------------------------------------------------------------
// param filename: name of mod file to add
//***************************************************************************

void add_local_mod(const char *filename) {
   // Note: this isn't accounting for /usr/share under Linux!!
   // In practice it doesn't matter, since Sol ended up in /opt instead, but
   // it'd be nice to fix it for the sake of consistency

   const char *basedir = PHYSFS_getBaseDir();
   char *buffer = (char*) malloc(strlen(basedir) + strlen(filename) + 1);
   if (buffer == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   sprintf(buffer, "%s%s", basedir, filename);
   add_mod(buffer);
}

//***************************************************************************
// load_mods
// Loads all the mods that have been requested so far.
//***************************************************************************

void load_mods(void) {
   // Add all mods requested
   for (size_t i = 0; i < num_mods; i++) {
#ifdef DEBUG
      fprintf(stderr, "Loading mod \"%s\"\n", mods[i]);
#endif
      if (!PHYSFS_mount(mods[i], "gamefs", 0))
         abort_program(ERR_LOADMOD, mods[i]);

      // Did the savegame get invalidated?
      Checksums new_checksums;
      retrieve_checksums(&new_checksums);
      if (altered_checksums(&checksums, &new_checksums)) {
         invalidate_savegame();
         checksums = new_checksums;
      }

      // Update tweaks (may revalidate the savegame)
      load_tweaks();
   }

   // Mods loaded, get rid of list
   free(mods);
   mods = NULL;
   num_mods = 0;
}

//***************************************************************************
// retrieve_checksums [internal]
// Retrieves the checksums of the files that can invalidate a savegame
//---------------------------------------------------------------------------
// param where: where to store checksums
//***************************************************************************

static void retrieve_checksums(Checksums *where) {
   // Compute scenes and tweaks
   where->scenes = get_file_checksum("system/scenes");
   where->tweaks = get_file_checksum("system/tweaks.ini");

   // Compute level checksums
   for (curr_scene = 0; curr_scene < get_num_scenes(); curr_scene++) {
      // Check if this scene is a level first
      SceneType type = get_scene_type();
      if (type != SCENE_LEVEL && type != SCENE_BONUS) {
         where->levels[curr_scene] = 0;
         continue;
      }

      // Compute checksum of scene
      char filename[0x80];
      sprintf(filename, "levels/%s.sol", get_scene_id());
      where->levels[curr_scene] = get_file_checksum(filename);
   }
}

//***************************************************************************
// altered_checksums
// Checks if the important checksums got altered since the last time they
// were fetched (this can happen when loading a mod).
//---------------------------------------------------------------------------
// param orig: original checksums
// param curr: current checksums
// return: 1 if altered, 0 if not
//***************************************************************************

static int altered_checksums(const Checksums *orig, const Checksums *curr) {
   // First of all: check scenes and tweaks
   if (orig->scenes != curr->scenes) return 1;
   if (orig->tweaks != curr->tweaks) return 1;

   // Now compare the altered levels, if any
   for (curr_scene = 0; curr_scene < get_num_scenes(); curr_scene++) {
      // Check if this scene is a level first
      SceneType type = get_scene_type();
      if (type != SCENE_LEVEL && type != SCENE_BONUS)
         continue;

      // Compare checksums
      if (orig->levels[curr_scene] != curr->levels[curr_scene])
         return 1;
   }

   // OK, seems nothing was changed
   return 0;
}

//***************************************************************************
// deinit_filesystem
// Deinitializes the file subsystem
//***************************************************************************

void deinit_filesystem(void) {
   // Deinitialize PhysFS
   // Yes, I know this function can fail, but if it does then we're pretty
   // much stuck forever and it'd mean something went *very* wrong. In our
   // case it may be better to just continue with the clean-up procedure and
   // let the OS free all the resources once we quit.
   if (PHYSFS_isInit())
      PHYSFS_deinit();
}

//***************************************************************************
// enable_user_filesystem
// Enables access to the user filesystem. Disabled by default for security
// reasons. You need to enable this before accessing files from outside the
// sandbox. Remember to disable it once not needed anymore.
//***************************************************************************

void enable_user_filesystem(void) {
#ifdef _WIN32
   // On Windows, try to mount every drive by hand
   // To-do: query which drives are available and only mount those?
   int mounted = 0;
   char real_path[] = "?:";
   char fake_path[] = "userfs/?/";
   for (char i = 'A'; i <= 'Z'; i++) {
      real_path[0] = i;
      fake_path[7] = i;
      if (PHYSFS_mount(real_path, fake_path, 0))
         mounted = 1;
   }

   // Mounted nothing? Seriously?
   if (!mounted)
      abort_program(ERR_UNKNOWN, NULL);
#else
   // On *nix, just mount root and you get everything
   if (!PHYSFS_mount("/", "userfs", 0))
      abort_program(ERR_UNKNOWN, NULL);
#endif

   // Enable symbolic links, since the user may rely on them
   PHYSFS_permitSymbolicLinks(1);
}

//***************************************************************************
// disable_user_filesystem
// Disables access to the user filesystem. Call this once you're done with it
// for security reasons.
//***************************************************************************

void disable_user_filesystem(void) {
#ifdef _WIN32
   // On Windows, we mounted every drive we could
   // Just unmount all, nothing will happen for those not mounted
   char real_path[] = "?:";
   for (char i = 'A'; i <= 'Z'; i++) {
      real_path[0] = i;
      PHYSFS_removeFromSearchPath(real_path);
   }
#else
   // On *nix, only root is mounted
   PHYSFS_removeFromSearchPath("/");
#endif

   // No need for symbolic links anymore
   PHYSFS_permitSymbolicLinks(0);
}

//***************************************************************************
// do_open_file [internal]
// The actual function that opens a file (open_file and open_user_file are
// wrappers for this one).
//---------------------------------------------------------------------------
// param filename: name of file
// param mode: access permissions
// returns: pointer to file on success, NULL on failure
//***************************************************************************

static File *do_open_file(const char *filename, FileMode mode) {
   // Make sure the filesystem is initialized yet! (if you wonder what
   // situation can result in this being called without PhysicsFS being
   // setup... try passing -h to the command line)
   if (!PHYSFS_isInit())
      return NULL;

   // If accessing the file for writing, we'll need to set up the write
   // directory and determine what filename to use with it, which is going to
   // suck really hard x_x;
   if (mode == FILE_WRITE) {
      // Determine what's the directory of this path
      // This is because we can't get the real directory of a file that
      // doesn't exist, which prevents using FILE_WRITE for creating new
      // files. This should let us do it. Of course that directory may not
      // exist either, but then we should fail if that's the case :P
      size_t fake_dir_len = (size_t)(strrchr(filename, '/') - filename);
      char *fake_dir = (char *) malloc(fake_dir_len + 1);
      if (fake_dir == NULL)
         abort_program(ERR_NOMEMORY, NULL);
      strncpy(fake_dir, filename, fake_dir_len);
      fake_dir[fake_dir_len] = '\0';

      // Try to set up write directory
      const char *real_dir = PHYSFS_getRealDir(fake_dir);
      free(fake_dir);
      if (real_dir == NULL)
         return NULL;
      if (!PHYSFS_setWriteDir(real_dir))
         return NULL;

      // Determine where is it mounted
      const char *mount = PHYSFS_getMountPoint(real_dir);
      if (mount == NULL)
         return NULL;

      // Figure out what filename should we use
      // The mount point should be part of the original filename, so...
      filename += strlen(mount);
      if (*filename == '/')
         filename++;
   }

   // Allocate structure
   File *file = (File *) malloc(sizeof(File));
   if (file == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   file->mode = mode;
   file->handle = NULL;

   // Try to open file
   switch (mode) {
      case FILE_READ:
         file->handle = PHYSFS_openRead(filename); break;
      case FILE_WRITE:
         file->handle = PHYSFS_openWrite(filename); break;
      default:
         file->handle = NULL; break;
   }

   // Couldn't open file?
   if (file->handle == NULL) {
      free(file);
      return NULL;
   }

   // Success!
   return file;
}

//***************************************************************************
// open_file
// Opens a file from the sandboxed game filesystem.
//---------------------------------------------------------------------------
// param filename: name of file
// param mode: access permissions
// returns: pointer to file on success, NULL on failure
//***************************************************************************

File *open_file(const char *filename, FileMode mode) {
   // Don't allow writing here
   // This makes the parameter seem redundant, but so much stuff called this
   // function by the time it was split that it was easier to just keep the
   // parameter around
   if (mode != FILE_READ)
      return NULL;

   // Generate actual filename in the sandbox
   char *fake_name = (char*) malloc(strlen(filename) + 7 + 1);
   if (fake_name == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   sprintf(fake_name, "gamefs/%s", filename);

   // Try to open file
   File *file = do_open_file(fake_name, mode);
   free(fake_name);
   return file;
}

//***************************************************************************
// open_save_file
// Opens a file from the sandboxed user save data.
//---------------------------------------------------------------------------
// param filename: name of file
// param mode: access permissions
// returns: pointer to file on success, NULL on failure
//***************************************************************************

File *open_save_file(const char *filename, FileMode mode) {
   // Generate actual filename in the sandbox
   char *fake_name = (char*) malloc(strlen(filename) + 7 + 1);
   if (fake_name == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   sprintf(fake_name, "savefs/%s", filename);

   // Try to open file
   File *file = do_open_file(fake_name, mode);
   free(fake_name);
   return file;
}

//***************************************************************************
// open_user_file
// Opens a file from the user filesystem. The user filesystem must have been
// enabled in order for this function to work.
//---------------------------------------------------------------------------
// param filename: name of file
// param mode: access permissions
// returns: pointer to file on success, NULL on failure
//***************************************************************************

File *open_user_file(const char *filename, FileMode mode) {
   // Determine to which filename this is mapped
   char *fake_name = NULL;
   size_t length = strlen(filename);
#ifdef _WIN32
   // Generate PhysicsFS filename
   fake_name = (char *) malloc(length + 7 + 1);
   if (fake_name == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   sprintf(fake_name, "userfs/%s", filename);

   // Drive name should be uppercase
   // PhysicsFS *needs* this one to be case sensitive
   fake_name[7] = toupper(fake_name[7]);

   // Make sure all backslashes are turned into slashes
   // Similar deal with the colon of the drive name
   // PhysicsFS only understands slashes
   fake_name[8] = '/';
   for (char *ptr = &fake_name[9]; *ptr != '\0'; ptr++)
      if (*ptr == '\\') *ptr = '/';
#else
   // Generate PhysicsFS filename
   fake_name = (char *) malloc(length + 6 + 1);
   if (fake_name == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   sprintf(fake_name, "userfs%s", filename);
#endif

   // Try to open file
   File *file = do_open_file(fake_name, mode);

   // Done!
   free(fake_name);
   return file;
}

//***************************************************************************
// read_file
// Reads bytes from a file and puts them into a buffer in RAM
//---------------------------------------------------------------------------
// param file: pointer to file handle
// param buffer: where to store the data
// param size: amount of bytes to read
// return: zero on success, non-zero on failure
//***************************************************************************

int read_file(File *file, void *buffer, size_t size) {
   // Lets make our life easier
   return read_file_ex(file, buffer, size) != size;
}

//***************************************************************************
// read_file_ex
// Like read_file, but instead of returning a success/failure status, it
// returns the amount of bytes read from the file (if you want to make sure
// it's an error or not when it returns less than requested check if it's the
// end of the file).
//---------------------------------------------------------------------------
// param file: pointer to file handle
// param buffer: where to store the data
// param size: amount of bytes to read
// return: number of bytes read
//***************************************************************************

size_t read_file_ex(File *file, void *buffer, size_t size) {
   // Safeguards
   if (file == NULL)
      return 0;
   if (file->mode != FILE_READ)
      return 0;

   // For pointer arithmetic shenanigans
   uint8_t *buf = (uint8_t *) buffer;

   // To keep track of the actual size since PhysFS is limited to 32-bit
   // sizes :/
   size_t len = 0;
   int64_t out;

   // PhysFS can only read up to 0xFFFFFFFF bytes of data in a single call,
   // so if the size is larger we need to split it up into multiple calls
   // (...why are we reading above 4GB though?)
   while (size >= UINT32_MAX) {
      out = PHYSFS_read(file->handle, buf, 1, UINT32_MAX);
      if (out > 0) {
         size -= out;
         len += out;
         buf += out;
      }
      if (out < UINT32_MAX)
         return len;
   }

   // Read the remaining data
   if (size > 0) {
      out = PHYSFS_read(file->handle, buf, 1, size);
      if (out > 0)
         len += out;
   }

   // Return how many bytes were read
   return len;
}

//***************************************************************************
// write_file
// Writes bytes into a file from a buffer in RAM
//---------------------------------------------------------------------------
// param file: pointer to file handle
// param buffer: data to be written
// param size: amount of bytes to write
// return: zero on success, non-zero on failure
//***************************************************************************

int write_file(File *file, const void *buffer, size_t size) {
   // Safeguards
   if (file == NULL)
      return -1;
   if (file->mode != FILE_WRITE)
      return -1;

   // For pointer arithmetic shenanigans
   uint8_t *buf = (uint8_t *) buffer;

   // PhysicsFS only supports writing 4GB at most, so let's split the writes
   // into chunks if we're asked to write even more
   while (size >= UINT32_MAX) {
      if (PHYSFS_write(file->handle, buf, 1, UINT32_MAX) != UINT32_MAX)
         return -1;
      size -= UINT32_MAX;
      buf += UINT32_MAX;
   }

   // Try to write the remainder
   if (size > 0) {
      if (PHYSFS_write(file->handle, buf, 1, size) != (int64_t) size)
         return -1;
   }

   // Success!
   return 0;
}

//***************************************************************************
// write_uint16_le
// Writes a little endian 32-bit unsigned integer into a file.
//---------------------------------------------------------------------------
// param file: pointer to file handle
// param value: value to be written
// return: zero on success, non-zero on failure
//***************************************************************************

int write_uint16_le(File *file, uint16_t value) {
   // Split value into bytes
   uint8_t blob[2] = {
      value,
      value >> 8
   };

   // Write value into the file
   return write_file(file, blob, 2);
}

//***************************************************************************
// write_uint32_le
// Writes a little endian 32-bit unsigned integer into a file.
//---------------------------------------------------------------------------
// param file: pointer to file handle
// param value: value to be written
// return: zero on success, non-zero on failure
//***************************************************************************

int write_uint32_le(File *file, uint32_t value) {
   // Split value into bytes
   uint8_t blob[4] = {
      value,
      value >> 8,
      value >> 16,
      value >> 24
   };

   // Write value into the file
   return write_file(file, blob, 4);
}

//***************************************************************************
// close_file
// Closes the specified file handle
//---------------------------------------------------------------------------
// param file: pointer to file handle (becomes invalid after call)
//***************************************************************************

void close_file(File *file) {
   // Okaaay...
   if (file == NULL)
      return;

   // Close file handle in PhysFS
   // To-do: this function *can* fail (although if that happens, we're pretty
   // much screwed up, right?). Figure out what should we do in the rare case
   // this happens.
   PHYSFS_close(file->handle);

   // We can get rid of the file structure now
   free(file);
}

//***************************************************************************
// end_of_file
// Checks if a file counter is at the end or not.
//---------------------------------------------------------------------------
// param file: pointer to file handle
// return: zero if not at end of file, non-zero if at end of file
//***************************************************************************

int end_of_file(File *file) {
   // Okaaay...
   if (file == NULL)
      return 1;

   // Call the relevant function
   return PHYSFS_eof(file->handle);
}

//***************************************************************************
// get_default_dir
// Determines a default directory for the level editor file select dialog.
// Make sure to free the returned string after calling this function!
//---------------------------------------------------------------------------
// return: name of default directory
//***************************************************************************

char *get_default_dir(void) {
   // Originally there was only this function, but later I needed to retrieve
   // the home directory elsewhere and I wanted to keep the functionality
   // separate even if in practice right now they behave the same. If we ever
   // allow the default directory to not be the home directory, this function
   // will change while get_home_dir will remain untouched.
   return get_home_dir();
}

//***************************************************************************
// get_home_dir
// Determines the home directory of the user. Make sure to free the returned
// string after calling this function!
//---------------------------------------------------------------------------
// return: name of home directory
//***************************************************************************

char *get_home_dir(void) {
   // Where we'll store our generated path
   char *buffer = NULL;

#ifdef _WIN32
   // Retrieve My Documents directory
   wchar_t wbuffer[MAX_PATH+1];
   if (SHGetFolderPathW(NULL, CSIDL_MYDOCUMENTS, NULL, SHGFP_TYPE_CURRENT,
   wbuffer) != S_OK)
      goto nodir;

   // Convert path to UTF-8
   buffer = utf16_to_utf8(wbuffer);
   goto done;

nodir:
   // Uh oh
   buffer = (char *) malloc(4);
   if (buffer == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   strcpy(buffer, "C:\\");
#else
   // Try getting the home directory first
   char *home = getenv("HOME");
   if (home == NULL)
      goto nodir;

   size_t len = strlen(home);
   if (home[0] != '/') len++;
   buffer = (char *) malloc(len+1);
   if (buffer == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   sprintf(buffer, "%s%s",
      home[0] != '/' ? "/" : "", home);
   goto done;

nodir:
   // Huh... No home directory specified? There are more complex ways to
   // retrieve it, but I won't bother with that at all (we just need
   // somewhere to start from), so we'll stick to the root directory as a
   // fallback and just let the user browse where needed.
   buffer = (char *) malloc(2);
   if (buffer == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   strcpy(buffer, "/");
#endif

done:
   // Return path
   return buffer;
}

//***************************************************************************
// get_pictures_dir
// Determines the pictures directory of the user. Make sure to free the
// returned string after calling this function!
//---------------------------------------------------------------------------
// return: name of pictures directory
//***************************************************************************

char *get_pictures_dir(void) {
   // Where we'll store our generated path
   char *buffer = NULL;

#ifdef _WIN32
   // Retrieve My Pictures directory
   wchar_t wbuffer[MAX_PATH+1];
   if (SHGetFolderPathW(NULL, CSIDL_MYPICTURES, NULL, SHGFP_TYPE_CURRENT,
   wbuffer) != S_OK)
      goto nodir;

   // Convert path to UTF-8
   buffer = utf16_to_utf8(wbuffer);
   return buffer;
#else
   // To-do: actually query the directory
   // For now just resort to the fallback...
   (void) buffer;
   goto nodir;
#endif

nodir:
   // Resort to a fallback
   return get_home_dir();
}

//***************************************************************************
// get_config_dir
// Determines the directory where the game's configuration files reside. Make
// sure to free the returned string after calling this function!
//---------------------------------------------------------------------------
// return: name of configuration directory
//***************************************************************************

char *get_config_dir(void) {
   // Get where the saved files are stored
   const char *dir = PHYSFS_getRealDir("/savefs/savedata");

   // Make a copy of the string
   char *buffer = (char *) malloc(strlen(dir) + 1);
   if (buffer == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   strcpy(buffer, dir);

   // Done
   return buffer;
}

//***************************************************************************
// get_manual_dir
// Determines the directory where the game's manual resides. Make sure to
// free the returned string after calling this function!
//---------------------------------------------------------------------------
// return: name of manual directory
//***************************************************************************

char *get_manual_dir(void) {
   // Get where the executable is stored
   const char *exedir = PHYSFS_getBaseDir();

   // Append the manual subdirectory to it
   char *buffer = (char *) malloc(strlen(exedir) + 8);
   if (buffer == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   sprintf(buffer, "%s/manual", exedir);

   // Done
   return buffer;
}

//***************************************************************************
// create_dir_list
// Creates a new empty directory list
//---------------------------------------------------------------------------
// return: pointer to directory list or NULL on failure
//***************************************************************************

Dir *create_dir_list(void) {
   // Allocate memory
   Dir *list = (Dir *) malloc(sizeof(Dir));
   if (list == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Initialize fields
   list->num_dirs = 0;
   list->num_files = 0;
   list->dirs = NULL;
   list->files = NULL;

   // Done!
   return list;
}

//***************************************************************************
// add_dir_to_list
// Adds a directory to the directory list
//---------------------------------------------------------------------------
// param list: pointer to list
// param path: path to add as directory
//***************************************************************************

void add_dir_to_list(Dir *list, const char *path) {
   // Make a copy of the filename (since we need to keep it)
   char *temp = malloc(strlen(path) + 1);
   if (temp == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   strcpy(temp, path);

   // Push filename into the directory list
   list->num_dirs++;
   list->dirs = (char **) realloc(list->dirs,
      sizeof(char **) * list->num_dirs);
   if (list->dirs == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   list->dirs[list->num_dirs-1] = temp;
}

//***************************************************************************
// add_file_to_list
// Adds a file to the directory list
//---------------------------------------------------------------------------
// param list: pointer to list
// param path: path to add as file
//***************************************************************************

void add_file_to_list(Dir *list, const char *path) {
   // Make a copy of the filename (since we need to keep it)
   char *temp = malloc(strlen(path) + 1);
   if (temp == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   strcpy(temp, path);

   // Push filename into the file list
   list->num_files++;
   list->files = (char **) realloc(list->files,
      sizeof(char **) * list->num_files);
   if (list->files == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   list->files[list->num_files-1] = temp;
}

//***************************************************************************
// get_dir_list
// Tries to retrieve a list of the files in the specified directory.
//---------------------------------------------------------------------------
// param dirname: name of directory to check for
// return: pointer to directory list or NULL on failure
//***************************************************************************

Dir *get_dir_list(const char *dirname) {
   // Create empty directory list
   Dir *list = create_dir_list();

   // Get internal name of directory
   char *real_name = (char *) malloc(strlen(dirname) + 0x10);
   if (real_name == NULL)
      abort_program(ERR_NOMEMORY, NULL);
#ifdef _WIN32
      sprintf(real_name, "/userfs/%s", dirname);
      real_name[9] = '/';
      for (char *temp = real_name; *temp != '\0'; temp++)
         if (*temp == '\\') *temp = '/';
#else
      sprintf(real_name, "/userfs%s", dirname);
#endif

   // So I don't have to calculate this all the time
   size_t real_name_len = strlen(real_name);

   // Scan what's in the directory
   char **data = PHYSFS_enumerateFiles(real_name);
   for (char **curr = data; *curr != NULL; curr++) {
      // To make our lives easier
      const char *entry_name = *curr;

#ifndef _WIN32
      // Skip hidden files <Sik> How do we skip those on Windows? (··?)
      // In case you wonder, files ending with ~ are back-ups
      if (entry_name[0] == '.' || entry_name[strlen(entry_name)-1] == '~')
         continue;
#endif

      // Retrieve the full filename of this entry so we can get PhysicsFS
      // tells us its information
      char *temp = (char*) malloc(strlen(entry_name) + real_name_len + 2);
      if (temp == NULL)
         abort_program(ERR_NOMEMORY, NULL);
      sprintf(temp, "%s/%s", real_name, entry_name);

      // Add entry to the list
      if (PHYSFS_isDirectory(temp))
         add_dir_to_list(list, entry_name);
      else
         add_file_to_list(list, entry_name);

      // There, done
      free(temp);
   }

   // Done with this
   free(real_name);
   PHYSFS_freeList(data);

   // Sort names as needed
   if (list->num_dirs)
      qsort(list->dirs, list->num_dirs, sizeof(char *), sort_dir_list);
   if (list->num_files)
      qsort(list->files, list->num_files, sizeof(char *), sort_dir_list);

   // Success!
   return list;
}

//***************************************************************************
// get_media_list
// Gets a list of the media available on this system
//---------------------------------------------------------------------------
// return: pointer to directory list or NULL on failure
//***************************************************************************

Dir *get_media_list(void) {
#if defined(_WIN32)
   // Start with an empty list
   // We'll add paths on our own
   Dir *list = create_dir_list();

   // Get a list of which drives are present
   uint32_t drives = GetLogicalDrives();

   // Try every possible letter
   // To-do: find out how to avoid errors when floppy disks aren't present
   // (is this just an issue with the test computer? it's kind of broken in
   // all sorts of ways after all)
   for (unsigned i = 0; i < 26; i++) {
      // Skip this drive if not present
      if (!(drives & 1 << i))
         continue;

      // Determine name of this drive
      char buffer[] = "?:";
      buffer[0] = 'A' + i;

      // Add drive to list
      add_dir_to_list(list, buffer);
   }

   // Done!
   return list;
#elif defined(__linux__)
   Dir *list = create_dir_list();
   add_dir_to_list(list, "/");

   // Done!
   return list;
#else
   // We have no idea how to retrieve the list of media under this platform,
   // so we'll just return nothing for now...
   return create_dir_list();
#endif
}

//***************************************************************************
// sort_dir_list [internal]
// Function used to sort the filenames in a directory list.
//---------------------------------------------------------------------------
// param str1: pointer to first string
// param str2: pointer to second string
// return: whatever qsort wants
//***************************************************************************

static int sort_dir_list(const void *str1, const void *str2) {
   // Get the proper string pointers...
   const char *s1 = *((const char **)str1);
   const char *s2 = *((const char **)str2);

   // Just sort them normally
   // To-do: any way to figure out if we can use strverscmp without having to
   // resort to tool-specific stuff like autotools? Because that'd make for a
   // much better comparison methinks.
   return strcmp(s1, s2);
}

//***************************************************************************
// filter_dir_list
// Filters the files in a directory list to keep only those with a given
// extension (used by the level editor to show only level files).
//---------------------------------------------------------------------------
// param list: pointer to directory list
// param extension: extension of files to leave (e.g. ".sol")
//---------------------------------------------------------------------------
// notes: extension is NOT case sensitive.
//***************************************************************************

void filter_dir_list(Dir *list, const char *extension) {
   // Get length of extension
   size_t extlen = strlen(extension);

   // Scan all file entries
   for (size_t i = 0; i < list->num_files; i++) {
      // To keep track if file is allowed by the filter
      int valid;

      // Check if this file is allowed
      size_t namelen = strlen(list->files[i]);
      if (namelen < extlen) {
         valid = 0;
      } else {
         valid = 1;
         const char *ptr1 = list->files[i] + namelen - extlen;
         const char *ptr2 = extension;
         while (*ptr2) {
            if (tolower(*ptr1) != tolower(*ptr2)) {
               valid = 0;
               break;
            }
            ptr1++;
            ptr2++;
         }
      }

      // Hack(?): mark filename as empty for now if not valid
      // Then we clean up the list after this loop
      if (!valid)
         list->files[i][0] = '\0';
   }

   // Clean up list from filtered out files
   size_t new_count = 0;
   for (size_t i = 0, last = 0; i < list->num_files; i++) {
      if (list->files[i][0] != '\0') {
         new_count++;
         list->files[last] = list->files[i];
         last++;
      }
   }

   // Set new list size
   // To-do: resize list... (lazy)
   list->num_files = new_count;
}

//***************************************************************************
// free_dir_list
// Deallocates a directory list.
//---------------------------------------------------------------------------
// param list: pointer to directory list (becomes invalid after call)
//***************************************************************************

void free_dir_list(Dir *list) {
   // Deallocate the memory for the filenames
   for (size_t i = 0; i < list->num_dirs; i++)
      free(list->dirs[i]);
   for (size_t i = 0; i < list->num_files; i++)
      free(list->files[i]);

   // Deallocate the memory for the lists
   if (list->dirs) free(list->dirs);
   if (list->files) free(list->files);

   // Deallocate the memory for the structure itself
   free(list);
}

//***************************************************************************
// find_basename
// Looks where the base name (i.e. sans-directory) starts in a path.
//---------------------------------------------------------------------------
// param str: pointer to filename
// return: where base name starts
//***************************************************************************

const char *find_basename(const char *str) {
#ifdef _WIN32
   // Look for BOTH directory separators
   // Yeah, thank you Microsoft for using both slashes >.>'
   const char *ptr1 = strrchr(str, '/');
   const char *ptr2 = strrchr(str, '\\');

   // If no directory separator was found, return the string as is (since it
   // is the base name itself!). If a directory separator was found, check
   // which one appears later (since somebody may attempt to mix both
   // separators in the same path >_>).
   if (!ptr1 && !ptr2)
      return str;
   else if (ptr1 > ptr2)
      return &ptr1[1];
   else
      return &ptr2[1];
#else
   // Look for the directory separator
   const char *ptr = strrchr(str, '/');

   // If the directory separator was found then return the character right
   // after it (which is where the base name starts), otherwise just return
   // the string as-is (since it *is* the base name)
   return ptr ? &ptr[1] : str;
#endif
}

//***************************************************************************
// is_root_dir
// Checks if a directory is root (has no parent) or not.
//---------------------------------------------------------------------------
// param path: path of directory to check
// return: zero if has a parent, non-zero if root
//***************************************************************************

int is_root_dir(const char *path) {
#ifdef _WIN32
   // Drive root directories are considered to be "root" (since this function
   // is used to determine if there are any parent directories or not) Yes, I
   // suppose this could be done in a cleaner way, but whatever...
   if (!isalpha(path[0])) return 0;
   return !strcmp(&path[1], ":");
#else
   // The root directory is always "/", so just compare against that.
   return !strcmp(path, "/");
#endif
}

//***************************************************************************
// get_parent_dir
// Takes a path and returns a new path to its parent directory. Remember to
// free the string when you don't need it anymore.
//---------------------------------------------------------------------------
// param path: path of directory to check
// return: path of parent directory
//***************************************************************************

char *get_parent_dir(const char *path) {
   // No parent?
   if (is_root_dir(path)) {
      // We really can't go up, so just return the same path... though
      // allocate memory for it since the return value is expected to be
      // freed
      char *new_path = (char *) malloc(strlen(path) + 1);
      if (new_path == NULL)
         abort_program(ERR_NOMEMORY, NULL);
      strcpy(new_path, path);

      // There we go
      return new_path;
   }

   // OK, go check for the parent directory
   else {
      // Find the point in the path where the base directory ends
      const char *end = find_basename(path) - 1;

      // Hack due to "/" being used for the root directory >_> (without this
      // trying to go there would result in "" which file APIs don't seem to
      // like). And yes, this should work fine because according to the
      // standard NULL should be 0, which means it's smaller than every valid
      // pointer.
      if (end <= &path[1]) end = &path[1];

      // Allocate memory to store the new path
      size_t len = end - path;
      char *new_path = (char *) malloc(len + 1);
      if (new_path == NULL)
         abort_program(ERR_NOMEMORY, NULL);
      memcpy(new_path, path, len);
      new_path[len] = '\0';

      // Done!
      return new_path;
   }
}

//***************************************************************************
// append_path
// Appends a filename or dirname to a path and returns a new path for it.
// Remember to free the string when you don't need it anymore.
//---------------------------------------------------------------------------
// param path: path of original directory
// param dir: which filename/dirname to append
// return: path of target directory
//***************************************************************************

char *append_path(const char *path, const char *dir) {
   // Quick hack to avoid crashes in case code is dumb
   // *cough*leveleditor*cough*
   if (path == NULL)
      path = "";
   if (dir == NULL)
      dir = "";

   // Determine length of original path
   // If it had a trailing backslash, throw it away
   int has_slash = 0;
   size_t len = strlen(path);
   if (path[len-1] == '/') has_slash = 1;
#ifdef _WIN32
   else if (path[len-1] == '\\') has_slash = 1;
#endif
   if (has_slash) len--;

   // Increase length to make room for the directory to be appended
   // We also take into account the new slash in place
   len += strlen(dir) + 1;

   // Allocate memory for the new path
   char *new_path = (char *) malloc(len + 1);
   if (new_path == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Generate the new path
   // Using backslashes under Windows for the sake of consistency (since this
   // path may be shown on screen in the file selector). We also avoid
   // writing the separating slash if it was already there (i.e. the original
   // path had a trailing slash).
   sprintf(new_path, "%s%s%s", path,
#ifdef _WIN32
      has_slash ? "" : "\\",
#else
      has_slash ? "" : "/",
#endif
      dir);

   // Done!
   return new_path;
}

//***************************************************************************
// read_line
// Reads a line from a file. Returns a pointer to a buffer with the line,
// which must be deallocated by calling free when done with it. Returns NULL
// only when there's an issue with the file, NOT when running out of memory
// (in which case abort_program is called).
//---------------------------------------------------------------------------
// param file: pointer to file handle
// return: pointer to line on success (remember to free), NULL on failure
//***************************************************************************

// Parameters to tweak the buffer where the line is read. Both numbers must
// be greather than zero, otherwise there is no restriction. Tweaking these
// values will affect performance.
#define BUF_SIZE 0x80   // How much to allocate initially
#define BUF_STEP 0x20   // How much to allocate when increasing

char *read_line(File *file) {
   // No more lines to read?
   if (end_of_file(file))
      return NULL;

   // To keep track of the size of the buffer.
   size_t bufsize = BUF_SIZE;
   size_t bufpos = 0;

   // Allocate initial buffer
   char *buffer = (char *) malloc(bufsize);
   if (buffer == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Keep reading characters until we're done
   for (;;) {
      // Read next character
      char ch;
      if (read_file(file, &ch, 1))
         break;

      // Newline?
      // This code will handle the backslash-newline combination properly. If
      // that's the case then we just get rid of it, otherwise cut the line
      // here instead.
      if (ch == '\n' || ch == '\r') {
         if (bufpos > 0 && buffer[bufpos-1] == '\\')
            bufpos--;
         else
            break;
      }

      // Huh...
      if (ch == '\0')
         continue;

      // Put character in the buffer!
      buffer[bufpos] = ch;
      bufpos++;

      // Allocate more memory?
      // The -1 is to make room for the null character
      if (bufpos == bufsize - 1) {
         bufsize += BUF_STEP;
         buffer = realloc(buffer, bufsize);
         if (buffer == NULL)
            abort_program(ERR_NOMEMORY, NULL);
      }
   }

   // Make sure to put the null character!
   buffer[bufpos] = '\0';

   // Return a pointer to the buffer with the line
   // Remember to free it when not needed anymore!
   return buffer;
}

//***************************************************************************
// get_file_checksum
// Computes a checksum for the specified file. Used to tell when a file was
// (most likely) modified.
//---------------------------------------------------------------------------
// param filename: name of file
// return: checksum
//***************************************************************************

uint64_t get_file_checksum(const char *filename) {
   // Open file if possible
   File *file = open_file(filename, FILE_READ);
   if (file == NULL) {
#ifdef DEBUG
      fprintf(stderr, "Error: no checksum for \"%s\"\n", filename);
#endif
      return 0;
   }

   // Intermediate checksums
   uint64_t add_checksum = 0;
   uint64_t xor_checksum = 0;
   uint64_t not_checksum = 0;
   uint64_t sub_checksum = 0;

   // Scan entire file
   while (!end_of_file(file)) {
      // Get byte
      uint8_t byte;
      if (read_file(file, &byte, 1))
         break;

      // Update intermediate checksums
      add_checksum = add_checksum << 8 | add_checksum >> 56;
      add_checksum += byte;
      xor_checksum ^= add_checksum;
      not_checksum = ~xor_checksum;
      sub_checksum -= not_checksum;
      add_checksum += sub_checksum;
   }

   // Compute final checksum
   uint64_t result;
   result = add_checksum;
   result = result << 16 | result >> 48;
   result ^= xor_checksum;
   result = result << 16 | result >> 48;
   result += not_checksum;
   result = result << 16 | result >> 48;
   result ^= sub_checksum;

   if (result == 0)
      result = 1;

   // Debugging, etc.
#ifdef DEBUG
   fprintf(stderr, "Checksum for \"%s\": %04X %04X %04X %04X\n", filename,
      (uint16_t)(result >> 48), (uint16_t)(result >> 32),
      (uint16_t)(result >> 16), (uint16_t)(result));
#endif

   // Done
   close_file(file);
   return result;
}

//***************************************************************************
// create_ini
// Creates a new (empty) INI object.
//---------------------------------------------------------------------------
// return: pointer to new INI object
//***************************************************************************

Ini *create_ini(void) {
   // Allocate memory for the structure
   Ini *ini = (Ini *) malloc(sizeof(Ini));
   if (ini == NULL)
      abort_program(ERR_NOMEMORY, NULL);

   // Initialize it
   ini->num_sections = 0;
   ini->sections = NULL;

   // Return new INI object
   return ini;
}

//***************************************************************************
// load_ini
// Reads an INI file and creates an INI object out of its contents.
//---------------------------------------------------------------------------
// param filename: name of file to read
// return: pointer to new INI object
//***************************************************************************

Ini *load_ini(const char *filename) {
   // Create new INI object
   Ini *ini = create_ini();

   // Open file to read
   File *file;
   int patch_ok = 0;
   if (strncmp(filename, "savedata/", 9) == 0) {
      file = open_save_file(filename, FILE_READ);
   } else {
      file = open_file(filename, FILE_READ);
      patch_ok = (strstr(filename, ".ini.patch") == NULL) ? 1 : 0;
   }
   if (file == NULL)
      return ini;

   // Where we store the name of the current section
   // If it's NULL, it's the global (nameless) section
   char *section = NULL;

   // Go through all lines in the file
   for (;;) {
      // Get next line
      char *line = read_line(file);
      if (line == NULL) break;
      char *ptr = line;

      // Skip initial whitespace
      // Note: the & 0x80 check is done because some implementations of
      // the C library treat bytes above 0x7F as being whitespace
      while (isspace(*ptr) && !(*ptr & 0x80))
         ptr++;

      // Blank? Comment?
      if (*ptr == '\0' || *ptr == '#') {
         free(line);
         continue;
      }

      // Section?
      if (*ptr == '[') {
         // Get start of section name
         // Skip initial whitespace
         do {
            ptr++;
         } while (isspace(*ptr) && !(*ptr & 0x80));
         char *start = ptr;

         // Get end of section name
         // Skip trailing whitespace
         // Note: the & 0x80 check is done because some implementations of
         // the C library treat bytes above 0x7F as being whitespace
         char *end = strchr(start, ']');
         if (end == NULL || end == start) {
            free(line);
            continue;
         }
         do {
            end--;
         } while (isspace(*end) && !(*end & 0x80));
         if (*end == '[') {
            free(line);
            continue;
         }
         end++;

         // End string at the end of the section name
         // This will make our lives easier :P
         *end = '\0';

         // Validate section name
         // Also make section name lowercase while we're at it
         for (ptr = start; ptr != end; ptr++) {
            *ptr = tolower(*ptr);
            if (isalnum(*ptr)) continue;
            if (*ptr == '_') continue;
            break;
         }
         if (ptr != end) {
            free(line);
            continue;
         }

         // Retrieve section name
         if (section != NULL)
            free(section);
         section = (char *) malloc(strlen(start) + 1);
         if (section == NULL)
            abort_program(ERR_NOMEMORY, NULL);
         strcpy(section, start);
      }

      // Variable?
      else {
         // Get where the equals sign is
         char *equals = strchr(ptr, '=');
         if (equals == NULL) {
            free(line);
            continue;
         }

         // Get start of variable name
         char *namestart = ptr;
         if (namestart == equals) {
            free(line);
            continue;
         }

         // Get end of variable name
         // Skip trailing whitespace
         // Note: the & 0x80 check is done because some implementations of
         // the C library treat bytes above 0x7F as being whitespace
         char *nameend = equals;
         do {
            nameend--;
         } while (isspace(*nameend) && !(*nameend & 0x80));
         nameend++;

         // Get start of variable value
         // Skip initial whitespace
         // Note: the & 0x80 check is done because some implementations of
         // the C library treat bytes above 0x7F as being whitespace
         char *valstart = equals;
         do {
            valstart++;
         } while (isspace(*valstart) && !(*valstart & 0x80));
         if (*valstart == '\0') {
            free(line);
            continue;
         }

         // Get end of variable value
         // Skip trailing whitespace
         // Note: the & 0x80 check is done because some implementations of
         // the C library treat bytes above 0x7F as being whitespace
         char *valend = valstart + strlen(valstart);
         do {
            valend--;
         } while (isspace(*valend) && !(*valend & 0x80));
         valend++;

         // Mark the end of both strings
         // This will make our lives easier :P
         *nameend = '\0';
         *valend = '\0';

         // Validate variable name
         // Also make variable name lowercase while we're at it
         for (ptr = namestart; ptr != nameend; ptr++) {
            *ptr = tolower(*ptr);
            if (isalnum(*ptr)) continue;
            if (*ptr == '_') continue;
            break;
         }
         if (ptr != nameend) {
            free(line);
            continue;
         }

         // Store variable in INI object
         set_ini_var(ini, section, namestart, valstart);
      }

      // Done with the line
      free(line);
   }

   // Free up stuff
   if (section != NULL)
      free(section);
   close_file(file);

   // If the INI file can be patched, look for it
   if (patch_ok) {
      // Append .patch suffix
      char *buffer = malloc(strlen(filename) + 7);
      if (buffer == NULL)
         abort_program(ERR_NOMEMORY, NULL);
      sprintf(buffer, "%s.patch", filename);

      // Load patched INI
      Ini *patch = load_ini(buffer);
      if (patch != NULL) {
         // Apply the patches
         for (unsigned s = 0; s < patch->num_sections; s++) {
            const IniSect *sect = &patch->sections[s];
            for (unsigned v = 0; v < sect->num_variables; v++) {
               const IniVar *var = &sect->variables[v];
               set_ini_var(ini, sect->name, var->name, var->value);
            }
         }
      }

      // Done
      destroy_ini(patch);
      free(buffer);
   }

   // Done
   return ini;
}

//***************************************************************************
// get_ini_var
// Retrieves the value of a variable from an INI object. If the variable
// isn't found, then it can return a fallback string instead (will return an
// empty string if no fallback is specified).
//---------------------------------------------------------------------------
// param ini: INI object to look into
// param sectname: name of the section
// param varname: name of the variable
// param fallback: fallback string
//***************************************************************************

const char *get_ini_var(const Ini *ini, const char *sectname,
const char *varname, const char *fallback) {
   // No section name, assume global section (nameless section)
   if (sectname == NULL)
      sectname = "";

   // Look for the specified section
   for (size_t i = 0; i < ini->num_sections; i++) {
      IniSect *section = &ini->sections[i];

      // Wrong section?
      if (strcmp(sectname, section->name) != 0)
         continue;

      // Look for a variable with the specified name within this section
      for (size_t i = 0; i < section->num_variables; i++) {
         IniVar *variable = &section->variables[i];

         // Wrong variable?
         if (strcmp(varname, variable->name) != 0)
            continue;

         // Found it, return its value!
         return variable->value;
      }

      // If we got here then the variable doesn't exist...
      // Stopping since no other section will have the same name
      break;
   }

   // Didn't find the variable, return fallback string
   return fallback == NULL ? "" : fallback;
}

//***************************************************************************
// set_ini_var
// Sets the value of a variable in an INI object. The variable is created if
// it doesn't exist yet.
//---------------------------------------------------------------------------
// param ini: INI object to modify
// param sectname: name of the section
// param varname: name of the variable
// param value: new value for variable
//***************************************************************************

void set_ini_var(Ini *ini, const char *sectname, const char *varname,
const char *value) {
   // No section name, assume global section (nameless section)
   if (sectname == NULL)
      sectname = "";

   // Find a section with this name
   IniSect *section = NULL;
   for (size_t i = 0; i < ini->num_sections; i++) {
      if (!strcmp(sectname, ini->sections[i].name)) {
         section = &ini->sections[i];
         break;
      }
   }

   // Section doesn't exist? Create one!
   if (section == NULL) {
      ini->num_sections++;
      ini->sections = (IniSect *) realloc(ini->sections,
         sizeof(IniSect) * ini->num_sections);
      if (ini->sections == NULL)
         abort_program(ERR_NOMEMORY, NULL);
      section = &ini->sections[ini->num_sections - 1];

      // Store section name
      section->name = (char *) malloc(strlen(sectname)+1);
      if (section->name == NULL)
         abort_program(ERR_NOMEMORY, NULL);
      strcpy(section->name, sectname);

      // Section has no variables yet!
      section->num_variables = 0;
      section->variables = NULL;
   }

   // Find a variable with this name
   IniVar *variable = NULL;
   for (size_t i = 0; i < section->num_variables; i++) {
      if (!strcmp(varname, section->variables[i].name)) {
         variable = &section->variables[i];
         break;
      }
   }

   // Variable doesn't exist? Create one!
   if (variable == NULL) {
      section->num_variables++;
      section->variables = (IniVar *) realloc(section->variables,
         sizeof(IniVar) * section->num_variables);
      if (section->variables == NULL)
         abort_program(ERR_NOMEMORY, NULL);
      variable = &section->variables[section->num_variables - 1];

      // Store variable name
      variable->name = (char *) malloc(strlen(varname)+1);
      if (variable->name == NULL)
         abort_program(ERR_NOMEMORY, NULL);
      strcpy(variable->name, varname);

      // Variable has no value yet!
      variable->value = NULL;
   }

   // Store new value into variable
   if (variable->value)
      free(variable->value);
   variable->value = (char *) malloc(strlen(value)+1);
   if (variable->value == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   strcpy(variable->value, value);
}

//***************************************************************************
// save_ini
// Writes an INI file.
//---------------------------------------------------------------------------
// param filename: name of file to write
// param ini: pointer to INI object
// return: zero on success or non-zero on failure
//***************************************************************************

int save_ini(const char *filename, const Ini *ini) {
   // Open file
   File *file = open_save_file(filename, FILE_WRITE);
   if (file == NULL)
      return -1;

   // We'll need this a lot
   char *buffer;
   size_t len;

   // Go through all sections in the INI object
   for (size_t i = 0; i < ini->num_sections; i++) {
      IniSect *section = &ini->sections[i];

      // Generate string with the header line
      if (section->name && section->name[0] != '\0') {
         len = strlen(section->name) + 3;
         buffer = (char *) malloc(len+1);
         if (buffer == NULL)
            abort_program(ERR_NOMEMORY, NULL);
         sprintf(buffer, "[%s]\n", section->name);

         // Write line
         if (write_file(file, buffer, len)) {
            free(buffer);
            close_file(file);
            return -1;
         }
         free(buffer);
      }

      // Go through all variables in the section
      for (size_t i = 0; i < section->num_variables; i++) {
         IniVar *variable = &section->variables[i];

         // Generate string with the line
         len = strlen(variable->name) + strlen(variable->value) + 2;
         buffer = (char *) malloc(len+1);
         if (buffer == NULL)
            abort_program(ERR_NOMEMORY, NULL);
         sprintf(buffer, "%s=%s\n", variable->name, variable->value);

         // Write line
         if (write_file(file, buffer, len)) {
            free(buffer);
            close_file(file);
            return -1;
         }
         free(buffer);
      }
   }

   // Success!
   close_file(file);
   return 0;
}

//***************************************************************************
// destroy_ini
// Destroys an INI object, freeing up all the resources it was using.
//---------------------------------------------------------------------------
// param ini: pointer to INI object (becomes invalid after call)
//***************************************************************************

void destroy_ini(Ini *ini) {
   // Go through all sections in the INI object
   for (size_t i = 0; i < ini->num_sections; i++) {
      IniSect *section = &ini->sections[i];

      // Free all the variable data in the section
      for (size_t i = 0; i < section->num_variables; i++) {
         IniVar *variable = &section->variables[i];
         if (variable->name)
            free(variable->name);
         if (variable->value)
            free(variable->value);
      }

      // Get rid of variables
      if (section->variables)
         free(section->variables);
   }

   // Get rid of sections
   if (ini->sections)
      free(ini->sections);

   // Get rid of INI object
   free(ini);
}
