//***************************************************************************
// "editor.c"
// Level editor
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include <stdint.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "main.h"
#include "editor.h"
#include "file.h"
#include "input.h"
#include "level.h"
#include "loading.h"
#include "menu.h"
#include "reader.h"
#include "settings.h"
#include "text.h"
#include "video.h"

// Data of the map being currently edited
static Map map = { 0, 0, 0, 0, THEME_VIRTUAL, NULL };

// Different screens making up the editor
// Most of the time we'll stay in the main GUI (maybe switching often to the
// tile/object selectors), where occasionally going into the load and save
// dialogs.
typedef enum {
   EDSCRN_MAIN,         // Main editor GUI
   EDSCRN_COLL,         // Select collision type
   EDSCRN_OBJ,          // Select object type
   EDSCRN_INFO,         // Level info dialog
   EDSCRN_LOAD,         // Load level dialog
   EDSCRN_SAVE,         // Save level dialog
   EDSCRN_HELP          // Editor help
} EditorScreen;
static EditorScreen editor_screen;

// Scroll coordinates of the editor region (in tiles)
static unsigned region_x;
static unsigned region_y;

// Dimensions (in tiles) of the tilemap region
// We need to calculate them on the fly since how many tiles can be shown on
// screen depends on the current resolution.
static int region_width;
static int region_height;

// A timer used to count the delay between each step when scrolling. Without
// this we'd be scrolling at one tile per frame = 60 tiles per second =
// so fast it'd be impossible to use.
#define SCROLL_DELAY 5
static unsigned scroll_timer;

// How many tiles to move in "fast scrolling" mode
#define FAST_SCROLL 10

// Different editing modes
typedef enum {
   EDMODE_TILEMAP,         // Collision placement
   EDMODE_OBJECTS,         // Object placement
   NUM_EDMODES             // Number of editor modes
} EditorMode;

// List of objects arranged in a more logical way
static const LevelObjID obj_table[] = {
   LVOBJ_NONE,

   // Items
   LVOBJ_HEART, LVOBJ_INVINCIBILITY, LVOBJ_SHIELD, LVOBJ_DANCER,
      LVOBJ_BLUESTAR,
   LVOBJ_CHECKPOINT, LVOBJ_GOAL, LVOBJ_SPRING, LVOBJ_SSPRING, LVOBJ_BALLOON,
   LVOBJ_WINGS, LVOBJ_SPIDERPOW, LVOBJ_HAMMER, LVOBJ_PARASOL,

   // Scenery
   LVOBJ_PLATFORM, LVOBJ_PLATFORM_H, LVOBJ_PLATFORM_V, LVOBJ_PLATFORM_B,
   LVOBJ_PUSHABLE, LVOBJ_OBSTRUCTION, LVOBJ_SCENERYSMALL, LVOBJ_SCENERYBIG,
   LVOBJ_GENERATOR, LVOBJ_MAINGENERATOR,

   // Enemies
   LVOBJ_FLAMER, LVOBJ_SPRAYER, LVOBJ_TURRET, LVOBJ_ROLLER,
   LVOBJ_GRABBER, LVOBJ_SPIDER, LVOBJ_HEATER, LVOBJ_BOMB,
   LVOBJ_BOSS1, LVOBJ_BOSS2, LVOBJ_BOSS3,
   LVOBJ_BOSS4, LVOBJ_BOSS5,

   // Hazards
   LVOBJ_FIRE, LVOBJ_LIQUIDHAZARD, LVOBJ_BOUNCINGHAZARD, LVOBJ_STALACTITE,
   LVOBJ_SPIKES_U, LVOBJ_SPIKES_D, LVOBJ_SPIKES_L, LVOBJ_SPIKES_R,
   LVOBJ_CRUSHER, LVOBJ_BUZZSAW, LVOBJ_BUZZSAW_C, LVOBJ_BUZZSAW_F,
   LVOBJ_COIL_F, LVOBJ_COIL_C,

   // Switches
   LVOBJ_SWITCH_1, LVOBJ_SWITCH_2, LVOBJ_SWITCH_3, LVOBJ_SWITCH_4,
   LVOBJ_SWITCH_5, LVOBJ_SWITCH_6, LVOBJ_SWITCH_7, LVOBJ_SWITCH_8,
   LVOBJ_DOOR_1, LVOBJ_DOOR_2, LVOBJ_DOOR_3, LVOBJ_DOOR_4,
   LVOBJ_DOOR_5, LVOBJ_DOOR_6, LVOBJ_DOOR_7, LVOBJ_DOOR_8
};

// Type of tile/object we'll place when pressing the mouse button
static EditorMode editor_mode;
static TileType coll_type;
static unsigned obj_type;

// How many steps can be undone at most
#define MAX_UNDO 0x400

// Possible actions that can be recorded in the undo stack
// This should be a typedef, but those are stored as ints, and we're going to
// use uint8_t instead to reduce space usage.
enum {
   UNDO_TILEMAP,           // Tilemap was modified
   UNDO_OBJECT,            // Object was modified
   UNDO_FLAGS              // Flags were modified
};

// Where the undo data is stored
typedef struct {
   uint8_t type;           // What got modified
   uint8_t old_value;      // Value before modification
   uint8_t new_value;      // Value after modification
   uint16_t x;             // X coordinate of tile
   uint16_t y;             // Y coordinate of tile
} UndoStack;
static UndoStack undo_stack[MAX_UNDO];
static unsigned undo_pos;         // Where the next undo step will go
static unsigned undo_count;       // How many steps are in the stack
static unsigned redo_count;       // How many steps can be redone

// Different states for a button
typedef enum {
   BSTATE_RELEASED,        // Released (normal state)
   BSTATE_PRESSED,         // Pressed (being held down)
   BSTATE_DISABLED,        // Disabled (can't be used)
   NUM_BSTATE              // Number of button states
} ButtonState;

// IDs for each of the buttons in the editor UI
typedef enum {
   // *** MAIN GUI ***
   BUTTON_NEW,             // New level
   BUTTON_INFO,            // Level information
   BUTTON_LOAD,            // Load level
   BUTTON_SAVE,            // Save level
   BUTTON_PLAY,            // Play level
   BUTTON_TILEMAP,         // Tilemap mode
   BUTTON_OBJECTS,         // Objects mode
   BUTTON_TILE,            // Select tile
   BUTTON_UNDO,            // Undo last action
   BUTTON_REDO,            // Redo last action
   BUTTON_HELP,            // Show help screen
   BUTTON_QUIT,            // Quit editor
   BUTTON_UP,              // Scroll up
   BUTTON_DOWN,            // Scroll down
   BUTTON_LEFT,            // Scroll left
   BUTTON_RIGHT,           // Scroll right
   BUTTON_MARKER_X,        // Horizontal scroll marker
   BUTTON_MARKER_Y,        // Vertical scroll marker
   BUTTON_SCROLL_XL,       // Horizontal scrollbar background (left)
   BUTTON_SCROLL_XR,       // Horizontal scrollbar background (right)
   BUTTON_SCROLL_YU,       // Vertical scrollbar background (up)
   BUTTON_SCROLL_YD,       // Vertical scrollbar background (down)

   // *** LEVEL INFO ***
   IBUTTON_APPLY,          // "Apply" button
   IBUTTON_CANCEL,         // "Cancel" button
   IBUTTON_INC_U,          // Add tiles at the top
   IBUTTON_DEC_U,          // Remove tiles from the top
   IBUTTON_INC_D,          // Add tiles at the bottom
   IBUTTON_DEC_D,          // Remove tiles from the bottom
   IBUTTON_INC_L,          // Add tiles at the left
   IBUTTON_DEC_L,          // Remove tiles from the left
   IBUTTON_INC_R,          // Add tiles at the right
   IBUTTON_DEC_R,          // Remove tiles from the right

   // *** FILE SELECT ***
   FBUTTON_OK,             // "Load"/"Save" button
   FBUTTON_CANCEL,         // "Cancel" button
   FBUTTON_LISTUP,         // Scroll file list up
   FBUTTON_LISTDOWN,       // Scroll file list down
   FBUTTON_LISTMARKER,     // File list scroll marker
   FBUTTON_MEDIAUP,        // Scroll media list up
   FBUTTON_MEDIADOWN,      // Scroll media list down
   FBUTTON_MEDIAMARKER,    // Media list scroll marker
   FBUTTON_SCROLL_LU,      // File list scrollbar (up)
   FBUTTON_SCROLL_LD,      // File list scrollbar (down)
   FBUTTON_SCROLL_MU,      // Media list scrollbar (up)
   FBUTTON_SCROLL_MD,      // Media list scrollbar (down)
   FBUTTON_DIRUP,          // Go a directory up

   // *** HELP ***
   HBUTTON_PREV,           // Previous page
   HBUTTON_NEXT,           // Next page
   HBUTTON_BACK,           // Return to editor

   // *** OBJECT SELECT ***
   OBUTTON_ITEM,           // Items
   OBUTTON_SCENERY,        // Scenery
   OBUTTON_ENEMY,          // Enemies
   OBUTTON_HAZARD,         // Hazards
   OBUTTON_SWITCH,         // Switches

   NUM_BUTTONS             // Number of buttons
} ButtonID;

// Where information for each editor UI button is stored
typedef struct {
   int x1, y1;                   // Top-left coordinates (inclusive)
   int x2, y2;                   // Bottom-right coordinates (inclusive)
   ButtonState state;            // Current state
   unsigned scroller: 1;         // Use scrollbar semantics

   Sprite *sprite[NUM_BSTATE];   // Sprites to use for this button
   const char *name;             // Short description (can be NULL for none)
   void (*func)(void);           // Function associated with this button
} Button;
static Button buttons[NUM_BUTTONS] = {
   { 0,0,0,0, BSTATE_RELEASED, 0, { NULL, NULL, NULL }, NULL, NULL }
};

// Keep track over which button is the cursor
// This is NULL when the cursor isn't over any button
static Button *over_button;

// Whatever button the user is interacting with currently. You can start
// interacting by clicking on a button, and it will remain the current one
// until you release the mouse. No other buttons can get triggered until
// you click again.
static Button *curr_button;

// IDs for each of the labels in the editor UI
// This doesn't count stuff that has buttons
typedef enum {
   // *** LEVEL INFO ***
   ILABEL_THEME1,          // Level theme names
   ILABEL_THEME2,
   ILABEL_THEME3,
   ILABEL_THEME4,
   ILABEL_THEME5,
   ILABEL_THEME6,
   ILABEL_THEME7,
   ILABEL_THEME8,
   ILABEL_TOP,             // Top border
   ILABEL_BOTTOM,          // Bottom border
   ILABEL_LEFT,            // Left border
   ILABEL_RIGHT,           // Right border

   // *** FILE SELECT ***
   FLABEL_LIST1,           // File list items (I think 7 is enough but eh)
   FLABEL_LIST2,
   FLABEL_LIST3,
   FLABEL_LIST4,
   FLABEL_LIST5,
   FLABEL_LIST6,
   FLABEL_LIST7,
   FLABEL_LIST8,
   FLABEL_MEDIA1,          // Media list items (same deal as above)
   FLABEL_MEDIA2,
   FLABEL_MEDIA3,
   FLABEL_MEDIA4,
   FLABEL_MEDIA5,
   FLABEL_MEDIA6,
   FLABEL_MEDIA7,
   FLABEL_MEDIA8,

   // *** HELP ***
   HLABEL_LEFT1, HLABEL_LEFT2, HLABEL_LEFT3,       // Left column
   HLABEL_LEFT4, HLABEL_LEFT5, HLABEL_LEFT6,
   HLABEL_LEFT7, HLABEL_LEFT8, HLABEL_LEFT9,
   HLABEL_RIGHT1, HLABEL_RIGHT2, HLABEL_RIGHT3,    // Right column
   HLABEL_RIGHT4, HLABEL_RIGHT5, HLABEL_RIGHT6,
   HLABEL_RIGHT7, HLABEL_RIGHT8, HLABEL_RIGHT9,

   // *** OBJECT SELECT ***
   OLABEL_TILE,            // Any tile, really

   NUM_LABELS              // Number of labels
} LabelID;

// Where information for each label is stored
typedef struct {
   int x1, y1;                   // Top-left coordinates (inclusive)
   int x2, y2;                   // Bottom-right coordinates (inclusive)
   const char *caption;          // Caption to be shown
   const char *strparam;         // String parameter (NULL if none)
   int intparam;                 // Integer parameter (if above is NULL)
} Label;
static Label labels[NUM_LABELS] = {
   { 0, 0, 0, 0, NULL, NULL, 0 }
};

// Used to hold the collision/object choice grids
#define MAX_CHOICE 0x20
static struct {
   unsigned value;
   int16_t x;
   int16_t y;
   const Sprite *sprite;
} choices[MAX_CHOICE];
static unsigned num_choices;
static int over_choice;

// Include the choice grid data
// In a separate file to make up space
#include "editor_choices.h"

// Dimensions and such for the level info dialog
#define INFO_BUTTON_X1 (screen_cx - 0x30)    // X for "Apply" button
#define INFO_BUTTON_X2 (screen_cx + 0x30)    // X for "Cancel" button
#define INFO_BUTTON_Y (screen_h - 0x10)      // Y for both buttons

#define INFO_THEME_YTITLE (0x10)             // Theme title Y coordinate
#define INFO_THEME_YTHUMB (0x20)             // Theme thumbnails coordinates
#define INFO_THEME_YTHUMB2 (0x3F)
#define INFO_THEME_XTHUMB (screen_cx - (NUM_THEMES * 0x24 - 4) / 2)
#define INFO_THEME_XTHUMB2 (screen_cx + (NUM_THEMES * 0x24 - 4) / 2)

#define INFO_SIZE_YTITLE (0x50)              // Size title Y coordinate
#define INFO_SIZE_YCURR (0xA0)               // Current size Y coordinate
#define INFO_SIZE_YNEW (0xB0)                // New size Y coordinate

#define INFO_SIZE_X1 (screen_cx - 0x80)      // X of ↑ and ← labels
#define INFO_SIZE_X2 (screen_cx - 0x70)      // X of - for ↑ and ←
#define INFO_SIZE_X3 (screen_cx - 0x40)      // X of ↑ and ← values
#define INFO_SIZE_X4 (screen_cx - 0x28)      // X of + for ↑ and ←
#define INFO_SIZE_X5 (screen_cx + 0x10)      // X of ↓ and → labels
#define INFO_SIZE_X6 (screen_cx + 0x20)      // X of - for ↓ and →
#define INFO_SIZE_X7 (screen_cx + 0x50)      // X of ↓ and → values
#define INFO_SIZE_X8 (screen_cx + 0x68)      // X of + for ↓ and →
#define INFO_SIZE_Y1B (0x60)                 // Y of ↑ and ↓ buttons
#define INFO_SIZE_Y1T (0x68)                 // Y of ↑ and ↓ labels
#define INFO_SIZE_Y2B (0x78)                 // Y of ← and → buttons
#define INFO_SIZE_Y2T (0x80)                 // Y of ← and → labels

// To hold the values set in the level info dialog before they're applied
// These values are only applied to the map when the user clicks "Apply"
static struct {
   // Level theme
   Theme theme;

   // New map dimensions (for convenience)
   unsigned new_width;
   unsigned new_height;

   // Amount of tiles to add/remove at each border
   // > 0 means add, < 0 means remove, 0 means do nothing
   int resize_up;
   int resize_down;
   int resize_left;
   int resize_right;
} info_temp;

// Dimensions and such for the file selector dialogs
#define FILE_TITLE_X (screen_cx)             // X coordinate for title
#define FILE_TITLE_Y (0x10)                  // Y coordinate for title
#define FILE_BUTTON_X1 (screen_cx - 0x30)    // X for "Load"/"Save" button
#define FILE_BUTTON_X2 (screen_cx + 0x30)    // X for "Cancel" button
#define FILE_BUTTON_Y (screen_h - 0x10)      // Y for both buttons
#define FILE_PATH_X (screen_cx)              // X coordinate for path
#define FILE_PATH_Y (screen_h - 0x28)        // Y coordinate for path

#define FILE_LIST_X1 (0x08)                  // File list dimensions
#define FILE_LIST_X2 (screen_w - 0x60)
#define FILE_LIST_Y1 (0x38)
#define FILE_LIST_Y2 (screen_h - 0x38)

#define FILE_NAME_X1 (0x08)                  // Filename textbox dimensions
#define FILE_NAME_X2 (screen_w - 0x08)
#define FILE_NAME_Y1 (0x20)
#define FILE_NAME_Y2 (0x2F)

#define FILE_MEDIA_X1 (screen_w - 0x50)      // Media list dimensions
#define FILE_MEDIA_X2 (screen_w - 0x18)
#define FILE_MEDIA_Y1 (0x38)
#define FILE_MEDIA_Y2 (screen_h - 0x38)

// Filename being entered in the file selector
static struct {
   char *path;             // Path of filename
   size_t len;             // Size of path (in CHARACTERS)
   size_t pos;             // Position of the cursor
   int32_t scroll;         // Scroll position of the field
} filename = {
   NULL, 0, 0, 0
};

// Directory lists
typedef struct {
   Dir *entries;           // List of entries
   unsigned pos;           // Scroll position
   unsigned size;          // How many entries can be shown on screen
   size_t count;           // How many entries are in the list in total
   unsigned selected;      // Which entry is selected (UINT_MAX = none)
} FList;

static FList file_list;    // List of files in current directory
static FList media_list;   // List of storage media

// Path for the directory being currently browsed in the file selector
// If NULL it means we haven't entered the file selector yet and a default
// path will be used when we first get into it
static char *current_path = NULL;

// Used to tell which part of the dialog has focus in the file selector
enum {
   FFOCUS_NONE,            // Nothing
   FFOCUS_NAME,            // Filename textbox
   FFOCUS_LIST,            // File list
   FFOCUS_MEDIA            // Media list
};
static unsigned file_focus;

// Used to blink the textbox cursor
static uint8_t cursor_blink;

// Dimensions and such for the object list buttons
#define OBJLIST_X (screen_cx - 0x70)   // Reference coordinate
#define OBJLIST_X1 (OBJLIST_X - 0x30)  // Left limit
#define OBJLIST_X2 (OBJLIST_X + 0x2F)  // Right limit

#define OBJLIST_Y (screen_cy)          // Reference coordinate
#define OBJLIST_Y1 (OBJLIST_Y - 0x30)  // First button
#define OBJLIST_Y2 (OBJLIST_Y - 0x1C)  // Second button
#define OBJLIST_Y3 (OBJLIST_Y - 0x08)  // Third button
#define OBJLIST_Y4 (OBJLIST_Y + 0x0C)  // Fourth button
#define OBJLIST_Y5 (OBJLIST_Y + 0x20)  // Fifth button

// Dimensions for the help screens
#define HELP_X1 (screen_cx - 60)       // Left column (shortcuts)
#define HELP_X2 (screen_cx - 40)       // Right column (descriptions)
#define HELP_Y1 (screen_cy - 0x40)     // Position of first line
#define HELP_Y(n) (HELP_Y1+(n)*0x10)   // To get the position of any line

#define HELP_TX (screen_cx)            // Title horizontal coordinate
#define HELP_TY (HELP_Y(0) - 0x18)     // Title vertical coordinate
#define HELP_PX (screen_cx)            // Page horizontal coordinate
#define HELP_PY (HELP_Y(8) + 0x18)     // Page vertical coordinate

#define HELP_PBX1 (screen_cx - 0x60)   // Previous page button X coordinate
#define HELP_PBX2 (screen_cx + 0x40)   // Next page button X coordinate
#define HELP_PBY (HELP_Y(8) + 0x10)    // Page buttons Y coordinate
#define HELP_PBXB (screen_cx - 0x98)   // Back button X coordinate
#define HELP_PBYB (HELP_Y(0) - 0x20)   // Back button Y coordinate

// Help page to show
enum {
   HELPPAGE_MOVEMENT,      // Movement page
   HELPPAGE_COMMANDS,      // Commands page
   HELPPAGE_MISC,          // Miscellaneous page
   NUM_HELPPAGES           // Number of help pages
};
static unsigned help_page;

// Possible sizes to show objects
enum {
   OBJSIZE_NORMAL,     // Normal (in the edit window)
   OBJSIZE_PREVIEW,    // Preview (in the GUI)
   NUM_OBJSIZES
};

// Graphics used by the level editor
static GraphicsSet *gfxset_editor = NULL;
static GraphicsSet *gfxset_objs[NUM_OBJSIZES] = { NULL, NULL };
static Sprite *spr_blank_tile;
static Sprite *spr_coord_digit[10];
static Sprite *spr_spawn;
static Sprite *spr_folder;
static Sprite *spr_coll_tile[NUM_TILETYPES];
static Sprite *spr_obj_tile[NUM_LVOBJIDS][NUM_OBJSIZES];
static Sprite *spr_themes[NUM_THEMES];
static Sprite *spr_disabled_theme;

// Message box stuff
static struct {
   int visible;                  // Set if visible
   char *message;                // Message to show
   int32_t x1, y1, x2, y2;       // Size of box
} messagebox;

// Private function prototypes
static void load_editor(void);

static void run_editor_gui(void);
static void run_tile_select(void);
static void run_level_info(void);
static void run_file_selector(void);
static void run_editor_help(void);

static void draw_editor_gui(void);
static void draw_tile_select(void);
static void draw_level_info(void);
static void draw_file_selector(void);
static void draw_editor_help(void);

static void draw_frame(int, int, int, int);
static void reset_buttons(void);
static void reset_labels(void);
static void process_buttons(void);
static void set_disabled(ButtonID, int);
static void draw_buttons(void);
static void init_choices(const uint8_t *, int32_t, int32_t);
static void process_choices(void);
static void draw_choices(void);

static void init_editor_gui(void);
static void init_level_info(void);
static void init_level_info_ui(void);
static void init_file_selector(void);
static void init_file_selector_ui(void);
static void init_help(void);
static void init_object_buttons(void);

static void update_file_list(void);
static void update_media_list(void);
static void nuke_filename(void);
static void set_filename(const char *);
static void go_to_dir(const char *);

static void button_new(void);
static void button_info(void);
static void button_load(void);
static void button_save(void);
static void button_play(void);
static void button_tilemap(void);
static void button_objects(void);
static void button_tile(void);
static void button_undo(void);
static void button_redo(void);
static void button_help(void);
static void button_quit(void);
static void button_scroll_up(void);
static void button_scroll_down(void);
static void button_scroll_left(void);
static void button_scroll_right(void);
static void button_scroll_up_fast(void);
static void button_scroll_down_fast(void);
static void button_scroll_left_fast(void);
static void button_scroll_right_fast(void);
static void move_up(void);
static void move_down(void);
static void move_left(void);
static void move_right(void);

static void ibutton_apply(void);
static void ibutton_cancel(void);
static void ibutton_inc_up(void);
static void ibutton_dec_up(void);
static void ibutton_inc_down(void);
static void ibutton_dec_down(void);
static void ibutton_inc_left(void);
static void ibutton_dec_left(void);
static void ibutton_inc_right(void);
static void ibutton_dec_right(void);

static void fbutton_ok(void);
static void fbutton_cancel(void);
static void fbutton_list_up(void);
static void fbutton_list_down(void);
static void fbutton_list_up_fast(void);
static void fbutton_list_down_fast(void);
static void fbutton_media_up(void);
static void fbutton_media_down(void);
static void fbutton_media_up_fast(void);
static void fbutton_media_down_fast(void);
static void fbutton_dir_up(void);

static void hbutton_previous_page(void);
static void hbutton_next_page(void);
static void hbutton_back(void);

static void obutton_items(void);
static void obutton_scenery(void);
static void obutton_enemies(void);
static void obutton_hazards(void);
static void obutton_switches(void);

static unsigned scroll_add(unsigned, unsigned, unsigned);
static unsigned scroll_sub(unsigned, unsigned);
static void new_map(unsigned, unsigned);
static int load_map(const char *);
static int save_map(const char *);
static void unload_map(void);
static UndoStack *push_undo(void);

static void show_messagebox(const char *);
static void run_messagebox(void);
static void draw_messagebox(void);

// Sprite names for each kind of collision tile type
// Use an empty string to indicate no sprite
static const char *coll_names[NUM_TILETYPES] = {
   "",
   "type_solid",
   "type_nwse_1",
   "type_nwse_2",
   "type_nesw_1",
   "type_nesw_2",
   "type_empty_bg",
   "type_floor_bg",
   "type_nwse_1_bg",
   "type_nwse_2_bg",
   "type_nesw_1_bg",
   "type_nesw_2_bg",
   "type_hidden",
   "type_bridge",
   "type_belt_left",
   "type_belt_right"
};

// Sprite names for each kind of object type
// They're in the same order as level object IDs
// Use an empty string to indicate no sprite
static const char *obj_names[NUM_LVOBJIDS] = {
   "",
   "flamer",
   "sprayer",
   "turret",
   "roller",
   "grabber",
   "spider",
   "heater",
   "bomb",
   "spring",
   "pushable",
   "platform_idle",
   "platform_horizontal",
   "platform_vertical",
   "platform_breakable",
   "heart",
   "invincibility",
   "checkpoint",
   "spikes_up",
   "spikes_down",
   "scenery_small",
   "scenery_big",
   "superspring",
   "spikes_left",
   "spikes_right",
   "obstruction",
   "shield",
   "crusher",
   "liquid_hazard",
   "goal",
   "wings_powerup",
   "hammer_powerup",
   "dancer",
   "fire",
   "parasol_powerup",
   "spider_powerup",
   "stalactite",
   "switch_1",
   "switch_2",
   "switch_3",
   "switch_4",
   "switch_5",
   "switch_6",
   "switch_7",
   "switch_8",
   "door_1",
   "door_2",
   "door_3",
   "door_4",
   "door_5",
   "door_6",
   "door_7",
   "door_8",
   "bouncing_hazard",
   "floor_coil",
   "ceiling_coil",
   "buzzsaw",
   "ceiling_buzzsaw",
   "floor_buzzsaw",
   "boss_spawn_1",
   "boss_spawn_5",
   "boss_spawn_2",
   "boss_spawn_4",
   "boss_spawn_3",
   "boss_spawn_6",
   "blue_star",
   "balloon",
   "generator",
   "main_generator"
};

// Sprite names for each theme
static const char *theme_names[] = {
   "virtual",
   "park",
   "sewer",
   "harbor",
   "desert",
   "cave",
   "factory",
   "carnival"
};

//***************************************************************************
// init_editor
// Initializes the level editor.
//***************************************************************************

void init_editor(void) {
   // Reset cursor in one-switch mode
   input.oneswitch.dir = 0;

   // Start in the main editor GUI
   editor_screen = EDSCRN_MAIN;

   // Start with a blank map, unless we just come from testing the map that
   // was being edited, in which case we should keep it
   if (prev_game_mode != GAMEMODE_INGAME)
      new_map(MIN_MAP_WIDTH, MIN_MAP_HEIGHT);

   // Load level editor graphics
   loading_screen(load_editor);

#define SPRITE(where, name) where = get_sprite(gfxset_editor, name)
   // Retrieve the sprites for the grid
   SPRITE(spr_blank_tile, "blank_tile");
   SPRITE(spr_coord_digit[0], "coord_digit_0");
   SPRITE(spr_coord_digit[1], "coord_digit_1");
   SPRITE(spr_coord_digit[2], "coord_digit_2");
   SPRITE(spr_coord_digit[3], "coord_digit_3");
   SPRITE(spr_coord_digit[4], "coord_digit_4");
   SPRITE(spr_coord_digit[5], "coord_digit_5");
   SPRITE(spr_coord_digit[6], "coord_digit_6");
   SPRITE(spr_coord_digit[7], "coord_digit_7");
   SPRITE(spr_coord_digit[8], "coord_digit_8");
   SPRITE(spr_coord_digit[9], "coord_digit_9");
   SPRITE(spr_spawn, "spawn");
   SPRITE(spr_folder, "folder");

   // Not part of the grid but since it's loaded on its own...
   SPRITE(spr_disabled_theme, "disabled_theme");
#undef SPRITE

   // Retrieve the sprites for each tile
   for (unsigned i = 0; i < NUM_TILETYPES; i++)
      spr_coll_tile[i] = get_sprite(gfxset_editor, coll_names[i]);

   // Retrieve the sprites for each object
   for (unsigned i = 0; i < NUM_LVOBJIDS; i++) {
      spr_obj_tile[i][OBJSIZE_NORMAL] =
         get_sprite(gfxset_objs[OBJSIZE_NORMAL], obj_names[i]);
      spr_obj_tile[i][OBJSIZE_PREVIEW] =
         get_sprite(gfxset_objs[OBJSIZE_PREVIEW], obj_names[i]);
   }

   // Retrieve the sprites for each theme thumbnail
   for (unsigned i = 0; i < NUM_THEMES; i++) {
      char buffer[0x40];
      sprintf(buffer, "theme_%s", theme_names[i]);
      spr_themes[i] = get_sprite(gfxset_editor, buffer);
   }

   // Initialize some button properties to a default value
   // We'll fill them in as needed in the following code
   // Buttons that don't get some data assigned don't need it
   for (unsigned i = 0; i < NUM_BUTTONS; i++) {
      buttons[i].sprite[BSTATE_RELEASED] = NULL;
      buttons[i].sprite[BSTATE_PRESSED] = NULL;
      buttons[i].sprite[BSTATE_DISABLED] = NULL;
      buttons[i].name = NULL;
      buttons[i].func = NULL;
      buttons[i].scroller = 0;
   }

   // Assign the sprites to use for each button in the toolbar
#define ASSIGN(name, state, id) \
   buttons[name].sprite[state] = get_sprite(gfxset_editor, id);

   ASSIGN(BUTTON_NEW, BSTATE_RELEASED, "button_new");
   ASSIGN(BUTTON_NEW, BSTATE_PRESSED, "button_new_press");
   ASSIGN(BUTTON_INFO, BSTATE_RELEASED, "button_info");
   ASSIGN(BUTTON_INFO, BSTATE_PRESSED, "button_info_press");
   ASSIGN(BUTTON_LOAD, BSTATE_RELEASED, "button_load");
   ASSIGN(BUTTON_LOAD, BSTATE_PRESSED, "button_load_press");
   ASSIGN(BUTTON_SAVE, BSTATE_RELEASED, "button_save");
   ASSIGN(BUTTON_SAVE, BSTATE_PRESSED, "button_save_press");
   ASSIGN(BUTTON_PLAY, BSTATE_RELEASED, "button_play");
   ASSIGN(BUTTON_PLAY, BSTATE_PRESSED, "button_play_press");
   ASSIGN(BUTTON_TILEMAP, BSTATE_RELEASED, "button_tilemap");
   ASSIGN(BUTTON_TILEMAP, BSTATE_PRESSED, "button_tilemap_press");
   ASSIGN(BUTTON_OBJECTS, BSTATE_RELEASED, "button_objects");
   ASSIGN(BUTTON_OBJECTS, BSTATE_PRESSED, "button_objects_press");
   ASSIGN(BUTTON_UNDO, BSTATE_RELEASED, "button_undo");
   ASSIGN(BUTTON_UNDO, BSTATE_PRESSED, "button_undo_press");
   ASSIGN(BUTTON_UNDO, BSTATE_DISABLED, "button_undo_off");
   ASSIGN(BUTTON_REDO, BSTATE_RELEASED, "button_redo");
   ASSIGN(BUTTON_REDO, BSTATE_PRESSED, "button_redo_press");
   ASSIGN(BUTTON_REDO, BSTATE_DISABLED, "button_redo_off");
   ASSIGN(BUTTON_HELP, BSTATE_RELEASED, "button_help");
   ASSIGN(BUTTON_HELP, BSTATE_PRESSED, "button_help_press");
   ASSIGN(BUTTON_QUIT, BSTATE_RELEASED, "button_quit");
   ASSIGN(BUTTON_QUIT, BSTATE_PRESSED, "button_quit_press");

   // Assign the sprites to use for the scrollbar
   ASSIGN(BUTTON_UP, BSTATE_RELEASED, "scroll_up");
   ASSIGN(BUTTON_UP, BSTATE_PRESSED, "scroll_up_press");
   ASSIGN(BUTTON_UP, BSTATE_DISABLED, "scroll_up_off");
   ASSIGN(BUTTON_DOWN, BSTATE_RELEASED, "scroll_down");
   ASSIGN(BUTTON_DOWN, BSTATE_PRESSED, "scroll_down_press");
   ASSIGN(BUTTON_DOWN, BSTATE_DISABLED, "scroll_down_off");
   ASSIGN(BUTTON_LEFT, BSTATE_RELEASED, "scroll_left");
   ASSIGN(BUTTON_LEFT, BSTATE_PRESSED, "scroll_left_press");
   ASSIGN(BUTTON_LEFT, BSTATE_DISABLED, "scroll_left_off");
   ASSIGN(BUTTON_RIGHT, BSTATE_RELEASED, "scroll_right");
   ASSIGN(BUTTON_RIGHT, BSTATE_PRESSED, "scroll_right_press");
   ASSIGN(BUTTON_RIGHT, BSTATE_DISABLED, "scroll_right_off");
   ASSIGN(BUTTON_MARKER_X, BSTATE_RELEASED, "scroll_marker_x");
   ASSIGN(BUTTON_MARKER_X, BSTATE_PRESSED, "scroll_marker_x_press");
   ASSIGN(BUTTON_MARKER_Y, BSTATE_RELEASED, "scroll_marker_y");
   ASSIGN(BUTTON_MARKER_Y, BSTATE_PRESSED, "scroll_marker_y_press");

   // Buttons used in the level info dialog
   ASSIGN(IBUTTON_APPLY, BSTATE_RELEASED, "dialog_ok");
   ASSIGN(IBUTTON_APPLY, BSTATE_PRESSED, "dialog_ok_press");
   ASSIGN(IBUTTON_CANCEL, BSTATE_RELEASED, "dialog_cancel");
   ASSIGN(IBUTTON_CANCEL, BSTATE_PRESSED, "dialog_cancel_press");
   ASSIGN(IBUTTON_INC_U, BSTATE_RELEASED, "ibutton_inc");
   ASSIGN(IBUTTON_INC_U, BSTATE_PRESSED, "ibutton_inc_press");
   ASSIGN(IBUTTON_INC_U, BSTATE_DISABLED, "ibutton_inc_off");
   ASSIGN(IBUTTON_DEC_U, BSTATE_RELEASED, "ibutton_dec");
   ASSIGN(IBUTTON_DEC_U, BSTATE_PRESSED, "ibutton_dec_press");
   ASSIGN(IBUTTON_DEC_U, BSTATE_DISABLED, "ibutton_dec_off");
   ASSIGN(IBUTTON_INC_D, BSTATE_RELEASED, "ibutton_inc");
   ASSIGN(IBUTTON_INC_D, BSTATE_PRESSED, "ibutton_inc_press");
   ASSIGN(IBUTTON_INC_D, BSTATE_DISABLED, "ibutton_inc_off");
   ASSIGN(IBUTTON_DEC_D, BSTATE_RELEASED, "ibutton_dec");
   ASSIGN(IBUTTON_DEC_D, BSTATE_PRESSED, "ibutton_dec_press");
   ASSIGN(IBUTTON_DEC_D, BSTATE_DISABLED, "ibutton_dec_off");
   ASSIGN(IBUTTON_INC_L, BSTATE_RELEASED, "ibutton_inc");
   ASSIGN(IBUTTON_INC_L, BSTATE_PRESSED, "ibutton_inc_press");
   ASSIGN(IBUTTON_INC_L, BSTATE_DISABLED, "ibutton_inc_off");
   ASSIGN(IBUTTON_DEC_L, BSTATE_RELEASED, "ibutton_dec");
   ASSIGN(IBUTTON_DEC_L, BSTATE_PRESSED, "ibutton_dec_press");
   ASSIGN(IBUTTON_DEC_L, BSTATE_DISABLED, "ibutton_dec_off");
   ASSIGN(IBUTTON_INC_R, BSTATE_RELEASED, "ibutton_inc");
   ASSIGN(IBUTTON_INC_R, BSTATE_PRESSED, "ibutton_inc_press");
   ASSIGN(IBUTTON_INC_R, BSTATE_DISABLED, "ibutton_inc_off");
   ASSIGN(IBUTTON_DEC_R, BSTATE_RELEASED, "ibutton_dec");
   ASSIGN(IBUTTON_DEC_R, BSTATE_PRESSED, "ibutton_dec_press");
   ASSIGN(IBUTTON_DEC_R, BSTATE_DISABLED, "ibutton_dec_off");

   // Buttons used in the file select dialogs
   ASSIGN(FBUTTON_OK, BSTATE_RELEASED, "dialog_ok");
   ASSIGN(FBUTTON_OK, BSTATE_PRESSED, "dialog_ok_press");
   ASSIGN(FBUTTON_CANCEL, BSTATE_RELEASED, "dialog_cancel");
   ASSIGN(FBUTTON_CANCEL, BSTATE_PRESSED, "dialog_cancel_press");
   ASSIGN(FBUTTON_LISTUP, BSTATE_RELEASED, "scroll_up");
   ASSIGN(FBUTTON_LISTUP, BSTATE_PRESSED, "scroll_up_press");
   ASSIGN(FBUTTON_LISTUP, BSTATE_DISABLED, "scroll_up_off");
   ASSIGN(FBUTTON_LISTDOWN, BSTATE_RELEASED, "scroll_down");
   ASSIGN(FBUTTON_LISTDOWN, BSTATE_PRESSED, "scroll_down_press");
   ASSIGN(FBUTTON_LISTDOWN, BSTATE_DISABLED, "scroll_down_off");
   ASSIGN(FBUTTON_LISTMARKER, BSTATE_RELEASED, "scroll_marker_y");
   ASSIGN(FBUTTON_LISTMARKER, BSTATE_PRESSED, "scroll_marker_y_press");
   ASSIGN(FBUTTON_MEDIAUP, BSTATE_RELEASED, "scroll_up");
   ASSIGN(FBUTTON_MEDIAUP, BSTATE_PRESSED, "scroll_up_press");
   ASSIGN(FBUTTON_MEDIAUP, BSTATE_DISABLED, "scroll_up_off");
   ASSIGN(FBUTTON_MEDIADOWN, BSTATE_RELEASED, "scroll_down");
   ASSIGN(FBUTTON_MEDIADOWN, BSTATE_PRESSED, "scroll_down_press");
   ASSIGN(FBUTTON_MEDIADOWN, BSTATE_DISABLED, "scroll_down_off");
   ASSIGN(FBUTTON_MEDIAMARKER, BSTATE_RELEASED, "scroll_marker_y");
   ASSIGN(FBUTTON_MEDIAMARKER, BSTATE_PRESSED, "scroll_marker_y_press");
   ASSIGN(FBUTTON_DIRUP, BSTATE_RELEASED, "fbutton_up");
   ASSIGN(FBUTTON_DIRUP, BSTATE_PRESSED, "fbutton_up_press");
   ASSIGN(FBUTTON_DIRUP, BSTATE_DISABLED, "fbutton_up_off");

   // Buttons used in the help menu
   ASSIGN(HBUTTON_PREV, BSTATE_RELEASED, "scroll_left");
   ASSIGN(HBUTTON_PREV, BSTATE_PRESSED, "scroll_left_press");
   ASSIGN(HBUTTON_PREV, BSTATE_DISABLED, "scroll_left_off");
   ASSIGN(HBUTTON_NEXT, BSTATE_RELEASED, "scroll_right");
   ASSIGN(HBUTTON_NEXT, BSTATE_PRESSED, "scroll_right_press");
   ASSIGN(HBUTTON_NEXT, BSTATE_DISABLED, "scroll_right_off");
   ASSIGN(HBUTTON_BACK, BSTATE_RELEASED, "button_quit");
   ASSIGN(HBUTTON_BACK, BSTATE_PRESSED, "button_quit_press");

   // Buttons used in the object list
   ASSIGN(OBUTTON_ITEM, BSTATE_RELEASED, "objtype");
   ASSIGN(OBUTTON_ITEM, BSTATE_PRESSED, "objtype_press");
   ASSIGN(OBUTTON_SCENERY, BSTATE_RELEASED, "objtype");
   ASSIGN(OBUTTON_SCENERY, BSTATE_PRESSED, "objtype_press");
   ASSIGN(OBUTTON_ENEMY, BSTATE_RELEASED, "objtype");
   ASSIGN(OBUTTON_ENEMY, BSTATE_PRESSED, "objtype_press");
   ASSIGN(OBUTTON_HAZARD, BSTATE_RELEASED, "objtype");
   ASSIGN(OBUTTON_HAZARD, BSTATE_PRESSED, "objtype_press");
   ASSIGN(OBUTTON_SWITCH, BSTATE_RELEASED, "objtype");
   ASSIGN(OBUTTON_SWITCH, BSTATE_PRESSED, "objtype_press");
#undef ASSIGN

   // Associate buttons with functions
   // Buttons without a function associated are handled differently
   buttons[BUTTON_NEW].func = button_new;
   buttons[BUTTON_INFO].func = button_info;
   buttons[BUTTON_LOAD].func = button_load;
   buttons[BUTTON_SAVE].func = button_save;
   buttons[BUTTON_PLAY].func = button_play;
   buttons[BUTTON_TILEMAP].func = button_tilemap;
   buttons[BUTTON_OBJECTS].func = button_objects;
   buttons[BUTTON_TILE].func = button_tile;
   buttons[BUTTON_UNDO].func = button_undo;
   buttons[BUTTON_REDO].func = button_redo;
   buttons[BUTTON_HELP].func = button_help;
   buttons[BUTTON_QUIT].func = button_quit;
   buttons[BUTTON_UP].func = button_scroll_up;
   buttons[BUTTON_DOWN].func = button_scroll_down;
   buttons[BUTTON_LEFT].func = button_scroll_left;
   buttons[BUTTON_RIGHT].func = button_scroll_right;
   buttons[BUTTON_SCROLL_XL].func = button_scroll_left_fast;
   buttons[BUTTON_SCROLL_XR].func = button_scroll_right_fast;
   buttons[BUTTON_SCROLL_YU].func = button_scroll_up_fast;
   buttons[BUTTON_SCROLL_YD].func = button_scroll_down_fast;

   buttons[IBUTTON_APPLY].func = ibutton_apply;
   buttons[IBUTTON_CANCEL].func = ibutton_cancel;
   buttons[IBUTTON_INC_U].func = ibutton_inc_up;
   buttons[IBUTTON_DEC_U].func = ibutton_dec_up;
   buttons[IBUTTON_INC_D].func = ibutton_inc_down;
   buttons[IBUTTON_DEC_D].func = ibutton_dec_down;
   buttons[IBUTTON_INC_L].func = ibutton_inc_left;
   buttons[IBUTTON_DEC_L].func = ibutton_dec_left;
   buttons[IBUTTON_INC_R].func = ibutton_inc_right;
   buttons[IBUTTON_DEC_R].func = ibutton_dec_right;

   buttons[FBUTTON_OK].func = fbutton_ok;
   buttons[FBUTTON_CANCEL].func = fbutton_cancel;
   buttons[FBUTTON_LISTUP].func = fbutton_list_up;
   buttons[FBUTTON_LISTDOWN].func = fbutton_list_down;
   buttons[FBUTTON_MEDIAUP].func = fbutton_media_up;
   buttons[FBUTTON_MEDIADOWN].func = fbutton_media_down;
   buttons[FBUTTON_SCROLL_LU].func = fbutton_list_up_fast;
   buttons[FBUTTON_SCROLL_LD].func = fbutton_list_down_fast;
   buttons[FBUTTON_SCROLL_MU].func = fbutton_media_up_fast;
   buttons[FBUTTON_SCROLL_MD].func = fbutton_media_down_fast;
   buttons[FBUTTON_DIRUP].func = fbutton_dir_up;

   buttons[HBUTTON_PREV].func = hbutton_previous_page;
   buttons[HBUTTON_NEXT].func = hbutton_next_page;
   buttons[HBUTTON_BACK].func = hbutton_back;

   buttons[OBUTTON_ITEM].func = obutton_items;
   buttons[OBUTTON_SCENERY].func = obutton_scenery;
   buttons[OBUTTON_ENEMY].func = obutton_enemies;
   buttons[OBUTTON_HAZARD].func = obutton_hazards;
   buttons[OBUTTON_SWITCH].func = obutton_switches;

   // Set button names
   // Buttons without names don't have a caption to show
   buttons[BUTTON_NEW].name = text.editor.b_new;
   buttons[BUTTON_INFO].name = text.editor.b_info;
   buttons[BUTTON_LOAD].name = text.editor.b_load;
   buttons[BUTTON_SAVE].name = text.editor.b_save;
   buttons[BUTTON_PLAY].name = text.editor.b_play;
   buttons[BUTTON_TILEMAP].name = text.editor.b_tilemap;
   buttons[BUTTON_OBJECTS].name = text.editor.b_objects;
   buttons[BUTTON_TILE].name = text.editor.b_select;
   buttons[BUTTON_UNDO].name = text.editor.b_undo;
   buttons[BUTTON_REDO].name = text.editor.b_redo;
   buttons[BUTTON_HELP].name = text.editor.b_help;
   buttons[BUTTON_QUIT].name = text.editor.b_quit;

   buttons[IBUTTON_APPLY].name = text.level_info.b_apply;
   buttons[IBUTTON_CANCEL].name = text.level_info.b_cancel;

   buttons[FBUTTON_CANCEL].name = text.file_select.b_cancel;

   // Set label text
   labels[ILABEL_THEME1].caption = text.level_info.themes[0];
   labels[ILABEL_THEME2].caption = text.level_info.themes[1];
   labels[ILABEL_THEME3].caption = text.level_info.themes[2];
   labels[ILABEL_THEME4].caption = text.level_info.themes[3];
   labels[ILABEL_THEME5].caption = text.level_info.themes[4];
   labels[ILABEL_THEME6].caption = text.level_info.themes[5];
   labels[ILABEL_THEME7].caption = text.level_info.themes[6];
   labels[ILABEL_THEME8].caption = text.level_info.themes[7];
   labels[ILABEL_TOP].caption = text.level_info.top;
   labels[ILABEL_BOTTOM].caption = text.level_info.bottom;
   labels[ILABEL_LEFT].caption = text.level_info.left;
   labels[ILABEL_RIGHT].caption = text.level_info.right;

   // The following buttons have scrollbar semantics, i.e. their action is
   // constantly repeated while held down, rather than being a single-click
   // action
   buttons[BUTTON_UP].scroller = 1;
   buttons[BUTTON_DOWN].scroller = 1;
   buttons[BUTTON_LEFT].scroller = 1;
   buttons[BUTTON_RIGHT].scroller = 1;
   buttons[BUTTON_SCROLL_XL].scroller = 1;
   buttons[BUTTON_SCROLL_XR].scroller = 1;
   buttons[BUTTON_SCROLL_YU].scroller = 1;
   buttons[BUTTON_SCROLL_YD].scroller = 1;
   buttons[IBUTTON_INC_U].scroller = 1;
   buttons[IBUTTON_DEC_U].scroller = 1;
   buttons[IBUTTON_INC_D].scroller = 1;
   buttons[IBUTTON_DEC_D].scroller = 1;
   buttons[IBUTTON_INC_L].scroller = 1;
   buttons[IBUTTON_DEC_L].scroller = 1;
   buttons[IBUTTON_INC_R].scroller = 1;
   buttons[IBUTTON_DEC_R].scroller = 1;
   buttons[FBUTTON_LISTUP].scroller = 1;
   buttons[FBUTTON_LISTDOWN].scroller = 1;
   buttons[FBUTTON_MEDIAUP].scroller = 1;
   buttons[FBUTTON_MEDIADOWN].scroller = 1;
   buttons[FBUTTON_SCROLL_LU].scroller = 1;
   buttons[FBUTTON_SCROLL_LD].scroller = 1;
   buttons[FBUTTON_SCROLL_MU].scroller = 1;
   buttons[FBUTTON_SCROLL_MD].scroller = 1;

   // Reset message box
   messagebox.visible = 0;
   messagebox.message = NULL;

   // Initialize the main GUI
   init_editor_gui();

   // Reset help page to the first one
   // Not using constants because we mean the first page, not any specific
   // page by content
   help_page = 0;

   // Make screen visible
   fade_on();
}

//***************************************************************************
// load_editor [internal]
// Loads the editor assets. Run during the loading screen.
//***************************************************************************

static void load_editor(void) {
   // Main GUI graphics
   gfxset_editor = load_graphics_set("graphics/editor_gui");
   set_loading_total(1, 3);

   // Graphics for the objects
   gfxset_objs[OBJSIZE_NORMAL] =
      load_graphics_set("graphics/editor_obj_normal");
   set_loading_total(2, 3);
   gfxset_objs[OBJSIZE_PREVIEW] =
      load_graphics_set("graphics/editor_obj_preview");
   set_loading_total(3, 3);
}

//***************************************************************************
// run_editor
// Processes a logic frame in the level editor.
//***************************************************************************

void run_editor(void) {
   // One-switch controls? (this is crazy...)
   if (settings.one_switch) {
      // Always fake the cursor being inside
      input.cursor.inside = 1;

      // Fake mouse motion by holding down the button
      if (input.oneswitch.hold && !(input.oneswitch.dir & 1))
         input.oneswitch.dir++;
      if (!input.oneswitch.hold && (input.oneswitch.dir & 1))
         input.oneswitch.dir++;

      // Execute fake mouse motion if needed
      switch (input.oneswitch.dir) {
         // Moving left
         case 1:
            input.cursor.x += 2;
            if (input.cursor.x > screen_w - 1)
               input.cursor.x = screen_w - 1;
            break;

         // Moving down
         case 3:
            input.cursor.y += 2;
            if (input.cursor.y > screen_h - 1)
               input.cursor.y = screen_h - 1;
            break;

         // Moving left
         case 5:
            input.cursor.x -= 2;
            if (input.cursor.x < 0)
               input.cursor.x = 0;
            break;

         // Moving up
         case 7:
            input.cursor.y -= 2;
            if (input.cursor.y < 0)
               input.cursor.y = 0;
            break;

         // Not moving
         default:
            break;
      }

      // Moving automatically kills scroller buttons
      if ((input.oneswitch.dir & 1) && curr_button != NULL &&
      curr_button->scroller) {
         curr_button->state = BSTATE_RELEASED;
         curr_button = NULL;
      }

      // Trigger buttons when tapping
      if (input.oneswitch.tap && over_button != NULL &&
      over_button->state != BSTATE_DISABLED) {
         // Scroll buttons should be held down
         if (over_button->scroller) {
            over_button->state = (over_button->state == BSTATE_RELEASED) ?
               BSTATE_PRESSED : BSTATE_RELEASED;
            curr_button = (over_button->state == BSTATE_PRESSED) ?
               over_button : NULL;
         }

         // Most buttons should trigger as usual
         else {
            over_button->func();
         }
      }

      // While we're at it, let's just fake cursor clicks
      // (non-button controls need this)
      if (over_button == NULL)
         input.cursor.click = input.oneswitch.tap;
   }

   // If showing the message box then let it take over
   if (messagebox.visible) {
      run_messagebox();
      return;
   }

   // Run the logic for whatever screen we're on
   switch (editor_screen) {
      case EDSCRN_MAIN: run_editor_gui(); break;
      case EDSCRN_COLL: run_tile_select(); break;
      case EDSCRN_OBJ: run_tile_select(); break;
      case EDSCRN_INFO: run_level_info(); break;
      case EDSCRN_LOAD: run_file_selector(); break;
      case EDSCRN_SAVE: run_file_selector(); break;
      case EDSCRN_HELP: run_editor_help(); break;
      default: break;
   }
}

//***************************************************************************
// run_editor_gui [internal]
// Processes the logic for the main editor GUI.
//***************************************************************************

static void run_editor_gui(void) {
   // Calculate how many tiles can we fit on screen
   // We add some margin for the extra UI elements
   region_width = (screen_w - 96) >> 5;
   region_height = (screen_h - 16) >> 5;

   // Draw in the map when tapping in it in one-switch mode
   if (settings.one_switch && input.oneswitch.tap &&
   input.cursor.x < (region_width << 5) &&
   input.cursor.y < (region_height << 5)) {
      // Determine on which tile it's placed
      int x = (input.cursor.x >> 5) + region_x;
      int y = (input.cursor.y >> 5) + region_y;

      // Tilemap mode?
      if (editor_mode == EDMODE_TILEMAP) {
         int clear = map.data[x][y].coll == coll_type;

         input.editor.place = clear ? 0 : 1;
         input.editor.remove = clear ? 1 : 0;
      }

      // Object mode?
      else {
         int same = map.data[x][y].obj == obj_table[obj_type];
         int flipped = map.data[x][y].flags & MAPFLAG_FLIP;

         input.editor.place = !same ? 1 : 0;
         input.editor.flip = same && !flipped ? 1 : 0;
         input.editor.remove = same && flipped ? 1 : 0;
      }
   }

   // Is the cursor inside the editing area?
   // Also make sure we aren't pressing any GUI buttons... (conflicts with
   // holding down the main mouse button - yes, this can happen if you click
   // the button then drag outside without releasing, it's extremely likely
   // to happen with the scrollbars actually)
   if (input.cursor.inside &&
   input.cursor.x < (region_width << 5) &&
   input.cursor.y < (region_height << 5) &&
   curr_button == NULL) {
      // Determine on which tile it's placed
      int x = (input.cursor.x >> 5) + region_x;
      int y = (input.cursor.y >> 5) + region_y;

      // Flipping and placing share the same button
      // Use this flag to know when it's OK to flip
      int flip_ok = 0;

      // Place tile?
      if (input.editor.place) {
         if (editor_mode == EDMODE_TILEMAP) {
            if (map.data[x][y].coll != coll_type) {
               UndoStack *ptr = push_undo();
               ptr->type = UNDO_TILEMAP;
               ptr->old_value = map.data[x][y].coll;
               ptr->new_value = coll_type;
               ptr->x = x;
               ptr->y = y;
               map.data[x][y].coll = coll_type;
            }
         } else {
            if (map.data[x][y].obj != obj_table[obj_type]) {
               UndoStack *ptr = push_undo();
               ptr->type = UNDO_OBJECT;
               ptr->old_value = map.data[x][y].obj;
               ptr->new_value = obj_table[obj_type];
               ptr->x = x;
               ptr->y = y;
               map.data[x][y].obj = obj_table[obj_type];
               map.data[x][y].flags &= ~MAPFLAG_FLIP;
            } else {
               flip_ok = 1;
            }
         }
      }

      // Clear tile?
      else if (input.editor.remove) {
         if (editor_mode == EDMODE_TILEMAP) {
            if (map.data[x][y].coll != TILE_EMPTY) {
               UndoStack *ptr = push_undo();
               ptr->type = UNDO_TILEMAP;
               ptr->old_value = map.data[x][y].coll;
               ptr->new_value = TILE_EMPTY;
               ptr->x = x;
               ptr->y = y;
               map.data[x][y].coll = TILE_EMPTY;
            }
         } else {
            if (map.data[x][y].obj != LVOBJ_NONE) {
               UndoStack *ptr = push_undo();
               ptr->type = UNDO_OBJECT;
               ptr->old_value = map.data[x][y].obj;
               ptr->new_value = LVOBJ_NONE;
               ptr->x = x;
               ptr->y = y;
               map.data[x][y].obj = LVOBJ_NONE;
            }
         }
      }

      // Pick type?
      else if (input.editor.pick) {
         if (editor_mode == EDMODE_TILEMAP)
            coll_type = map.data[x][y].coll;
         else {
            for (unsigned i = 0; i < NUM_LVOBJIDS; i++) {
               if (obj_table[i] == map.data[x][y].obj)
                  obj_type = i;
            }
         }
      }

      // Set spawn point?
      else if (input.editor.spawn) {
         map.spawn_x = x;
         map.spawn_y = y;
      }

      // Flip object?
      if (input.editor.flip && flip_ok) {
         uint8_t old_flags = map.data[x][y].flags;
         map.data[x][y].flags ^= MAPFLAG_FLIP;

         // Add action to undo stack
         if (map.data[x][y].obj != LVOBJ_NONE) {
            UndoStack *ptr = push_undo();
            ptr->type = UNDO_FLAGS;
            ptr->old_value = old_flags;
            ptr->new_value = map.data[x][y].flags;
            ptr->x = x;
            ptr->y = y;
         }
      }
   }

   // Calculate coordinates of the scrolling buttons
   buttons[BUTTON_UP].x1 = region_width << 5;
   buttons[BUTTON_UP].y1 = 0;
   buttons[BUTTON_UP].x2 = buttons[BUTTON_UP].x1 + 0x0F;
   buttons[BUTTON_UP].y2 = buttons[BUTTON_UP].y1 + 0x1F;

   buttons[BUTTON_DOWN].x1 = region_width << 5;
   buttons[BUTTON_DOWN].y1 = (region_height << 5) - 0x20;
   buttons[BUTTON_DOWN].x2 = buttons[BUTTON_DOWN].x1 + 0x0F;
   buttons[BUTTON_DOWN].y2 = buttons[BUTTON_DOWN].y1 + 0x1F;

   buttons[BUTTON_LEFT].x1 = 0;
   buttons[BUTTON_LEFT].y1 = region_height << 5;
   buttons[BUTTON_LEFT].x2 = buttons[BUTTON_LEFT].x1 + 0x1F;
   buttons[BUTTON_LEFT].y2 = buttons[BUTTON_LEFT].y1 + 0x0F;

   buttons[BUTTON_RIGHT].x1 = (region_width << 5) - 0x20;
   buttons[BUTTON_RIGHT].y1 = region_height << 5;
   buttons[BUTTON_RIGHT].x2 = buttons[BUTTON_RIGHT].x1 + 0x1F;
   buttons[BUTTON_RIGHT].y2 = buttons[BUTTON_RIGHT].y1 + 0x0F;

   // Calculate coordinates of the scroll markers
   buttons[BUTTON_MARKER_X].y1 = buttons[BUTTON_LEFT].y1;
   buttons[BUTTON_MARKER_X].y2 = buttons[BUTTON_LEFT].y2;
   buttons[BUTTON_MARKER_Y].x1 = buttons[BUTTON_UP].x1;
   buttons[BUTTON_MARKER_Y].x2 = buttons[BUTTON_UP].x2;

   // Update the sensors for the horizontal scrollbar background
   buttons[BUTTON_SCROLL_XL].x1 = buttons[BUTTON_LEFT].x2 + 1;
   buttons[BUTTON_SCROLL_XL].x2 = buttons[BUTTON_MARKER_X].x1 - 1;
   buttons[BUTTON_SCROLL_XL].y1 = buttons[BUTTON_LEFT].y1 + 4;
   buttons[BUTTON_SCROLL_XL].y2 = buttons[BUTTON_LEFT].y2 - 4;
   buttons[BUTTON_SCROLL_XR].x1 = buttons[BUTTON_MARKER_X].x2 + 1;
   buttons[BUTTON_SCROLL_XR].x2 = buttons[BUTTON_RIGHT].x1 - 1;
   buttons[BUTTON_SCROLL_XR].y1 = buttons[BUTTON_RIGHT].y1 + 4;
   buttons[BUTTON_SCROLL_XR].y2 = buttons[BUTTON_RIGHT].y2 - 4;

   // Update the sensors for the vertical scrollbar background
   buttons[BUTTON_SCROLL_YU].x1 = buttons[BUTTON_UP].x1 + 4;
   buttons[BUTTON_SCROLL_YU].x2 = buttons[BUTTON_UP].x2 - 4;
   buttons[BUTTON_SCROLL_YU].y1 = buttons[BUTTON_UP].y2 + 1;
   buttons[BUTTON_SCROLL_YU].y2 = buttons[BUTTON_MARKER_Y].y1 - 1;
   buttons[BUTTON_SCROLL_YD].x1 = buttons[BUTTON_DOWN].x1 + 4;
   buttons[BUTTON_SCROLL_YD].x2 = buttons[BUTTON_DOWN].x2 - 4;
   buttons[BUTTON_SCROLL_YD].y1 = buttons[BUTTON_MARKER_Y].y2 + 1;
   buttons[BUTTON_SCROLL_YD].y2 = buttons[BUTTON_DOWN].y1 - 1;

   // Disable scrolling buttons if we're at the relevant extreme of the
   // tilemap (indicate we can't scroll in that direction anymore)
   set_disabled(BUTTON_LEFT, region_x == 0);
   set_disabled(BUTTON_UP, region_y == 0);
   set_disabled(BUTTON_RIGHT, region_x >= map.width - region_width);
   set_disabled(BUTTON_DOWN, region_y >= map.height - region_height);

   // Set the state of the undo/redo buttons depending if the relevant
   // action can be performed or not.
   buttons[BUTTON_UNDO].state = (undo_count == 0) ?
      BSTATE_DISABLED : BSTATE_RELEASED;
   buttons[BUTTON_REDO].state = (redo_count == 0) ?
      BSTATE_DISABLED : BSTATE_RELEASED;

   // Process all buttons
   process_buttons();

   // Override the state of the mode buttons to reflect the current mode
   if (editor_mode == EDMODE_TILEMAP)
      buttons[BUTTON_TILEMAP].state = BSTATE_PRESSED;
   if (editor_mode == EDMODE_OBJECTS)
      buttons[BUTTON_OBJECTS].state = BSTATE_PRESSED;

   // Mouse wheel up selects the next tile type
   if (input.cursor.wheel_up || input.editor.next) {
      if (editor_mode == EDMODE_TILEMAP) {
         coll_type++;
         if (coll_type == NUM_TILETYPES)
            coll_type = 0;
      } else {
         obj_type++;
         if (obj_type == NUM_LVOBJIDS-1)
            obj_type = 0;
      }
   }

   // Mouse wheel down selects the previous tile type
   if (input.cursor.wheel_down || input.editor.prev) {
      if (editor_mode == EDMODE_TILEMAP) {
         if (coll_type == 0)
            coll_type = NUM_TILETYPES;
         coll_type--;
      } else {
         if (obj_type == 0)
            obj_type = NUM_LVOBJIDS-1;
         obj_type--;
      }
   }

   // Remove redundancies
   if (editor_mode == EDMODE_TILEMAP)
      input.editor.b_tilemap = 0;
   if (editor_mode == EDMODE_OBJECTS)
      input.editor.b_objects = 0;

   // Key shortcuts
   if (input.editor.b_new)
      button_new();
   if (input.editor.b_info)
      button_info();
   if (input.editor.b_load)
      button_load();
   if (input.editor.b_save)
      button_save();
   if (input.editor.b_play)
      button_play();
   if (input.editor.b_tilemap)
      button_tilemap();
   if (input.editor.b_objects)
      button_objects();
   if (input.editor.b_undo && buttons[BUTTON_UNDO].state == BSTATE_RELEASED)
      button_undo();
   if (input.editor.b_redo && buttons[BUTTON_REDO].state == BSTATE_RELEASED)
      button_redo();
   if (input.editor.b_help)
      button_help();
   if (input.editor.b_quit)
      button_quit();

   // Use the arrows to move around
   if (scroll_timer == 0) {
      int move_speed = input.editor.fast ? FAST_SCROLL : 1;
      if (input.editor.up)
         for (int i = 0; i < move_speed; i++) move_up();
      if (input.editor.down)
         for (int i = 0; i < move_speed; i++) move_down();
      if (input.editor.left)
         for (int i = 0; i < move_speed; i++) move_left();
      if (input.editor.right)
         for (int i = 0; i < move_speed; i++) move_right();
   }

   // Make sure the scroll coordinates aren't outbounds
   if (region_x > map.width - region_width)
      region_x = map.width - region_width;
   if (region_y > map.height - region_height)
      region_y = map.height - region_height;

   // Calculate position of the horizontal scroll marker
   int range = (buttons[BUTTON_RIGHT].x1 - 0x18) -
      (buttons[BUTTON_LEFT].x2 + 1);
   buttons[BUTTON_MARKER_X].x1 = (buttons[BUTTON_LEFT].x2 + 1) +
      range * region_x / (map.width - region_width);
   buttons[BUTTON_MARKER_X].x2 = buttons[BUTTON_MARKER_X].x1 + 0x17;

   // Calculate position of the vertical scroll marker
   range = (buttons[BUTTON_DOWN].y1 - 0x18) -
      (buttons[BUTTON_UP].y2 + 1);
   buttons[BUTTON_MARKER_Y].y1 = (buttons[BUTTON_UP].y2 + 1) +
      range * region_y / (map.height - region_height);
   buttons[BUTTON_MARKER_Y].y2 = buttons[BUTTON_MARKER_Y].y1 + 0x17;
}

//***************************************************************************
// run_tile_select [internal]
// Processes the logic for the dialogs for selecting the tile type.
//***************************************************************************

static void run_tile_select(void) {
   // Return to the editor when pressing Esc
   if (input.menu.cancel) {
      init_editor_gui();
      editor_screen = EDSCRN_MAIN;
   }

   // Handle the GUI
   process_buttons();
   process_choices();
}

//***************************************************************************
// run_level_info [internal]
// Processes the logic for the level info dialog.
//***************************************************************************

static void run_level_info(void) {
   // Update buttons
   process_buttons();

   // Clicked on a theme thumbnail? Switch to that theme then
   // Note that only virtual and park are available in the demo version
   if (input.cursor.click &&
   input.cursor.x >= INFO_THEME_XTHUMB &&
   input.cursor.x <= INFO_THEME_XTHUMB2 &&
   input.cursor.y >= INFO_THEME_YTHUMB &&
   input.cursor.y <= INFO_THEME_YTHUMB2 &&
   (input.cursor.x - INFO_THEME_XTHUMB) % 0x24 < 0x20) {
      Theme theme = (input.cursor.x - INFO_THEME_XTHUMB) / 0x24;
      if (!settings.demo || theme <= THEME_PARK)
         info_temp.theme = theme;
   }

   // Enable/disable resizing buttons depending if it's possible to resize
   // in their direction or not
   set_disabled(IBUTTON_INC_U, info_temp.new_height >= MAX_MAP_HEIGHT);
   set_disabled(IBUTTON_DEC_U, info_temp.new_height <= MIN_MAP_HEIGHT);
   set_disabled(IBUTTON_INC_D, info_temp.new_height >= MAX_MAP_HEIGHT);
   set_disabled(IBUTTON_DEC_D, info_temp.new_height <= MIN_MAP_HEIGHT);
   set_disabled(IBUTTON_INC_L, info_temp.new_width >= MAX_MAP_WIDTH);
   set_disabled(IBUTTON_DEC_L, info_temp.new_width <= MIN_MAP_WIDTH);
   set_disabled(IBUTTON_INC_R, info_temp.new_width >= MAX_MAP_WIDTH);
   set_disabled(IBUTTON_DEC_R, info_temp.new_width <= MIN_MAP_WIDTH);
}

//***************************************************************************
// run_file_selector [internal]
// Processes the logic for the file select dialogs.
//***************************************************************************

static void run_file_selector(void) {
   // Update buttons
   process_buttons();

   // Quit by pressing Esc
   if (input.menu.cancel)
      fbutton_cancel();

   // Select next field?
   if (input.editor.ui_next) {
      switch (file_focus) {
         case FFOCUS_NAME: file_focus = FFOCUS_LIST; break;
         case FFOCUS_LIST: file_focus = FFOCUS_MEDIA; break;
         default: file_focus = FFOCUS_NAME; break;
      }
   }

   // Select previous field?
   if (input.editor.ui_prev) {
      switch (file_focus) {
         case FFOCUS_NAME: file_focus = FFOCUS_MEDIA; break;
         case FFOCUS_MEDIA: file_focus = FFOCUS_LIST; break;
         default: file_focus = FFOCUS_NAME; break;
      }
   }

   // Ensure *something* is selected
   if (input.editor.ui_next || input.editor.ui_prev) {
      switch (file_focus) {
         case FFOCUS_LIST:
            if (file_list.selected >= file_list.count) {
               file_list.selected = 0;
               file_list.pos = 0;
            }
            break;
         case FFOCUS_MEDIA:
            if (media_list.selected >= media_list.count) {
               media_list.selected = 0;
               media_list.pos = 0;
            }
            break;
         default:
            break;
      }
   }

   // Select next entry?
   if (input.editor.ui_down || input.editor.ui_xdown) {
      unsigned count = input.editor.ui_xdown ? 10 : 1;
      for (unsigned i = 0; i < count; i++) switch (file_focus) {
         case FFOCUS_LIST:
            if (file_list.selected < file_list.count - 1)
               file_list.selected++;
            break;
         case FFOCUS_MEDIA:
            if (media_list.selected < media_list.count - 1)
               media_list.selected++;
            break;
         default:
            break;
      }
   }

   // Select previous entry?
   if (input.editor.ui_up || input.editor.ui_xup) {
      unsigned count = input.editor.ui_xup ? 10 : 1;
      for (unsigned i = 0; i < count; i++) switch (file_focus) {
         case FFOCUS_LIST:
            if (file_list.selected > 0)
               file_list.selected--;
            break;
         case FFOCUS_MEDIA:
            if (media_list.selected > 0)
               media_list.selected--;
            break;
         default:
            break;
      }
   }

   // Last entry?
   if (input.editor.ui_end) {
      switch (file_focus) {
         case FFOCUS_LIST:
            file_list.selected = file_list.count - 1;
            break;
         case FFOCUS_MEDIA:
            media_list.selected = media_list.count - 1;
            break;
         default:
            break;
      }
   }

   // First entry?
   if (input.editor.ui_home) {
      switch (file_focus) {
         case FFOCUS_LIST:
            file_list.selected = 0;
            break;
         case FFOCUS_MEDIA:
            media_list.selected = 0;
            break;
         default:
            break;
      }
   }

   // Prevent selection from going off-scroll!
   if (input.editor.ui_down || input.editor.ui_up ||
   input.editor.ui_xdown || input.editor.ui_xup ||
   input.editor.ui_end || input.editor.ui_home) {
      switch (file_focus) {
         case FFOCUS_LIST:
            if (file_list.selected < file_list.pos)
               file_list.pos = file_list.selected;
            if (file_list.selected >= file_list.pos + file_list.size)
               file_list.pos = file_list.selected - file_list.size + 1;
            break;
         case FFOCUS_MEDIA:
            if (media_list.selected < media_list.pos)
               media_list.pos = media_list.selected;
            if (media_list.selected >= media_list.pos + media_list.size)
               media_list.pos = media_list.selected - media_list.size + 1;
            break;
         default:
            break;
      }
   }

   // Scrolling up using the mouse wheel?
   if (input.cursor.wheel_up) {
      // Scroll file list?
      if (input.cursor.x >= FILE_LIST_X1 &&
      input.cursor.x <= FILE_LIST_X2 &&
      input.cursor.y >= FILE_LIST_Y1 &&
      input.cursor.y <= FILE_LIST_Y1 + (signed) file_list.size * 0x10) {
         fbutton_list_up_fast();
      }

      // Scroll media list?
      else if (input.cursor.x >= FILE_MEDIA_X1 &&
      input.cursor.x <= FILE_MEDIA_X2 &&
      input.cursor.y >= FILE_MEDIA_Y1 &&
      input.cursor.y <= FILE_MEDIA_Y1 + (signed) media_list.size * 0x10) {
         fbutton_media_up_fast();
      }
   }

   // Scrolling down using the mouse wheel?
   if (input.cursor.wheel_down) {
      // Scroll file list?
      if (input.cursor.x >= FILE_LIST_X1 &&
      input.cursor.x <= FILE_LIST_X2 &&
      input.cursor.y >= FILE_LIST_Y1 &&
      input.cursor.y <= FILE_LIST_Y1 + (signed) file_list.size * 0x10) {
         fbutton_list_down_fast();
      }

      // Scroll media list?
      else if (input.cursor.x >= FILE_MEDIA_X1 &&
      input.cursor.x <= FILE_MEDIA_X2 &&
      input.cursor.y >= FILE_MEDIA_Y1 &&
      input.cursor.y <= FILE_MEDIA_Y1 + (signed) media_list.size * 0x10) {
         fbutton_media_down_fast();
      }
   }

   // Update file list scrollbar if it can be scrolled
   if (file_list.count > file_list.size) {
      // Make scrollbar buttons usable
      buttons[FBUTTON_LISTMARKER].x1 = buttons[FBUTTON_LISTUP].x1;
      buttons[FBUTTON_LISTMARKER].x2 = buttons[FBUTTON_LISTUP].x2;
      set_disabled(FBUTTON_LISTUP, file_list.pos <= 0);
      set_disabled(FBUTTON_LISTDOWN, file_list.pos >=
         file_list.count - file_list.size);
      set_disabled(FBUTTON_SCROLL_LU, 0);
      set_disabled(FBUTTON_SCROLL_LD, 0);

      // Determine position of the marker
      int range = (buttons[FBUTTON_LISTDOWN].y1 - 0x18) -
         buttons[FBUTTON_LISTUP].y2;
      buttons[FBUTTON_LISTMARKER].y1 = (buttons[FBUTTON_LISTUP].y2 + 1) +
         range * file_list.pos / (file_list.count - file_list.size);
      buttons[FBUTTON_LISTMARKER].y2 =
         buttons[FBUTTON_LISTMARKER].y1 + 0x17;

      // Determine size of the scrollbar
      buttons[FBUTTON_SCROLL_LU].y2 = buttons[FBUTTON_LISTMARKER].y1 - 1;
      buttons[FBUTTON_SCROLL_LD].y1 = buttons[FBUTTON_LISTMARKER].y2 + 1;
   }

   // File list too short for scrolling?
   else {
      // Make scrollbar buttons unusable
      buttons[FBUTTON_LISTMARKER].x1 = INT_MIN;
      buttons[FBUTTON_LISTMARKER].x2 = INT_MIN;
      set_disabled(FBUTTON_LISTUP, 1);
      set_disabled(FBUTTON_LISTDOWN, 1);
      set_disabled(FBUTTON_SCROLL_LU, 1);
      set_disabled(FBUTTON_SCROLL_LD, 1);
   }

   // Update media list scrollbar if it can be scrolled
   if (media_list.count > media_list.size) {
      // Make scrollbar buttons usable
      buttons[FBUTTON_MEDIAMARKER].x1 = buttons[FBUTTON_MEDIAUP].x1;
      buttons[FBUTTON_MEDIAMARKER].x2 = buttons[FBUTTON_MEDIAUP].x2;
      set_disabled(FBUTTON_MEDIAUP, media_list.pos <= 0);
      set_disabled(FBUTTON_MEDIADOWN, media_list.pos >=
         media_list.count - media_list.size);
      set_disabled(FBUTTON_SCROLL_LU, 0);
      set_disabled(FBUTTON_SCROLL_LD, 0);

      // Determine position of the marker
      int range = (buttons[FBUTTON_MEDIADOWN].y1 - 0x18) -
         buttons[FBUTTON_MEDIAUP].y2;
      buttons[FBUTTON_MEDIAMARKER].y1 = (buttons[FBUTTON_MEDIAUP].y2 + 1) +
         range * media_list.pos / (media_list.count - media_list.size);
      buttons[FBUTTON_MEDIAMARKER].y2 =
         buttons[FBUTTON_MEDIAMARKER].y1 + 0x17;

      // Determine size of the scrollbar
      buttons[FBUTTON_SCROLL_MU].y2 = buttons[FBUTTON_MEDIAMARKER].y1 - 1;
      buttons[FBUTTON_SCROLL_MD].y1 = buttons[FBUTTON_MEDIAMARKER].y2 + 1;
   }

   // Media list too short for scrolling?
   else {
      // Make scrollbar buttons unusable
      buttons[FBUTTON_MEDIAMARKER].x1 = INT_MIN;
      buttons[FBUTTON_MEDIAMARKER].x2 = INT_MIN;
      set_disabled(FBUTTON_MEDIAUP, 1);
      set_disabled(FBUTTON_MEDIADOWN, 1);
      set_disabled(FBUTTON_SCROLL_MU, 1);
      set_disabled(FBUTTON_SCROLL_MD, 1);
   }

   // Clicked inside one of the non-button controls?
   if (input.cursor.click && curr_button == NULL) {
      // Stop any text input
      SDL_StopTextInput();

      // Selected a file?
      if (input.cursor.x >= FILE_LIST_X1 &&
      input.cursor.x <= FILE_LIST_X2 &&
      input.cursor.y >= FILE_LIST_Y1 &&
      input.cursor.y <= FILE_LIST_Y1 + (signed) file_list.size * 0x10) {
         // Check which entry is the cursor on
         unsigned pos = file_list.pos +
            (input.cursor.y - FILE_LIST_Y1) / 0x10;

         // If this entry was already selected then proceed to trigger it
         if (pos == file_list.selected && file_focus == FFOCUS_LIST) {
            // Directory? (go there)
            if (pos < file_list.entries->num_dirs) {
               // Determine the path of the target directory
               char *path = append_path(current_path,
                  file_list.entries->dirs[pos]);

               // Go to the new path
               go_to_dir(path);
               free(path);
            }

            // File?
            else {
               // Figure which file is selected
               pos -= file_list.entries->num_dirs;

               // Accept file (as if the user had clicked "Load"/"Save")
               set_filename(file_list.entries->files[pos]);
               fbutton_ok();
            }
         }

         // Mark this entry as the current one
         else {
            file_list.selected = pos;
            if (pos < file_list.entries->num_dirs)
               set_filename(file_list.entries->dirs[pos]);
            else {
               pos -= file_list.entries->num_dirs;
               set_filename(file_list.entries->files[pos]);
            }
         }

         // Give this list focus
         file_focus = FFOCUS_LIST;
      }

      // Selected media?
      else if (input.cursor.x >= FILE_MEDIA_X1 &&
      input.cursor.x <= FILE_MEDIA_X2 &&
      input.cursor.y >= FILE_MEDIA_Y1 &&
      input.cursor.y <= FILE_MEDIA_Y1 + (signed) media_list.size * 0x10) {
         // Check which entry is the cursor on
         unsigned pos = media_list.pos +
            (input.cursor.y - FILE_MEDIA_Y1) / 0x10;

         // If this entry was already selected then proceed to go there
         if (pos == media_list.selected && file_focus == FFOCUS_MEDIA) {
            if (pos == 0) {
               char *dir = get_default_dir();
               go_to_dir(dir);
               free(dir);
            } else
               go_to_dir(media_list.entries->dirs[pos-1]);
         }

         // Mark this entry as the current one if not selected
         else {
            media_list.selected = pos;
         }

         // Give this list focus
         file_focus = FFOCUS_MEDIA;
      }

      // Clicked inside the filename textbox?
      else if (input.cursor.x >= FILE_NAME_X1 &&
      input.cursor.x <= FILE_NAME_X2 &&
      input.cursor.y >= FILE_NAME_Y1 &&
      input.cursor.y <= FILE_NAME_Y2) {
         // Determine new position of the cursor
         filename.pos = get_char_at_pixel(filename.path,
            input.cursor.x + filename.scroll - FILE_NAME_X1 - 4);

         // Give focus to the textbox
         file_focus = FFOCUS_NAME;
         cursor_blink = 0;

         // Start text input
         SDL_StartTextInput();
      }

      // Clicked on nothing?
      else {
         // Make everything lose focus
         file_focus = FFOCUS_NONE;
      }
   }

   // Entering filename?
   if (file_focus == FFOCUS_NAME) {
      // Entered a character?
      if (input.ime.entered != NULL) {
         // No filename entered yet? (quick hack)
         if (filename.path == NULL) {
            filename.path = malloc(1);
            if (filename.path == NULL)
               abort_program(ERR_NOMEMORY, NULL);
            filename.path[0] = 0;
         }

         // Split filename into two
         const char *split_point = get_char_at_pos(filename.path,
            filename.pos);

         size_t before_len = (size_t)(split_point - filename.path);
         char *before = (char *) malloc(before_len + 1);
         if (before == NULL)
            abort_program(ERR_NOMEMORY, NULL);
         strncpy(before, filename.path, before_len);
         before[before_len] = 0;

         size_t after_len = strlen(split_point);
         char *after = (char *) malloc(after_len + 1);
         if (after == NULL)
            abort_program(ERR_NOMEMORY, NULL);
         strncpy(after, split_point, after_len);
         after[after_len] = 0;

         // Get rid of old filename
         free(filename.path);

         // Put all pieces together
         size_t total_len = before_len + strlen(input.ime.entered) +
            after_len;
         filename.path = malloc(total_len + 1);
         if (filename.path == NULL)
            abort_program(ERR_NOMEMORY, NULL);
         sprintf(filename.path, "%s%s%s", before, input.ime.entered, after);

         // Get rid of temporary stuff
         free(before);
         free(after);

         // Advance cursor
         filename.pos += get_utf8_strlen(input.ime.entered);

         // Update length of filename
         filename.len = get_utf8_strlen(filename.path);
      }

      // Move cursor to the left?
      else if (input.ime.left) {
         if (filename.pos > 0)
            filename.pos--;
      }

      // Move cursor to the right?
      else if (input.ime.right) {
         if (filename.pos < filename.len)
            filename.pos++;
      }

      // Move cursor to the beginning?
      else if (input.ime.start) {
         filename.pos = 0;
      }

      // Move cursor to the end?
      else if (input.ime.end) {
         filename.pos = filename.len;
      }

      // Delete a character?
      else if ((input.ime.back && filename.pos > 0) ||
      (input.ime.del && filename.pos < filename.len)) {
         // Go back one position if using backspace
         if (input.ime.back)
            filename.pos--;

         // No filename entered yet? (quick hack)
         if (filename.path == NULL) {
            filename.path = malloc(1);
            if (filename.path == NULL)
               abort_program(ERR_NOMEMORY, NULL);
            filename.path[0] = 0;
         }

         // Split filename into two
         const char *split_point = get_char_at_pos(filename.path,
            filename.pos);

         size_t before_len = (size_t)(split_point - filename.path);
         char *before = (char *) malloc(before_len + 1);
         if (before == NULL)
            abort_program(ERR_NOMEMORY, NULL);
         strncpy(before, filename.path, before_len);
         before[before_len] = 0;

         split_point += get_utf8_charlen(split_point);
         size_t after_len = strlen(split_point);
         char *after = (char *) malloc(after_len + 1);
         if (after == NULL)
            abort_program(ERR_NOMEMORY, NULL);
         strncpy(after, split_point, after_len);
         after[after_len] = 0;

         // Get rid of old filename
         free(filename.path);

         // Put all pieces together
         size_t total_len = before_len + after_len;
         filename.path = malloc(total_len + 1);
         if (filename.path == NULL)
            abort_program(ERR_NOMEMORY, NULL);
         sprintf(filename.path, "%s%s", before, after);

         // Get rid of temporary stuff
         free(before);
         free(after);

         // Update length of filename
         filename.len = get_utf8_strlen(filename.path);
      }
   }

   // Update scrolling of the filename field if needed
   {
      // Determine positions to compare
      int32_t min = FILE_NAME_X1 + 4;
      int32_t max = FILE_NAME_X2 - 4;
      int32_t x = min + calc_char_pos(filename.path, filename.pos) -
         filename.scroll;

      // Too far to the left?
      if (x < min)
         filename.scroll += x - min;

      // Too far to the right?
      if (x > max)
         filename.scroll += x - max;
   }

   // Update blinking the textbox cursor
   cursor_blink++;
}

//***************************************************************************
// run_editor_help [internal]
// Processes the logic for the help screen.
//***************************************************************************

static void run_editor_help(void) {
   // Make prev/next buttons usable only when it makes sense
   set_disabled(HBUTTON_PREV, help_page <= 0);
   set_disabled(HBUTTON_NEXT, help_page >= NUM_HELPPAGES-1);

   // Update buttons
   process_buttons();

   // Toggle pages with the shortcuts
   if (input.menu.left)
      hbutton_previous_page();
   if (input.menu.right)
      hbutton_next_page();

   // Esc quits the menu immediately
   if (input.menu.cancel)
      hbutton_back();
}

//***************************************************************************
// draw_editor
// Draws everything in the level editor
//***************************************************************************

void draw_editor(void) {
   // Clear the background
   clear_screen(0x606060);

   // Determine default cursor
   set_cursor(over_button != NULL ? CURSOR_HAND : CURSOR_ARROW);

   // Draw the GUI for the current screen
   switch (editor_screen) {
      case EDSCRN_MAIN: draw_editor_gui(); break;
      case EDSCRN_COLL: draw_tile_select(); break;
      case EDSCRN_OBJ: draw_tile_select(); break;
      case EDSCRN_INFO: draw_level_info(); break;
      case EDSCRN_LOAD: draw_file_selector(); break;
      case EDSCRN_SAVE: draw_file_selector(); break;
      case EDSCRN_HELP: draw_editor_help(); break;
      default: break;
   }

   // Draw message box on top of everything if visible
   if (messagebox.visible) {
      draw_messagebox();
   }

   // Check if there's something to show on the screen reader
   else {
      // If we're hovering over a button check its caption
      if (over_button != NULL && over_button->name != NULL) {
         set_reader_text(over_button->name);
      }

      // Check if the cursor is over a label otherwise
      else {
         for (unsigned i = 0; i < NUM_LABELS; i++) {
            if (input.cursor.x < labels[i].x1) continue;
            if (input.cursor.y < labels[i].y1) continue;
            if (input.cursor.x > labels[i].x2) continue;
            if (input.cursor.y > labels[i].y2) continue;
            if (labels[i].caption == NULL) continue;

            if (labels[i].strparam != NULL)
               set_reader_text_str(labels[i].caption, labels[i].strparam);
            else
               set_reader_text_int(labels[i].caption, labels[i].intparam);
         }
      }
   }
}

//***************************************************************************
// draw_editor_gui [internal]
// Draws the main GUI of the editor
//***************************************************************************

static void draw_editor_gui(void) {
   // If the cursor is within the tilemap area then show a cross
   if (input.cursor.x < (signed) (region_width << 5) &&
   input.cursor.y < (signed) (region_height << 5))
      set_cursor(CURSOR_CROSS);

   // Draw grid
   for (int tile_y = 0; tile_y < region_height; tile_y++)
   for (int tile_x = 0; tile_x < region_width; tile_x++) {
      // Calculate where this tile will be drawn
      int base_x = tile_x << TILE_SIZE_BIT;
      int base_y = tile_y << TILE_SIZE_BIT;

      // Draw slot
      draw_sprite(spr_blank_tile, base_x, base_y, SPR_NOFLIP);

      // Draw X coordinate
      char buffer[6];
      sprintf(buffer, "%u", region_x + tile_x);
      for (unsigned i = 0; buffer[i]; i++)
         draw_sprite(spr_coord_digit[buffer[i] - '0'],
         base_x + (i * 5) + 2, base_y + 2, SPR_NOFLIP);

      // Draw Y coordinate
      sprintf(buffer, "%u", region_y + tile_y);
      for (unsigned i = 0; buffer[i]; i++)
         draw_sprite(spr_coord_digit[buffer[i] - '0'],
         base_x + (i * 5) + 2, base_y + 9, SPR_NOFLIP);

      // Draw the collision tile here
      draw_sprite(spr_coll_tile[map.data
         [region_x + tile_x][region_y + tile_y].coll],
         base_x, base_y, SPR_NOFLIP);
   }

   // Draw objects placed in the map that are within the scroll window
   // Doing this separately because otherwise the grid will get drawn over
   // objects that overflow the tile boundaries. Also drawing outbounds since
   // some objects spawn over multiple tiles so we may need to render some
   // objects outside the visible region.
   for (int tile_y = -2; tile_y < region_height + 2; tile_y++)
   for (int tile_x = -2; tile_x < region_width + 2; tile_x++) {
      // Get coordinates of the tile here
      int x = region_x + tile_x;
      int y = region_y + tile_y;

      // Skip if outbounds
      if (x < 0 || x >= (signed) map.width ||
      y < 0 || y >= (signed) map.height)
         continue;

      // Get sprite of this object, if any
      // If NULL (no sprite) then keep going
      Sprite *spr = spr_obj_tile[map.data[x][y].obj][OBJSIZE_NORMAL];
      if (spr == NULL)
         continue;

      // Calculate where this tile will be drawn
      int base_x = tile_x << TILE_SIZE_BIT;
      int base_y = tile_y << TILE_SIZE_BIT;

      // Draw object
      if (spr != NULL) {
         draw_sprite(spr,
         base_x + 0x10 - (spr->width >> 1),
         base_y + 0x10 - (spr->height >> 1),
         (map.data[x][y].flags & MAPFLAG_FLIP) ?
            SPR_HFLIP : SPR_NOFLIP);
      }
   }

   // Redraw the background outside the editing window because otherwise big
   // objects will overflow into this area...
   fill_rectangle(region_width << TILE_SIZE_BIT, 0,
      screen_w, screen_h, 0x606060);
   fill_rectangle(0, region_height << TILE_SIZE_BIT,
      region_width << TILE_SIZE_BIT, screen_h, 0x606060);

   // Draw spawn point if visible
   if (map.spawn_x >= region_x && map.spawn_y >= region_y &&
   map.spawn_x < region_x + region_width &&
   map.spawn_y < region_y + region_height) {
      draw_sprite(spr_spawn,
         (map.spawn_x - region_x) << TILE_SIZE_BIT,
         (map.spawn_y - region_y) << TILE_SIZE_BIT,
         SPR_NOFLIP);
   }

   // Draw the scrollbar backgrounds
   // <Sik> Since I know somebody *will* get confused: this is *not* mixing
   // up X and Y coordinates, this is going through the scrollbar button IDs
   // (BUTTON_SCROLL_XL,BUTTON_SCROLL_XR,BUTTON_SCROLL_YU,BUTTON_SCROLL_YD),
   // so yes, this is correct.
   for (unsigned i = BUTTON_SCROLL_XL; i <= BUTTON_SCROLL_YD; i++) {
      fill_rectangle(buttons[i].x1, buttons[i].y1,
                     buttons[i].x2, buttons[i].y2,
                     (curr_button == &buttons[i] &&
                     over_button == &buttons[i]) ?
                     0x404040 : 0x808080);
   }

   // Draw currently selected tile type
   draw_sprite(spr_blank_tile,
      screen_w - 0x28, screen_h - 0x60, SPR_NOFLIP);
   if (editor_mode == EDMODE_TILEMAP) {
      draw_sprite(spr_coll_tile[coll_type],
         screen_w - 0x28, screen_h - 0x60, SPR_NOFLIP);
   } else {
      Sprite *spr = spr_obj_tile[obj_table[obj_type]][OBJSIZE_PREVIEW];
      if (spr != NULL) {
         draw_sprite(spr, screen_w - 0x18 - (spr->width >> 1),
         screen_h - 0x50 - (spr->height >> 1), SPR_NOFLIP);
      }
   }

   // Draw all buttons
   draw_buttons();

   // Draw the name of the button the cursor is over, if any
   if (over_button != NULL && over_button->name != NULL) {
      draw_text(over_button->name, screen_w - 0x2A, screen_h - 4,
         FONT_LIT, ALIGN_BOTTOM);
      //set_reader_text(over_button->name);
   }
}

//***************************************************************************
// draw_tile_select [internal]
// Draws the dialogs for selecting the tile type.
//***************************************************************************

static void draw_tile_select(void) {
   // Use the arrow by default unless it's on something clickable
   if (over_choice != -1)
      set_cursor(CURSOR_HAND);

   // Draw GUI
   draw_choices();
   draw_buttons();

   // Draw button names
   if (editor_screen == EDSCRN_OBJ) {
      draw_text(text.tile_select.items, OBJLIST_X, OBJLIST_Y1+8,
         FONT_LIT, ALIGN_CENTER);
      draw_text(text.tile_select.scenery, OBJLIST_X, OBJLIST_Y2+8,
         FONT_LIT, ALIGN_CENTER);
      draw_text(text.tile_select.enemies, OBJLIST_X, OBJLIST_Y3+8,
         FONT_LIT, ALIGN_CENTER);
      draw_text(text.tile_select.hazards, OBJLIST_X, OBJLIST_Y4+8,
         FONT_LIT, ALIGN_CENTER);
      draw_text(text.tile_select.switches, OBJLIST_X, OBJLIST_Y5+8,
         FONT_LIT, ALIGN_CENTER);
   }

   // Draw title
   draw_text(editor_screen == EDSCRN_COLL ?
      text.tile_select.title_coll : text.tile_select.title_obj,
      screen_cx, 0x10, FONT_LIT, ALIGN_CENTER);

   // Draw name of type if selected
   if (over_choice != -1) {
      // Get the name of the thing
      const char *caption;
      if (editor_screen == EDSCRN_COLL) {
         caption = text.tile_select.coll_names[choices[over_choice].value];
      } else {
         caption = text.tile_select.obj_names[choices[over_choice].value];
      }

      // Now draw it :P
      draw_text(caption, screen_cx, screen_h - 0x10, FONT_LIT, ALIGN_CENTER);
      labels[OLABEL_TILE].caption = caption;
   }

   // Er, let the screen reader know to not show anything
   // (unless hovering a button but that's another issue)
   else {
      labels[OLABEL_TILE].caption = NULL;
   }
}

//***************************************************************************
// draw_level_info [internal]
// Draws the dialog used to edit the level properties
//***************************************************************************

static void draw_level_info(void) {
   // Draw level theme section title
   draw_text_str(text.level_info.level_theme,
      text.level_info.themes[info_temp.theme],
      screen_cx, INFO_THEME_YTITLE, FONT_LIT, ALIGN_CENTER);

   // Draw level theme thumbnails
   int x = INFO_THEME_XTHUMB;
   for (unsigned i = 0; i < NUM_THEMES; i++, x += 0x24) {
      draw_frame(x-1, INFO_THEME_YTHUMB-1, x+0x20, INFO_THEME_YTHUMB+0x20);
      draw_sprite(spr_themes[i], x, INFO_THEME_YTHUMB, SPR_NOFLIP);
   }

   // In the demo version only virtual and park are available
   // Show a cross over all other themes in that case
   if (settings.demo) {
      int x = INFO_THEME_XTHUMB + 0x24 * 2;
      for (unsigned i = 2; i < NUM_THEMES; i++, x += 0x24)
         draw_sprite(spr_disabled_theme, x, INFO_THEME_YTHUMB, SPR_NOFLIP);
   }

   // Draw border for selected level theme
   x = INFO_THEME_XTHUMB + info_temp.theme * 0x24;
   draw_rectangle(x + 1, INFO_THEME_YTHUMB + 1,
      x + 0x1E, INFO_THEME_YTHUMB + 0x1E, settings.box_lit[1]);
   draw_rectangle(x, INFO_THEME_YTHUMB,
      x + 0x1F, INFO_THEME_YTHUMB + 0x1F, settings.box_lit[0]);
   draw_rectangle(x - 1, INFO_THEME_YTHUMB - 1,
      x + 0x20, INFO_THEME_YTHUMB + 0x20, settings.box_lit[0]);
   draw_rectangle(x - 2, INFO_THEME_YTHUMB - 2,
      x + 0x21, INFO_THEME_YTHUMB + 0x21, settings.box_lit[1]);
   draw_hline(x - 2, INFO_THEME_YTHUMB + 0x22, x + 0x21,
      settings.box_lit[2]);

   // Draw map size section title
   draw_text(text.level_info.map_size,
      screen_cx, INFO_SIZE_YTITLE, FONT_LIT, ALIGN_CENTER);

   // A buffer to store the strings with numbers
   // This is actually larger than we need, we should be able to get away
   // with just 14 bytes. Better to err on the safe side, though.
   // <Sik> Wait a second, 14? Aren't these numbers the level dimensions,
   // which are 16-bit? Doesn't that mean only 7 bytes are really needed?
   char buffer[0x20];

   // Draw labels for the map resizing controls
   draw_text("↑", INFO_SIZE_X1, INFO_SIZE_Y1T, FONT_LIT, ALIGN_CENTER);
   draw_text("↓", INFO_SIZE_X5, INFO_SIZE_Y1T, FONT_LIT, ALIGN_CENTER);
   draw_text("←", INFO_SIZE_X1, INFO_SIZE_Y2T, FONT_LIT, ALIGN_CENTER);
   draw_text("→", INFO_SIZE_X5, INFO_SIZE_Y2T, FONT_LIT, ALIGN_CENTER);

   // Draw amount of tiles to add/remove to the top
   sprintf(buffer, info_temp.resize_up ? "%+d" : "0",
      info_temp.resize_up);
   draw_frame(INFO_SIZE_X3 - 0x18, INFO_SIZE_Y1T - 0x08,
              INFO_SIZE_X3 + 0x17, INFO_SIZE_Y1T + 0x07);
   draw_text(buffer, INFO_SIZE_X3, INFO_SIZE_Y1T, FONT_LIT, ALIGN_CENTER);

   // Draw amount of tiles to add/remove to the bottom
   sprintf(buffer, info_temp.resize_down ? "%+d" : "0",
      info_temp.resize_down);
   draw_frame(INFO_SIZE_X7 - 0x18, INFO_SIZE_Y1T - 0x08,
              INFO_SIZE_X7 + 0x17, INFO_SIZE_Y1T + 0x07);
   draw_text(buffer, INFO_SIZE_X7, INFO_SIZE_Y1T, FONT_LIT, ALIGN_CENTER);

   // Draw amount of tiles to add/remove to the left
   sprintf(buffer, info_temp.resize_left ? "%+d" : "0",
      info_temp.resize_left);
   draw_frame(INFO_SIZE_X3 - 0x18, INFO_SIZE_Y2T - 0x08,
              INFO_SIZE_X3 + 0x17, INFO_SIZE_Y2T + 0x07);
   draw_text(buffer, INFO_SIZE_X3, INFO_SIZE_Y2T, FONT_LIT, ALIGN_CENTER);

   // Draw amount of tiles to add/remove to the right
   sprintf(buffer, info_temp.resize_right ? "%+d" : "0",
      info_temp.resize_right);
   draw_frame(INFO_SIZE_X7 - 0x18, INFO_SIZE_Y2T - 0x08,
              INFO_SIZE_X7 + 0x17, INFO_SIZE_Y2T + 0x07);
   draw_text(buffer, INFO_SIZE_X7, INFO_SIZE_Y2T, FONT_LIT, ALIGN_CENTER);

   // Draw current map size
   sprintf(buffer, "%u×%u", map.width, map.height);
   draw_text(text.level_info.current_size, screen_cx,
      INFO_SIZE_YCURR, FONT_LIT, ALIGN_RIGHT);
   draw_text(buffer, screen_cx + 0x10,
      INFO_SIZE_YCURR, FONT_LIT, ALIGN_LEFT);

   // Draw new map size
   sprintf(buffer, "%u×%u", info_temp.new_width, info_temp.new_height);
   draw_text(text.level_info.new_size, screen_cx,
      INFO_SIZE_YNEW, FONT_LIT, ALIGN_RIGHT);
   draw_text(buffer, screen_cx + 0x10,
      INFO_SIZE_YNEW, FONT_LIT, ALIGN_LEFT);

   // Draw "Apply" and "Cancel" buttons
   draw_buttons();
   draw_text(text.level_info.b_apply, INFO_BUTTON_X1, INFO_BUTTON_Y,
      over_button == &buttons[IBUTTON_APPLY] ? FONT_LIT : FONT_DIM,
      ALIGN_CENTER);
   draw_text(text.level_info.b_cancel, INFO_BUTTON_X2, INFO_BUTTON_Y,
      over_button == &buttons[IBUTTON_CANCEL] ? FONT_LIT : FONT_DIM,
      ALIGN_CENTER);

   // Update screen reader labels
   labels[ILABEL_TOP].intparam = info_temp.resize_up;
   labels[ILABEL_BOTTOM].intparam = info_temp.resize_down;
   labels[ILABEL_LEFT].intparam = info_temp.resize_left;
   labels[ILABEL_RIGHT].intparam = info_temp.resize_right;
}

//***************************************************************************
// draw_file_selector [internal]
// Draws the dialog for selecting files
//***************************************************************************

static void draw_file_selector(void) {
   // Use the I-beam if the cursor is over the filename textbox, otherwise
   // use the hand on buttons or the normal arrow otherwise
   if (input.cursor.x >= FILE_NAME_X1 &&
   input.cursor.x <= FILE_NAME_X2 &&
   input.cursor.y >= FILE_NAME_Y1 &&
   input.cursor.y <= FILE_NAME_Y2)
      set_cursor(CURSOR_IBEAM);

   // Draw name of entered file
   fill_rectangle(FILE_NAME_X1, FILE_NAME_Y1,
      FILE_NAME_X2, FILE_NAME_Y2, 0xA0A0A0);
   if (filename.path != NULL) {
      draw_text(filename.path,
         FILE_NAME_X1 + 4 - filename.scroll, FILE_NAME_Y1,
         file_focus == FFOCUS_NAME ? FONT_LIT : FONT_DIM,
         ALIGN_TOPLEFT);
   }

   // If the filename box is active draw the cursor
   // Also make the cursor blink to bring in the user's attention
   if (file_focus == FFOCUS_NAME) {
      // Calculate position of the cursor
      int32_t x = FILE_NAME_X1 + 4 - filename.scroll +
         calc_char_pos(filename.path, filename.pos);

      // Set position of candidate list
      int list_x = x;
      int list_y = FILE_NAME_Y2;
      virtual_to_real_coord(&list_x, &list_y);

      SDL_Rect list_rect;
      list_rect.x = list_x;
      list_rect.y = list_y;
      list_rect.w = 1;
      list_rect.h = 1;

      SDL_SetTextInputRect(&list_rect);

      // Draw cursor
      if (cursor_blink & 0x08)
         draw_vline(x, FILE_NAME_Y1, FILE_NAME_Y2, 0xFFFFFF);
   }

   // Draw file list
   for (unsigned i = 0; i < 8; i++)
      labels[FLABEL_LIST1 + i].caption = NULL;
   fill_rectangle(FILE_LIST_X1, FILE_LIST_Y1,
      FILE_LIST_X2, FILE_LIST_Y2, 0xA0A0A0);
   for (unsigned i = 0; i < file_list.size; i++) {
      // Get which entry to show
      unsigned pos = i + file_list.pos;

      // Highlight the background if this entry is selected
      if (pos == file_list.selected && file_focus == FFOCUS_LIST) {
         int where = (pos - file_list.pos) * 0x10;
         fill_rectangle(FILE_LIST_X1, FILE_LIST_Y1 + where,
            FILE_LIST_X2, FILE_LIST_Y1 + where + 0x0F, 0xC0C0C0);
      }

      // Some entries (i.e. directories) show a small icon before their name,
      // some don't. Specify which icon to use in this variable.
      const Sprite *icon = NULL;

      // Determine string for this entry
      const char *str;
      if (pos < file_list.entries->num_dirs) {
         icon = spr_folder;
         str = file_list.entries->dirs[pos];
      } else {
         str = file_list.entries->files[pos -
               file_list.entries->num_dirs];
      }

      // Files don't show any icon next to them, but directories do, and we
      // need to make room for them. We use this variable to know when we
      // need to make room for an icon and when not.
      int offset = icon ? icon->width + 4 : 0;

      // Draw name of file/directory
      draw_text(str,
         FILE_LIST_X1 + 4 + offset,
         FILE_LIST_Y1 + (i << 4),
         file_list.selected == pos && file_focus == FFOCUS_LIST ?
         FONT_LIT : FONT_DIM, ALIGN_TOPLEFT);
      labels[FLABEL_LIST1 + i].caption = str;

      // Draw icon, if any
      if (icon) {
         draw_sprite(icon, FILE_LIST_X1 + 4, FILE_LIST_Y1 + (i << 4),
         SPR_NOFLIP);
      }
   }

   // Draw media list
   for (unsigned i = 0; i < 8; i++)
      labels[FLABEL_MEDIA1 + i].caption = NULL;
   fill_rectangle(FILE_MEDIA_X1, FILE_MEDIA_Y1,
      FILE_MEDIA_X2, FILE_MEDIA_Y2, 0xA0A0A0);
   for (unsigned i = 0; i < media_list.size; i++) {
      // Get which entry to show
      unsigned pos = i + media_list.pos;

      // Highlight the background if this entry is selected
      if (pos == media_list.selected && file_focus == FFOCUS_MEDIA) {
         int where = (pos - media_list.pos) * 0x10;
         fill_rectangle(FILE_MEDIA_X1, FILE_MEDIA_Y1 + where,
            FILE_MEDIA_X2, FILE_MEDIA_Y1 + where + 0x0F, 0xC0C0C0);
      }

      // Determine string for this entry
      const char *str;
      if (pos == 0)
         str = text.file_select.home;
#ifndef _WIN32
      else if (strcmp(media_list.entries->dirs[pos-1], "/") == 0)
         str = text.file_select.root;
#endif
      else
         str = find_basename(media_list.entries->dirs[pos-1]);

      // Draw name of media
      draw_text(str, FILE_MEDIA_X1 + 4, FILE_MEDIA_Y1 + (i << 4),
         media_list.selected == pos && file_focus == FFOCUS_MEDIA ?
         FONT_LIT : FONT_DIM, ALIGN_TOPLEFT);
      labels[FLABEL_MEDIA1 + i].caption = str;
   }

   // Hide overflows
   fill_rectangle(0, FILE_NAME_Y1,
      FILE_NAME_X1, FILE_NAME_Y2, 0x606060);
   fill_rectangle(FILE_NAME_X2, FILE_NAME_Y1,
      screen_w, FILE_NAME_Y2, 0x606060);
   fill_rectangle(FILE_LIST_X2, FILE_LIST_Y1,
      FILE_MEDIA_X1, FILE_LIST_Y2, 0x606060);
   fill_rectangle(FILE_MEDIA_X2, FILE_MEDIA_Y1,
      screen_w, FILE_MEDIA_Y2, 0x606060);

   // Draw frames
   draw_frame(FILE_NAME_X1, FILE_NAME_Y1-1,
              FILE_NAME_X2, FILE_NAME_Y2+1);
   draw_frame(FILE_LIST_X1, FILE_LIST_Y1-1,
              FILE_LIST_X2, FILE_LIST_Y2+1);
   draw_frame(FILE_MEDIA_X1, FILE_MEDIA_Y1-1,
              FILE_MEDIA_X2, FILE_MEDIA_Y2+1);

   // Draw current path
   draw_text(current_path, FILE_PATH_X, FILE_PATH_Y,
      FONT_LIT, ALIGN_CENTER);

   // Draw the scrollbar backgrounds
   for (unsigned i = FBUTTON_SCROLL_LU; i <= FBUTTON_SCROLL_MD; i++) {
      if (buttons[i].state == BSTATE_DISABLED)
         continue;
      fill_rectangle(buttons[i].x1, buttons[i].y1,
                     buttons[i].x2, buttons[i].y2,
                     (curr_button == &buttons[i] &&
                     over_button == &buttons[i]) ?
                     0x404040 : 0x808080);
   }

   // Draw buttons
   draw_buttons();
   draw_text(editor_screen == EDSCRN_LOAD ?
      text.file_select.b_load : text.file_select.b_save,
      FILE_BUTTON_X1, FILE_BUTTON_Y,
      over_button == &buttons[FBUTTON_OK] ? FONT_LIT : FONT_DIM,
      ALIGN_CENTER);
   draw_text(text.file_select.b_cancel, FILE_BUTTON_X2, FILE_BUTTON_Y,
      over_button == &buttons[FBUTTON_CANCEL] ? FONT_LIT : FONT_DIM,
      ALIGN_CENTER);

   // Draw title
   draw_text(editor_screen == EDSCRN_LOAD ?
      text.file_select.load_title : text.file_select.save_title,
      FILE_TITLE_X, FILE_TITLE_Y, FONT_LIT, ALIGN_CENTER);
}

//***************************************************************************
// draw_editor_help [internal]
// Draws the help screen
//***************************************************************************

static void draw_editor_help(void) {
   // Reset all labels
   for (unsigned i = 0; i < 9; i++) {
      labels[HLABEL_LEFT1 + i].caption = "";
      labels[HLABEL_RIGHT1 + i].caption = "";
   }

   // Draw the limits of the area where the help text can appear
   for (unsigned y = 0; y < 9; y++) {
      draw_frame(screen_cx - 152, HELP_Y(y) - 8,
                 HELP_X1 + 7, HELP_Y(y) + 7);
      draw_frame(HELP_X2 - 8, HELP_Y(y) - 8,
                 screen_cx + 151, HELP_Y(y) + 7);
   }

   // To make code more readable (since we do this *a lot*)
#define LINE(pos, name) \
   draw_text(text.editor_help.name[0], HELP_X1, HELP_Y(pos), \
      FONT_LIT, ALIGN_RIGHT); \
   draw_text(text.editor_help.name[1], HELP_X2, HELP_Y(pos), \
      FONT_LIT, ALIGN_LEFT); \
   labels[HLABEL_LEFT1 + (pos)].caption = text.editor_help.name[0]; \
   labels[HLABEL_RIGHT1 + (pos)].caption = text.editor_help.name[1];

   // Determine what to show based on the current page
   switch (help_page) {
      // Movement page
      case HELPPAGE_MOVEMENT:
         LINE(0, scroll);
         LINE(1, scroll_fast);
         LINE(2, draw_tiles);
         LINE(3, erase_tiles);
         LINE(4, copy_tile);
         LINE(5, start_point);
         LINE(6, flip_object);
         LINE(7, select_tile);
         LINE(8, select_tile_2);
         break;

      // Commands page
      case HELPPAGE_COMMANDS:
         LINE(0, new_level);
         LINE(1, load_level);
         LINE(2, save_level);
         LINE(3, edit_info);
         LINE(4, play_level);
         LINE(5, tilemap_mode);
         LINE(6, objects_mode);
         break;

      // Miscellaneous page
      case HELPPAGE_MISC:
         LINE(0, show_help);
         LINE(1, quit_editor);
         break;

      // Unknown page :(
      default:
         draw_text("???", screen_cx, screen_cy, FONT_LIT, ALIGN_CENTER);
         break;
   }

   // Done with this
#undef LINE

   // Draw buttons
   draw_buttons();

   // Draw title
   draw_text(text.editor_help.title, HELP_TX, HELP_TY,
      FONT_LIT, ALIGN_CENTER);

   // Draw page number
   draw_text_int(text.editor_help.page, help_page + 1, HELP_PX, HELP_PY,
      FONT_LIT, ALIGN_CENTER);
}

//***************************************************************************
// draw_frame [internal]
// Draws a sunken frame at the specified coordinates
//---------------------------------------------------------------------------
// param x1: left side
// param y1: top side
// param x2: right side
// param y2: bottom side
//***************************************************************************

static void draw_frame(int x1, int y1, int x2, int y2) {
   draw_hline(x1+1, y1, x2-1, 0x404040);
   draw_hline(x1+1, y2, x2-1, 0x808080);
   draw_vline(x1, y1, y2-1, 0x404040);
   draw_vline(x2, y1+1, y2, 0x808080);
   draw_pixel(x1, y1, 0x202020);
   draw_pixel(x2, y2, 0xA0A0A0);
}

//***************************************************************************
// deinit_editor
// Deinitializes the level editor.
//***************************************************************************

void deinit_editor(void) {
   // Get rid of map data, *unless* we're going to test it, in which case
   // we should leave the map data loaded since it will be used in the new
   // game mode
   if (next_game_mode != GAMEMODE_INGAME) {
      unload_map();

      // We get rid of this here so the current path doesn't get reset when
      // the player decides to test the level (that gets annoying!)
      if (current_path) {
         free(current_path);
         current_path = NULL;
      }
   }

   // Unload graphics
   if (gfxset_editor) {
      destroy_graphics_set(gfxset_objs[OBJSIZE_PREVIEW]);
      destroy_graphics_set(gfxset_objs[OBJSIZE_NORMAL]);
      destroy_graphics_set(gfxset_editor);

      gfxset_objs[OBJSIZE_PREVIEW] = NULL;
      gfxset_objs[OBJSIZE_NORMAL] = NULL;
      gfxset_editor = NULL;
   }

   // Get rid of file selector stuff if needed
   if (file_list.entries) {
      free_dir_list(file_list.entries);
      file_list.entries = NULL;
   }
   if (media_list.entries) {
      free_dir_list(media_list.entries);
      media_list.entries = NULL;
   }
}

//***************************************************************************
// reset_buttons [internal]
// Initializes all buttons with default properties
//***************************************************************************

static void reset_buttons(void) {
   // Go through all buttons
   for (unsigned i = 0; i < NUM_BUTTONS; i++) {
      // Put them outbounds (so they can't be interacted with)
      buttons[i].x1 = INT_MIN;
      buttons[i].y1 = INT_MIN;
      buttons[i].x2 = INT_MIN;
      buttons[i].y2 = INT_MIN;

      // Not pressed yet
      buttons[i].state = BSTATE_RELEASED;
   }

   // Not interacting with any buttons yet
   over_button = NULL;
   curr_button = NULL;

   // Reset timer for buttons with scrollbar semantics
   scroll_timer = 0;
}

//***************************************************************************
// reset_labels [internal]
// Initializes all labels with default properties
//***************************************************************************

static void reset_labels(void) {
   // Make all labels outbounds
   for (unsigned i = 0; i < NUM_LABELS; i++) {
      labels[i].x1 = INT_MIN;
      labels[i].y1 = INT_MIN;
      labels[i].x2 = INT_MIN;
      labels[i].y2 = INT_MIN;
   }
}

//***************************************************************************
// process_buttons [internal]
// Processes and updates the status of all the buttons, calling any callbacks
// as needed if any button is triggered
//***************************************************************************

static void process_buttons(void) {
   // Mouse button not pressed? (ignore this im one-switch mode since it's
   // handled differently in that case)
   if (!input.cursor.left && !settings.one_switch) {
      // If the user just released the mouse button over the button he/she
      // had clicked, then trigger that button if it has an action associated
      // with it.
      if (curr_button != NULL && curr_button == over_button &&
      curr_button->func != NULL && !curr_button->scroller)
         curr_button->func();

      // Current button not current anymore
      curr_button = NULL;
   }

   // Check over which button is the cursor
   over_button = NULL;
   for (unsigned i = 0; i < NUM_BUTTONS; i++) {
      // Disabled? Don't bother with it
      if (buttons[i].state == BSTATE_DISABLED)
         continue;

      // Is the cursor over it?
      if (input.cursor.x >= buttons[i].x1 &&
      input.cursor.x <= buttons[i].x2 &&
      input.cursor.y >= buttons[i].y1 &&
      input.cursor.y <= buttons[i].y2) {
         over_button = &buttons[i];

         // Did we just click this button?
         if (input.cursor.click)
            curr_button = over_button;

         // Done looking for buttons, we found one already
         break;
      }
   }

   // Update the state of all buttons
   for (unsigned i = 0; i < NUM_BUTTONS; i++) {
      // Disabled? Don't bother with it
      if (buttons[i].state == BSTATE_DISABLED)
         continue;

      // If it's a button with scrollbar semantics and it's pressed, trigger
      // its callback every so often for as long as it's held down
      if (buttons[i].state == BSTATE_PRESSED && buttons[i].scroller &&
      scroll_timer == 0 && buttons[i].func != NULL) {
         buttons[i].func();
         scroll_timer = SCROLL_DELAY;
      }

      // Again, ignore this in one-switch mode since this is handled in a
      // different way in that case
      if (!settings.one_switch) {
         // Assume button isn't pressed
         buttons[i].state = BSTATE_RELEASED;

         // Pressed?
         if (over_button == &buttons[i] && curr_button == &buttons[i] &&
         input.cursor.left)
            buttons[i].state = BSTATE_PRESSED;
      }
   }

   // Reduce delay before calling the callback for scroller buttons again
   if (scroll_timer > 0)
      scroll_timer--;
}

//***************************************************************************
// set_disabled [internal]
// Toggles whether a button is disabled or not
//---------------------------------------------------------------------------
// param id: ID of button to modify
// param set: zero to enable, non-zero to disable
//***************************************************************************

static void set_disabled(ButtonID id, int set) {
   // Make the button disabled?
   if (set)
      buttons[id].state = BSTATE_DISABLED;

   // Make the button enabled?
   else if (buttons[id].state == BSTATE_DISABLED)
      buttons[id].state = BSTATE_RELEASED;
}

//***************************************************************************
// draw_buttons [internal]
// Draws all the buttons
//***************************************************************************

static void draw_buttons(void) {
   // Should be self-explanatory...
   // The sprite to use depends on the current state of the button
   // Invisible buttons get ignored by draw_sprite since they would attempt
   // to pass NULL as the sprite (draw_sprite will do nothing in that case)
   for (unsigned i = 0; i < NUM_BUTTONS; i++) {
      draw_sprite(buttons[i].sprite[buttons[i].state],
      buttons[i].x1, buttons[i].y1, SPR_NOFLIP);
   }
}

//***************************************************************************
// init_choices [internal]
// Initializes a choice grid based on the given layout
//---------------------------------------------------------------------------
// param layout: data for the grid layout
// param base_x: horizontal center of grid
// param base_y: vertical center of grid
//***************************************************************************

static void init_choices(const uint8_t *layout, int32_t base_x,
int32_t base_y) {
   // Reset choices
   num_choices = 0;

   // Get number of rows
   unsigned num_rows = *layout++;
   base_y -= num_rows * 20;
   base_y += 4;

   // Scan all rows
   while (num_rows--) {
      // Initial coordinates for this row
      int32_t x = base_x;
      int32_t y = base_y;

      // Get number of items in this row
      unsigned num_items = *layout++;
      x -= num_items * 20;
      x += 4;

      // Scan all items
      while (num_items--) {
         // Get which item is it
         uint8_t item = *layout++;

         // Get sprite for this choice
         const Sprite *sprite = NULL;
         if (item != 0) switch (editor_screen) {
            case EDSCRN_COLL:
               sprite = spr_coll_tile[item];
               break;
            case EDSCRN_OBJ:
               sprite = spr_obj_tile[item][OBJSIZE_PREVIEW];
               break;
            default:
               break;
         }

         // Add choice
         choices[num_choices].x = x;
         choices[num_choices].y = y;
         choices[num_choices].sprite = sprite;
         choices[num_choices].value = item;
         num_choices++;

         // Advance horizontal position
         x += 40;
      }

      // Advance vertical position
      base_y += 40;
   }

   // Set up the global label for the screen
   reset_labels();
   labels[OLABEL_TILE].x1 = 0;
   labels[OLABEL_TILE].y1 = 0;
   labels[OLABEL_TILE].x2 = screen_w;
   labels[OLABEL_TILE].y2 = screen_h;
}

//***************************************************************************
// process_choices [internal]
// Processes the logic for the choice grids
//***************************************************************************

static void process_choices(void) {
   // No choice hovered by default
   over_choice = -1;

   // See if the cursor is over any of the choices
   for (unsigned i = 0; i < num_choices; i++) {
      if (input.cursor.x >= choices[i].x &&
          input.cursor.y >= choices[i].y &&
          input.cursor.x <= choices[i].x + 31 &&
          input.cursor.y <= choices[i].y + 31)
      {
         over_choice = i;
         break;
      }
   }

   // Selected something?
   if (over_choice != -1 && input.cursor.click) {
      // Set current tile based on choice
      if (editor_screen == EDSCRN_COLL) {
         coll_type = choices[over_choice].value;
      } else {
         for (unsigned i = 0; ; i++)
         if (obj_table[i] == choices[over_choice].value) {
            obj_type = i;
            break;
         }
      }

      // Return to the main GUI
      init_editor_gui();
      editor_screen = EDSCRN_MAIN;
   }
}

//***************************************************************************
// draw_choices [internal]
// Draws all choices when selecting a collision or object tile
//***************************************************************************

static void draw_choices(void) {
   // Draw all available choices
   for (unsigned i = 0; i < num_choices; i++) {
      // Draw frame around object
      draw_frame(choices[i].x, choices[i].y,
                 choices[i].x+31, choices[i].y+31);

      // Draw sprite, if any
      if (choices[i].sprite != NULL) {
         draw_sprite(choices[i].sprite,
            choices[i].x+16 - (choices[i].sprite->width >> 1),
            choices[i].y+16 - (choices[i].sprite->height >> 1),
            SPR_NOFLIP);
      }
   }
}

//***************************************************************************
// init_editor_gui [internal]
// Initializes the main editor GUI. Called not only when entering the level
// editor, but also when returning back from the file select dialogs (since
// those impose their own GUI).
//***************************************************************************

static void init_editor_gui(void) {
   // This function only initializes the UI, so...
   set_reinit_menu_func(init_editor_gui);

   // Remove all buttons
   reset_buttons();
   reset_labels();

   // Calculate where the two columns of buttons are positioned
   // They're positioned relative to the right side of the screen
   unsigned left_column = screen_w - 0x4C;
   unsigned right_column = screen_w - 0x28;

   // Position the buttons at the top
   buttons[BUTTON_NEW].x1 = left_column;
   buttons[BUTTON_NEW].y1 = 4;
   buttons[BUTTON_NEW].x2 = buttons[BUTTON_NEW].x1 + 0x1F;
   buttons[BUTTON_NEW].y2 = buttons[BUTTON_NEW].y1 + 0x0F;

   buttons[BUTTON_INFO].x1 = right_column;
   buttons[BUTTON_INFO].y1 = 4;
   buttons[BUTTON_INFO].x2 = buttons[BUTTON_INFO].x1 + 0x1F;
   buttons[BUTTON_INFO].y2 = buttons[BUTTON_INFO].y1 + 0x0F;

   buttons[BUTTON_LOAD].x1 = left_column;
   buttons[BUTTON_LOAD].y1 = 24;
   buttons[BUTTON_LOAD].x2 = buttons[BUTTON_LOAD].x1 + 0x1F;
   buttons[BUTTON_LOAD].y2 = buttons[BUTTON_LOAD].y1 + 0x0F;

   buttons[BUTTON_SAVE].x1 = right_column;
   buttons[BUTTON_SAVE].y1 = 24;
   buttons[BUTTON_SAVE].x2 = buttons[BUTTON_SAVE].x1 + 0x1F;
   buttons[BUTTON_SAVE].y2 = buttons[BUTTON_SAVE].y1 + 0x0F;

   buttons[BUTTON_PLAY].x1 = left_column;
   buttons[BUTTON_PLAY].y1 = 44;
   buttons[BUTTON_PLAY].x2 = buttons[BUTTON_PLAY].x1 + 0x3F + 4;
   buttons[BUTTON_PLAY].y2 = buttons[BUTTON_PLAY].y1 + 0x0F;

   // Position the buttons at the bottom
   buttons[BUTTON_TILEMAP].x1 = left_column;
   buttons[BUTTON_TILEMAP].y1 = screen_h - 0x60;
   buttons[BUTTON_TILEMAP].x2 = buttons[BUTTON_TILEMAP].x1 + 0x1F;
   buttons[BUTTON_TILEMAP].y2 = buttons[BUTTON_TILEMAP].y1 + 0x0F;

   buttons[BUTTON_OBJECTS].x1 = left_column;
   buttons[BUTTON_OBJECTS].y1 = screen_h - 0x50;
   buttons[BUTTON_OBJECTS].x2 = buttons[BUTTON_OBJECTS].x1 + 0x1F;
   buttons[BUTTON_OBJECTS].y2 = buttons[BUTTON_OBJECTS].y1 + 0x0F;

   buttons[BUTTON_TILE].x1 = right_column;
   buttons[BUTTON_TILE].y1 = screen_h - 0x60;
   buttons[BUTTON_TILE].x2 = buttons[BUTTON_TILE].x1 + 0x1F;
   buttons[BUTTON_TILE].y2 = buttons[BUTTON_TILE].y1 + 0x1F;

   buttons[BUTTON_UNDO].x1 = left_column;
   buttons[BUTTON_UNDO].y1 = screen_h - 0x3C;
   buttons[BUTTON_UNDO].x2 = buttons[BUTTON_UNDO].x1 + 0x1F;
   buttons[BUTTON_UNDO].y2 = buttons[BUTTON_UNDO].y1 + 0x0F;

   buttons[BUTTON_REDO].x1 = right_column;
   buttons[BUTTON_REDO].y1 = screen_h - 0x3C;
   buttons[BUTTON_REDO].x2 = buttons[BUTTON_REDO].x1 + 0x1F;
   buttons[BUTTON_REDO].y2 = buttons[BUTTON_REDO].y1 + 0x0F;

   buttons[BUTTON_HELP].x1 = left_column;
   buttons[BUTTON_HELP].y1 = screen_h - 0x28;
   buttons[BUTTON_HELP].x2 = buttons[BUTTON_HELP].x1 + 0x1F;
   buttons[BUTTON_HELP].y2 = buttons[BUTTON_HELP].y1 + 0x0F;

   buttons[BUTTON_QUIT].x1 = right_column;
   buttons[BUTTON_QUIT].y1 = screen_h - 0x28;
   buttons[BUTTON_QUIT].x2 = buttons[BUTTON_QUIT].x1 + 0x1F;
   buttons[BUTTON_QUIT].y2 = buttons[BUTTON_QUIT].y1 + 0x0F;
}

//***************************************************************************
// init_level_info [internal]
// Initialize the level info dialog. Call this to be able to use its GUI, and
// also to initialize the temporary settings.
//***************************************************************************

static void init_level_info(void) {
   // By default use the same theme
   info_temp.theme = map.theme;

   // By default the map size is unchanged
   info_temp.new_width = map.width;
   info_temp.new_height = map.height;

   // The above implies borders are untouched...
   info_temp.resize_up = 0;
   info_temp.resize_down = 0;
   info_temp.resize_left = 0;
   info_temp.resize_right = 0;

   // Initialize UI
   init_level_info_ui();
}

//***************************************************************************
// init_level_info_ui [internal]
// Sets up the UI for the above function.
//***************************************************************************

static void init_level_info_ui(void) {
   set_reinit_menu_func(init_level_info_ui);

   // Remove all buttons
   reset_buttons();
   reset_labels();

   // Set up labels for level themes
   int32_t x = INFO_THEME_XTHUMB;
   int32_t y = INFO_THEME_YTHUMB;
   for (unsigned i = 0; i < NUM_THEMES; i++) {
      unsigned id = ILABEL_THEME1 + i;
      labels[id].x1 = x;
      labels[id].y1 = y;
      labels[id].x2 = x + 0x1F;
      labels[id].y2 = y + 0x1F;
      labels[id].caption = text.level_info.themes[i];
      x += 0x24;
   }

   // Set up labels for resizing buttons
   labels[ILABEL_TOP].x1 = INFO_SIZE_X2;
   labels[ILABEL_TOP].y1 = INFO_SIZE_Y1B;
   labels[ILABEL_TOP].x2 = INFO_SIZE_X4 + 0x17;
   labels[ILABEL_TOP].y2 = INFO_SIZE_Y1B + 0x0F;

   labels[ILABEL_BOTTOM].x1 = INFO_SIZE_X6;
   labels[ILABEL_BOTTOM].y1 = INFO_SIZE_Y1B;
   labels[ILABEL_BOTTOM].x2 = INFO_SIZE_X8 + 0x17;
   labels[ILABEL_BOTTOM].y2 = INFO_SIZE_Y1B + 0x0F;

   labels[ILABEL_LEFT].x1 = INFO_SIZE_X2;
   labels[ILABEL_LEFT].y1 = INFO_SIZE_Y2B;
   labels[ILABEL_LEFT].x2 = INFO_SIZE_X4 + 0x17;
   labels[ILABEL_LEFT].y2 = INFO_SIZE_Y2B + 0x0F;

   labels[ILABEL_RIGHT].x1 = INFO_SIZE_X6;
   labels[ILABEL_RIGHT].y1 = INFO_SIZE_Y2B;
   labels[ILABEL_RIGHT].x2 = INFO_SIZE_X8 + 0x17;
   labels[ILABEL_RIGHT].y2 = INFO_SIZE_Y2B + 0x0F;

   // Set up buttons for tiles at the top
   buttons[IBUTTON_INC_U].x1 = INFO_SIZE_X4;
   buttons[IBUTTON_INC_U].x2 = INFO_SIZE_X4 + 0x17;
   buttons[IBUTTON_INC_U].y1 = INFO_SIZE_Y1B;
   buttons[IBUTTON_INC_U].y2 = INFO_SIZE_Y1B + 0x0F;
   buttons[IBUTTON_DEC_U].x1 = INFO_SIZE_X2;
   buttons[IBUTTON_DEC_U].x2 = INFO_SIZE_X2 + 0x17;
   buttons[IBUTTON_DEC_U].y1 = INFO_SIZE_Y1B;
   buttons[IBUTTON_DEC_U].y2 = INFO_SIZE_Y1B + 0x0F;

   // Set up buttons for tiles at the bottom
   buttons[IBUTTON_INC_D].x1 = INFO_SIZE_X8;
   buttons[IBUTTON_INC_D].x2 = INFO_SIZE_X8 + 0x17;
   buttons[IBUTTON_INC_D].y1 = INFO_SIZE_Y1B;
   buttons[IBUTTON_INC_D].y2 = INFO_SIZE_Y1B + 0x0F;
   buttons[IBUTTON_DEC_D].x1 = INFO_SIZE_X6;
   buttons[IBUTTON_DEC_D].x2 = INFO_SIZE_X6 + 0x17;
   buttons[IBUTTON_DEC_D].y1 = INFO_SIZE_Y1B;
   buttons[IBUTTON_DEC_D].y2 = INFO_SIZE_Y1B + 0x0F;

   // Set up buttons for tiles at the left
   buttons[IBUTTON_INC_L].x1 = INFO_SIZE_X4;
   buttons[IBUTTON_INC_L].x2 = INFO_SIZE_X4 + 0x17;
   buttons[IBUTTON_INC_L].y1 = INFO_SIZE_Y2B;
   buttons[IBUTTON_INC_L].y2 = INFO_SIZE_Y2B + 0x0F;
   buttons[IBUTTON_DEC_L].x1 = INFO_SIZE_X2;
   buttons[IBUTTON_DEC_L].x2 = INFO_SIZE_X2 + 0x17;
   buttons[IBUTTON_DEC_L].y1 = INFO_SIZE_Y2B;
   buttons[IBUTTON_DEC_L].y2 = INFO_SIZE_Y2B + 0x0F;

   // Set up buttons for tiles at the right
   buttons[IBUTTON_INC_R].x1 = INFO_SIZE_X8;
   buttons[IBUTTON_INC_R].x2 = INFO_SIZE_X8 + 0x17;
   buttons[IBUTTON_INC_R].y1 = INFO_SIZE_Y2B;
   buttons[IBUTTON_INC_R].y2 = INFO_SIZE_Y2B + 0x0F;
   buttons[IBUTTON_DEC_R].x1 = INFO_SIZE_X6;
   buttons[IBUTTON_DEC_R].x2 = INFO_SIZE_X6 + 0x17;
   buttons[IBUTTON_DEC_R].y1 = INFO_SIZE_Y2B;
   buttons[IBUTTON_DEC_R].y2 = INFO_SIZE_Y2B + 0x0F;

   // Set up "Apply" button
   buttons[IBUTTON_APPLY].x1 = INFO_BUTTON_X1 - 0x28;
   buttons[IBUTTON_APPLY].y1 = INFO_BUTTON_Y - 0x08;
   buttons[IBUTTON_APPLY].x2 = INFO_BUTTON_X1 + 0x27;
   buttons[IBUTTON_APPLY].y2 = INFO_BUTTON_Y + 0x07;

   // Set up "Cancel" button
   buttons[IBUTTON_CANCEL].x1 = INFO_BUTTON_X2 - 0x28;
   buttons[IBUTTON_CANCEL].y1 = INFO_BUTTON_Y - 0x08;
   buttons[IBUTTON_CANCEL].x2 = INFO_BUTTON_X2 + 0x27;
   buttons[IBUTTON_CANCEL].y2 = INFO_BUTTON_Y + 0x07;
}

//***************************************************************************
// init_file_selector [internal]
// Initialize the file select dialogs. Call this to be able to use their GUI.
//***************************************************************************

static void init_file_selector(void) {
   // Enable access to the real filesystem
   // Doing this now so drives are updated on Windows (pointless on Linux
   // since you can't remove /, and if you can you have bigger issues)
   enable_user_filesystem();

   // Set up default path if needed
   if (current_path == NULL) {
      char *dir = get_default_dir();
      go_to_dir(dir);
      free(dir);
   }

   // Set up UI
   init_file_selector_ui();

   // Set up the lists
   update_file_list();
   update_media_list();

   // Reset focus
   file_focus = FFOCUS_NONE;
}

//***************************************************************************
// init_file_selector_ui [internal]
// Sets up the UI for the above function.
//***************************************************************************

static void init_file_selector_ui(void) {
   set_reinit_menu_func(init_file_selector_ui);

   // Remove all buttons
   reset_buttons();
   reset_labels();

   // Set up "Load"/"Save" button
   buttons[FBUTTON_OK].x1 = FILE_BUTTON_X1 - 0x28;
   buttons[FBUTTON_OK].y1 = FILE_BUTTON_Y - 0x08;
   buttons[FBUTTON_OK].x2 = FILE_BUTTON_X1 + 0x27;
   buttons[FBUTTON_OK].y2 = FILE_BUTTON_Y + 0x07;

   // Set up "Cancel" button
   buttons[FBUTTON_CANCEL].x1 = FILE_BUTTON_X2 - 0x28;
   buttons[FBUTTON_CANCEL].y1 = FILE_BUTTON_Y - 0x08;
   buttons[FBUTTON_CANCEL].x2 = FILE_BUTTON_X2 + 0x27;
   buttons[FBUTTON_CANCEL].y2 = FILE_BUTTON_Y + 0x07;

   // File list scrollbar
   buttons[FBUTTON_LISTUP].x1 = FILE_LIST_X2;
   buttons[FBUTTON_LISTUP].y1 = FILE_LIST_Y1;
   buttons[FBUTTON_LISTUP].x2 = FILE_LIST_X2 + 0x0F;
   buttons[FBUTTON_LISTUP].y2 = FILE_LIST_Y1 + 0x1F;

   buttons[FBUTTON_LISTDOWN].x1 = FILE_LIST_X2;
   buttons[FBUTTON_LISTDOWN].y1 = FILE_LIST_Y2 - 0x1F;
   buttons[FBUTTON_LISTDOWN].x2 = FILE_LIST_X2 + 0x0F;
   buttons[FBUTTON_LISTDOWN].y2 = FILE_LIST_Y2;

   buttons[FBUTTON_LISTMARKER].x1 = FILE_LIST_X2;
   buttons[FBUTTON_LISTMARKER].x2 = FILE_LIST_X2 + 0x0F;

   buttons[FBUTTON_SCROLL_LU].x1 = FILE_LIST_X2 + 4;
   buttons[FBUTTON_SCROLL_LU].x2 = FILE_LIST_X2 + 0x0B;
   buttons[FBUTTON_SCROLL_LU].y1 = FILE_LIST_Y1 + 0x20;

   buttons[FBUTTON_SCROLL_LD].x1 = FILE_LIST_X2 + 4;
   buttons[FBUTTON_SCROLL_LD].x2 = FILE_LIST_X2 + 0x0B;
   buttons[FBUTTON_SCROLL_LD].y2 = FILE_LIST_Y2 - 0x20;

   // Media list scrollbar
   buttons[FBUTTON_MEDIAUP].x1 = FILE_MEDIA_X2;
   buttons[FBUTTON_MEDIAUP].y1 = FILE_MEDIA_Y1;
   buttons[FBUTTON_MEDIAUP].x2 = FILE_MEDIA_X2 + 0x0F;
   buttons[FBUTTON_MEDIAUP].y2 = FILE_MEDIA_Y1 + 0x1F;

   buttons[FBUTTON_MEDIADOWN].x1 = FILE_MEDIA_X2;
   buttons[FBUTTON_MEDIADOWN].y1 = FILE_MEDIA_Y2 - 0x1F;
   buttons[FBUTTON_MEDIADOWN].x2 = FILE_MEDIA_X2 + 0x0F;
   buttons[FBUTTON_MEDIADOWN].y2 = FILE_MEDIA_Y2;

   buttons[FBUTTON_MEDIAMARKER].x1 = FILE_MEDIA_X2;
   buttons[FBUTTON_MEDIAMARKER].x2 = FILE_MEDIA_X2 + 0x0F;

   buttons[FBUTTON_SCROLL_MU].x1 = FILE_MEDIA_X2 + 4;
   buttons[FBUTTON_SCROLL_MU].x2 = FILE_MEDIA_X2 + 0x0B;
   buttons[FBUTTON_SCROLL_MU].y1 = FILE_MEDIA_Y1 + 0x20;

   buttons[FBUTTON_SCROLL_MD].x1 = FILE_MEDIA_X2 + 4;
   buttons[FBUTTON_SCROLL_MD].x2 = FILE_MEDIA_X2 + 0x0B;
   buttons[FBUTTON_SCROLL_MD].y2 = FILE_MEDIA_Y2 - 0x20;

   // "Up" button
   buttons[FBUTTON_DIRUP].x1 = 0x08;
   buttons[FBUTTON_DIRUP].y1 = 0x08;
   buttons[FBUTTON_DIRUP].x2 = 0x27;
   buttons[FBUTTON_DIRUP].y2 = 0x17;

   // What this button shows to the screen reader depends on whether we're
   // loading or saving
   buttons[FBUTTON_OK].name = (editor_screen == EDSCRN_LOAD) ?
      text.file_select.b_load : text.file_select.b_save;

   // Set up labels
   for (int32_t y = FILE_LIST_Y1; y < FILE_LIST_Y2; y += 0x10) {
      unsigned i = ((y - FILE_LIST_Y1) >> 4) + FLABEL_LIST1;
      labels[i].x1 = FILE_LIST_X1;
      labels[i].x2 = FILE_LIST_X2;
      labels[i].y1 = y;
      labels[i].y2 = y + 0x0F;
      labels[i].strparam = "{PARAM}"; // hack :P
   }
   for (int32_t y = FILE_MEDIA_Y1; y < FILE_MEDIA_Y2; y += 0x10) {
      unsigned i = ((y - FILE_MEDIA_Y1) >> 4) + FLABEL_MEDIA1;
      labels[i].x1 = FILE_MEDIA_X1;
      labels[i].x2 = FILE_MEDIA_X2;
      labels[i].y1 = y;
      labels[i].y2 = y + 0x0F;
      labels[i].strparam = "{PARAM}"; // hack :P
   }
}

//***************************************************************************
// init_help [internal]
// Initialize the help screen dialog. Call this to be able to use its GUI.
//***************************************************************************

static void init_help(void) {
   set_reinit_menu_func(init_help);

   // Remove all buttons
   reset_buttons();
   reset_labels();

   // Set previous/next page buttons
   buttons[HBUTTON_PREV].x1 = HELP_PBX1;
   buttons[HBUTTON_PREV].y1 = HELP_PBY;
   buttons[HBUTTON_PREV].x2 = HELP_PBX1 + 0x1F;
   buttons[HBUTTON_PREV].y2 = HELP_PBY + 0x0F;

   buttons[HBUTTON_NEXT].x1 = HELP_PBX2;
   buttons[HBUTTON_NEXT].y1 = HELP_PBY;
   buttons[HBUTTON_NEXT].x2 = HELP_PBX2 + 0x1F;
   buttons[HBUTTON_NEXT].y2 = HELP_PBY + 0x0F;

   // Set back button
   buttons[HBUTTON_BACK].x1 = HELP_PBXB;
   buttons[HBUTTON_BACK].y1 = HELP_PBYB;
   buttons[HBUTTON_BACK].x2 = HELP_PBXB + 0x1F;
   buttons[HBUTTON_BACK].y2 = HELP_PBYB + 0x0F;

   // Set label locations
   for (unsigned i = 0; i < 9; i++) {
      unsigned id;

      id = HLABEL_LEFT1 + i;
      labels[id].x1 = screen_cx - 152;
      labels[id].x2 = HELP_X1 + 7;
      labels[id].y1 = HELP_Y(i) - 8;
      labels[id].y2 = HELP_Y(i) + 7;

      id = HLABEL_RIGHT1 + i;
      labels[id].x1 = HELP_X2 - 8;
      labels[id].x2 = screen_cx + 151;
      labels[id].y1 = HELP_Y(i) - 8;
      labels[id].y2 = HELP_Y(i) + 7;
   }
}

//***************************************************************************
// init_object_buttons [internal]
// Initialize the buttons for the object select dialog.
//***************************************************************************

static void init_object_buttons(void) {
   // Remove all buttons
   reset_buttons();
   reset_labels();

   // "Items" button
   buttons[OBUTTON_ITEM].x1 = OBJLIST_X1;
   buttons[OBUTTON_ITEM].y1 = OBJLIST_Y1;
   buttons[OBUTTON_ITEM].x2 = OBJLIST_X2;
   buttons[OBUTTON_ITEM].y2 = OBJLIST_Y1 + 0x0F;
   buttons[OBUTTON_ITEM].name = text.tile_select.items;

   // "Scenery" button
   buttons[OBUTTON_SCENERY].x1 = OBJLIST_X1;
   buttons[OBUTTON_SCENERY].y1 = OBJLIST_Y2;
   buttons[OBUTTON_SCENERY].x2 = OBJLIST_X2;
   buttons[OBUTTON_SCENERY].y2 = OBJLIST_Y2 + 0x0F;
   buttons[OBUTTON_SCENERY].name = text.tile_select.scenery;

   // "Enemies" button
   buttons[OBUTTON_ENEMY].x1 = OBJLIST_X1;
   buttons[OBUTTON_ENEMY].y1 = OBJLIST_Y3;
   buttons[OBUTTON_ENEMY].x2 = OBJLIST_X2;
   buttons[OBUTTON_ENEMY].y2 = OBJLIST_Y3 + 0x0F;
   buttons[OBUTTON_ENEMY].name = text.tile_select.enemies;

   // "Hazards" button
   buttons[OBUTTON_HAZARD].x1 = OBJLIST_X1;
   buttons[OBUTTON_HAZARD].y1 = OBJLIST_Y4;
   buttons[OBUTTON_HAZARD].x2 = OBJLIST_X2;
   buttons[OBUTTON_HAZARD].y2 = OBJLIST_Y4 + 0x0F;
   buttons[OBUTTON_HAZARD].name = text.tile_select.hazards;

   // "Switches" button
   buttons[OBUTTON_SWITCH].x1 = OBJLIST_X1;
   buttons[OBUTTON_SWITCH].y1 = OBJLIST_Y5;
   buttons[OBUTTON_SWITCH].x2 = OBJLIST_X2;
   buttons[OBUTTON_SWITCH].y2 = OBJLIST_Y5 + 0x0F;
   buttons[OBUTTON_SWITCH].name = text.tile_select.switches;
}

//***************************************************************************
// update_file_list [internal]
// Updates the files list with the files in this directory
//***************************************************************************

static void update_file_list(void) {
   // Get rid of the old file list if needed
   if (file_list.entries) {
      free_dir_list(file_list.entries);
      file_list.entries = NULL;
   }

   // Get list of files in this directory
   file_list.entries = get_dir_list(current_path);
   if (file_list.entries == NULL)
      abort_program(ERR_FILELIST, current_path);
   filter_dir_list(file_list.entries, ".sol");

   // Check how many entries need to be in this list
   file_list.count = file_list.entries->num_files +
                     file_list.entries->num_dirs;

   // Scroll the list back to the beginning
   file_list.pos = 0;

   // Determine how many entries can be shown at the same time
   file_list.size = (FILE_LIST_Y2 - FILE_LIST_Y1) / 0x10;
   if (file_list.count < file_list.size)
      file_list.size = file_list.count;

   // Unselect everything
   file_list.selected = UINT_MAX;
   media_list.selected = UINT_MAX;

   // Enable the "go dir up" button only if there's a parent directory
   set_disabled(FBUTTON_DIRUP, is_root_dir(current_path));
}

//***************************************************************************
// update_media_list [internal]
// Updates the media list with the storage media currently available
//***************************************************************************

static void update_media_list(void) {
   // Get rid of the old media list if any
   if (media_list.entries) {
      free_dir_list(media_list.entries);
      media_list.entries = NULL;
   }

   // Get a list of the available media
   media_list.entries = get_media_list();
   if (media_list.entries == NULL)
      abort_program(ERR_MEDIALIST, NULL);

   // Check how many entries need to be in this list
   media_list.count = media_list.entries->num_dirs + 1;

   // Scroll the list back to the beginning
   media_list.pos = 0;

   // Determine how many entries can be shown at the same time
   media_list.size = (FILE_MEDIA_Y2 - FILE_MEDIA_Y1) / 0x10;
   if (media_list.count < media_list.size)
      media_list.size = media_list.count;
}

//***************************************************************************
// nuke_filename [internal]
// Sets the filename in the file selector to "" (empty).
//***************************************************************************

static void nuke_filename(void) {
   // Get rid of current path
   if (filename.path != NULL) {
      free(filename.path);
      filename.path = NULL;
   }

   // Reset cursor
   filename.len = 0;
   filename.pos = 0;
   filename.scroll = 0;
}

//***************************************************************************
// set_filename [internal]
// Changes the current filename in the textbox to the specified one.
//---------------------------------------------------------------------------
// param name: new filename
//***************************************************************************

static void set_filename(const char *name) {
   // Get rid of current path
   if (filename.path != NULL) {
      free(filename.path);
      filename.path = NULL;
   }

   // Store the new filename
   filename.path = (char *) malloc(strlen(name) + 1);
   if (filename.path == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   strcpy(filename.path, name);

   // Adjust cursor
   filename.len = get_utf8_strlen(name);
   filename.pos = 0;
   filename.scroll = 0;
}

//***************************************************************************
// go_to_dir [internal]
// Goes to the specified directory (in the file selector)
//---------------------------------------------------------------------------
// param path: path of new directory
//***************************************************************************

static void go_to_dir(const char *path) {
   // Get rid of the old path if needed
   if (current_path != NULL)
      free(current_path);

   // Save the new path
   current_path = (char *) malloc(strlen(path) + 1);
   if (current_path == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   strcpy(current_path, path);

   // Get the file list for this path
   update_file_list();

   // Reset filename
   nuke_filename();
}

//***************************************************************************
// button_new [internal]
// The code for the "New" button. It clears everything and leaves a new blank
// map. Used to start new levels.
//***************************************************************************

static void button_new(void) {
   // ...
   // To-do: check if map has changed and ask the user if he wants to save
   // before resetting the map data.
   // ...

   // Clear map
   new_map(MIN_MAP_WIDTH, MIN_MAP_HEIGHT);
}

//***************************************************************************
// button_info [internal]
// The code for the "Info" button. It goes into the dialog where the user can
// edit the level properties.
//***************************************************************************

static void button_info(void) {
   // Switch into the level info dialog
   editor_screen = EDSCRN_INFO;
   init_level_info();
}

//***************************************************************************
// button_load [internal]
// The code for the "Load" button. It goes into the load level dialog.
//***************************************************************************

static void button_load(void) {
   // ...
   // To-do: check if map has changed and ask the user if he wants to save
   // before trying to load a new level.
   // ...

   // Switch into the file selector
   editor_screen = EDSCRN_LOAD;
   init_file_selector();
}

//***************************************************************************
// button_save [internal]
// The code for the "Save" button. It goes into the save level dialog.
//***************************************************************************

static void button_save(void) {
   // Switch into the file selector
   editor_screen = EDSCRN_SAVE;
   init_file_selector();
}

//***************************************************************************
// button_play [internal]
// The code for the "Play" button. It goes into the ingame mode to test the
// map being currently edited. It's also used to just go play a custom level
// not loaded as a mod.
//***************************************************************************

static void button_play(void) {
   // Start game
   reset_spawn_point();
   fade_off_and_switch(GAMEMODE_INGAME);
}

//***************************************************************************
// button_tilemap, button_objects [internal]
// Code for the buttons that change the editor mode.
//***************************************************************************

static void button_tilemap(void) {
   editor_mode = EDMODE_TILEMAP;
}
static void button_objects(void) {
   editor_mode = EDMODE_OBJECTS;
}

//***************************************************************************
// button_tile [internal]
// Code for the button to change the tile type.
//***************************************************************************

static void button_tile(void) {
   // Extremely ugly hack, but it works...
   // Should have made the switching their own functions >.<
   set_reinit_menu_func(button_tile);

   // NOW switch mode
   switch (editor_mode) {
      // Go into collision tile select
      case EDMODE_TILEMAP:
         editor_screen = EDSCRN_COLL;
         reset_buttons();
         reset_labels();
         init_choices(choices_coll, screen_cx, screen_cy);
         break;

      // Go into object tile select
      case EDMODE_OBJECTS:
         editor_screen = EDSCRN_OBJ;
         init_object_buttons();
         init_choices(choices_items, screen_cx + 0x28, screen_cy);
         break;

      // What?
      default:
         break;
   }
}

//***************************************************************************
// button_undo [internal]
// The code for the "Undo" button. It undoes the last action (if any). Note
// that there's a limit to how many actions can be undone (determined by
// MAX_UNDO).
//***************************************************************************

static void button_undo(void) {
   // Nothing to undo? (the button should *not* be enabled if this is the
   // case, but whatever, let's just put this here...)
   if (undo_count == 0)
      return;

   // Check what action to undo and act accordingly
   const UndoStack *ptr = &undo_stack[undo_pos];
   switch (ptr->type) {
      // Placed a collision tile?
      case UNDO_TILEMAP:
         map.data[ptr->x][ptr->y].coll = ptr->old_value;
         break;

      // Placed an object?
      case UNDO_OBJECT:
         map.data[ptr->x][ptr->y].obj = ptr->old_value;
         break;

      // Changed the flags?
      case UNDO_FLAGS:
         map.data[ptr->x][ptr->y].flags = ptr->old_value;
         break;

      // ...
      default:
         abort_program(ERR_UNKNOWN, NULL);
   }

   // Get position of the last undo action and make it the current one
   if (undo_pos == 0)
      undo_pos = MAX_UNDO;
   undo_pos--;

   // One less action available to undo, one more available to redo...
   undo_count--;
   redo_count++;
}

//***************************************************************************
// button_redo [internal]
// The code for the "Redo" button. It redoes the last undone action (if any).
//***************************************************************************

static void button_redo(void) {
   // Nothing to redo? (the button should *not* be enabled if this is the
   // case, but whatever, let's just put this here...)
   if (redo_count == 0)
      return;

   // Determine position of next slot available to redo
   undo_pos++;
   if (undo_pos == MAX_UNDO)
      undo_pos = 0;

   // Check what action to redo and act accordingly
   const UndoStack *ptr = &undo_stack[undo_pos];
   switch (ptr->type) {
      // Placed a collision tile?
      case UNDO_TILEMAP:
         map.data[ptr->x][ptr->y].coll = ptr->new_value;
         break;

      // Placed an object?
      case UNDO_OBJECT:
         map.data[ptr->x][ptr->y].obj = ptr->new_value;
         break;

      // Changed the flags?
      case UNDO_FLAGS:
         map.data[ptr->x][ptr->y].flags = ptr->new_value;
         break;

      // ...
      default:
         abort_program(ERR_UNKNOWN, NULL);
   }

   // One less action available to redo, one more available to undo...
   undo_count++;
   redo_count--;
}

//***************************************************************************
// button_help [internal]
// The code for the "Help" button. Shows the help screen.
//***************************************************************************

static void button_help(void) {
   // Switch to the help screen
   editor_screen = EDSCRN_HELP;
   init_help();
}

//***************************************************************************
// button_quit [internal]
// The code for the "Quit" button. Quits the editor and goes back to the
// title screen.
//***************************************************************************

static void button_quit(void) {
   // ...
   // To-do: check if map has changed and ask the user if he wants to save
   // before quitting.
   // ...

   // Switch to the title screen
   fade_off_and_switch(GAMEMODE_TITLE);
}

//***************************************************************************
// button_scroll_up [internal]
// The code for the button used to scroll up the editor window.
//***************************************************************************

static void button_scroll_up(void) {
   if (input.editor.fast)
      region_y = scroll_sub(region_y, FAST_SCROLL);
   else
      region_y = scroll_sub(region_y, 1);
}

//***************************************************************************
// button_scroll_down [internal]
// The code for the button used to scroll down the editor window.
//***************************************************************************

static void button_scroll_down(void) {
   if (input.editor.fast)
      region_y = scroll_add(region_y, FAST_SCROLL,
      map.height - region_height);
   else
      region_y = scroll_add(region_y, 1,
      map.height - region_height);
}

//***************************************************************************
// button_scroll_left [internal]
// The code for the button used to scroll left the editor window.
//***************************************************************************

static void button_scroll_left(void) {
   if (input.editor.fast)
      region_x = scroll_sub(region_x, FAST_SCROLL);
   else
      region_x = scroll_sub(region_x, 1);
}

//***************************************************************************
// button_scroll_right [internal]
// The code for the button used to scroll right the editor window.
//***************************************************************************

static void button_scroll_right(void) {
   if (input.editor.fast)
      region_x = scroll_add(region_x, FAST_SCROLL,
      map.width - region_width);
   else
      region_x = scroll_add(region_x, 1,
      map.width - region_width);
}

//***************************************************************************
// button_scroll_up_fast [internal]
// The code for pressing over the vertical scrollbar background in the editor
// window to scroll upwards.
//***************************************************************************

static void button_scroll_up_fast(void) {
   region_y = scroll_sub(region_y, FAST_SCROLL);
}

//***************************************************************************
// button_scroll_down_fast [internal]
// The code for pressing over the vertical scrollbar background in the editor
// window to scroll downwards.
//***************************************************************************

static void button_scroll_down_fast(void) {
   region_y = scroll_add(region_y, FAST_SCROLL,
      map.height - region_height);
}

//***************************************************************************
// button_scroll_left_fast [internal]
// The code for pressing over the horizontal scrollbar background in the
// editor window to scroll to the left.
//***************************************************************************

static void button_scroll_left_fast(void) {
   region_x = scroll_sub(region_x, FAST_SCROLL);
}

//***************************************************************************
// button_scroll_right_fast [internal]
// The code for pressing over the horizontal scrollbar background in the
// editor window to scroll to the right.
//***************************************************************************

static void button_scroll_right_fast(void) {
   region_x = scroll_add(region_x, FAST_SCROLL,
      map.width - region_width);
}

//***************************************************************************
// move_up [internal]
// The code for moving the cursor up one tile.
//***************************************************************************

static void move_up(void) {
   scroll_timer = SCROLL_DELAY;

   int tile_y = input.cursor.y >> 5;
   if (tile_y == 0)
      region_y = scroll_sub(region_y, 1);
   else if (tile_y < region_height)
      input.cursor.y -= 0x20;
   else
      input.cursor.y = (region_height << 5) - 0x10;

   input.cursor.x >>= 5;   input.cursor.y >>= 5;
   input.cursor.x <<= 5;   input.cursor.y <<= 5;
   input.cursor.x += 0x10; input.cursor.y += 0x10;
}

//***************************************************************************
// move_down [internal]
// The code for moving the cursor down one tile.
//***************************************************************************

static void move_down(void) {
   scroll_timer = SCROLL_DELAY;

   int tile_y = input.cursor.y >> 5;
   if (tile_y == region_height - 1)
      region_y = scroll_add(region_y, 1, map.height - region_height);
   else if (tile_y < region_height)
      input.cursor.y += 0x20;
   else
      input.cursor.y = (region_height << 5) - 0x10;

   input.cursor.x >>= 5;   input.cursor.y >>= 5;
   input.cursor.x <<= 5;   input.cursor.y <<= 5;
   input.cursor.x += 0x10; input.cursor.y += 0x10;
}

//***************************************************************************
// move_left [internal]
// The code for moving the cursor left one tile.
//***************************************************************************

static void move_left(void) {
   scroll_timer = SCROLL_DELAY;

   int tile_x = input.cursor.x >> 5;
   if (tile_x == 0)
      region_x = scroll_sub(region_x, 1);
   else if (tile_x < region_width)
      input.cursor.x -= 0x20;
   else
      input.cursor.x = (region_width << 5) - 0x10;

   input.cursor.x >>= 5;   input.cursor.y >>= 5;
   input.cursor.x <<= 5;   input.cursor.y <<= 5;
   input.cursor.x += 0x10; input.cursor.y += 0x10;
}

//***************************************************************************
// move_right [internal]
// The code for moving the cursor right one tile.
//***************************************************************************

static void move_right(void) {
   scroll_timer = SCROLL_DELAY;

   int tile_x = input.cursor.x >> 5;
   if (tile_x == region_width - 1)
      region_x = scroll_add(region_x, 1, map.width - region_width);
   else if (tile_x < region_width)
      input.cursor.x += 0x20;
   else
      input.cursor.x = (region_width << 5) - 0x10;

   input.cursor.x >>= 5;   input.cursor.y >>= 5;
   input.cursor.x <<= 5;   input.cursor.y <<= 5;
   input.cursor.x += 0x10; input.cursor.y += 0x10;
}

//***************************************************************************
// ibutton_apply [internal]
// The code for the "Apply" button in the level info dialog. Returns to the
// main editor GUI and applies the changes made to the level.
//---------------------------------------------------------------------------
// To-do: optimize map resizing
//***************************************************************************

static void ibutton_apply(void) {
   // Implement new theme
   map.theme = info_temp.theme;

   // To make the code more readable, etc.
   uint16_t new_width = info_temp.new_width;
   uint16_t new_height = info_temp.new_height;

   // Only bother messing with the map itself if the size changed
   if (info_temp.resize_up != 0 || info_temp.resize_down != 0 ||
   info_temp.resize_left != 0 || info_temp.resize_right != 0) {
      // Allocate enough memory for the new map
      MapTile **new_data = (MapTile **) malloc(sizeof(MapTile *) *
         new_width);
      if (new_data == NULL)
         abort_program(ERR_NOMEMORY, NULL);
      for (unsigned i = 0; i < new_width; i++) {
         new_data[i] = (MapTile *) calloc(new_height, sizeof(MapTile));
         if (new_data[i] == NULL)
            abort_program(ERR_NOMEMORY, NULL);
      }

      // Determine which part of the old map to keep
      uint16_t src_x1 = 0;
      uint16_t src_x2 = map.width - 1;
      uint16_t src_y1 = 0;
      uint16_t src_y2 = map.height - 1;
      if (info_temp.resize_left < 0)
         src_x1 -= info_temp.resize_left;
      if (info_temp.resize_right < 0)
         src_x2 += info_temp.resize_right;
      if (info_temp.resize_up < 0)
         src_y1 -= info_temp.resize_up;
      if (info_temp.resize_down < 0)
         src_y2 += info_temp.resize_down;

      // Determine where the old map will be copied
      uint16_t dest_x = 0;
      uint16_t dest_y = 0;
      if (info_temp.resize_left > 0)
         dest_x = info_temp.resize_left;
      if (info_temp.resize_up > 0)
         dest_y = info_temp.resize_up;

      // Copy data from the old map to the new one
      // <Sik> WTF I had to put the explicit cast to unsigned there because
      // apparently substracting two uint16_t will result in a signed type...
      // Gotta love type promotion rules
      for (unsigned x = 0; x <= (unsigned)(src_x2 - src_x1); x++)
      for (unsigned y = 0; y <= (unsigned)(src_y2 - src_y1); y++) {
         memcpy(&new_data[x + dest_x][y + dest_y],
            &map.data[x + src_x1][y + src_y1],
            sizeof(MapTile));
      }

      // Adjust scrolling position to make up for the resizing
      region_x += dest_x;
      region_y += dest_y;
      if (region_x > info_temp.new_width - region_width)
         region_x = info_temp.new_width - region_width;
      if (region_y > info_temp.new_height - region_height)
         region_y = info_temp.new_height - region_height;

      // Store the new tilemap data and get rid of the old one
      unload_map();
      map.data = new_data;
      map.width = new_width;
      map.height = new_height;

      // Adjust spawn point
      if ((int)map.spawn_x >= -info_temp.resize_left)
         map.spawn_x += info_temp.resize_left;
      else
         map.spawn_x = 0;
      if ((int)map.spawn_y >= -info_temp.resize_up)
         map.spawn_y += info_temp.resize_up;
      else
         map.spawn_y = 0;
      if (map.spawn_x >= map.width)
         map.spawn_x = map.width - 1;
      if (map.spawn_y >= map.height)
         map.spawn_y = map.height - 1;
   }

   // For now trash the undo stack...
   // To-do: implement undoing level property changes
   undo_pos = 0;
   undo_count = 0;
   redo_count = 0;

   // Return to the main GUI
   editor_screen = EDSCRN_MAIN;
   init_editor_gui();
}

//***************************************************************************
// ibutton_cancel [internal]
// The code for the "Cancel" button in the level info dialog. Returns to the
// main editor GUI without doing anything.
//***************************************************************************

static void ibutton_cancel(void) {
   // Return to the main GUI
   editor_screen = EDSCRN_MAIN;
   init_editor_gui();
}

//***************************************************************************
// ibutton_inc_up [internal]
// The code for the button to add tiles from the top in the level info
// dialog.
//***************************************************************************

static void ibutton_inc_up(void) {
   // Ensure the map isn't too big
   if (info_temp.new_height >= MAX_MAP_HEIGHT)
      return;

   // Resize map
   info_temp.new_height++;
   info_temp.resize_up++;
}

//***************************************************************************
// ibutton_dec_up [internal]
// The code for the button to remove tiles from the top in the level info
// dialog.
//***************************************************************************

static void ibutton_dec_up(void) {
   // Ensure the map isn't too small
   if (info_temp.new_height <= MIN_MAP_HEIGHT)
      return;

   // Resize map
   info_temp.new_height--;
   info_temp.resize_up--;
}

//***************************************************************************
// ibutton_inc_down [internal]
// The code for the button to add tiles from the bottom in the level info
// dialog.
//***************************************************************************

static void ibutton_inc_down(void) {
   // Ensure the map isn't too big
   if (info_temp.new_height >= MAX_MAP_HEIGHT)
      return;

   // Resize map
   info_temp.new_height++;
   info_temp.resize_down++;
}

//***************************************************************************
// ibutton_dec_down [internal]
// The code for the button to remove tiles from the bottom in the level info
// dialog.
//***************************************************************************

static void ibutton_dec_down(void) {
   // Ensure the map isn't too small
   if (info_temp.new_height <= MIN_MAP_HEIGHT)
      return;

   // Resize map
   info_temp.new_height--;
   info_temp.resize_down--;
}

//***************************************************************************
// ibutton_inc_left [internal]
// The code for the button to add tiles from the left in the level info
// dialog.
//***************************************************************************

static void ibutton_inc_left(void) {
   // Ensure the map isn't too big
   if (info_temp.new_width >= MAX_MAP_WIDTH)
      return;

   // Resize map
   info_temp.new_width++;
   info_temp.resize_left++;
}

//***************************************************************************
// ibutton_dec_left [internal]
// The code for the button to remove tiles from the left in the level info
// dialog.
//***************************************************************************

static void ibutton_dec_left(void) {
   // Ensure the map isn't too small
   if (info_temp.new_width <= MIN_MAP_WIDTH)
      return;

   // Resize map
   info_temp.new_width--;
   info_temp.resize_left--;
}

//***************************************************************************
// ibutton_inc_right [internal]
// The code for the button to add tiles from the right in the level info
// dialog.
//***************************************************************************

static void ibutton_inc_right(void) {
   // Ensure the map isn't too big
   if (info_temp.new_width >= MAX_MAP_WIDTH)
      return;

   // Resize map
   info_temp.new_width++;
   info_temp.resize_right++;
}

//***************************************************************************
// ibutton_dec_right [internal]
// The code for the button to remove tiles from the right in the level info
// dialog.
//***************************************************************************

static void ibutton_dec_right(void) {
   // Ensure the map isn't too small
   if (info_temp.new_width <= MIN_MAP_WIDTH)
      return;

   // Resize map
   info_temp.new_width--;
   info_temp.resize_right--;
}

//***************************************************************************
// fbutton_ok [internal]
// The code for the "Load"/"Save" button in the file selector. Selects
// whatever filename has been entered.
//***************************************************************************

static void fbutton_ok(void) {
   // Load file?
   if (editor_screen == EDSCRN_LOAD) {
      // Get full filename
      char *path = append_path(current_path, filename.path);

      // Try to load file
      if (load_map(path)) {
         new_map(MIN_MAP_WIDTH, MIN_MAP_HEIGHT);
         show_messagebox(text.editor_error.load_level);
      }

      // Nice try
      else if (settings.demo && map.theme > THEME_PARK) {
         new_map(MIN_MAP_WIDTH, MIN_MAP_HEIGHT);
         show_messagebox(text.editor_error.demo_error);
      }

      // Done with the path
      free(path);
   }

   // Save file?
   else if (editor_screen == EDSCRN_SAVE) {
      // Get full filename
      char *path = append_path(current_path, filename.path);

      // ...
      // To-do: check if the user wants to overwrite a file
      // ...

      // Try to save file
      if (save_map(path)) {
         show_messagebox(text.editor_error.save_level);
      }

      // Done with the path
      free(path);
   }

   // Disable access to the real filesystem (for security reasons)
   disable_user_filesystem();

   // Return to the main GUI
   editor_screen = EDSCRN_MAIN;
   init_editor_gui();
}

//***************************************************************************
// fbutton_cancel [internal]
// The code for the "Cancel" button in the file selector. Returns to the main
// editor GUI without doing anything.
//***************************************************************************

static void fbutton_cancel(void) {
   // Disable access to the real filesystem (for security reasons)
   disable_user_filesystem();

   // Return to the main GUI
   editor_screen = EDSCRN_MAIN;
   init_editor_gui();
}

//***************************************************************************
// fbutton_list_up [internal]
// The code for the button used to scroll up the files list, one entry at a
// time.
//***************************************************************************

static void fbutton_list_up(void) {
   file_list.pos = scroll_sub(file_list.pos, 1);
}

//***************************************************************************
// fbutton_list_down [internal]
// The code for the button used to scroll down the file list, one entry at
// a time.
//***************************************************************************

static void fbutton_list_down(void) {
   file_list.pos = scroll_add(file_list.pos, 1,
      file_list.count - file_list.size);
}

//***************************************************************************
// fbutton_list_up_fast [internal]
// The code for the button used to scroll up the files list, several entries
// at a time.
//***************************************************************************

static void fbutton_list_up_fast(void) {
   file_list.pos = scroll_sub(file_list.pos, 3);
}

//***************************************************************************
// fbutton_list_down_fast [internal]
// The code for the button used to scroll down the file list, several entries
// at a time.
//***************************************************************************

static void fbutton_list_down_fast(void) {
   file_list.pos = scroll_add(file_list.pos, 3,
      file_list.count - file_list.size);
}

//***************************************************************************
// fbutton_media_up [internal]
// The code for the button used to scroll up the media list, one entry at a
// time.
//***************************************************************************

static void fbutton_media_up(void) {
   media_list.pos = scroll_sub(media_list.pos, 1);
}

//***************************************************************************
// fbutton_media_down [internal]
// The code for the button used to scroll down the media list, one entry at a
// time.
//***************************************************************************

static void fbutton_media_down(void) {
   media_list.pos = scroll_add(media_list.pos, 1,
      media_list.count - media_list.size);
}

//***************************************************************************
// fbutton_media_up_fast [internal]
// The code for the button used to scroll up the media list, several entries
// at a time.
//***************************************************************************

static void fbutton_media_up_fast(void) {
   media_list.pos = scroll_sub(media_list.pos, 1);
}

//***************************************************************************
// fbutton_media_down_fast [internal]
// The code for the button used to scroll down the media list, several
// entries at a time.
//***************************************************************************

static void fbutton_media_down_fast(void) {
   media_list.pos = scroll_add(media_list.pos, 1,
      media_list.count - media_list.size);
}

//***************************************************************************
// fbutton_dir_up [internal]
// The code for the button used to go up a directory.
//***************************************************************************

static void fbutton_dir_up(void) {
   char *path = get_parent_dir(current_path);
   go_to_dir(path);
   free(path);
}

//***************************************************************************
// hbutton_previous_page [internal]
// The code for the previous page button in the help screen.
//***************************************************************************

static void hbutton_previous_page(void) {
   if (help_page > 0)
      help_page--;
}

//***************************************************************************
// hbutton_next_page [internal]
// The code for the next page button in the help screen.
//***************************************************************************

static void hbutton_next_page(void) {
   if (help_page < NUM_HELPPAGES-1)
      help_page++;
}

//***************************************************************************
// hbutton_back [internal]
// The code for the back button in the help screen. Returns to the main
// editor GUI.
//***************************************************************************

static void hbutton_back(void) {
   // Return to the main GUI
   editor_screen = EDSCRN_MAIN;
   init_editor_gui();
}

//***************************************************************************
// obutton_* [internal]
// Code for the buttons that make up the list in the object dialog
//***************************************************************************

static void obutton_items(void) {
   init_choices(choices_items, screen_cx + 0x28, screen_cy);
}

static void obutton_scenery(void) {
   init_choices(choices_scenery, screen_cx + 0x28, screen_cy);
}

static void obutton_enemies(void) {
   init_choices(choices_enemies, screen_cx + 0x28, screen_cy);
}

static void obutton_hazards(void) {
   init_choices(choices_hazards, screen_cx + 0x28, screen_cy);
}

static void obutton_switches(void) {
   init_choices(choices_switches, screen_cx + 0x28, screen_cy);
}

//***************************************************************************
// scroll_add, scroll_sub [internal]
// Helper functions for scrollbar behavior. Updates the specified value
// taking into account the caps.
//---------------------------------------------------------------------------
// param original: original value
// param amount: amount to add/substract
// param cap: upper limit (for scroll_add, always 0 for scroll_sub)
//***************************************************************************

static unsigned scroll_add(unsigned original, unsigned amount, unsigned cap){
   return original + amount > cap ? cap : original + amount;
}
static unsigned scroll_sub(unsigned original, unsigned amount) {
   return amount > original ? 0 : original - amount;
}

//***************************************************************************
// new_map [internal]
// Creates a new map. This is what happens when you click "New".
//---------------------------------------------------------------------------
// param width: width in tiles of new map
// param height: height in tiles of new map
//***************************************************************************

static void new_map(unsigned width, unsigned height) {
   // Get rid of the previous map, if any
   unload_map();

   // Reset editor status
   editor_mode = EDMODE_TILEMAP;
   coll_type = TILE_SOLID;
   obj_type = 1;

   // Reset the undo stack
   undo_pos = 0;
   undo_count = 0;
   redo_count = 0;

   // Set up initial properties
   map.width = width;
   map.height = height;
   map.spawn_x = 0;
   map.spawn_y = 0;
   map.theme = THEME_VIRTUAL;

   // Allocate memory for the collision map
   // Using calloc to allocate the rows because it zeroes out the memory,
   // something we'd be doing anyways.
   map.data = (MapTile **) malloc(sizeof(MapTile *) * map.width);
   if (map.data == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   for (unsigned i = 0; i < map.width; i++) {
      map.data[i] = (MapTile *) calloc(sizeof(MapTile), map.height);
      if (map.data[i] == NULL)
         abort_program(ERR_NOMEMORY, NULL);
   }

   // Reset scrolling
   region_x = 0;
   region_y = 0;
}

//***************************************************************************
// load_map
// Loads the map from a file.
//---------------------------------------------------------------------------
// param name: name of file to read
// return: zero on success, non-zero on failure
//***************************************************************************

static int load_map(const char *name) {
   // Open file
   File *file = open_user_file(name, FILE_READ);
   if (file == NULL)
      return -1;

   // Buffer to store the data we read
   uint8_t buffer[4];

   // Check magic number
   if (read_file(file, buffer, 4)) {
      close_file(file);
      return -1;
   }
   if (memcmp(buffer, "\x1ASol", 4) != 0) {
      close_file(file);
      return -1;
   }

   // Check version
   if (read_file(file, buffer, 2)) {
      close_file(file);
      return -1;
   }
   uint16_t version = buffer[0] << 8 | buffer[1];
   if (version != 0x100) {
      close_file(file);
      return -1;
   }

   // Get level dimensions
   if (read_file(file, buffer, 4)) {
      close_file(file);
      return -1;
   }
   uint16_t width = buffer[0] << 8 | buffer[1];
   uint16_t height = buffer[2] << 8 | buffer[3];
   if (width < MIN_MAP_WIDTH || height < MIN_MAP_HEIGHT) {
      close_file(file);
      return -1;
   }

   // Make new map with those dimensions
   new_map(width, height);

   // Get player spawn position
   if (read_file(file, buffer, 4)) {
      close_file(file);
      return -1;
   }
   map.spawn_x = buffer[0] << 8 | buffer[1];
   map.spawn_y = buffer[2] << 8 | buffer[3];

   // Get level theme
   if (read_file(file, buffer, 1)) {
      close_file(file);
      return -1;
   }
   if (buffer[0] >= NUM_THEMES) {
      close_file(file);
      return -1;
   }
   map.theme = buffer[0];

   // Read tilemap
   for (unsigned y = 0; y < height; y++)
   for (unsigned x = 0; x < width; x++) {
      // Read this tile's data
      if (read_file(file, buffer, 3)) {
         close_file(file);
         return -1;
      }

      // Huh, invalid flags were set?
      if (buffer[2] & MAPFLAG_RESERVED) {
         close_file(file);
         return -1;
      }

      // Store the tile info in the map
      map.data[x][y].coll = buffer[0];
      map.data[x][y].obj = buffer[1];
      map.data[x][y].flags = buffer[2];
   }

   // Done!
   close_file(file);
   return 0;
}

//***************************************************************************
// save_map
// Saves the map into a file.
//---------------------------------------------------------------------------
// param name: name of file to write
// return: zero on success, non-zero on failure
//***************************************************************************

static int save_map(const char *name) {
   // Open file
   File *file = open_user_file(name, FILE_WRITE);
   if (file == NULL)
      return -1;

   // Quick buffer used to store the bytes we're going to write
   // Remember, maintaining endianess is important! :P
   uint8_t buffer[4];

   // Write magic number
   buffer[0] = 0x1A;
   buffer[1] = 'S';
   buffer[2] = 'o';
   buffer[3] = 'l';
   if (write_file(file, buffer, 4)) {
      close_file(file);
      return -1;
   }

   // Write version
   buffer[0] = 0x01;
   buffer[1] = 0x00;
   if (write_file(file, buffer, 2)) {
      close_file(file);
      return -1;
   }

   // Write level dimensions
   buffer[0] = map.width >> 8;
   buffer[1] = map.width;
   buffer[2] = map.height >> 8;
   buffer[3] = map.height;
   if (write_file(file, buffer, 4)) {
      close_file(file);
      return -1;
   }

   // Write spawn position
   buffer[0] = map.spawn_x >> 8;
   buffer[1] = map.spawn_x;
   buffer[2] = map.spawn_y >> 8;
   buffer[3] = map.spawn_y;
   if (write_file(file, buffer, 4)) {
      close_file(file);
      return -1;
   }

   // Write the level theme
   buffer[0] = map.theme;
   if (write_file(file, buffer, 1)) {
      close_file(file);
      return -1;
   }

   // Write all tiles into the file
   for (unsigned y = 0; y < map.height; y++)
   for (unsigned x = 0; x < map.width; x++) {
      buffer[0] = map.data[x][y].coll;
      buffer[1] = map.data[x][y].obj;
      buffer[2] = map.data[x][y].flags;
      if (write_file(file, buffer, 3)) {
         close_file(file);
         return -1;
      }
   }

   // Done!
   close_file(file);
   return 0;
}

//***************************************************************************
// unload_map [internal]
// Unloads the data for the current map.
//***************************************************************************

static void unload_map(void) {
   // Deallocate collision map
   if (map.data != NULL) {
      for (unsigned i = 0; i < map.width; i++)
         free(map.data[i]);
      free(map.data);
      map.data = NULL;
   }
}

//***************************************************************************
// push_undo [internal]
// Requests a slot in the undo stack for recording a new action.
//---------------------------------------------------------------------------
// return: pointer to undo stack slot
//***************************************************************************

static UndoStack *push_undo(void) {
   // Determine position of next available undo stack slot
   // (yes, I know this method will skip the first slot when it's first used,
   // but that doesn't matter since it's a ring buffer after all)
   undo_pos++;
   if (undo_pos == MAX_UNDO)
      undo_pos = 0;

   // Increase undo counter. If there were actions that could be redone,
   // they're lost forever now. If there are too many actions that could be
   // undone, the oldest one is lost forever too.
   redo_count = 0;
   if (undo_count < MAX_UNDO)
      undo_count++;

   // Return a pointer to the available slot we found
   return &undo_stack[undo_pos];
}

//***************************************************************************
// get_editor_map
// Returns a pointer to the editor map data. This pointer is *const*, so no,
// you can't modify the map outside the editor itself. Returns a NULL pointer
// if the map data is not present.
//---------------------------------------------------------------------------
// return: pointer to map data (or NULL if not loaded)
//***************************************************************************

const Map *get_editor_map(void) {
   if (map.data == NULL) return NULL;
   return &map;
}

//***************************************************************************
// show_messagebox
// Shows a message box.
//---------------------------------------------------------------------------
// param message: message to show.
//***************************************************************************

static void show_messagebox(const char *message) {
   // Make message box visible
   messagebox.visible = 1;

   // Add "press or click" line and store message for future use
   char *str = (char *) malloc(strlen(message) +
      strlen(text.editor_error.press_key) + 2);
   if (str == NULL)
      abort_program(ERR_NOMEMORY, NULL);
   sprintf(str, "%s\n%s", message, text.editor_error.press_key);
   messagebox.message = str;

   // Replace backslashes with newlines
   for (char *ptr = str; *ptr != '\0'; ptr++)
      if (*ptr == '\\') *ptr = '\n';

   // Determine proportions of the message
   int32_t width = calc_text_len(str);
   int32_t height = calc_text_height(str);

   // Assign message box coordinates
   width >>= 1;
   height >>= 1;
   messagebox.x1 = screen_cx - width - 16;
   messagebox.x2 = screen_cx + width + 16;
   messagebox.y1 = screen_cy - height - 8;
   messagebox.y2 = screen_cy + height + 8;
}

//***************************************************************************
// run_messagebox [internal]
// Executes the code for the message box.
//***************************************************************************

static void run_messagebox(void) {
   // Quit by pressing a button
   if (input.menu.accept || input.cursor.click) {
      messagebox.visible = 0;
      free(messagebox.message);
      messagebox.message = NULL;
   }
}

//***************************************************************************
// draw_messagebox [internal]
// Draws the message box.
//***************************************************************************

static void draw_messagebox(void) {
   // Always set the cursor as an arrow
   set_cursor(CURSOR_ARROW);

   // For the sake of making things more readable
   int32_t x1 = messagebox.x1;
   int32_t y1 = messagebox.y1;
   int32_t x2 = messagebox.x2;
   int32_t y2 = messagebox.y2;

   // Dim everything below us
   dim_screen(0x80);

   // Draw box itself
   fill_rectangle(x1, y1, x2, y2, 0x606060);
   draw_hline(x1, y1, x2-1, 0x808080);
   draw_vline(x1, y1, y2-1, 0x808080);
   draw_hline(x1+1, y2, x2, 0x404040);
   draw_vline(x2, y1+1, y2, 0x404040);
   draw_rectangle(x1-1, y1-1, x2+1, y2+1, 0x000000);

   // Draw error message
   draw_text(messagebox.message, screen_cx, screen_cy,
      FONT_LIT, ALIGN_CENTER);
   set_reader_text(messagebox.message);
}
