//***************************************************************************
// "bosses.c"
// Code for boss objects.
//---------------------------------------------------------------------------
// Sol engine
// Copyright ©2015, 2016 Azura Sun
//
// This file is part of Sol.
//
// Sol is free software: you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version.
//
// Sol is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Sol. If not, see <http://www.gnu.org/licenses/>.
//***************************************************************************

// Required headers
#include <stddef.h>
#include <stdint.h>
#include "main.h"
#include "bosses.h"
#include "ingame.h"
#include "editor.h"
#include "level.h"
#include "objects.h"
#include "physics.h"
#include "player.h"
#include "projectile.h"
#include "savegame.h"
#include "scene.h"
#include "settings.h"
#include "sound.h"
#include "tables.h"
#include "tally.h"
#include "video.h"

// Possible animations for a boss
typedef enum {
   BO_ANIM_1_IDLE,         // Bulldozer: idle
   BO_ANIM_1_DRIVE,        // Bulldozer: driving
   BO_ANIM_1_DEAD,         // Bulldozer: dead

   BO_ANIM_2_IDLE,         // Trash truck: idle
   BO_ANIM_2_DRIVE,        // Trash truck: driving
   BO_ANIM_2_SHOOT,        // Trash truck: shooting
   BO_ANIM_2_DEAD,         // Trash truck: dead

   BO_ANIM_3_BACK,         // Helicopter: going backwards
   BO_ANIM_3_CENTER,       // Helicopter: idling
   BO_ANIM_3_FRONT,        // Helicopter: going forwards
   BO_ANIM_3_DEAD,         // Helicopter: dead

   BO_ANIM_3G_DROP,        // Helicopter grenade: dropping
   BO_ANIM_3G_IDLE,        // Helicopter grenade: idle

   BO_ANIM_3S_SHIP_SIDE,   // Ship body (side)
   BO_ANIM_3S_SHIP_BODY,   // Ship body (middle)
   BO_ANIM_3S_SHIP_WINDOW, // Ship window
   BO_ANIM_3S_CABIN_SIDE,  // Ship cabin (side)
   BO_ANIM_3S_CABIN_BODY,  // Ship cabin (middle)

   BO_ANIM_4_IDLE,         // Oil truck: idle
   BO_ANIM_4_DRIVE,        // Oil truck: driving
   BO_ANIM_4_DEAD,         // Oil truck: dead
   BO_ANIM_4T_IDLE,        // Oil truck trailer: idle
   BO_ANIM_4T_DRIVE,       // Oil truck trailer: driving

   BO_ANIM_5_IDLE,         // Driller: idling
   BO_ANIM_5_DRILL,        // Driller: drilling
   BO_ANIM_5_JUMP,         // Driller: jump
   BO_ANIM_5_DEAD,         // Driller: dead

   BO_ANIM_6_BASE,         // Mr. Evil machine: base
   BO_ANIM_6_FLAME,        // Mr. Evil machine: flame
   BO_ANIM_6_LAUNCHERIDLE, // Mr. Evil machine: missile launcher idle
   BO_ANIM_6_LAUNCHERSHOOT,// Mr. Evil machine: missile launcher shooting
   BO_ANIM_6_HANDIDLE,     // Mr. Evil machine: hand idle
   BO_ANIM_6_HANDPUNCH1,   // Mr. Evil machine: hand punching (readying)
   BO_ANIM_6_HANDPUNCH2,   // Mr. Evil machine: hand punching (punching)
   BO_ANIM_6_HANDPUNCH3,   // Mr. Evil machine: hand punching (retiring)
   BO_ANIM_6_EVILIDLE,     // Mr. Evil machine: Mr. Evil idle
   BO_ANIM_6_EVILSHOOT,    // Mr. Evil machine: Mr. Evil shooting
   BO_ANIM_6_EVILPANIC,    // Mr. Evil machine: Mr. Evil panicking

   BO_ANIM_6M_IN,          // Incoming missiles
   BO_ANIM_6M_OUT,         // Outgoing missiles

   BO_ANIM_DRIVER,         // Fleeing driver
   NUM_BO_ANIM             // Number of animations
} BossAnim;

// Where boss graphics are stored
static GraphicsSet *gfxset_boss = NULL;
static AnimFrame *anim_boss[NUM_BO_ANIM];

// Names of each animation
static const char *anim_names[] = {
   "bulldozer_idle",
   "bulldozer_drive",
   "bulldozer_dead",

   "truck_idle",
   "truck_drive",
   "truck_shoot",
   "truck_dead",

   "helicopter_back",
   "helicopter_center",
   "helicopter_front",
   "helicopter_dead",

   "grenade_drop",
   "grenade_idle",

   "ship_side",
   "ship_body",
   "ship_window",
   "cabin_side",
   "cabin_body",

   "oil_truck_idle",
   "oil_truck_drive",
   "oil_truck_dead",
   "oil_trailer_idle",
   "oil_trailer_drive",

   "driller_idle",
   "driller_drill",
   "driller_jump",
   "driller_dead",

   "mrevil_base",
   "mrevil_flame",
   "mrevil_launcher_idle",
   "mrevil_launcher_shoot",
   "mrevil_hand_idle",
   "mrevil_hand_punch_1",
   "mrevil_hand_punch_2",
   "mrevil_hand_punch_3",
   "mrevil_evil_idle",
   "mrevil_evil_shoot",
   "mrevil_evil_panic",

   "missile_in",
   "missile_out",

   "driver"
};

// Private function prototypes
static void boss_common(Object *);

// Common parameters
#define BOSS_HITS_E     8        // Health of a boss in easy
#define BOSS_HITS_N     10       // Health of a boss in normal
#define BOSS_HITS_H     12       // Health of a boss in hard
#define BOSS_WEIGHT     0x40     // Weight of bosses

// Bulldozer boss parameters
//#define BOSS1_SPEED_E   0x300    // Speed when driving in easy
//#define BOSS1_SPEED_N   0x400    // Speed when driving in normal
//#define BOSS1_SPEED_H   0x500    // Speed when driving in hard
#define BOSS1_SPEED_E   0x500    // Maximum speed when driving in easy
#define BOSS1_SPEED_N   0x600    // Maximum speed when driving in normal
#define BOSS1_SPEED_H   0x700    // Maximum speed when driving in hard
#define BOSS1_STEP_E    0x40     // Speed change after every hit in easy
#define BOSS1_STEP_N    0x60     // Speed change after every hit in normal
#define BOSS1_STEP_H    0x80     // Speed change after every hit in hard
#define BOSS1_BOUNCE    0x300    // How much to bounce

// Trash truck boss parameters
#define BOSS2_RECOIL    6        // How long does the recoil last
#define BOSS2_DELAY_E   150      // Delay between shoots in easy
#define BOSS2_DELAY_N   120      // Delay between shoots in normal
#define BOSS2_DELAY_H   90       // Delay between shoots in hard

// Helicopter boos parameters
#define BOSS3_MAXSPEED  0x400    // How fast it can go
#define BOSS3_ACCEL_E   0x10     // How fast it accelerates in easy
#define BOSS3_ACCEL_N   0x18     // How fast it accelerates in normal
#define BOSS3_ACCEL_H   0x20     // How fast it accelerates in hard

// Oil truck boss parameters
#define BOSS4_ACCEL_E   0x08     // How fast it accelerates in easy
#define BOSS4_ACCEL_N   0x0C     // How fast it accelerates in normal
#define BOSS4_ACCEL_H   0x10     // How fast it accelerates in hard
#define BOSS4_SMOKE     0x200    // How fast the smoke rises
#define BOSS4_TRAILER   0x40     // Distance of trailer from truck

// Driller boss parameters
#define BOSS5_DELAY     30       // Delay between jumps
#define BOSS5_JUMP      0x700    // How far the boss jumps
#define BOSS5_SPEED     0x200    // How fast the boss moves when jumping
#define BOSS5_FACTOR    (7*8)    // Used to calculate the speed in hard

// Mr. Evil boss parameters
#define BOSS6_ACCELSPEED   0x10  // How quickly the machine accelerates
#define BOSS6_SPEEDCAP     0x200 // Maximum speed at which it can move
#define BOSS6_LAUNCHER_X   0x38  // X coordinate of missile launchers
#define BOSS6_LAUNCHER_Y   0x10  // Y coordinate of missile launchers
#define BOSS6_HAND_X       0x50  // Base X coordinate of hands
#define BOSS6_HAND_Y       0x10  // Base Y coordinate of hands

//***************************************************************************
// load_bosses
// Loads the bosses assets.
//***************************************************************************

void load_bosses(void) {
   // Load graphics
   gfxset_boss = load_graphics_set("graphics/bosses");

   // Get a list of all the animations we need
   for (size_t i = 0; i < NUM_BO_ANIM; i++)
      anim_boss[i] = get_anim(gfxset_boss, anim_names[i]);
}

//***************************************************************************
// unload_bosses
// Frees up the resources taken up by bosses assets.
//***************************************************************************

void unload_bosses(void) {
   // Unload graphics
   if (gfxset_boss) {
      destroy_graphics_set(gfxset_boss);
      gfxset_boss = NULL;
   }
}

//***************************************************************************
// init_boss_spawn
// Initializes the boss spawn point objects
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void init_boss_spawn(Object *obj) {
   // For the sake of physics (we want the player to only hurt us when
   // landing on top of us, OK?)
   obj->y -= 0x18;

   // Determine highest location for the collision box
   int32_t tile_x = obj->x >> TILE_SIZE_BIT;
   int32_t min_y = obj->y >> TILE_SIZE_BIT;
   for (;;) {
      if (min_y <= 0) break;
      LevelTile *tile = get_tile(tile_x, min_y);
      if (tile->collision != TILE_EMPTY && tile->collision != TILE_EMPTY_BG)
         break;
      min_y--;
   }

   // Determine lowest location for the collision box
   int32_t max_y = obj->y >> TILE_SIZE_BIT;
   int32_t limit_y = camera.limit_bottom >> TILE_SIZE_BIT;
   for (;;) {
      if (max_y >= limit_y) break;
      LevelTile *tile = get_tile(tile_x, max_y);
      if (tile->collision != TILE_EMPTY && tile->collision != TILE_EMPTY_BG)
         break;
      max_y++;
   }

   // Set up collision box
   set_hitbox(obj, -0x28, 0x27,
      (min_y << TILE_SIZE_BIT) - obj->y,
      (max_y << TILE_SIZE_BIT) + obj->y + TILE_SIZE-1);

   // Don't interact with player yet
   obj->health = 0;
}

//***************************************************************************
// run_boss_spawn
// Logic for boss spawn point objects
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void run_boss_spawn(Object *obj) {
   // Still not triggered?
   if (obj->timer == 0) {
      // Look for nearby players
      for (Object *other = get_first_object(OBJGROUP_PLAYER); other != NULL;
      other = other->next) {
         // Player overlapping our area?
         if (collision(obj, other)) {
            // Last boss spawns directly
            if (obj->type == OBJ_BOSS6_SPAWN) {
               add_object(OBJ_BOSS6, obj->x, obj->y + 0x100, obj->dir);
               play_ingame_bgm(BGM_FINALBOSS);
               delete_object(obj);
               return;
            }

            // Normal boss, proceed as usual
            obj->timer = 1;
            obj->x += obj->dir ? 0x100 : -0x100;
            set_hitbox(obj, -0x28, 0x27, -0x10, 0x27);

            // Change music but only if we aren't the helicopter boss, since
            // that one already had the music changed by the ship
            if (obj->type != OBJ_BOSS3_SPAWN)
               play_ingame_bgm(BGM_BOSS);
         }
      }

      // Create trailer if we're the oil truck and we're just spawning
      if (obj->timer && obj->type == OBJ_BOSS4_SPAWN) {
         obj->otherobj = add_object(OBJ_TRAILER,
            obj->x + (obj->dir ? -BOSS4_TRAILER : BOSS4_TRAILER),
            obj->y, obj->dir);
         obj->otherobj->otherobj = obj;
      }

      // Done for now
      return;
   }

   // Move in
   if (obj->timer <= 0x80) {
      // Determine which animation to use
      unsigned anim;
      switch (obj->type) {
         case OBJ_BOSS2_SPAWN: anim = BO_ANIM_2_DRIVE; break;
         case OBJ_BOSS3_SPAWN: anim = BO_ANIM_3_FRONT; break;
         case OBJ_BOSS4_SPAWN: anim = BO_ANIM_4_DRIVE; break;
         case OBJ_BOSS5_SPAWN: anim = BO_ANIM_5_DRILL; break;
         default:              anim = BO_ANIM_1_DRIVE; break;
      }
      set_object_anim(obj, anim_boss[anim]);

      // Walk into the scene
      // Not using real physics here so we can move through walls
      obj->x -= obj->dir ? 2 : -2;
   }

   // Wait a bit so player realizes what we are
   else {
      // At this point we already behave like the boss (logic aside), so
      // run the common boss stuff
      boss_common(obj);

      // Determine which animations to use
      unsigned anim1, anim2;
      switch (obj->type) {
         case OBJ_BOSS2_SPAWN:
            anim1 = BO_ANIM_2_IDLE;
            anim2 = BO_ANIM_2_DRIVE;
            break;

         case OBJ_BOSS3_SPAWN:
            anim1 = BO_ANIM_3_CENTER;
            anim2 = BO_ANIM_3_CENTER;
            break;

         case OBJ_BOSS4_SPAWN:
            anim1 = BO_ANIM_4_IDLE;
            anim2 = BO_ANIM_4_DRIVE;
            break;

         case OBJ_BOSS5_SPAWN:
            anim1 = BO_ANIM_5_IDLE;
            anim2 = BO_ANIM_5_DRILL;
            break;

         default:
            anim1 = BO_ANIM_1_IDLE;
            anim2 = BO_ANIM_1_DRIVE;
            break;
      }
      set_object_anim(obj, anim_boss[obj->timer < 0xC0 ? anim1 : anim2]);

      // Let boss fall as needed
      if (obj->type != OBJ_BOSS3_SPAWN)
         obj->gravity += BOSS_WEIGHT;
      apply_physics(obj);
   }

   // Update animation
   obj->timer++;

   // When we settle set the health so player can start interacting with us
   // (both attack us and get hurt by us)
   if (obj->timer == 0x80) {
      switch (get_difficulty()) {
         case DIFF_EASY: obj->health = BOSS_HITS_E; break;
         case DIFF_HARD: obj->health = BOSS_HITS_H; break;
         default:        obj->health = BOSS_HITS_N; break;
      }
   }

   // Turn into the boss after a while
   if (obj->timer == 0x100) {
      // Switch to the proper object type
      switch (obj->type) {
         case OBJ_BOSS1_SPAWN: obj->type = OBJ_BOSS1; break;
         case OBJ_BOSS2_SPAWN: obj->type = OBJ_BOSS2; break;
         case OBJ_BOSS3_SPAWN: obj->type = OBJ_BOSS3; break;
         case OBJ_BOSS4_SPAWN: obj->type = OBJ_BOSS4; break;
         case OBJ_BOSS5_SPAWN: obj->type = OBJ_BOSS5; break;
         default: obj->type = OBJ_DUMMY; break;
      }

      // Helicopter boss is too hard
      // Yes I know this location is odd but I need to ensure it's done only
      // once or otherwise this will underflow and result in a boss with a
      // ton of health
      if (obj->type == OBJ_BOSS3)
         obj->health -= 2;

      // Reset stuff bosses use for their state
      obj->timer = 0;
   }

   // If we're the oil truck keep trailer together with us
   if (obj->type == OBJ_BOSS4_SPAWN) {
      obj->otherobj->x = obj->x + (obj->dir ?
         BOSS4_TRAILER : -BOSS4_TRAILER);
      obj->otherobj->y = obj->y;
      refresh_hitbox(obj->otherobj);
   }
}

//***************************************************************************
// run_boss1
// Logic for the bulldozer boss object
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void run_boss1(Object *obj) {
   // Do boss stuff
   boss_common(obj);
   if (obj->dead) {
      set_object_anim(obj, anim_boss[BO_ANIM_1_DEAD]);
      return;
   }

   // Determine speed at which we're going
   if (obj->jumping)
      obj->speed = 0;
   else {
      switch (get_difficulty()) {
         case DIFF_EASY:
            obj->speed = BOSS1_SPEED_E;
            obj->speed -= BOSS1_STEP_E * obj->health;
            break;
         case DIFF_HARD:
            obj->speed = BOSS1_SPEED_H;
            obj->speed -= BOSS1_STEP_H * obj->health;
            break;
         default:
            obj->speed = BOSS1_SPEED_N;
            obj->speed -= BOSS1_STEP_N * obj->health;
            break;
      }
      if (obj->dir)
         obj->speed = -obj->speed;
   }

   // Get physics going
   obj->gravity += BOSS_WEIGHT;
   apply_physics(obj);

   // Bounce when hitting a wall
   if (!obj->jumping && obj->on_wall) {
      obj->jumping = 1;
      obj->on_ground = 0;
      obj->gravity = -BOSS1_BOUNCE;
      play_2d_sfx(SFX_CRASH, obj->x, obj->y);
      apply_quake(0x10);
   }

   // Turn around after we bounced
   if (obj->jumping && obj->on_ground) {
      obj->jumping = 0;
      obj->dir ^= 1;
   }

   // Determine animation to show
   set_object_anim(obj, anim_boss[obj->jumping ?
      BO_ANIM_1_IDLE : BO_ANIM_1_DRIVE]);
}

//***************************************************************************
// run_boss2
// Logic for the trash truck boss object
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void run_boss2(Object *obj) {
   // Do boss stuff
   boss_common(obj);
   if (obj->dead) {
      set_object_anim(obj, anim_boss[BO_ANIM_2_DEAD]);
      return;
   }

   // Shoot!
   if (obj->timer == 0) {
      // Spawn a barrel (or whatever is the bouncing hazard in the current
      // level, in the case of mods)
      Object *other = add_object(OBJ_BOUNCINGHAZARD,
         obj->x + (obj->dir ? -0x10 : 0x10), obj->y - 0x10,
         obj->dir);
      other->speed += (get_random(0x80) + 0x80) * (obj->dir ? -1 : 1);
      other->gravity = -get_random(0x200);

      // Make noise!
      play_2d_sfx(SFX_SHOOT, obj->x, obj->y);
   }

   // Update timer
   obj->timer++;

   // Time to shoot again?
   unsigned delay;
   switch (get_difficulty()) {
      case DIFF_EASY: delay = BOSS2_DELAY_E; break;
      case DIFF_HARD: delay = BOSS2_DELAY_H; break;
      default:        delay = BOSS2_DELAY_N; break;
   }
   if (obj->timer == delay)
      obj->timer = 0;

   // Get physics going
   obj->gravity += BOSS_WEIGHT;
   apply_physics(obj);

   // Determine animation to show
   set_object_anim(obj, anim_boss[obj->timer < BOSS2_RECOIL ?
      BO_ANIM_2_SHOOT : BO_ANIM_2_IDLE]);
}

//***************************************************************************
// run_boss3
// Logic for the helicopter boss object
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void run_boss3(Object *obj) {
   // Do boss stuff
   boss_common(obj);
   if (obj->dead) {
      set_object_anim(obj, anim_boss[BO_ANIM_3_DEAD]);
      return;
   }

   // Determine how fast the helicopter accelerates
   int32_t accel;
   switch (get_difficulty()) {
      case DIFF_EASY: accel = BOSS3_ACCEL_E; break;
      case DIFF_HARD: accel = BOSS3_ACCEL_H; break;
      default:        accel = BOSS3_ACCEL_N; break;
   }

   // Find a player to chase
   Object *player = get_first_object(OBJGROUP_PLAYER);

   // Check if we should flee away from the player or go for it
   // Used to prevent the player from just pogoing on the boss
   if (player == NULL)
      obj->crouching = 0;
   else if (obj->invulnerability)
      obj->crouching = 1;
   else if (player->y > obj->y)
      obj->crouching = 0;

   // This used to be a more complex condition to determine if we should
   // panic away from the player, but in practice it just made things harder
   // so we settled with the condition above... This sticks for the sake
   // of not rewriting.
   int unfair = /*player->type == OBJ_PLAYERWINGS && player->y < obj->y &&
      (player->active || !player->jumping)*/ obj->crouching;

   // Chase the player
   if (player == NULL || player->dead) {
      obj->speed += obj->dir ? accel : accel;
   } else if ((obj->x > player->x && !unfair) ||
              (obj->x < player->x && unfair))
      obj->speed -= accel;
   else if ((obj->x < player->x && !unfair) ||
            (obj->x > player->x && unfair))
      obj->speed += accel;

   // Don't move too fast!
   if (obj->speed > BOSS3_MAXSPEED)
      obj->speed = BOSS3_MAXSPEED;
   if (obj->speed < -BOSS3_MAXSPEED)
      obj->speed = -BOSS3_MAXSPEED;

   // Make helicopter move
   apply_physics(obj);

   // Throw a grenade every so often
   obj->timer++;
   if (obj->timer == 60) {
      obj->timer = 0;
      play_2d_sfx(SFX_DROP, obj->x, obj->y);
      add_object(OBJ_GRENADE, obj->x, obj->y + 0x20, 0);
   }

   // Determine animation to show
   int32_t show_speed = obj->speed;
   if (obj->dir) show_speed = -show_speed;
   if (show_speed >= 0x40)
      set_object_anim(obj, anim_boss[BO_ANIM_3_FRONT]);
   else if (show_speed <= -0x40)
      set_object_anim(obj, anim_boss[BO_ANIM_3_BACK]);
   else
      set_object_anim(obj, anim_boss[BO_ANIM_3_CENTER]);
}

//***************************************************************************
// run_boss4
// Logic for the oil truck boss object
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void run_boss4(Object *obj) {
   // Place trailer where it belongs
   obj->otherobj->x = obj->x + (obj->dir ? BOSS4_TRAILER : -BOSS4_TRAILER);
   obj->otherobj->y = obj->y;
   refresh_hitbox(obj->otherobj);

   // Do boss stuff
   boss_common(obj);
   if (obj->dead) {
      set_object_anim(obj, anim_boss[BO_ANIM_4_DEAD]);
      if (!obj->current) delete_object(obj->otherobj);
      return;
   }

   // Determine how fast we accelerate
   int32_t accel;
   switch (get_difficulty()) {
      case DIFF_EASY: accel = BOSS4_ACCEL_E; break;
      case DIFF_HARD: accel = BOSS4_ACCEL_H; break;
      default:        accel = BOSS4_ACCEL_N; break;
   }

   // Accelerate towards the player
   Object *player = get_first_object(OBJGROUP_PLAYER);
   if (player == NULL)
      obj->speed += accel;
   else if (obj->x <= player->x)
      obj->speed += accel;
   else
      obj->speed -= accel;

   // Get physics going
   int32_t old_speed = obj->speed;
   obj->gravity += BOSS_WEIGHT;
   apply_physics(obj);

   // Crashed against a wall?
   if (obj->on_wall) {
      // Just slow down
      obj->speed = old_speed >> 1;

      // Break stuff around
      if (old_speed > 0) {
         break_stuff(obj->abs_hitbox.x2 + 1, obj->y - 0x38);
         break_stuff(obj->abs_hitbox.x2 + 1, obj->y - 0x18);
         break_stuff(obj->abs_hitbox.x2 + 1, obj->y + 0x08);
      } else if (old_speed < 0) {
         break_stuff(obj->abs_hitbox.x1 - 1, obj->y - 0x38);
         break_stuff(obj->abs_hitbox.x1 - 1, obj->y - 0x18);
         break_stuff(obj->abs_hitbox.x1 - 1, obj->y + 0x08);
      }

      // Make noise!
      play_sfx(SFX_BREAK);

      // Rumble rumble
      apply_quake(0x10);
   }

   // Release toxic gas
   obj->timer++;
   if (obj->timer == 4) {
      Object *other = add_object(OBJ_TOXICGAS, obj->x, obj->y - 0x18, 0);
      other->gravity = -BOSS4_SMOKE;
      obj->timer = 0;
   }

   // Determine animation to show
   set_object_anim(obj, anim_boss[BO_ANIM_4_DRIVE]);
}

//***************************************************************************
// run_boss5
// Logic for the driller boss object
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void run_boss5(Object *obj) {
   // Do boss stuff
   boss_common(obj);
   if (obj->dead) {
      set_object_anim(obj, anim_boss[BO_ANIM_5_DEAD]);
      return;
   }

   // Get the main player for the sake of aiming later
   Object *player = get_first_object(OBJGROUP_PLAYER);

   // Already jumping?
   if (obj->jumping) {
      // Landed?
      if (obj->on_ground) {
         obj->jumping = 0;
         obj->timer = 0;
      }
   }

   // Nope, drilling yet
   else {
      // Stop moving!
      obj->speed = 0;

      // Jump now?
      obj->timer++;
      if (obj->timer == BOSS5_DELAY) {
         // Break stuff under us!
         break_stuff(obj->x - 0x28, obj->y + 0x30);
         break_stuff(obj->x - 0x10, obj->y + 0x30);
         break_stuff(obj->x + 0x10, obj->y + 0x30);
         break_stuff(obj->x + 0x27, obj->y + 0x30);

         // Make boss jump!
         obj->jumping = 1;
         obj->on_ground = 0;
         obj->gravity = -BOSS5_JUMP;

         // Determine which direction to go... depending on difficulty
         switch (get_difficulty()) {
            // Easy: ...just jump forwards
            case DIFF_EASY:
               // Move forwards
               obj->speed = obj->dir ? -BOSS5_SPEED : BOSS5_SPEED;
               break;

            // Hard: attempt to accurately land on the player
            case DIFF_HARD:
               // We need a player for this...
               if (player != NULL) {
                  // Face where the player is
                  obj->dir = obj->x > player->x ? 1 : 0;

                  // Calculate the speed we need so that we land we end up
                  // right over the player (assuming we're at the same
                  // height, at least)
                  obj->speed = (player->x - obj->x) * 0x100 / BOSS5_FACTOR;
               }
               break;

            // Normal: just go towards the player direction
            default:
               // Look towards player, if possible
               if (player != NULL)
                  obj->dir = obj->x > player->x ? 1 : 0;

               // Move forwards
               obj->speed = obj->dir ? -BOSS5_SPEED : BOSS5_SPEED;
               break;
         }
      }
   }

   // Get physics going
   // The speed hack is so the boss can't get stuck if it hits a wall (by
   // drilling itself into a deep hole, basically)
   int32_t temp_speed = obj->speed;
   obj->gravity += BOSS_WEIGHT;
   apply_physics(obj);
   if (!obj->on_ground && obj->on_wall) {
      obj->speed = -temp_speed;
      obj->dir ^= 1;
   }

   // Determine animation to show
   set_object_anim(obj, anim_boss[BO_ANIM_5_DRILL]);
}

//***************************************************************************
// boss_common [internal]
// Functionality common to all bosses
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

static void boss_common(Object *obj) {
   // Exploding?
   if (obj->dead) {
      // Spawn explosions every so often
      if ((obj->timer & 3) == 0) {
         Object *explosion = add_object(OBJ_SMOKE,
            obj->abs_hitbox.x1 + get_random(
            obj->abs_hitbox.x2 - obj->abs_hitbox.x1),
            obj->abs_hitbox.y1 + get_random(
            obj->abs_hitbox.y2 - obj->abs_hitbox.y1), 0);
         set_object_anim(explosion,
            retrieve_object_anim(OB_ANIM_EXPLOSION));
      }
      if ((obj->timer & 7) == 0)
         play_2d_sfx(SFX_EXPLOSION, obj->x, obj->y);
      obj->timer++;

      // ...and go boom after a while :D
      if (obj->timer == 240) {
         // Calculate center point
         int32_t x = (obj->abs_hitbox.x1 + obj->abs_hitbox.x2) >> 1;
         int32_t y = (obj->abs_hitbox.y1 + obj->abs_hitbox.y2) >> 1;

         // Cause a huge explosion
         play_2d_sfx(SFX_BIGEXPLOSION, obj->x, obj->y);
         Object *explosion = add_object(OBJ_SMOKE, x, y, 0);
         set_object_anim(explosion,
            retrieve_object_anim(OB_ANIM_BIGEXPLOSION));

         // Throw scraps everywhere
         for (unsigned i = 0; i < 0x20; i++) {
            int32_t speed = -0x300 + get_random(0x600);
            int32_t gravity = -0x600 + get_random(0x600);
            Object *scrap = add_object(OBJ_SCRAPGEAR + get_random(1),
               x, y, speed < 0 ? 1 : 0);
            scrap->speed = speed;
            scrap->gravity = gravity;
         }

         // Shake the screen!
         apply_quake(0x20);

         // Signal the end of the level
         finish_level();

         // Boss is done
         delete_object(obj);
      }
   }

   // Invulnerable? (can't hurt player nor get hurt by the player)
   if (obj->invulnerability) {
      obj->invulnerability--;
      return;
   }
}

//***************************************************************************
// hurt_boss
// Hurts a boss object.
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void hurt_boss(Object *obj) {
   // Reduce how many hits are left
   obj->health--;

   // Uh oh, dead?
   // (timer is reset for the explosions)
   if (obj->health == 0) {
      obj->invulnerability = 0;
      obj->dead = 1;
      obj->timer = 0;

      // Make driver flee!
      add_object(OBJ_DRIVER, obj->x + (obj->dir ? -28 : 28),
         obj->y, obj->dir);

      // Add bonus while we're at it
      add_enemy_bonus(1000);

      // No need to be angry anymore :)
      reset_anger();

      // Update save if needed
      if (get_editor_map() == NULL)
         advance_savegame_scene();
   }

   // OK, no, give some grace period where the player and the boss can't
   // hurt each other then
   else {
      play_sfx(SFX_BREAK);
      obj->invulnerability = 60;
   }
}

//***************************************************************************
// init_ship
// Initializes the ship where the boss 3 fight happens
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void init_ship(Object *obj) {
   // Not feeling like even dealing with this...
   obj->dir = 0;

   // For the sake of our sanity
   obj->base_x = obj->x -= 80-32;
   obj->base_y = obj->y -= 112-32;

   // Think this hitbox is ridiculous? Remember what this thing is!
   set_hitbox(obj, 16, 0xFFFF, 16, 191);

   // Create initial chunk for the ship
   Object *piece = add_object(OBJ_SHIPHICHUNK, 0, 0, 0);
   piece = add_object(OBJ_SHIPHICHUNK, 0, 0, 0);
   piece->base_x = 0;
   piece->base_y = 0;
   set_object_anim(piece, anim_boss[BO_ANIM_3S_SHIP_SIDE]);
   obj->otherobj = piece;

   // Create the rest of the ship body
   Object *parent = obj->otherobj;
   for (unsigned i = 0; i < 0x40; i++) {
      piece = add_object(OBJ_SHIPHICHUNK, 0, 0, 0);
      piece->base_x = 0x80 + (i << 6);
      piece->base_y = i ? 1 : 0;
      set_object_anim(piece, anim_boss[BO_ANIM_3S_SHIP_BODY]);
      parent->otherobj = piece;
      parent = piece;
   }

   // Create the windows on the ship body
   for (unsigned i = 0; i < 0x20; i++) {
      piece = add_object(OBJ_SHIPHICHUNK, 0, 0, 0);
      piece->base_x = 0xE0 + (i << 7);
      piece->base_y = 0x40;
      set_object_anim(piece, anim_boss[BO_ANIM_3S_SHIP_WINDOW]);
      parent->otherobj = piece;
      parent = piece;
   }

   // Create the side of the cabin
   piece = add_object(OBJ_SHIPLOCHUNK, 0, 0, 0);
   piece->base_x = 0xC0;
   piece->base_y = -0x70;
      set_object_anim(piece, anim_boss[BO_ANIM_3S_CABIN_SIDE]);
   parent->otherobj = piece;
   parent = piece;

   // Create the rest of the cabin body
   for (unsigned i = 0; i < 0x20; i++) {
      piece = add_object(OBJ_SHIPLOCHUNK, 0, 0, 0);
      piece->base_x = 0x100 + (i << 7);
      piece->base_y = -0x70;
      set_object_anim(piece, anim_boss[BO_ANIM_3S_CABIN_BODY]);
      parent->otherobj = piece;
      parent = piece;
   }

   // Get all the pieces in place
   run_ship(obj);
}

//***************************************************************************
// run_ship
// Logic for the ship where the boss 3 fight happens
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void run_ship(Object *obj) {
   // Make ship wobble
   int32_t old_y = obj->y;
   int32_t wobble = sines[game_anim << 1 & 0xFF] >> 4;
   obj->y = obj->base_y + wobble;
   obj->gravity = (obj->y - old_y) << 8;

   // Ready to sail?
   if (!obj->active) {
      // Check if all players are onboard
      int ready = 1;
      for (Object *player = get_first_object(OBJGROUP_PLAYER);
      player != NULL; player = player->next) {
         if (player->x < obj->x + 0x20 || !player->on_ground) {
            ready = 0;
            break;
         }
      }

      // Can we sail now?
      if (ready && get_first_object(OBJGROUP_PLAYER)) {
         obj->active = 1;
         play_ingame_bgm(BGM_BOSS);
         add_object(OBJ_BOSS3_SPAWN, obj->x + 0x20*15, obj->y - 0x20*3, 1);
      }
   }

   // Already sailed?
   else {
      // Create water every so often
      if (obj->x % TILE_SIZE == 0) {
         Object *water = add_object(OBJ_LIQUIDHAZARD,
            camera.limit_right + 0x10, camera.limit_bottom - 0x0F, 0);
         water->timer = 3600;
      }

      // Moooove onwaaards!
      obj->x++;

      // Carry all objects with us (except water, d'oh)
      // Ensure helicopter is always carried or it'll change height!
      // (it took up to 1.3 for me to notice this...)
      for (ObjGroup group = 0; group < NUM_OBJGROUPS; group++) {
         for (Object *other = get_first_object(group); other != NULL;
         other = other->next) {
            if (other == obj)
               continue;
            if (other->type == OBJ_LIQUIDHAZARD)
               continue;
            if (other->x >= obj->x || other->type == OBJ_BOSS3) {
               other->x++;
               other->y += obj->y - old_y;
            }
         }
      }

      // Make room for moving around
      camera.limit_right++;
   }

   // Update the position of all the ship pieces
   for (Object *piece = obj->otherobj; piece != NULL;
   piece = piece->otherobj) {
      piece->x = obj->x + piece->base_x;
      piece->y = obj->y + piece->base_y;
   }
}

//***************************************************************************
// init_grenade
// Initializes the grenade object
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void init_grenade(Object *obj) {
   // Set collision box
   set_hitbox(obj, -0x10, 0x0F, -0x10, 0x0F);
}

//***************************************************************************
// run_grenade
// Logic for the grenade object
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void run_grenade(Object *obj) {
   // Fall
   obj->gravity += 0x80;
   apply_physics(obj);

   // Explode after a while
   if (obj->on_ground) {
      // Wait a second...
      obj->timer++;
      if (obj->timer == 60) {
         // Temporarily change our hitbox to reflect the recoil area
         set_hitbox(obj, -0x18, 0x18, -0x80, 0x0F);
         refresh_hitbox(obj);

         // If there's any player above us, make it recoil!
         for (Object *player = get_first_object(OBJGROUP_PLAYER);
         player != NULL; player = player->next) {
            // Er...
            if (player->dead)
               continue;

            // Colliding with it?
            if (!collision(obj, player))
               continue;

            // Recoil!
            player->jumping = 1;
            player->gravity = -0x800;
         }

         // Scatter pieces around to hint about the recoil
         for (unsigned i = 0; i < 0x10; i++) {
            // Determine speed
            int32_t speed = -0x80 + get_random(0x100);
            int32_t gravity = -0x800 + get_random(0x400);

            // Create object
            Object *other = add_object(get_random(1) ? OBJ_SCRAPGEAR :
               OBJ_SCRAPSPRING, obj->x, obj->y, speed >= 0 ? 1 : 0);
            other->speed = speed;
            other->gravity = gravity;
         }

         // Kaboom!
         add_object(OBJ_BIGEXPLOSION, obj->x, obj->y, 0);
         delete_object(obj);
         return;
      }
   }

   // Set animation
   set_object_anim(obj, obj->on_ground ?
      anim_boss[BO_ANIM_3G_IDLE] :
      anim_boss[BO_ANIM_3G_DROP]);
}

//***************************************************************************
// init_boss4_trailer
// Initializes the trailer of the oil truck boss object
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void init_boss4_trailer(Object *obj) {
   // Set collision box
   set_hitbox(obj, -0x40, 0x3F, -0x18, 0x1F);
}

//***************************************************************************
// run_boss4_trailer
// Logic for the trailer of the oil truck boss object
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void run_boss4_trailer(Object *obj) {
   // Check if some player has been harmed by us
   for (Object *player = get_first_object(OBJGROUP_PLAYER);
   player != NULL; player = player->next) {
      // Hurt the player if it has touched us
      if (collision(obj, player))
         hurt_player(player, obj->x);
   }

   // Exploding?
   if (obj->timer) {
      // Create explosion every so often
      if ((obj->timer & 7) == 0) {
         // Determine how far can the explosions spawn
         int32_t range_x = (obj->abs_hitbox.x2 - obj->abs_hitbox.x1) >> 1;
         int32_t range_y = (obj->abs_hitbox.y2 - obj->abs_hitbox.y1) >> 1;

         // Create an explosion
         add_object(OBJ_BIGEXPLOSION,
            obj->x + get_random(range_x) - (range_x >> 1),
            obj->y + get_random(range_y) - (range_x >> 1) + 0x10,
            0);
      }

      // Kill the boss after a while
      if (obj->timer == 0x20 && obj->otherobj != NULL
      && !obj->otherobj->dead) {
         obj->otherobj->health = 1;
         hurt_boss(obj->otherobj);
      }

      // Keep going
      obj->timer++;
   }

   // Check if we touched some fire
   else for (Object *other = get_first_object(OBJGROUP_HAZARD);
   other != NULL; other = other->next) {
      if (other->type == OBJ_FIRE && collision(obj, other)) {
         // Start explosions
         obj->timer++;

         // Make a huge explosion in the middle
         add_object(OBJ_BIGEXPLOSION, obj->x, obj->y + 0x10, 0);

         // OK, exploding already, done
         break;
      }
   }

   // Determine animation to show
   if (obj->otherobj)
      set_object_anim(obj, anim_boss[obj->otherobj->type == OBJ_BOSS4 ?
      BO_ANIM_4T_DRIVE : BO_ANIM_4T_IDLE]);
   else
      set_object_anim(obj, anim_boss[BO_ANIM_4T_IDLE]);
}

//***************************************************************************
// init_boss6
// Initialization for the Mr. Evil machine boss
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void init_boss6(Object *obj) {
   // Make machine rise
   obj->timer = 0x24;
   obj->jumping = 1;
   obj->gravity = -BOSS6_SPEEDCAP;

   // We're going to create lots of objects for the machine and we need to
   // keep track of them somehow
   Object *other;

   // Create missile launchers
   other = add_object(OBJ_B6_LAUNCHER, 0, 0, 0);
   other->otherobj = obj;
   run_mrevil_launcher(other);

   other = add_object(OBJ_B6_LAUNCHER, 0, 0, 1);
   other->otherobj = obj;
   run_mrevil_launcher(other);

   // Create flame coming out of machine
   other = add_object(OBJ_B6_FLAME, 0, 0, 0);
   other->otherobj = obj;
   run_mrevil_flame(other);

   // Create base of machine
   other = add_object(OBJ_B6_BASE, 0, 0, 0);
   other->otherobj = obj;
   run_mrevil_base(other);

   // Create Mr. Evil
   other = add_object(OBJ_B6_EVIL, 0, 0, 0);
   other->otherobj = obj;
   run_mrevil_evil(other);

   // Create hands
   other = add_object(OBJ_B6_HAND, 0, 0, 0);
   other->otherobj = obj;
   run_mrevil_hand(other);

   other = add_object(OBJ_B6_HAND, 0, 0, 1);
   other->otherobj = obj;
   run_mrevil_hand(other);

   // Collision box used when punching
   set_hitbox(obj, -48, 47, -32, 31);
}

//***************************************************************************
// run_boss6
// Logic for the Mr. Evil machine boss
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void run_boss6(Object *obj) {
   // Are we finished?
   if (obj->dead) {
      // Make place rumble since it's exploding
      apply_quake(0x20);

      // Cause explosions
      /*if ((obj->timer & 0x01) == 0)*/ {
         Object *player = get_first_object(OBJGROUP_PLAYER);
         if (player == NULL) player = obj;
         add_object(OBJ_ENDEXPLOSION, player->x - 192 + get_random(384),
            player->y - 120 + get_random(240), 0);
      }
      if ((obj->timer & 0x0F) == 0) {
         play_sfx(SFX_EXPLOSION);
      }

      // Finish level?
      //if (obj->timer > 0) {
         obj->timer--;
         if (obj->timer == 0)
            finish_level();
      //}

      // Do nothing
      return;
   }

   // Make machine rise
   if (obj->jumping) {
      obj->y -= 0x08;
      obj->base_y -= 0x08;
      obj->timer--;
      if (obj->timer == 0)
         obj->jumping = 0;
      return;
   }

   // Launching missiles?
   if (!obj->active) {
      // Chase player
      Object *player = get_first_object(OBJGROUP_PLAYER);
      if (player != NULL) {
         // Determine where to go
         if (obj->x < player->x) obj->speed += BOSS6_ACCELSPEED;
         if (obj->x > player->x) obj->speed -= BOSS6_ACCELSPEED;
         if (obj->y < player->y) obj->gravity += BOSS6_ACCELSPEED;
         if (obj->y > player->y) obj->gravity -= BOSS6_ACCELSPEED;

         // Speed cap
         if (obj->speed < -BOSS6_SPEEDCAP) obj->speed = -BOSS6_SPEEDCAP;
         if (obj->speed > BOSS6_SPEEDCAP) obj->speed = BOSS6_SPEEDCAP;
         if (obj->gravity < -BOSS6_SPEEDCAP) obj->gravity = -BOSS6_SPEEDCAP;
         if (obj->gravity > BOSS6_SPEEDCAP) obj->gravity = BOSS6_SPEEDCAP;

         // Move
         obj->x += speed_to_int(obj->speed);
         obj->y += speed_to_int(obj->gravity);
      }

      // Launch every so often
      obj->timer++;
      if ((obj->timer & 0x3F) == 0) {
         add_object(OBJ_MISSILE_IN,
            obj->x - BOSS6_LAUNCHER_X,
            obj->y - BOSS6_LAUNCHER_Y, 0);
      } else if ((obj->timer & 0x3F) == 0x20) {
         add_object(OBJ_MISSILE_IN,
            obj->x + BOSS6_LAUNCHER_X,
            obj->y - BOSS6_LAUNCHER_Y, 0);
      }

      // Launched enough?
      if (obj->timer == 0x11F) {
         obj->active = 1;
         obj->timer = 0;
      }
   }

   // Punching?
   else {
      // Starting punch, determine where we'll go
      if (obj->timer == 0) {
         // Default in case there isn't any player
         obj->speed = 0;
         obj->gravity = 0;

         // Look for the player
         Object *player = get_first_object(OBJGROUP_PLAYER);
         if (player != NULL && !player->dead) {
            // Determine where the player is relative to us
            int32_t dist_x = player->x - obj->x;
            int32_t dist_y = (player->y - 24) - obj->y;

            // Calculate speed when we start moving
            obj->speed = (dist_x << 8) / 12;
            obj->gravity = (dist_y << 8) / 12;
         }
      }

      // Punch!
      if (obj->timer >= 0x20 && obj->timer < 0x2C) {
         obj->x += speed_to_int(obj->speed);
         obj->y += speed_to_int(obj->gravity);
      }

      // Impact time?
      if (obj->timer == 0x2C) {
         refresh_hitbox(obj);

         // Hurt players
         for (Object *player = get_first_object(OBJGROUP_PLAYER);
         player != NULL; player = player->next) {
            if (collision(obj, player))
               hurt_player(player, obj->x);
         }

         // Destroy generators
         for (Object *other = get_first_object(OBJGROUP_SCENERY);
         other != NULL; other = other->next) {
            // Make sure we're hitting this object
            if (!collision(obj, other))
               continue;

            // Normal generator?
            if (other->type == OBJ_GENERATOR) {
               other->dead = 1;
            }

            // Main generator?
            if (other->type == OBJ_MAINGENERATOR) {
               // Destroy it
               other->dead = 1;

               // Start ending sequence
               obj->dead = 1;
               obj->timer = 180;

               // Add bonus
               add_enemy_bonus(10000);

               // Update save if needed
               if (get_editor_map() == NULL)
                  advance_savegame_scene();
            }
         }
      }

      // Keep going
      obj->timer++;

      // Done?
      if (obj->timer == 0x40) {
         obj->timer = 0;
         obj->active = 0;
         obj->dir ^= 1;
      }
   }
}

//***************************************************************************
// init_mrevil_base
// Initialization for base of the Mr. Evil machine boss
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void init_mrevil_base(Object *obj) {
   // Set sprite
   set_object_anim(obj, anim_boss[BO_ANIM_6_BASE]);
}

//***************************************************************************
// run_mrevil_base
// Logic for the base of the Mr. Evil machine boss
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void run_mrevil_base(Object *obj) {
   // Set our coordinates same as our parent
   obj->x = obj->otherobj->x;
   obj->y = obj->otherobj->y;
}

//***************************************************************************
// init_mrevil_flame
// Initialization for the flame coming out of Mr. Evil machine boss
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void init_mrevil_flame(Object *obj) {
   // Set sprite
   set_object_anim(obj, anim_boss[BO_ANIM_6_FLAME]);
}

//***************************************************************************
// run_mrevil_flame
// Logic for the flame coming out of Mr. Evil machine boss
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void run_mrevil_flame(Object *obj) {
   // Set our coordinates same as our parent, but at the engine location
   obj->x = obj->otherobj->x;
   obj->y = obj->otherobj->y + 56;
}

//***************************************************************************
// run_mrevil_launcher
// Logic for the missile launchers of the Mr. Evil machine boss
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void run_mrevil_launcher(Object *obj) {
   // Set our coordinates
   obj->x = obj->otherobj->x + (obj->dir ?
      BOSS6_LAUNCHER_X : -BOSS6_LAUNCHER_X);
   obj->y = obj->otherobj->y - BOSS6_LAUNCHER_Y;

   // Determine whether this launcher is active
   unsigned active;
   if (obj->otherobj->active)
      active = 0;
   else {
      uint32_t timer = obj->otherobj->timer;
      if (obj->dir) timer -= 0x20;
      active = (timer & 0x3F) >= 0x30;
   }

   // Set our sprite
   set_object_anim(obj, anim_boss[active ?
      BO_ANIM_6_LAUNCHERIDLE : BO_ANIM_6_LAUNCHERSHOOT]);
}

//***************************************************************************
// run_mrevil_hand
// Logic for the hands of the Mr. Evil machine boss
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void run_mrevil_hand(Object *obj) {
   // DO NOTHING WHEN LEVEL IS OVER
   // Easier than checking and adjusting accordingly
   if (obj->otherobj->dead)
      return;

   // Set our coordinates
   obj->x = obj->otherobj->x + (obj->dir ? BOSS6_HAND_X : -BOSS6_HAND_X);
   obj->y = obj->otherobj->y + BOSS6_HAND_Y;

   // Punching?
   if (obj->otherobj->active) {
      // Punching with this hand?
      if (obj->otherobj->dir == obj->dir) {
         // Getting ready?
         if (obj->otherobj->timer < 0x20) {
            // Calculate our position
            if (obj->dir)
               obj->x += obj->otherobj->timer >> 2;
            else
               obj->x -= obj->otherobj->timer >> 2;

            // Set animation
            set_object_anim(obj, anim_boss[BO_ANIM_6_HANDPUNCH1]);
         }

         // Punching?
         if (obj->otherobj->timer >= 0x20 &&
         obj->otherobj->timer < 0x30) {
            // We need to continue from where we left
            obj->x += obj->dir ? 0x08 : -0x08;

            // Calculate where we're in the animation
            int32_t pos = obj->otherobj->timer - 0x20;
            if (pos > 12) pos = 12;

            // Calculate position
            int32_t dist_x = obj->otherobj->x - obj->x;
            obj->x += dist_x * pos / 12;

            // Set animation
            set_object_anim(obj, anim_boss[BO_ANIM_6_HANDPUNCH2]);
         }

         // Retiring?
         if (obj->otherobj->timer >= 0x30) {
            // Calculate where we're in the animation
            int32_t pos = obj->otherobj->timer - 0x30;
            if (pos > 0x10) pos = 0x10;
            pos = 0x0F - pos;

            // Calculate position
            int32_t dist_x = obj->otherobj->x - obj->x;
            obj->x += dist_x * pos / 0x10;

            // Set animation
            set_object_anim(obj, anim_boss[BO_ANIM_6_HANDPUNCH3]);
         }
      }
   }

   // Idle
   else {
      // Make hands wobble a bit
      if (obj->dir) {
         obj->x += cosines[obj->timer << 1 & 0xFF] >> 6;
         obj->y -= sines[obj->timer << 2 & 0xFF] >> 6;
      } else {
         obj->x += cosines[obj->timer << 1 & 0xFF] >> 6;
         obj->y += sines[obj->timer << 2 & 0xFF] >> 6;
      }
      obj->timer++;

      // Set graphic
      set_object_anim(obj, anim_boss[BO_ANIM_6_HANDIDLE]);
   }
}

//***************************************************************************
// run_mrevil_evil
// Logic for Mr. Evil himself
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void run_mrevil_evil(Object *obj) {
   // Set our coordinates
   obj->x = obj->otherobj->x;
   obj->y = obj->otherobj->y - 0x08;

   // Set our sprite
   if (obj->otherobj->dead)
      set_object_anim(obj, anim_boss[BO_ANIM_6_EVILPANIC]);
   else if (obj->otherobj->jumping)
      set_object_anim(obj, anim_boss[BO_ANIM_6_EVILIDLE]);
   else if (obj->otherobj->active)
      set_object_anim(obj, anim_boss[BO_ANIM_6_EVILIDLE]);
   else
      set_object_anim(obj, anim_boss[BO_ANIM_6_EVILSHOOT]);
}

//***************************************************************************
// init_missile_in
// Initialization for the Mr. Evil machine incoming missiles
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void init_missile_in(Object *obj) {
   // Set collision box
   set_hitbox(obj, -0x10, 0x0F, -0x10, 0x0F);

   // Set animation
   set_object_anim(obj, anim_boss[BO_ANIM_6M_IN]);

   // Get a player to target...
   Object *player = get_first_object(OBJGROUP_PLAYER);
   if (player == NULL)
      return;

   // Determine our target distance
   int32_t dist_x = player->x - obj->x;
   int32_t dist_y = player->y - obj->y;

   // Set our speed to reach the target
   obj->speed = (dist_x << 8) / 40;
   obj->gravity = (dist_y << 8) / 40;

   // Play shooting sound
   play_2d_sfx(SFX_MISSILE, obj->x, obj->y);
}

//***************************************************************************
// run_missile_in
// Logic for the Mr. Evil machine incoming missiles
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void run_missile_in(Object *obj) {
   // Move towards the player
   obj->x += speed_to_int(obj->speed);
   obj->y += speed_to_int(obj->gravity);
   obj->timer++;

   // Same depth as player?
   if (obj->timer == 40) {
      // Look up for collision with players
      refresh_hitbox(obj);
      for (Object *player = get_first_object(OBJGROUP_PLAYER);
      player != NULL; player = player->next) {
         // No collision?
         if (!collision(obj, player))
            continue;

         // Explode!
         add_object(OBJ_EXPLOSION, obj->x, obj->y, 0);
         delete_object(obj);
         return;
      }

      // Also explode if we hit the foreground
      LevelTile *tile = get_tile_by_pixel(obj->x, obj->y);
      if (tile->collision != 0) {
         add_object(OBJ_EXPLOSION, obj->x, obj->y, 0);
         apply_quake(0x10);
         delete_object(obj);
         return;
      }

      // Go away if no collision
      Object *other = add_object(OBJ_MISSILE_OUT, obj->x, obj->y, 0);
      other->speed = obj->speed;
      other->gravity = obj->gravity;
      delete_object(obj);
      return;
   }
}

//***************************************************************************
// init_missile_out
// Initialization for the Mr. Evil machine outgoing missiles
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void init_missile_out(Object *obj) {
   // Set animation
   set_object_anim(obj, anim_boss[BO_ANIM_6M_OUT]);
}

//***************************************************************************
// run_missile_out
// Logic for the Mr. Evil machine outgoing missiles
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void run_missile_out(Object *obj) {
   // Accelerate
   /*int32_t dist_x = obj->x - obj->base_x;
   int32_t dist_y = obj->y - obj->base_y;
   obj->speed += dist_x;
   obj->gravity += dist_y;*/
   obj->speed += obj->speed >> 5;
   obj->gravity += obj->gravity >> 5;

   // Keep moving
   obj->x += speed_to_int(obj->speed);
   obj->y += speed_to_int(obj->gravity);

   // Disappear when too far
   obj->timer++;
   if (obj->timer == 32) {
      delete_object(obj);
      return;
   }
}

//***************************************************************************
// init_driver
// Initializes a fleeing driver.
//---------------------------------------------------------------------------
// param obj: pointer to object
//***************************************************************************

void init_driver(Object *obj) {
   // Set sprite
   set_object_anim(obj, anim_boss[BO_ANIM_DRIVER]);

   // Set momentum
   obj->speed = obj->dir ? -0x180 : 0x180;
   obj->gravity = -0x300;
}
