#///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
# Flow3rEffects: Simple Demo Effects
# for On-Track party at CCCamp23 (Party Production)
# by sofaritze / hacksaar
# 18.08.2023
#///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

from st3m.application import Application, ApplicationContext
from st3m.reactor import Responder
import st3m.run
import math
import leds
import random

class DemoEffects(Application):

    def __init__(self, app_ctx: ApplicationContext) -> None:
        super().__init__(app_ctx)
        # Ignore the app_ctx for now.
        
    def __init__(self) -> None:
        self.effectTimer = 0

        self.demoMode = True
        self.timeOffset = 5000

        # Tunnel:
        self.numberOfRings = 6
        self.radiusIncrement = 22
        self.cameraVelocity = 80
        self.timeIndex = 0
        self.timeIncrement = 2
        self.lastX = -1
        self.lastY = -1
        self.lastRadius = -1

        # Rasterbars:
        self.rasterBarsPerformInit = True
        self.numberOfBars = 6
        self.barPosY = [0, 0, 0, 0, 0, 0]
        self.barSpeed = [0, 0, 0, 0, 0, 0]
        self.timeCounter = [0, 0, 0, 0, 0, 0]
        self.SCREENWIDTH = 240
        self.SCREENHEIGHT = 240
        self.ledPosYPercent = [0, 0.1, 0.2, 0.3, 0.4, 0.4, 0.42, 0.45, 0.5, 0.55, 
                               0.58, 0.6, 0.62, 0.72, 0.82, 0.92, 1, 0.95, 0.93, 0.92, 
                               0.85, 0.92, 0.93, 0.95, 1, 0.92, 0.82, 0.72, 0.62, 0.6, 
                               0.58, 0.55, 0.5, 0.45, 0.42, 0.4, 0.4, 0.3, 0.2, 0.1]

        # SPINNING CUBE
        self.vertices = [ # [8 * 3]
            -1, -1, 1, 
            1, -1, 1, 
            1, 1, 1, 
            -1, 1, 1, 
            -1, -1, -1, 
            1, -1, -1, 
            1, 1, -1, 
            -1, 1, -1] 
        self.segments = [ # [12 * 2]
            0, 1, 
            1, 2, 
            2, 3,
            3, 0, 
            0, 4, 
            1, 5, 
            2, 6, 
            3, 7, 
            4, 5, 
            5, 6, 
            6, 7, 
            7, 4] 
        self.vertices_count = 8
        self.verticesScaling = 60
        self.segment_count = 12
        self.cubeIterator = -0.3
        self.scaleX = 0.05
        self.scaleY = 0.05
        self.scaleZ = 0.05

        
        # STARFIELD
        self.numberOfStars = 40
        self.starsPositionX = [1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40]
        self.starsPositionY = [1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40]
        self.starsPositionZ = [1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40]
        self.starsPreviousPositionZ = [1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40]
        self.starSpeed = 0.5
        self.starfieldPerformInit = True
        self.zMin = 0
        self.zMax = 120 / 4


    def drawPixel(self, ctx: Context, x, y, r, g, b):
        ctx.rgb(255, 0, 0).move_to(x, y)
        ctx.rgb(255, 0, 0).rgb(r, g, b).stroke()

    def drawLine(self, ctx: Context, xStart, yStart, xEnd, yEnd):
        ctx.rgb(255, 0, 0).move_to(xStart, yStart)        
        ctx.rgb(255, 0, 0).line_to(xEnd, yEnd).stroke()

    #///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    # RASTERBARS
    #///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    def rasterbars(self, ctx: Context):
        #// init?
        if self.rasterBarsPerformInit:
            self.rasterBarsPerformInit = False
            self.barSpeed[0] = 0.1
            self.barSpeed[1] = 0.08
            self.barSpeed[2] = 0.07
            self.barSpeed[3] = 0.06
            self.barSpeed[4] = 0.05
            self.barSpeed[5] = 0.04

        ctx.begin_path()
        #draw bars:
        for i in range(0, self.numberOfBars):
            self.timeCounter[i] = self.timeCounter[i] + self.barSpeed[i]
            self.barPosY[i] = (math.sin(self.timeCounter[i]) + 1.0)/2.0 * (float)(self.SCREENHEIGHT)
            self.drawLine(ctx, -120, -120 + self.barPosY[i], 120, -120 + self.barPosY[i])
        ctx.close_path()

        ctx.rgb(255, 0, 0).round_rectangle(-118,-118,236,236,118).stroke()

        self.timeIndex = self.timeIndex + self.timeIncrement
        if self.timeIndex == 500:
            self.timeIndex = 0
        ledIndex = self.timeIndex / 2 / 500 * 40
        ledWidth = 3

        height01red = self.barPosY[0] / 240
        height01green = self.barPosY[1] / 240
        
        for i in range(0, 39):
            ledDistanceRed = abs(self.ledPosYPercent[i] - height01red)
            ledDistanceGreen = abs(self.ledPosYPercent[i] - height01green)
            
            # #Variante nur rot beidseitig
            # if (ledDistanceRed < 0.2):  
            #     leds.set_rgb(i, 1.0 - ledDistanceRed, 0, 0)
            # else:
            #     leds.set_rgb(i, 0, 0, 0)
                
            #Variante beidfarbig beidseitig
            leds.set_rgb(i, 1.0 - ledDistanceRed, 1.0 - ledDistanceGreen, 0)

            # #Variante getrennt:
            # if (i <=19):
            #     leds.set_rgb(i, 1.0 - ledDistanceRed, 0, 0)
            # else:
            #     leds.set_rgb(i, 0, 1.0 - ledDistanceGreen, 0)
           
        leds.update()

    #################################################################################################
    ## TUNNEL 
    #################################################################################################
    def tunnel(self, ctx: Context):
        ctx.rgb(0, 0, 0).rectangle(-120, -120, 240, 240).fill()

        for ring in range(0, self.numberOfRings - 1):  
            x = math.cos(2.0 * math.pi * (ring * self.cameraVelocity + self.timeIndex) / 500.0) * 8.0
            y = math.sin(2.0 * math.pi * (ring * self.cameraVelocity + self.timeIndex) / 500.0) * 10.0
            r = (self.numberOfRings - ring) * self.radiusIncrement
            self.timeIndex = self.timeIndex + self.timeIncrement
            if self.timeIndex == 500:
              self.timeIndex = 0

            ctx.rgb(255, 0, 0).round_rectangle(x - r, y - r, 2*r, 2*r, r).stroke()

            if (self.lastX != -1):
                ctx.begin_path()
                self.drawLine(ctx, self.lastX - self.lastRadius, self.lastY, x - r, y)
                self.drawLine(ctx, self.lastX + self.lastRadius, self.lastY, x + r, y)
                self.drawLine(ctx, self.lastX, self.lastY - self.lastRadius, x, y - r)
                self.drawLine(ctx, self.lastX, self.lastY + self.lastRadius, x, y + r)
                ctx.close_path()

            self.lastX = x
            self.lastY = y
            self.lastRadius = r

        self.lastX = -1
        self.lastY = -1

        ledIndex = self.timeIndex / 500 * 40
        ledWidth = 3

        for i in range(0, 39):
            ledDistance = abs(ledIndex -i - ledWidth) / 5
            if (i > ledIndex -ledWidth and i < ledIndex + ledWidth):  
                leds.set_rgb(i, 1.0 - ledDistance, 0, 0)
            else:
                if (ledDistance > 1):
                    ledDistance = 1
                leds.set_rgb(i, 0, ledDistance, 0)

        leds.update()

    #################################################################################################
    ## SPINNING CUBE 
    #################################################################################################
    def drawScreen(self, ctx:Context):
        rotateX = self.scaleX
        rotateY = math.sin(self.cubeIterator) * self.scaleY #//random(0, 10)
        rotateZ = math.cos(self.cubeIterator) * self.scaleZ
        self.rotate3D(rotateZ, rotateX, rotateY)

        ctx.begin_path()
        for i in range(0, self.segment_count):
            self.draw3dline(ctx, self.segments[i * 2 + 0], self.segments[i * 2 + 1])
        ctx.close_path()
        ctx.rgb(255, 0, 0).round_rectangle(0, 0, 0, 0, 0).stroke()
        
        leds.set_slew_rate(80)
        if((self.effectTimer % 500) < 100):
            for i in range(0, 39):
                leds.set_rgb(i, 1, 1, 1)
        else:
            for i in range(0, 39):
                leds.set_rgb(i, 0, 0, 0)            
        leds.update()

    def draw3dline(self, ctx: Context, vertex1, vertex2):
        x1 = self.vertices[vertex1 * 3 + 0] * self.verticesScaling
        y1 = self.vertices[vertex1 * 3 + 1] * self.verticesScaling
        x2 = self.vertices[vertex2 * 3 + 0] * self.verticesScaling
        y2 = self.vertices[vertex2 * 3 + 1] * self.verticesScaling
        self.drawLine(ctx, x1, y1, x2, y2)

        
    def rotate3D(self, x_theta, y_theta, z_theta):
        x_sinTheta = math.sin(x_theta)
        x_cosTheta = math.cos(x_theta)
        y_sinTheta = math.sin(y_theta)
        y_cosTheta = math.cos(y_theta)
        z_sinTheta = math.sin(z_theta)
        z_cosTheta = math.cos(z_theta)

        for n in range(0, self.vertices_count):
            if (x_theta > 0):
                y = self.vertices[n * 3 + 1]
                z = self.vertices[n * 3 + 2];
                self.vertices[n * 3 + 1] = y * x_cosTheta - z * x_sinTheta
                self.vertices[n * 3 + 2] = z * x_cosTheta + y * x_sinTheta
            if (y_theta > 0):
                x = self.vertices[n * 3 + 0]
                z = self.vertices[n * 3 + 2]
                self.vertices[n * 3 + 0] = x * y_cosTheta - z * y_sinTheta
                self.vertices[n * 3 + 2] = z * y_cosTheta + x * y_sinTheta
            if (z_theta > 0):
                x = self.vertices[n * 3 + 0]
                y = self.vertices[n * 3 + 1]
                self.vertices[n * 3 + 0] = x * z_cosTheta - y * z_sinTheta;
                self.vertices[n * 3 + 1] = y * z_cosTheta + x * z_sinTheta;
    
    def spinningcube(self, ctx: Context):
        leds.set_brightness(20)
        self.cubeIterator = self.cubeIterator + 0.1
        if (self.cubeIterator > 3.1415 / 2.0):
            self.cubeIterator = -3.1415 / 2.0
        self.drawScreen(ctx)



    #///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    # STARFIELD
    #///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    def resetStar(self, i):
        self.starsPositionX[i] = random.uniform(-6,6)
        self.starsPositionY[i] = random.uniform(-6,6)
        if self.starsPositionX[i] == 0:
            self.starsPositionX[i] = 1
        if (self.starsPositionY[i] == 0):
            self.starsPositionY[i] = 1
        self.starsPositionZ[i] = random.uniform(self.zMin, self.zMax)
        self.starsPreviousPositionZ[i] = -1

    def starfield(self, ctx: Context):
        #// init?
        if (self.starfieldPerformInit):
            self.starfieldPerformInit = False
            for i in range(0, self.numberOfStars):
                self.resetStar(i)
        
        ctx.line_width = 3

        ctx.begin_path()
        # draw stars:
        for i in range(0, self.numberOfStars):
            sx = self.starsPositionX[i] / self.starsPositionZ[i]
            sy = self.starsPositionY[i] / self.starsPositionZ[i]
            if (sx > 1):
                sx = 1
            if (sy > 1):
                sy = 1
            sx *= (float)(self.SCREENWIDTH / 2.0)
            sy *= (float)(self.SCREENWIDTH / 2.0)

            if (self.starsPreviousPositionZ[i] == -1):
                self.drawPixel(ctx, 
                              sx - self.SCREENWIDTH / 2.0, sy - self.SCREENHEIGHT / 2.0,
                              1, 0, 0)
            else:
                psx = self.starsPositionX[i] / self.starsPreviousPositionZ[i]
                psy = self.starsPositionY[i] / self.starsPreviousPositionZ[i]
      
                if (psx > 1):
                    psx = 1
                if (psy > 1):
                    psy = 1
                psx *= (float)(self.SCREENWIDTH / 2.0)
                psy *= (float)(self.SCREENWIDTH / 2.0)
                self.drawLine(ctx, sx,sy, psx, psy)
        
        ctx.close_path()

        ctx.rgb(255, 0, 0).round_rectangle(0, 0, 0, 0, 0).stroke()

        # update all star positions:
        for i in range(0, self.numberOfStars):
            self.starsPreviousPositionZ[i] = self.starsPositionZ[i]
            self.starsPositionZ[i] = self.starsPositionZ[i] - self.starSpeed
            if (self.starsPositionZ[i] < 1):
                self.resetStar(i)

        leds.set_slew_rate(10)

        for i in range(0, 39):
            leds.set_rgb(i, 0, 0, 0.1)

        for i in range(0, self.numberOfStars / 4):
            ledIndx = random.randint(0,39)
            leds.set_rgb(ledIndx, 1.0, 0, 0)
            
        leds.update()

    #################################################################################################
    ## INTRO / OUTRO TEXT 
    #################################################################################################
    def introoutro(self, ctx:Context, relativeTime):
        fadingLength = 1000
        fadingPeriod = 8000

        ctx.text_baseline = ctx.MIDDLE
        ctx.text_align = ctx.CENTER
        ctx.move_to(0, 0)
        ctx.font_size = 30
        ctx.font = "Camp Font 1"

        col = (relativeTime % fadingPeriod) / fadingLength
        if col>1:
            col = 1
        if(relativeTime > fadingLength):
            if (abs(fadingPeriod - (relativeTime % fadingPeriod))) < fadingLength:
                col = (fadingPeriod - (relativeTime % fadingPeriod)) / fadingLength
        ctx.rgb(col,0,0)

        if relativeTime < fadingPeriod:
            ctx.move_to(0, -50)
            ctx.text(f"sofaritze")
            ctx.move_to(0, -10)
            ctx.text(f"(hacksaar)")
            ctx.move_to(0, 30)
            ctx.text(f"@CCCamp23")

            leds.set_slew_rate(80)
            for i in range(0, 39):
                leds.set_rgb(i, 0, 0, col)
            leds.update()

        if relativeTime  >= fadingPeriod and relativeTime < 2 * fadingPeriod:
            ctx.move_to(0, -50)
            ctx.text(f"Simple demo")
            ctx.move_to(0, -10)
            ctx.text(f"effects on")
            ctx.move_to(0, 30)
            ctx.text(f"flow3r badge")

            leds.set_slew_rate(80)
            for i in range(0, 39):
                leds.set_rgb(i, 0, col, 0)
            leds.update()

        if relativeTime >= 48000 and relativeTime < 56000:
            ctx.move_to(0, -50)
            ctx.text(f"See you at")
            ctx.move_to(0, -10)
            ctx.text(f"Revision 2024")
            ctx.move_to(0, 30)
            ctx.text(f"#staysafe")

            leds.set_slew_rate(80)
            for i in range(0, 39):
                leds.set_rgb(i, col, col, col)
            leds.update()


    ######################################################################
    # Main effect loop:
    def draw(self, ctx: Context) -> None:
        # Paint the background black
        ctx.rgb(0, 0, 0).rectangle(-120, -120, 240, 240).fill()

        if self.demoMode == False:
            self.timeOffset = 0

        if (self.effectTimer >= 0 + self.timeOffset and self.effectTimer < 160000 + self.timeOffset):
            self.introoutro(ctx, self.effectTimer - self.timeOffset)

        if (self.effectTimer >= 16000 + self.timeOffset and self.effectTimer < 24000 + self.timeOffset):
            self.rasterbars(ctx)

        if (self.effectTimer >= 24000 + self.timeOffset and self.effectTimer < 32000 + self.timeOffset):
            self.starfield(ctx)

        if (self.effectTimer >= 32000 + self.timeOffset and self.effectTimer < 40000 + self.timeOffset):
            self.spinningcube(ctx)

        if (self.effectTimer >= 40000 + self.timeOffset and self.effectTimer < 48000 + self.timeOffset):
            self.tunnel(ctx)

        if (self.effectTimer >= 48000 + self.timeOffset and self.effectTimer < 56000 + self.timeOffset):
            self.introoutro(ctx, self.effectTimer - self.timeOffset)
        
        if (self.demoMode == False and self.effectTimer > 56000):
            self.effectTimer = 0


    def think(self, ins: InputState, delta_ms: int) -> None:
        direction = ins.buttons.app # -1 (left), 1 (right), or 2 (pressed)

        self.effectTimer = self.effectTimer + delta_ms


st3m.run.run_responder(DemoEffects())

