import moonlander.library.*;
import ddf.minim.*;

Moonlander moonlander;

float [][]perlin;
PVector []blob_particles;
float []blob_particle_lifes;
PVector [][]curlnoise;
int curl_width;
int curl_height;
float blob_max_life = 1.0f;
int previous_time;

PGraphics text_buffer;
PFont font_black;
PGraphics background_gradient;
String []big_texts;

PImage background_forest;
PImage img_leaf;
PImage img_shrub;
PImage img_root;
PImage img_endtext;

PVector []leaf_positions;

PVector get_random_position_in_mask(PGraphics mask) {
    // this is bad and can hang if unlucky
    while (true) {
        PVector pos = new PVector(
            random(mask.width), random(mask.height)
        );
        color c = mask.get(int(pos.x), int(pos.y));
        if (red(c) == 255) {
            return pos;
        }
    }
}

PVector []init_leaf_positions() {
    PVector []leaf_pos = new PVector[15];
    for (int i = 0; i < leaf_pos.length; i++) {
        leaf_pos[i] = new PVector(random(1.0f), random(1.0f));
    }
    return leaf_pos;
}

PVector []get_random_particles(int number) {
    PVector []particles = new PVector[number];

    for (int i = 0; i < number; i++) {
        particles[i] = new PVector(random(1.0f), random(1.0f));
    }

    return particles;
}

float []get_random_particle_lifes(int number, float max_life) {
    float []lifes = new float[number];

    for (int i = 0; i < number; i++) {
        lifes[i] = random(max_life);
    }

    return lifes;
}

void setup() {
    fullScreen(P2D);
    noCursor();
    background(0);
    //size(1920, 1080, P2D);

    big_texts = new String[5];
    big_texts[0] = "LISTEN";
    big_texts[1] = "TO THE";
    big_texts[2] = "NATURE";
    big_texts[3] = "AND BE";
    big_texts[4] = "INSPIRED";

    background_forest = loadImage("forest.jpg");
    img_leaf = loadImage("leaf.png");
    img_shrub = loadImage("shrub.png");
    img_root = loadImage("root.png");
    img_endtext = loadImage("endtext.png");

    background_gradient = createGraphics(width, height);
    drawGradient(
        background_gradient,
        width/2,
        height/2, width/2 + 200, 1, color(#02001c), color(#011247)
    );

    leaf_positions = init_leaf_positions();

    moonlander = Moonlander.initWithSoundtrack(
        this, "data/JellyfishInSpace_fadeout.mp3", 100, 4);
    moonlander.start("localhost", 1338, "data/syncdata.rocket");

    randomSeed(1337);
    noiseSeed(1337);

    text_buffer = createGraphics(1280, 256);
    font_black = createFont("Archivo-Black.ttf", 220);

    float current_time = 0.1f;

    int num_particles = 10000;

    blob_particles = get_random_particles(num_particles);
    blob_particle_lifes = get_random_particle_lifes(
        num_particles, blob_max_life);

    for (int i = 0; i < blob_particles.length; i++) {
        //print(blob_particles[i].x, "\n");
    }

    curl_width = 320;
    curl_height = 240;

    curlnoise = get_curl_noise(curl_width, curl_height, 0);

    previous_time = millis();

    frameRate(60);
}

PVector [][]get_curl_noise(int w, int h, float time) {
    PVector [][]curl_noise = new PVector[h][w];

    float noise_scale = 0.005f;
    float curl_delta = 0.01f;

    for (int y = 0; y < h; y++) {
        for (int x = 0; x < w; x++) {
            float x0 = noise(
                x*noise_scale - curl_delta,
                y*noise_scale,
                time);
            x0 = (x0 - 0.5) * 2.0;
            float x1 = noise(
                x*noise_scale + curl_delta,
                y*noise_scale,
                time);
            x1 = (x1 - 0.5) * 2.0;
            float y0 = noise(
                x*noise_scale,
                y*noise_scale - curl_delta,
                time);
            y0 = (y0 - 0.5) * 2.0;
            float y1 = noise(
                x*noise_scale,
                y*noise_scale + curl_delta,
                time);
            y1 = (y1 - 0.5) * 2.0;
            PVector noi = new PVector(
                (x1 - x0) / (curl_delta * 2),
                (y1 - y0) / (curl_delta * 2)
            );
            curl_noise[y][x] = new PVector(
                noi.y, -noi.x);
        }
    }

    return curl_noise;
}

PVector []move_particles(PVector []particles, PVector [][]vecfield, float timedelta) {
    PVector []particles_new = new PVector[particles.length];

    for (int i = 0; i < particles.length; i++) {
        PVector old_pos = particles[i];

        PVector vecfield_coords = new PVector(
            old_pos.x * vecfield[0].length,
            old_pos.y * vecfield.length
        );

        PVector direction =
            vecfield[
                constrain(int(vecfield_coords.y), 0, vecfield.length-1)
            ][
                constrain(int(vecfield_coords.x), 0, vecfield[0].length-1)
            ];

        direction.mult(timedelta);
        particles_new[i] = old_pos.add(direction);
    }

    return particles_new;
}

PVector maskcoord_to_viewcoord(PVector maskpos, PVector maskcoord) {
    return new PVector(
        (maskpos.x + maskcoord.x) / width,
        (maskpos.y + maskcoord.y) / height
    );
}

float []update_particle_lifes(
        PVector []particles, float []lifes, float timedelta, PVector mask_on_screen) {
    float []lifes_new = new float[lifes.length];

    for (int i = 0; i < lifes.length; i++) {
        if (lifes[i] <= 0) {
            lifes_new[i] = random(1.0f);
            //lifes_new[i] = 1.0f;
            PVector maskpos = get_random_position_in_mask(text_buffer);
            particles[i] = maskcoord_to_viewcoord(
                maskpos,
                mask_on_screen
            );
            continue;
        }

        lifes_new[i] = lifes[i] - timedelta;
    }

    return lifes_new;
}

void drawText(PGraphics buf, PFont font, String what) {
    buf.beginDraw();
    buf.background(0, 0);
    buf.fill(255);
    //buf.textSize(256);
    buf.textAlign(CENTER);
    buf.textFont(font);
    buf.text(what, 0, 0, buf.width, buf.height);
    buf.endDraw();
}

void drawGradient(
        PGraphics buf,
        float x, float y, int radius, int step, color a, color b) {

    buf.beginDraw();
    for (int r = radius; r > 0; r -= step) {
        color c = lerpColor(b, a, (float)r / radius);
        buf.fill(c);
        buf.noStroke();
        buf.ellipseMode(RADIUS);
        buf.ellipse(x, y, r, r);
    }
    buf.endDraw();
}

PVector []updateLeafs(PVector []positions, float timedelta) {
    PVector []new_positions = new PVector[positions.length];

    for (int i = 0; i < positions.length; i++) {
        if (positions[i].y > 1.03f) {
            positions[i].y = -0.1f;
            positions[i].x = random(1.0f);
        }
        new_positions[i] = new PVector(
            positions[i].x,
            positions[i].y + timedelta);
    }

    return new_positions;
}

void drawLeafs(
        PVector []positions, float current_time, float leaf_alpha) {
    int imagew = 128;

    //translate(width/2, height/2);

    for (int i = 0; i < positions.length; i++) {
        pushMatrix();

        PVector p = positions[i];

        translate(
            (p.x + sin(current_time + i)*0.1f)*width,
            p.y*height
        );
        rotate(PI * sin(current_time + i)*0.3f);
        rotate(PI/3.0);
        translate(-imagew / 2.0, -imagew / 2.0);

        tint(255, ((cos(p.y*2) + 1)*0.5)*255 * leaf_alpha);
        image(img_leaf, 0, 0, imagew, imagew);
        tint(255, 255);

        popMatrix();
    }
}

void drawShrubs(
        float current_time,
        float shrub_alpha, float shrub_reveal) {
    tint(255, shrub_alpha * 255);
    int imagew = 720;

    float ofsy = sin(current_time) * 30.0;
    float ofsr = sin(current_time) * 0.1f;

    pushMatrix();
    translate(
        imagew*0.2f - (1 - shrub_reveal)*imagew,
        height-(imagew*0.5f) + ofsy
    );
    rotate(PI*0.1f + ofsr);
    translate(-imagew / 2.0, -imagew / 2.0);
    image(img_shrub, 0, 0, imagew, imagew);
    popMatrix();

    pushMatrix();
    translate(
        width - (imagew*0.1f) + (1 - shrub_reveal)*imagew,
        height-(imagew*0.5f) + ofsy
    );
    rotate(PI*-0.1f - ofsr);
    scale(-1.0f, 1.0f);
    translate(-imagew / 2.0, -imagew / 2.0);
    image(img_shrub, 0, 0, imagew, imagew);
    popMatrix();

    tint(255, 255);
}

void drawRoot(float current_time, float root_alpha, float root_reveal) {
    tint(160, root_alpha * 255);
    image(
        img_root, 0,
        500 + (1 - root_reveal) * 300 + sin(current_time) * 10
    );
    tint(255, 255);
}

void drawEndtext(float text_reveal) {
    tint(255, text_reveal* 255);
    image(img_endtext, 0, 0);
    tint(255, 255);
}

void draw() {
    background(0);
    moonlander.update();

    //image(background_gradient, 0, 0);

    //for (int y = 0; y < height; y++) {
    //    for (int x = 0; x < width; x++) {
    //        stroke(curlnoise[y][x]*255.0);
    //        point(x, y);
    //    }
    //}

    float current_time = (float)moonlander.getCurrentTime();
    float timedelta = (millis() - previous_time) * 0.001f;
    float particle_alpha = (float)moonlander.getValue("particle_alpha");
    int text_i = moonlander.getIntValue("text_i");
    float shrub_alpha = (float)moonlander.getValue("shrub_alpha");
    float root_alpha = (float)moonlander.getValue("root_alpha");
    float stuff_reveal = (float)moonlander.getValue("stuff_reveal");
    float leaf_alpha = (float)moonlander.getValue("leaf_alpha");
    float bg_alpha = (float)moonlander.getValue("bg_alpha");
    float endtext_reveal = (float)moonlander.getValue("endtext_reveal");

    // background
    tint(90*bg_alpha);
    image(background_forest, 0, sin(current_time)*5 - 5);
    tint(255);

    previous_time = millis();

    PVector mask_on_screen = new PVector(
        width/2 - text_buffer.width/2,
        height/2 - text_buffer.height/2);

    float particle_life = (float) moonlander.getValue("particle_life");
    float noise_seed = (float) moonlander.getValue("noise_seed");

    curlnoise = get_curl_noise(curl_width, curl_height, noise_seed);
    blob_particles = move_particles(blob_particles, curlnoise, timedelta*0.1f);
    blob_particle_lifes = update_particle_lifes(
        blob_particles,
        blob_particle_lifes,
        timedelta*particle_life,
        mask_on_screen);

    drawText(text_buffer, font_black, big_texts[text_i]);

    leaf_positions = updateLeafs(leaf_positions, timedelta*0.1f);
    drawRoot(current_time, root_alpha, stuff_reveal);
    drawLeafs(leaf_positions, current_time, leaf_alpha);
    drawShrubs(current_time, shrub_alpha, stuff_reveal);
    drawEndtext(endtext_reveal);

    //tint(255, particle_alpha*0.1f*255);
    //image(
    //    text_buffer,
    //    width/2-text_buffer.width/2,
    //    height/2-text_buffer.height/2
    //);
    //tint(255, 255);

    // draw particles
    color pcolora = color(0xff98d585);
    color pcolorb = color(0x00161f13);

    strokeWeight(5);
    for (int i = 0; i < blob_particles.length; i++) {
        //float prb = blob_particle_lifes[i] * 255.0;
        color pc = lerpColor(pcolora, pcolorb, blob_particle_lifes[i]);
        stroke(
            pc,
            (blob_particle_lifes[i]+0.3f) * particle_alpha * 255.0);
        point(blob_particles[i].x*width, blob_particles[i].y*height);
    }


    if (moonlander.getIntValue("exit") == 1) {
        exit();
    }
}
