/**
 * Copyright (c) 2023 Sven Oliver "SvOlli" Moll
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 *
 * This program implements Apple Computer I emulation
 * for the Sorbus Computer
 */

#include <time.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include <pico/stdlib.h>
#include <pico/util/queue.h>
#include <pico/multicore.h>
#include <pico/platform.h>

#include "bus.h"
#include "cpudetect_apple1.h"
#include "a1hello.h"
#include "krusader.h"

// set this to 1 to run for 5 million cycles while keeping time
#define SPEED_TEST 0
// the delay between characters printed, the Apple 1 can only add
// a single characters per frame displayed at 60Hz
#define WRITE_DELAY_US 16666
// this is where the write protected area starts
// for Krusader you can change this to 0xE000
#define ROM_START (0xFF00)

uint32_t write_delay_us = WRITE_DELAY_US;

queue_t keyboard_queue;
queue_t display_queue;

uint8_t memory[0x10000]; // 64k of RAM/ROM
uint32_t state;
uint32_t address;

volatile bool enable_print = false;

volatile uint32_t cycles_left_reset = 5;
volatile uint32_t cycles_left_nmi   = 0;
volatile uint32_t cycles_left_irq   = 0;


// sequence to clear screen
// generated by shell: "clear | hd"
const uint8_t clear_screen_sequence[] =
{
   0x1b,0x5b,0x48,0x1b,0x5b,0x32,0x4a,0x1b,0x5b,0x33,0x4a,0
};


// small relocateable 6502 routine to print all characters to display
// assembled by hand
const uint8_t printloop[] =
{
   0xa2,0x00,0x8a,0x20,0xef,0xff,0xe8,0x18,0x90,0xf8
};


int apple2output( int c )
{
   c &= 0x7f;
   if( c == 0x0d )
   {
      // don't modify return below
   }
   else if( c <= 0x1f )
   {
      // bytes below space are not printed
      // (except for return), but still cause a delay
      c = 1;
   }
   else if( c >= 0x60 )
   {
      // bytes above underscore will be trimmed down
      c &= 0x5f;
   }
   return c;
}


int input2apple( int c )
{
   // handle special cases
   if( (c == 0x08) || (c == 0x7f) )
   {
      // backspace
      return 0xdf;
   }
   else if( (c == 0x60) || (c >= 0x7b) )
   {
      // "`{|}~" are meta keys
      return c;
   }
   // Apple 1 only has upper charset
   if( (c >= 'a') && (c <= 'z') )
   {
      c &= 0xdf;
   }
   return c | 0x80;
}


void run_console()
{
   int in = PICO_ERROR_TIMEOUT, out;
   int col = 0;
   uint64_t next_write = time_us_64();

   for(;;)
   {
      if( in == PICO_ERROR_TIMEOUT )
      {
         in = getchar_timeout_us(1000);
      }
      if( in != PICO_ERROR_TIMEOUT )
      {
         in = input2apple( in );
         if( in >= 0x80 )
         {
            // standard key, hand over to Apple 1
            if( queue_try_add( &keyboard_queue, &in ) )
            {
               // only get next key once this is processed
               in = PICO_ERROR_TIMEOUT;

               // problem: a never processed key can prevent from
               // handling the reset key. this can be only delayed
               // by expanding the input queue
            }
         }
         else
         {
            // meta key: print an move cursor over
            // -> output visible, but does not interfere
            putchar( in );
            putchar( '\b' );
            switch( in )
            {
               case '`':
                  cycles_left_reset = 8;
                  enable_print = false;
                  break;
               case '{':
                  write_delay_us = 0;
                  break;
               case '}':
                  write_delay_us = WRITE_DELAY_US;
                  break;
               case '~':
                  printf( "%s", clear_screen_sequence );
                  break;
               default:
                  break;
            }
            in = PICO_ERROR_TIMEOUT;
         }
      }

      if( queue_try_remove( &display_queue, &out ) )
      {
         out = apple2output( out );
         if( out >= 0x20 )
         {
            col++;
         }
         if( out == 0x0d )
         {
            putchar( '\n' );
            col = 0;
         }
         if( col > 40 )
         {
            putchar( '\n' );
            col = 1;
         }

         if( out > 0 )
         {
            while( time_us_64() < next_write )
            {
               tight_loop_contents();
            }
            if( out > 1 )
            {
               // 1 is an indicator of delay, but not print
               putchar( out );
            }
            next_write = time_us_64() + write_delay_us;
         }
      }

      tight_loop_contents();
   }
}


static inline void handle_ram()
{
   // assume that data bus direction is already set
   if( state & bus_config.mask_rw )
   {
      // read from memory and write to bus
      gpio_put_masked( bus_config.mask_data, ((uint32_t)memory[address]) << bus_config.shift_data );
   }
   else
   {
      // read from bus and write to memory write
      // WozMon (and Krusader) is in ROM, so only accept writes below romstart
      if( address < ROM_START )
      {
         memory[address] = (gpio_get_all() >> bus_config.shift_data); // truncate is intended
      }
   }
}


static inline void handle_pia()
{
   int c;
   switch( address )
   {
      case 0xD010: // PIA.A: keyboard
         if( state & bus_config.mask_rw )
         {
            // CPU reads register
            if( queue_try_remove( &keyboard_queue, &c ) )
            {
               memory[address] = input2apple( c );
            }
            gpio_put_masked( bus_config.mask_data, ((uint32_t)memory[address]) << bus_config.shift_data );
         }
         else
         {
            // CPU writes register
            memory[address] = (gpio_get_all() >> bus_config.shift_data);
         }
         break;
      case 0xD011: // PIA.A: control register
         if( state & bus_config.mask_rw )
         {
            // CPU reads register
            memory[address] &= 0x7f;
            memory[address] |= queue_is_empty( &keyboard_queue ) ? 0 : 0x80;
            gpio_put_masked( bus_config.mask_data, ((uint32_t)memory[address]) << bus_config.shift_data );
         }
         else
         {
            // CPU writes register
            memory[address] = (gpio_get_all() >> bus_config.shift_data);
         }
         break;
      case 0xD012: // PIA.B: display
         if( state & bus_config.mask_rw )
         {
            // CPU reads register
            memory[address] &= 0x7f;
            memory[address] |= queue_is_empty( &display_queue ) ? 0 : 0x80;
            gpio_put_masked( bus_config.mask_data, ((uint32_t)memory[address]) << bus_config.shift_data );
         }
         else
         {
            // CPU writes register
            memory[address] = (gpio_get_all() >> bus_config.shift_data);
            c = memory[address];
            if( enable_print )
            {
               queue_try_add( &display_queue, &c );
            }
         }
         break;
      case 0xD013: // PIA.B: control register
         if( state & bus_config.mask_rw )
         {
            // CPU reads register
            gpio_put_masked( bus_config.mask_data, ((uint32_t)memory[address]) << bus_config.shift_data );
            memory[address] &= 0x7f;
         }
         else
         {
            // CPU writes register
            memory[address] = (gpio_get_all() >> bus_config.shift_data);
            enable_print = true;
         }
         break;
      default:
         break;
   }
}


void run_bus()
{
#if SPEED_TEST
   uint64_t time_start, time_end;
   double time_exec;
   double time_mhz;
   uint32_t cyc;

   time_start = time_us_64();
   for(cyc = 0; cyc < 5000000; ++cyc)
#else
   // when not running as speed test run main loop forever
   for(;;)
#endif
   {
      if( cycles_left_reset )
      {
         int dummy;
         --cycles_left_reset;
         gpio_clr_mask( bus_config.mask_reset );
         while( queue_try_remove( &keyboard_queue, &dummy ) )
         {
            // just loop until queue is empty
         }
      }
      else
      {
         gpio_set_mask( bus_config.mask_reset );
      }

      if( cycles_left_nmi )
      {
         --cycles_left_nmi;
         gpio_clr_mask( bus_config.mask_nmi );
      }
      else
      {
         gpio_set_mask( bus_config.mask_nmi );
      }

      if( cycles_left_irq )
      {
         --cycles_left_irq;
         gpio_clr_mask( bus_config.mask_irq );
      }
      else
      {
         gpio_set_mask( bus_config.mask_irq );
      }

      // done: set clock to high
      gpio_set_mask( bus_config.mask_clock );

      // bus should be still valid from clock low
      state = gpio_get_all();

      // setup bus direction so I/O can settle
      if( state & bus_config.mask_rw )
      {
         // read from memory and write to bus
         gpio_set_dir_out_masked( bus_config.mask_data );
      }
      else
      {
         // read from bus and write to memory write
         gpio_set_dir_in_masked( bus_config.mask_data );
      }

      address = ((state & bus_config.mask_address) >> bus_config.shift_address);

      // setup data
      if( (address & 0xFFF0) == 0xD010 )
      {
         // handle_io
         handle_pia();
      }
      else
      {
         handle_ram();
      }

      // done: set clock to low
      gpio_clr_mask( bus_config.mask_clock );
   }

#if SPEED_TEST
   time_end = time_us_64();
   time_exec = (double)(time_end - time_start) / CLOCKS_PER_SEC / 10000;
   time_mhz = (double)cyc / time_exec;
   for(;;)
   {
      printf( "\rbus has terminated after %d cycles in %.06f seconds: %.0fHz ", cyc, time_exec, time_mhz );
      sleep_ms(2000);
   }
#endif
}


int main()
{
   // setup UART
   stdio_init_all();
   uart_set_translate_crlf( uart0, true );

   // setup between UART core and bus core
   queue_init( &keyboard_queue, sizeof(int), 256 );
   queue_init( &display_queue, sizeof(int), 1 );

   // clean out memory
   // yes, this is not original, but more userfriendly and also easier
   // to implement a real DRAM initstate profile
   // also we've got 64k addressable, which is not original as well
   memset( &memory[0x0000], 0x00, sizeof(memory) );

   // let's preload some code
   memcpy( &memory[0x02F0], &printloop[0], sizeof(printloop) );
   memcpy( &memory[0x0280], &cpudetect_0280[0], sizeof(cpudetect_0280) );
   memcpy( &memory[0x0800], &a1hello_0800[0], sizeof(a1hello_0800) );

   // and we need also setup the ROM
   // it's implemented as part of RAM with write disabled
   memcpy( &memory[0xE000], &krusader_e000[0], sizeof(krusader_e000) );

   // setup the bus and run the bus core
   bus_init();
   multicore_launch_core1( run_bus );

   // run interactive console -> should never return
   run_console();

   return 0;
}
