#version 450

struct LightProperties
{
    vec4 diffuse;
    vec4 direction;
    vec4 position;
    vec4 up;
    vec4 right;
    vec2 dimensions;
    vec2 _pad1;
    float intensity;
    float range;
    float cutoff;
    float roughness_modifier;
    int is_area;
    int type;
    int projector_sampler;
    float projector_intensity;
    int shadowmap_sampler0;
    int shadowmap_sampler1;
    int shadowmap_sampler2;
    int shadowmap_sampler3;
    float cascade_distance0;
    float cascade_distance1;
    float cascade_distance2;
    float cascade_distance3;
    mat4 mat_shadow_mv;
    mat4 mat_shadow_p[4];
    mat4 mat_shadow_mvp[4];
};

struct AngularInfo
{
    float NdotL;
    float NdotV;
    float NdotH;
    float LdotH;
    float VdotH;
};

struct MaterialInfo
{
    float perceptualRoughness;
    vec3 reflectance0;
    float alphaRoughness;
    vec3 diffuseColor;
    vec3 reflectance90;
    vec3 specularColor;
};

struct Cone
{
    float cosa;
    float h;
    vec3 c;
    vec3 v;
};

struct Ray
{
    vec3 o;
    vec3 d;
};

struct Rect
{
    vec3 center;
    vec3 dirx;
    vec3 diry;
    float halfx;
    float halfy;
    vec4 plane;
};

struct BasicDeferredParams
{
    mat4 mModelview;
    mat4 mCameraModelview;
    mat4 mProjection;
    vec4 camera_near_far_plane;
    vec3 camera_position;
};

struct GlobalVariables
{
    float time;
    float global_time;
    float time_step;
    int monotonic;
};

struct DeferredRenderLightsParams
{
    int width;
    int height;
    int lights_num;
    float env_map_intensity;
    float raytrace_scaling_factor;
    float raytrace_strength;
};

struct MaterialPropertiesGPU
{
    vec3 diffuse;
    float transparency;
    vec3 emissive;
    float roughness;
    vec3 triplanar_factor;
    float refraction;
    float normal_factor;
    float emissive_factor;
    float temporal_accumulation_factor;
    float shadowmap_bias;
    float metalness;
    int albedo_sampler;
    int emissive_sampler;
    int normal_sampler;
    int metalic_roughness_sampler;
    uint flags;
    uint _pad0;
    uint _pad1;
};

struct GeometryInformation
{
    uint vtx_num;
    uint surfaces_num;
    uint builtin_attribute_mask;
    uint _pad1;
    uint faces_num_per_surface[64];
};

layout(set = 1, binding = 1, std140) uniform BasicDeferredParamsBuffer
{
    layout(row_major) BasicDeferredParams basic_params;
} _1133;

layout(set = 1, binding = 2, std140) uniform GlobalVariablesBuffer
{
    GlobalVariables globals;
} _1653;

layout(set = 1, binding = 3, std140) uniform DeferredRenderLightsParamsBuffer
{
    DeferredRenderLightsParams render_lights_params;
} _2732;

layout(set = 1, binding = 4, std140) uniform MaterialPropertiesDataBuffer
{
    MaterialPropertiesGPU material_properties[512];
} materials;

layout(set = 1, binding = 5, std140) uniform LightPropertiesBuffer
{
    layout(row_major) LightProperties light_properties[16];
} lights;

layout(set = 1, binding = 6) uniform sampler2DArray s_BlueNoise;
layout(set = 1, binding = 7) uniform sampler2DShadow LightShadowmapCmpSamplers[16];
layout(set = 1, binding = 23) uniform sampler2D s_LTC2;
layout(set = 1, binding = 24) uniform samplerCube sEnviromentMap;
layout(set = 1, binding = 25) uniform sampler2D sAlbedo;
layout(set = 1, binding = 26) uniform usampler2D sNormalMaterial;
layout(set = 1, binding = 27) uniform sampler2D sDepth;
layout(set = 1, binding = 28) uniform usampler2D sMetalnessRoughnessMaterialTags;
layout(set = 1, binding = 29) uniform sampler2D sEmissive;
layout(set = 1, binding = 30) uniform sampler2D LightShadowmapSamplers[16];
layout(set = 1, binding = 46) uniform sampler2D LightProjectorSamplers[16];
layout(set = 1, binding = 62) uniform sampler2D s_LTC1;
layout(set = 1, binding = 63) uniform sampler2D sRaytrace;

layout(location = 0) in vec2 vTexcoord0;
layout(location = 1) in vec4 vFrustum;
layout(location = 0) out vec4 outColor;
bool twoSided;
bool clipless;

vec2 unpackSnorm2x15(uint d)
{
    return (vec2(uvec2(d, d >> uint(15)) & uvec2(32767u)) / vec2(16383.5)) - vec2(1.0);
}

vec3 decode_normal(uint data)
{
    uint param = data;
    vec2 v = unpackSnorm2x15(param);
    uint s = data & 1073741824u;
    vec3 n;
    n.x = v.x;
    n.y = v.y;
    n.z = sqrt(clamp(1.0 - dot(n.xy, n.xy), 0.0, 1.0)) * ((s > 0u) ? 1.0 : (-1.0));
    return n;
}

int decode_material(uint data)
{
    return int(data >> 31u);
}

float linearizeDepth(float d)
{
    return _1133.basic_params.camera_near_far_plane.z / ((_1133.basic_params.camera_near_far_plane.y + _1133.basic_params.camera_near_far_plane.x) - (d * _1133.basic_params.camera_near_far_plane.w));
}

vec3 positionFromDepth(vec4 vDirection, float depth)
{
    return _1133.basic_params.camera_position + (vDirection.xyz * depth);
}

void decode_metalness_roughness_material(uvec2 mrm, out float metalness, out float roughness, out uint material)
{
    metalness = float(mrm.x >> uint(8)) * 0.0039215688593685626983642578125;
    roughness = float((mrm.x >> uint(0)) & 255u) * 0.0039215688593685626983642578125;
    material = mrm.y;
}

vec3 saturate(vec3 a)
{
    return max(vec3(0.0), min(vec3(1.0), a));
}

bool inside_light_cone(vec3 p0, vec3 p, float angle, float height)
{
    float hsquared = height * height;
    float cosangle = angle;
    float cDistance = dot(p, p);
    bool _1359 = cDistance <= hsquared;
    bool _1370;
    if (_1359)
    {
        _1370 = dot(p0, p) >= (sqrt(cDistance) * cosangle);
    }
    else
    {
        _1370 = _1359;
    }
    return _1370;
}

bool intersect_cone(Cone s, Ray r, out float v)
{
    v = 0.0;
    vec3 param = s.v;
    vec3 param_1 = r.o - s.c;
    float param_2 = s.cosa;
    float param_3 = s.h;
    if (inside_light_cone(param, param_1, param_2, param_3))
    {
        return true;
    }
    vec3 co = r.o - s.c;
    float a = (dot(r.d, s.v) * dot(r.d, s.v)) - (s.cosa * s.cosa);
    float b = 2.0 * ((dot(r.d, s.v) * dot(co, s.v)) - ((dot(r.d, co) * s.cosa) * s.cosa));
    float c = (dot(co, s.v) * dot(co, s.v)) - ((dot(co, co) * s.cosa) * s.cosa);
    float det = (b * b) - ((4.0 * a) * c);
    if (det < 0.0)
    {
        return false;
    }
    det = sqrt(det);
    float t1 = ((-b) - det) / (2.0 * a);
    float t2 = ((-b) + det) / (2.0 * a);
    float t = t1;
    bool _1496 = t < 0.0;
    bool _1506;
    if (!_1496)
    {
        _1506 = (t2 > 0.0) && (t2 < t);
    }
    else
    {
        _1506 = _1496;
    }
    if (_1506)
    {
        t = t2;
    }
    if (t < 0.0)
    {
        return false;
    }
    vec3 cp = (r.o + (r.d * t)) - s.c;
    float h = dot(cp, s.v);
    if (h < 0.0)
    {
        cp = (r.o + (r.d * max(t1, t2))) - s.c;
        h = dot(cp, s.v);
        if (h < 0.0)
        {
            return false;
        }
    }
    if (h < s.h)
    {
        return true;
    }
    cp = (r.o + (r.d * max(t1, t2))) - s.c;
    h = dot(cp, s.v);
    bool _1580 = h > 0.0;
    bool _1587;
    if (_1580)
    {
        _1587 = h < s.h;
    }
    else
    {
        _1587 = _1580;
    }
    if (_1587)
    {
        return true;
    }
    return false;
}

AngularInfo getAngularInfo(vec3 pointToLight, vec3 normal, vec3 view)
{
    vec3 n = normalize(normal);
    vec3 v = normalize(view);
    vec3 l = normalize(pointToLight);
    vec3 h = normalize(l + v);
    float NdotL = dot(n, l);
    float NdotV = dot(n, v);
    float NdotH = clamp(dot(n, h), 0.0, 1.0);
    float LdotH = clamp(dot(l, h), 0.0, 1.0);
    float VdotH = clamp(dot(v, h), 0.0, 1.0);
    AngularInfo angularInfo = AngularInfo(NdotL, NdotV, NdotH, LdotH, VdotH);
    return angularInfo;
}

vec3 specularReflection(MaterialInfo materialInfo, AngularInfo angularInfo)
{
    return materialInfo.reflectance0 + ((materialInfo.reflectance90 - materialInfo.reflectance0) * pow(clamp(1.0 - angularInfo.VdotH, 0.0, 1.0), 5.0));
}

float visibilityOcclusion(MaterialInfo materialInfo, AngularInfo angularInfo)
{
    float NdotL = clamp(angularInfo.NdotL, 0.0, 1.0);
    float NdotV = clamp(angularInfo.NdotV, 0.0, 1.0);
    float alphaRoughnessSq = materialInfo.alphaRoughness * materialInfo.alphaRoughness;
    float GGXV = NdotL * sqrt(((NdotV * NdotV) * (1.0 - alphaRoughnessSq)) + alphaRoughnessSq);
    float GGXL = NdotV * sqrt(((NdotL * NdotL) * (1.0 - alphaRoughnessSq)) + alphaRoughnessSq);
    float GGX = GGXV + GGXL;
    if (GGX > 0.0)
    {
        return 0.5 / GGX;
    }
    return 0.0;
}

float microfacetDistribution(MaterialInfo materialInfo, AngularInfo angularInfo)
{
    float alphaRoughnessSq = materialInfo.alphaRoughness * materialInfo.alphaRoughness;
    float f = (((angularInfo.NdotH * alphaRoughnessSq) - angularInfo.NdotH) * angularInfo.NdotH) + 1.0;
    return alphaRoughnessSq / (((3.1415927410125732421875 * f) * f) + 9.9999999747524270787835121154785e-07);
}

vec3 diffuse(MaterialInfo materialInfo)
{
    return materialInfo.diffuseColor / vec3(3.1415927410125732421875);
}

vec3 getPointShade(vec3 pointToLight, MaterialInfo materialInfo, vec3 normal, vec3 view)
{
    vec3 param = pointToLight;
    vec3 param_1 = normal;
    vec3 param_2 = view;
    AngularInfo angularInfo = getAngularInfo(param, param_1, param_2);
    bool _1078 = angularInfo.NdotL > 0.0;
    bool _1085;
    if (!_1078)
    {
        _1085 = angularInfo.NdotV > 0.0;
    }
    else
    {
        _1085 = _1078;
    }
    if (_1085)
    {
        MaterialInfo param_3 = materialInfo;
        AngularInfo param_4 = angularInfo;
        vec3 F = specularReflection(param_3, param_4);
        MaterialInfo param_5 = materialInfo;
        AngularInfo param_6 = angularInfo;
        float Vis = visibilityOcclusion(param_5, param_6);
        MaterialInfo param_7 = materialInfo;
        AngularInfo param_8 = angularInfo;
        float D = microfacetDistribution(param_7, param_8);
        MaterialInfo param_9 = materialInfo;
        vec3 diffuseContrib = (vec3(1.0) - F) * diffuse(param_9);
        vec3 specContrib = (F * Vis) * D;
        return (diffuseContrib + specContrib) * clamp(angularInfo.NdotL, 0.0, 1.0);
    }
    return vec3(0.0);
}

vec2 VogelDiskSample(int sampleIndex, int samplesCount, float phi)
{
    float GoldenAngle = 2.400000095367431640625;
    float r = sqrt(float(sampleIndex) + 0.5) / sqrt(float(samplesCount));
    float theta = (float(sampleIndex) * GoldenAngle) + phi;
    float sine = sin(theta);
    float cosine = cos(theta);
    return vec2(r * cosine, r * sine);
}

float AvgBlockersDepthToPenumbra(float z_shadowMapView, float avgBlockersDepth)
{
    float penumbra = (z_shadowMapView - avgBlockersDepth) / avgBlockersDepth;
    penumbra = abs(penumbra);
    return clamp(1000.0 * penumbra, 0.0, 1.0);
}

float PenumbraFromOccluderSearch(sampler2D smpl, float gradientNoise, vec4 coords, float sampling_range, int samplesCount, float shadowmap_bias)
{
    float avgBlockersDepth = 0.0;
    float blockersCount = 0.0;
    float penumbraFilterMaxSize = sampling_range;
    bool _405 = coords.x < (-coords.w);
    bool _414;
    if (!_405)
    {
        _414 = coords.x > coords.w;
    }
    else
    {
        _414 = _405;
    }
    bool _424;
    if (!_414)
    {
        _424 = coords.y < (-coords.w);
    }
    else
    {
        _424 = _414;
    }
    bool _433;
    if (!_424)
    {
        _433 = coords.y > coords.w;
    }
    else
    {
        _433 = _424;
    }
    if (_433)
    {
        return 0.0;
    }
    vec2 shadowMapUV = ((coords.xy / vec2(coords.w)) * 0.5) + vec2(0.5);
    shadowMapUV.y = 1.0 - shadowMapUV.y;
    float z_shadowMapView = coords.z / coords.w;
    for (int i = 0; i < samplesCount; i++)
    {
        int param = i;
        int param_1 = samplesCount;
        float param_2 = gradientNoise;
        vec2 sampleUV = VogelDiskSample(param, param_1, param_2) * 1.25;
        sampleUV = shadowMapUV + (sampleUV * penumbraFilterMaxSize);
        float sampleDepth = texture(smpl, sampleUV).x;
        if (sampleDepth < z_shadowMapView)
        {
            avgBlockersDepth += sampleDepth;
            blockersCount += 1.0;
        }
    }
    if (blockersCount > 0.0)
    {
        avgBlockersDepth /= blockersCount;
        float param_3 = z_shadowMapView;
        float param_4 = avgBlockersDepth;
        return AvgBlockersDepthToPenumbra(param_3, param_4);
    }
    else
    {
        return 0.0;
    }
}

float sampleShadowPCFNoiseOffset(sampler2DShadow smpl, inout vec4 coords, ivec2 noise_offset, out float in_frustum, int samples, float sampling_range, float shadowmap_bias)
{
    float fact = 0.0;
    in_frustum = 0.0;
    if (coords.w <= 0.0)
    {
        return 1.0;
    }
    bool _527 = coords.x < (-coords.w);
    bool _536;
    if (!_527)
    {
        _536 = coords.x > coords.w;
    }
    else
    {
        _536 = _527;
    }
    bool _546;
    if (!_536)
    {
        _546 = coords.y < (-coords.w);
    }
    else
    {
        _546 = _536;
    }
    bool _555;
    if (!_546)
    {
        _555 = coords.y > coords.w;
    }
    else
    {
        _555 = _546;
    }
    bool _562;
    if (!_555)
    {
        _562 = coords.z < 0.0;
    }
    else
    {
        _562 = _555;
    }
    if (_562)
    {
        return 1.0;
    }
    in_frustum = 1.0;
    vec4 _566 = coords;
    float _571 = coords.w;
    vec2 _573 = (_566.xy * vec2(0.5)) + (vec2(0.5) * _571);
    coords.x = _573.x;
    coords.y = _573.y;
    coords.y /= coords.w;
    coords.y = 1.0 - coords.y;
    coords.y *= coords.w;
    uint h_samples = 0u;
    vec3 hash = (texelFetch(s_BlueNoise, ivec3(noise_offset & ivec2(127), 0), 0).xyz - vec3(0.5)) * 0.25;
    float bias = shadowmap_bias;
    float bias2 = 0.300000011920928955078125;
    int samples_coarse = int(min(3.0, ceil(float(samples) / 6.0)));
    int samples_fine = int(min(3.0, ceil(float(samples) / 2.0)));
    float sampling_r = 1.0;
    float y = -sampling_r;
    for (int iy = 0; iy < samples_coarse; iy++, y += ((2.0 * sampling_r) / float(samples_coarse - 1)))
    {
        float _652 = -sampling_r;
        float x = _652;
        int ix = 0;
        for (; ix < samples_coarse; ix++, x += ((2.0 * sampling_r) / float(samples_coarse - 1)))
        {
            vec2 xy = vec2(x + hash.x, y + hash.y);
            vec4 jittered_coords = coords + vec4((xy.x * sampling_range) * coords.w, (xy.y * sampling_range) * coords.w, bias, 0.0);
            vec4 _693 = jittered_coords;
            vec4 _696 = _693;
            _696.z = _693.w;
            float shadow = textureProjLod(smpl, vec4(_696.xy, _693.z, _696.z), 0.0);
            fact += shadow;
            h_samples++;
        }
    }
    if (fact == 0.0)
    {
        return fact;
    }
    if ((fact * (1.0 / float(h_samples))) >= 0.999499976634979248046875)
    {
        return fact * (1.0 / float(h_samples));
    }
    sampling_r = 0.89999997615814208984375;
    y = -sampling_r;
    for (int iy_1 = 0; iy_1 < samples_fine; iy_1++, y += ((2.0 * sampling_r) / float(samples_fine - 1)))
    {
        float _758 = -sampling_r;
        float x_1 = _758;
        int ix_1 = 0;
        for (; ix_1 < samples_fine; ix_1++, x_1 += ((2.0 * sampling_r) / float(samples_fine - 1)))
        {
            vec2 xy_1 = vec2(x_1 + hash.x, y + hash.y);
            vec4 _797 = coords + vec4((xy_1.x * sampling_range) * coords.w, (xy_1.y * sampling_range) * coords.w, bias, 0.0);
            vec4 _800 = _797;
            _800.z = _797.w;
            float shadow_1 = textureProjLod(smpl, vec4(_800.xy, _797.z, _800.z), 0.0);
            fact += shadow_1;
            h_samples++;
        }
    }
    return fact * (1.0 / float(h_samples));
}

vec4 sampleProjectorTexture(sampler2D smpl, inout vec4 coords)
{
    vec4 color = vec4(0.0);
    if (coords.w <= 0.0)
    {
        return color;
    }
    bool _1245 = coords.x < (-coords.w);
    bool _1254;
    if (!_1245)
    {
        _1254 = coords.x > coords.w;
    }
    else
    {
        _1254 = _1245;
    }
    bool _1264;
    if (!_1254)
    {
        _1264 = coords.y < (-coords.w);
    }
    else
    {
        _1264 = _1254;
    }
    bool _1273;
    if (!_1264)
    {
        _1273 = coords.y > coords.w;
    }
    else
    {
        _1273 = _1264;
    }
    if (_1273)
    {
        return color;
    }
    vec4 _1278 = coords;
    float _1282 = coords.w;
    vec2 _1284 = (_1278.xy * vec2(0.5)) + (vec2(0.5) * _1282);
    coords.x = _1284.x;
    coords.y = _1284.y;
    vec3 samp = coords.xyz / vec3(coords.w);
    color = texture(smpl, coords.xy / vec2(coords.w));
    return color;
}

float light_calculate_spot_attenuation(LightProperties light, vec3 pos)
{
    float cutoff = light.cutoff;
    float light_distance = length(light.position.xyz - pos);
    float falloff = dot(light.direction.xyz, (pos - light.position.xyz) / vec3(light_distance));
    float attenuation = 0.0;
    bool _862 = falloff > cutoff;
    bool _869;
    if (_862)
    {
        _869 = light_distance < light.range;
    }
    else
    {
        _869 = _862;
    }
    if (_869)
    {
        attenuation = 1.0 - ((1.0 - falloff) / (1.0 - cutoff));
        attenuation *= (1.0 - clamp(light_distance / light.range, 0.0, 1.0));
        attenuation = pow(attenuation, 2.0);
    }
    return attenuation;
}

void InitRect(LightProperties light, inout Rect rect)
{
    rect.dirx = -light.right.xyz;
    rect.diry = light.up.xyz;
    rect.center = light.position.xyz;
    rect.halfx = 0.5 * light.dimensions.x;
    rect.halfy = 0.5 * light.dimensions.y;
    vec3 rectNormal = cross(rect.dirx, rect.diry);
    rect.plane = vec4(rectNormal, -dot(rectNormal, rect.center));
}

void InitRectPoints(Rect rect, inout vec3 points[4])
{
    vec3 ex = rect.dirx * rect.halfx;
    vec3 ey = rect.diry * rect.halfy;
    points[0] = (rect.center - ex) - ey;
    points[1] = (rect.center + ex) - ey;
    points[2] = (rect.center + ex) + ey;
    points[3] = (rect.center - ex) + ey;
}

mat3 mul(mat3 m1, mat3 m2)
{
    return m1 * m2;
}

vec3 mul(mat3 m, vec3 v)
{
    return m * v;
}

vec3 IntegrateEdgeVec(vec3 v1, vec3 v2)
{
    float x = dot(v1, v2);
    float y = abs(x);
    float a = 0.8543984889984130859375 + ((0.4965155124664306640625 + (0.01452060043811798095703125 * y)) * y);
    float b = 3.41759395599365234375 + ((4.1616725921630859375 + y) * y);
    float v = a / b;
    float _1840;
    if (x > 0.0)
    {
        _1840 = v;
    }
    else
    {
        _1840 = (0.5 * inversesqrt(max(1.0 - (x * x), 1.0000000116860974230803549289703e-07))) - v;
    }
    float theta_sintheta = _1840;
    return cross(v1, v2) * theta_sintheta;
}

void ClipQuadToHorizon(inout vec3 L[5], inout int n)
{
    int config = 0;
    if (L[0].z > 0.0)
    {
        config++;
    }
    if (L[1].z > 0.0)
    {
        config += 2;
    }
    if (L[2].z > 0.0)
    {
        config += 4;
    }
    if (L[3].z > 0.0)
    {
        config += 8;
    }
    n = 0;
    if (config == 0)
    {
    }
    else
    {
        if (config == 1)
        {
            n = 3;
            L[1] = (L[0] * (-L[1].z)) + (L[1] * L[0].z);
            L[2] = (L[0] * (-L[3].z)) + (L[3] * L[0].z);
        }
        else
        {
            if (config == 2)
            {
                n = 3;
                L[0] = (L[1] * (-L[0].z)) + (L[0] * L[1].z);
                L[2] = (L[1] * (-L[2].z)) + (L[2] * L[1].z);
            }
            else
            {
                if (config == 3)
                {
                    n = 4;
                    L[2] = (L[1] * (-L[2].z)) + (L[2] * L[1].z);
                    L[3] = (L[0] * (-L[3].z)) + (L[3] * L[0].z);
                }
                else
                {
                    if (config == 4)
                    {
                        n = 3;
                        L[0] = (L[2] * (-L[3].z)) + (L[3] * L[2].z);
                        L[1] = (L[2] * (-L[1].z)) + (L[1] * L[2].z);
                    }
                    else
                    {
                        if (config == 5)
                        {
                            n = 0;
                        }
                        else
                        {
                            if (config == 6)
                            {
                                n = 4;
                                L[0] = (L[1] * (-L[0].z)) + (L[0] * L[1].z);
                                L[3] = (L[2] * (-L[3].z)) + (L[3] * L[2].z);
                            }
                            else
                            {
                                if (config == 7)
                                {
                                    n = 5;
                                    L[4] = (L[0] * (-L[3].z)) + (L[3] * L[0].z);
                                    L[3] = (L[2] * (-L[3].z)) + (L[3] * L[2].z);
                                }
                                else
                                {
                                    if (config == 8)
                                    {
                                        n = 3;
                                        L[0] = (L[3] * (-L[0].z)) + (L[0] * L[3].z);
                                        L[1] = (L[3] * (-L[2].z)) + (L[2] * L[3].z);
                                        L[2] = L[3];
                                    }
                                    else
                                    {
                                        if (config == 9)
                                        {
                                            n = 4;
                                            L[1] = (L[0] * (-L[1].z)) + (L[1] * L[0].z);
                                            L[2] = (L[3] * (-L[2].z)) + (L[2] * L[3].z);
                                        }
                                        else
                                        {
                                            if (config == 10)
                                            {
                                                n = 0;
                                            }
                                            else
                                            {
                                                if (config == 11)
                                                {
                                                    n = 5;
                                                    L[4] = L[3];
                                                    L[3] = (L[3] * (-L[2].z)) + (L[2] * L[3].z);
                                                    L[2] = (L[1] * (-L[2].z)) + (L[2] * L[1].z);
                                                }
                                                else
                                                {
                                                    if (config == 12)
                                                    {
                                                        n = 4;
                                                        L[1] = (L[2] * (-L[1].z)) + (L[1] * L[2].z);
                                                        L[0] = (L[3] * (-L[0].z)) + (L[0] * L[3].z);
                                                    }
                                                    else
                                                    {
                                                        if (config == 13)
                                                        {
                                                            n = 5;
                                                            L[4] = L[3];
                                                            L[3] = L[2];
                                                            L[2] = (L[2] * (-L[1].z)) + (L[1] * L[2].z);
                                                            L[1] = (L[0] * (-L[1].z)) + (L[1] * L[0].z);
                                                        }
                                                        else
                                                        {
                                                            if (config == 14)
                                                            {
                                                                n = 5;
                                                                L[4] = (L[3] * (-L[0].z)) + (L[0] * L[3].z);
                                                                L[0] = (L[1] * (-L[0].z)) + (L[0] * L[1].z);
                                                            }
                                                            else
                                                            {
                                                                if (config == 15)
                                                                {
                                                                    n = 4;
                                                                }
                                                            }
                                                        }
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    if (n == 3)
    {
        L[3] = L[0];
    }
    if (n == 4)
    {
        L[4] = L[0];
    }
}

float IntegrateEdge(vec3 v1, vec3 v2)
{
    vec3 param = v1;
    vec3 param_1 = v2;
    return IntegrateEdgeVec(param, param_1).z;
}

vec3 LTC_Evaluate(vec3 N, vec3 V, vec3 P, inout mat3 Minv, vec3 points[4], bool twoSided_1)
{
    vec3 T1 = normalize(V - (N * dot(V, N)));
    vec3 T2 = cross(N, T1);
    mat3 param = Minv;
    mat3 param_1 = transpose(mat3(vec3(T1), vec3(T2), vec3(N)));
    Minv = mul(param, param_1);
    mat3 param_2 = Minv;
    vec3 param_3 = points[0] - P;
    vec3 L[5];
    L[0] = mul(param_2, param_3);
    mat3 param_4 = Minv;
    vec3 param_5 = points[1] - P;
    L[1] = mul(param_4, param_5);
    mat3 param_6 = Minv;
    vec3 param_7 = points[2] - P;
    L[2] = mul(param_6, param_7);
    mat3 param_8 = Minv;
    vec3 param_9 = points[3] - P;
    L[3] = mul(param_8, param_9);
    float sum = 0.0;
    if (clipless)
    {
        vec3 dir = points[0] - P;
        vec3 lightNormal = cross(points[1] - points[0], points[3] - points[0]);
        bool behind = dot(dir, lightNormal) < 0.0;
        L[0] = normalize(L[0]);
        L[1] = normalize(L[1]);
        L[2] = normalize(L[2]);
        L[3] = normalize(L[3]);
        vec3 vsum = vec3(0.0);
        vec3 param_10 = L[0];
        vec3 param_11 = L[1];
        vsum += IntegrateEdgeVec(param_10, param_11);
        vec3 param_12 = L[1];
        vec3 param_13 = L[2];
        vsum += IntegrateEdgeVec(param_12, param_13);
        vec3 param_14 = L[2];
        vec3 param_15 = L[3];
        vsum += IntegrateEdgeVec(param_14, param_15);
        vec3 param_16 = L[3];
        vec3 param_17 = L[0];
        vsum += IntegrateEdgeVec(param_16, param_17);
        float len = length(vsum);
        float z = vsum.z / len;
        if (behind)
        {
            z = -z;
        }
        vec2 uv = vec2((z * 0.5) + 0.5, len);
        uv = (uv * 0.984375) + vec2(0.0078125);
        float scale = texture(s_LTC2, uv).w;
        sum = len * scale;
        if (behind && (!twoSided_1))
        {
            sum = 0.0;
        }
    }
    else
    {
        vec3 param_18[5] = L;
        int param_19;
        ClipQuadToHorizon(param_18, param_19);
        L = param_18;
        int n = param_19;
        if (n == 0)
        {
            return vec3(0.0);
        }
        L[0] = normalize(L[0]);
        L[1] = normalize(L[1]);
        L[2] = normalize(L[2]);
        L[3] = normalize(L[3]);
        L[4] = normalize(L[4]);
        vec3 param_20 = L[0];
        vec3 param_21 = L[1];
        sum += IntegrateEdge(param_20, param_21);
        vec3 param_22 = L[1];
        vec3 param_23 = L[2];
        sum += IntegrateEdge(param_22, param_23);
        vec3 param_24 = L[2];
        vec3 param_25 = L[3];
        sum += IntegrateEdge(param_24, param_25);
        if (n >= 4)
        {
            vec3 param_26 = L[3];
            vec3 param_27 = L[4];
            sum += IntegrateEdge(param_26, param_27);
        }
        if (n == 5)
        {
            vec3 param_28 = L[4];
            vec3 param_29 = L[0];
            sum += IntegrateEdge(param_28, param_29);
        }
        float _2602;
        if (twoSided_1)
        {
            _2602 = abs(sum);
        }
        else
        {
            _2602 = max(0.0, sum);
        }
        sum = _2602;
    }
    vec3 Lo_i = vec3(sum, sum, sum);
    return Lo_i;
}

float light_calculate_area_attenuation(LightProperties light, vec3 pos)
{
    float light_distance = length(light.position.xyz - pos);
    float attenuation = 0.0;
    if (light_distance < light.range)
    {
        attenuation = 1.0 - clamp(light_distance / light.range, 0.0, 1.0);
        attenuation = pow(attenuation, 2.0);
    }
    return attenuation;
}

float vol_rand(vec2 n)
{
    return fract(sin(dot(n, vec2(12.98980045318603515625, 4.141399860382080078125))) * 43758.546875);
}

float vol_noise(vec2 n)
{
    vec2 b = floor(n);
    vec2 f = smoothstep(vec2(0.0), vec2(1.0), fract(n));
    vec2 param = b;
    vec2 param_1 = b + vec2(1.0, 0.0);
    vec2 param_2 = b + vec2(0.0, 1.0);
    vec2 param_3 = b + vec2(1.0);
    return mix(mix(vol_rand(param), vol_rand(param_1), f.x), mix(vol_rand(param_2), vol_rand(param_3), f.x), f.y);
}

float sampleShadowDist(sampler2DShadow smpl, inout vec4 coords, out float in_frustum, inout vec4 projector_color, inout float shadow_dist)
{
    in_frustum = 0.0;
    if (coords.w <= 0.0)
    {
        return 0.0;
    }
    bool _1171 = coords.x < (-coords.w);
    bool _1180;
    if (!_1171)
    {
        _1180 = coords.x > coords.w;
    }
    else
    {
        _1180 = _1171;
    }
    bool _1190;
    if (!_1180)
    {
        _1190 = coords.y < (-coords.w);
    }
    else
    {
        _1190 = _1180;
    }
    bool _1199;
    if (!_1190)
    {
        _1199 = coords.y > coords.w;
    }
    else
    {
        _1199 = _1190;
    }
    if (_1199)
    {
        return 1.0;
    }
    in_frustum = 1.0;
    vec4 _1203 = coords;
    float _1207 = coords.w;
    vec2 _1209 = (_1203.xy * vec2(0.5)) + (vec2(0.5) * _1207);
    coords.x = _1209.x;
    coords.y = _1209.y;
    vec3 samp = coords.xyz / vec3(coords.w);
    vec4 _1223 = coords;
    vec4 _1226 = _1223;
    _1226.z = _1223.w;
    float shadow = textureProj(smpl, vec4(_1226.xy, _1223.z, _1226.z));
    projector_color = vec4(0.0);
    shadow_dist = 0.0;
    return shadow;
}

float calculate_shadow_dist_for_position(LightProperties light, vec3 world, float depth, out vec4 cascadeColor, inout vec4 projector_color, out float shadow_dist)
{
    vec4 vShadowCoords = light.mat_shadow_mvp[0] * vec4(world, 1.0);
    cascadeColor = vec4(1.0, 0.100000001490116119384765625, 0.100000001490116119384765625, 1.0);
    vec4 param = vShadowCoords;
    float param_1;
    vec4 param_2;
    float param_3;
    float _1334 = sampleShadowDist(LightShadowmapCmpSamplers[light.shadowmap_sampler0], param, param_1, param_2, param_3);
    float in_frustum = param_1;
    projector_color = param_2;
    shadow_dist = param_3;
    float shadow = _1334;
    shadow *= in_frustum;
    projector_color *= in_frustum;
    return shadow;
}

float volumetric_sample_shadow_spot(LightProperties light, vec4 frustum, float max_depth, inout float attenuation, inout vec4 projector_color)
{
    float fact = 0.0;
    float depth_dir = 0.100000001490116119384765625;
    float depth_pos = 0.0;
    attenuation = 0.0;
    vec2 param = gl_FragCoord.xy + vec2(_1653.globals.global_time);
    depth_pos += ((depth_dir * vol_noise(param)) * 2.5);
    int i = 0;
    vec4 param_6;
    vec4 param_7;
    float param_8;
    for (; i < 256; i++)
    {
        vec4 param_1 = frustum;
        float param_2 = depth_pos;
        vec3 ray_world = positionFromDepth(param_1, param_2);
        float lf = 0.0;
        float falloff = dot(light.direction.xyz, normalize(ray_world - light.position.xyz));
        if (falloff > light.cutoff)
        {
            float sample_attenuation = 1.0 - ((1.0 - falloff) / (1.0 - light.cutoff));
            sample_attenuation *= (1.0 - clamp(length(light.position.xyz - ray_world) / light.range, 0.0, 1.0));
            sample_attenuation = pow(sample_attenuation, 3.0);
            LightProperties param_3 = light;
            vec3 param_4 = ray_world;
            float param_5 = depth_pos;
            float _1733 = calculate_shadow_dist_for_position(param_3, param_4, param_5, param_6, param_7, param_8);
            vec4 cascade_color = param_6;
            vec4 projector = param_7;
            float shadow_dist = param_8;
            float shadow_value = _1733;
            lf = sample_attenuation * shadow_value;
            attenuation += sample_attenuation;
            vec4 _1750 = projector_color;
            vec3 _1752 = _1750.xyz + ((projector.xyz * (1.0 - shadow_value)) * sample_attenuation);
            projector_color.x = _1752.x;
            projector_color.y = _1752.y;
            projector_color.z = _1752.z;
        }
        fact += lf;
        depth_dir *= 1.0099999904632568359375;
        depth_pos += depth_dir;
        if (depth_pos > max_depth)
        {
            break;
        }
    }
    vec4 _1779 = projector_color;
    vec3 _1782 = _1779.xyz / vec3(float(i + 1));
    projector_color.x = _1782.x;
    projector_color.y = _1782.y;
    projector_color.z = _1782.z;
    return (attenuation - fact) / float(i + 1);
}

float computeLODFromRoughness(float perceptualRoughness)
{
    return perceptualRoughness * 4.0;
}

vec3 ibl(vec3 n, vec3 v, vec3 diffuseColor, vec3 f0, vec3 f90, float perceptualRoughness)
{
    vec3 r = reflect(v, n);
    vec3 Ld = textureLod(sEnviromentMap, r, 6.0).xyz * diffuseColor;
    float param = perceptualRoughness;
    float lod = computeLODFromRoughness(param);
    vec3 fake_spec = textureLod(sEnviromentMap, r, lod).xyz;
    return (Ld + fake_spec) * _2732.render_lights_params.env_map_intensity;
}

void main()
{
    twoSided = false;
    clipless = false;
    ivec2 screen_pos = ivec2(gl_FragCoord.xy);
    vec4 base_color = texture(sAlbedo, vTexcoord0);
    uint encoded_normal_material = texture(sNormalMaterial, vTexcoord0).x;
    uint param = encoded_normal_material;
    vec3 vNorm = decode_normal(param);
    uint param_1 = encoded_normal_material;
    int materialId = decode_material(param_1);
    float param_2 = texture(sDepth, vTexcoord0).x;
    float depth = linearizeDepth(param_2);
    vec4 param_3 = vFrustum;
    float param_4 = depth;
    vec3 world = positionFromDepth(param_3, param_4);
    outColor = vec4(0.0);
    uvec2 param_5 = textureLod(sMetalnessRoughnessMaterialTags, vTexcoord0, 0.0).xy;
    float param_6;
    float param_7;
    uint param_8;
    decode_metalness_roughness_material(param_5, param_6, param_7, param_8);
    float metalness = param_6;
    float roughness = param_7;
    uint material = param_8;
    if ((materials.material_properties[material].flags & 4096u) != 0u)
    {
        outColor = base_color;
        outColor.w = 0.0;
        return;
    }
    vec4 _2822 = base_color;
    vec3 _2824 = _2822.xyz * materials.material_properties[material].diffuse;
    base_color.x = _2824.x;
    base_color.y = _2824.y;
    base_color.z = _2824.z;
    vec3 base_emissive = texture(sEmissive, vTexcoord0).xyz * materials.material_properties[material].emissive;
    bool is_background = (materialId & 1) == 1;
    bool is_raytraced = (materials.material_properties[material].flags & 3u) != 0u;
    if (is_raytraced)
    {
        float color_magnitude = length(base_color.xyz);
        if (color_magnitude > 1.0)
        {
            vec3 emissive_from_base_color = base_color.xyz - (base_color.xyz / vec3(color_magnitude));
            vec4 _2872 = base_color;
            vec3 _2874 = _2872.xyz - emissive_from_base_color;
            base_color.x = _2874.x;
            base_color.y = _2874.y;
            base_color.z = _2874.z;
            base_emissive += emissive_from_base_color;
        }
        vec3 param_9 = base_color.xyz;
        vec3 _2887 = saturate(param_9);
        base_color.x = _2887.x;
        base_color.y = _2887.y;
        base_color.z = _2887.z;
    }
    vec3 f0 = vec3(0.039999999105930328369140625);
    float metallic = metalness;
    vec3 diffuseColor = (base_color.xyz * (vec3(1.0) - f0)) * (1.0 - metallic);
    vec3 specularColor = mix(f0, base_color.xyz, vec3(metallic));
    float alphaRoughness = roughness * roughness;
    vec3 specularEnvironmentR0 = specularColor;
    float reflectance = max(max(specularColor.x, specularColor.y), specularColor.z);
    vec3 specularEnvironmentR90 = vec3(1.0) * clamp(reflectance * 25.0, 0.0, 1.0);
    MaterialInfo materialInfo = MaterialInfo(roughness, specularEnvironmentR0, alphaRoughness, diffuseColor, specularEnvironmentR90, specularColor);
    vec3 view = normalize(_1133.basic_params.camera_position - world);
    float param_12;
    float param_24;
    Rect param_32;
    vec3 param_34[4];
    float param_52;
    vec4 param_53;
    for (int light_idx = 0; light_idx < _2732.render_lights_params.lights_num; light_idx++)
    {
        LightProperties _2973;
        _2973.diffuse = lights.light_properties[light_idx].diffuse;
        _2973.direction = lights.light_properties[light_idx].direction;
        _2973.position = lights.light_properties[light_idx].position;
        _2973.up = lights.light_properties[light_idx].up;
        _2973.right = lights.light_properties[light_idx].right;
        _2973.dimensions = lights.light_properties[light_idx].dimensions;
        _2973._pad1 = lights.light_properties[light_idx]._pad1;
        _2973.intensity = lights.light_properties[light_idx].intensity;
        _2973.range = lights.light_properties[light_idx].range;
        _2973.cutoff = lights.light_properties[light_idx].cutoff;
        _2973.roughness_modifier = lights.light_properties[light_idx].roughness_modifier;
        _2973.is_area = lights.light_properties[light_idx].is_area;
        _2973.type = lights.light_properties[light_idx].type;
        _2973.projector_sampler = lights.light_properties[light_idx].projector_sampler;
        _2973.projector_intensity = lights.light_properties[light_idx].projector_intensity;
        _2973.shadowmap_sampler0 = lights.light_properties[light_idx].shadowmap_sampler0;
        _2973.shadowmap_sampler1 = lights.light_properties[light_idx].shadowmap_sampler1;
        _2973.shadowmap_sampler2 = lights.light_properties[light_idx].shadowmap_sampler2;
        _2973.shadowmap_sampler3 = lights.light_properties[light_idx].shadowmap_sampler3;
        _2973.cascade_distance0 = lights.light_properties[light_idx].cascade_distance0;
        _2973.cascade_distance1 = lights.light_properties[light_idx].cascade_distance1;
        _2973.cascade_distance2 = lights.light_properties[light_idx].cascade_distance2;
        _2973.cascade_distance3 = lights.light_properties[light_idx].cascade_distance3;
        _2973.mat_shadow_mv = lights.light_properties[light_idx].mat_shadow_mv;
        _2973.mat_shadow_p[0] = lights.light_properties[light_idx].mat_shadow_p[0];
        _2973.mat_shadow_p[1] = lights.light_properties[light_idx].mat_shadow_p[1];
        _2973.mat_shadow_p[2] = lights.light_properties[light_idx].mat_shadow_p[2];
        _2973.mat_shadow_p[3] = lights.light_properties[light_idx].mat_shadow_p[3];
        _2973.mat_shadow_mvp[0] = lights.light_properties[light_idx].mat_shadow_mvp[0];
        _2973.mat_shadow_mvp[1] = lights.light_properties[light_idx].mat_shadow_mvp[1];
        _2973.mat_shadow_mvp[2] = lights.light_properties[light_idx].mat_shadow_mvp[2];
        _2973.mat_shadow_mvp[3] = lights.light_properties[light_idx].mat_shadow_mvp[3];
        LightProperties light = _2973;
        vec3 light_color = vec3(0.0);
        if ((light.type & 2) != 0)
        {
            if ((light.type & 64) != 0)
            {
                Cone cone = Cone(light.cutoff * 1.0, light.range * 1.5, light.position.xyz, light.direction.xyz);
                Ray ray = Ray(_1133.basic_params.camera_position, normalize(vFrustum.xyz));
                float vv = 0.0;
                Cone param_10 = cone;
                Ray param_11 = ray;
                bool _3016 = intersect_cone(param_10, param_11, param_12);
                vv = param_12;
                if (_3016 == false)
                {
                    continue;
                }
            }
        }
        vec3 pointToLight = light.position.xyz - world;
        if ((light.type & 1) != 0)
        {
            pointToLight = -light.direction.xyz;
        }
        float NdotL = dot(vNorm, normalize(pointToLight));
        if ((materialId & 1) == 1)
        {
            NdotL = 1.0;
        }
        vec3 param_13 = pointToLight;
        MaterialInfo param_14 = materialInfo;
        vec3 param_15 = vNorm;
        vec3 param_16 = view;
        vec3 lighting = (getPointShade(param_13, param_14, param_15, param_16) * light.intensity) * light.diffuse.xyz;
        float shadow = 0.0;
        bool _3071 = (materials.material_properties[material].flags & 512u) != 0u;
        bool _3078;
        if (_3071)
        {
            _3078 = (light.type & 8) != 0;
        }
        else
        {
            _3078 = _3071;
        }
        bool calculate_shadows = _3078;
        if (is_background)
        {
            shadow = 0.0;
            lighting = vec3(1.0);
            calculate_shadows = false;
        }
        if (((light.type & 3) != 0) && calculate_shadows)
        {
            vec4 vShadowCoords = (light.mat_shadow_p[0] * light.mat_shadow_mv) * vec4(world, 1.0);
            float in_frustum = 0.0;
            float penumbraSampleNoise = texelFetch(s_BlueNoise, ivec3(screen_pos & ivec2(127), 0), 0).x;
            float param_17 = penumbraSampleNoise;
            vec4 param_18 = vShadowCoords;
            float param_19 = max(0.5, roughness * light.roughness_modifier) / 1024.0;
            int param_20 = 8;
            float param_21 = materials.material_properties[material].shadowmap_bias;
            float penumbra = PenumbraFromOccluderSearch(LightShadowmapSamplers[light.shadowmap_sampler0], param_17, param_18, param_19, param_20, param_21);
            vec4 param_22 = vShadowCoords;
            ivec2 param_23 = screen_pos;
            int param_25 = 12;
            float param_26 = max(0.5, (penumbra * roughness) * light.roughness_modifier) / 1024.0;
            float param_27 = materials.material_properties[material].shadowmap_bias;
            float _3162 = sampleShadowPCFNoiseOffset(LightShadowmapCmpSamplers[light.shadowmap_sampler0], param_22, param_23, param_24, param_25, param_26, param_27);
            in_frustum = param_24;
            shadow = _3162;
            if (NdotL <= 0.0)
            {
                shadow = max(shadow, smoothstep(0.0, -0.0500000007450580596923828125, NdotL));
            }
        }
        light_color = lighting * (1.0 - shadow);
        if ((light.type & 16) != 0)
        {
            vec4 vShadowCoords_1 = (light.mat_shadow_p[0] * light.mat_shadow_mv) * vec4(world, 1.0);
            vec4 param_28 = vShadowCoords_1;
            vec4 _3202 = sampleProjectorTexture(LightProjectorSamplers[light.projector_sampler], param_28);
            light_color += (_3202.xyz * ((1.0 - shadow) * light.projector_intensity));
        }
        if ((light.type & 130) == 130)
        {
            LightProperties param_29 = light;
            vec3 param_30 = world;
            float attenuation = light_calculate_spot_attenuation(param_29, param_30);
            light_color *= attenuation;
        }
        if (light.is_area != 0)
        {
            if ((materialId & 1) == 1)
            {
                light_color = normalize(light.diffuse.xyz) * light.intensity;
            }
            else
            {
                LightProperties param_31 = light;
                InitRect(param_31, param_32);
                Rect rect = param_32;
                Rect param_33 = rect;
                InitRectPoints(param_33, param_34);
                vec3 points[4] = param_34;
                float ltc_roughness = roughness;
                float ltc_intensity = light.intensity;
                vec3 dcol = light.diffuse.xyz;
                vec3 scol = dcol;
                vec3 col = vec3(0.0);
                vec3 pos = world;
                vec3 N = vNorm;
                vec3 V = normalize(_1133.basic_params.camera_position - world);
                float ndotv = clamp(dot(N, V), 0.0500000007450580596923828125, 1.0);
                vec2 uv = vec2(ltc_roughness, sqrt(1.0 - ndotv));
                uv = (uv * 0.984375) + vec2(0.0078125);
                vec4 t1 = texture(s_LTC1, uv);
                vec4 t2 = texture(s_LTC2, uv);
                mat3 Minv = mat3(vec3(vec3(t1.x, 0.0, t1.y)), vec3(0.0, 1.0, 0.0), vec3(vec3(t1.z, 0.0, t1.w)));
                vec3 param_35 = N;
                vec3 param_36 = V;
                vec3 param_37 = pos;
                mat3 param_38 = Minv;
                vec3 param_39[4] = points;
                bool param_40 = twoSided;
                vec3 _3343 = LTC_Evaluate(param_35, param_36, param_37, param_38, param_39, param_40);
                vec3 spec = _3343;
                spec *= ((scol * t2.x) + ((vec3(1.0) - scol) * t2.y));
                vec3 param_41 = N;
                vec3 param_42 = V;
                vec3 param_43 = pos;
                mat3 param_44 = mat3(vec3(1.0, 0.0, 0.0), vec3(0.0, 1.0, 0.0), vec3(0.0, 0.0, 1.0));
                vec3 param_45[4] = points;
                bool param_46 = twoSided;
                vec3 _3372 = LTC_Evaluate(param_41, param_42, param_43, param_44, param_45, param_46);
                vec3 diff = _3372;
                col = ((spec * specularColor) + ((dcol * diff) * diffuseColor)) * ltc_intensity;
                light_color = col;
            }
            if ((light.type & 128) == 128)
            {
                LightProperties param_47 = light;
                vec3 param_48 = world;
                float attenuation_1 = light_calculate_area_attenuation(param_47, param_48);
                light_color *= attenuation_1;
            }
        }
        if (is_raytraced)
        {
            vec2 p = vTexcoord0 * _2732.render_lights_params.raytrace_scaling_factor;
            vec4 raytraceColor = texture(sRaytrace, p);
            vec4 _3416 = raytraceColor;
            vec3 _3418 = _3416.xyz * _2732.render_lights_params.raytrace_strength;
            raytraceColor.x = _3418.x;
            raytraceColor.y = _3418.y;
            raytraceColor.z = _3418.z;
            light_color = (light_color * (1.0 - raytraceColor.w)) + (specularColor * raytraceColor.xyz);
        }
        if ((light.type & 64) != 0)
        {
            float attenuation_2 = 0.0;
            vec4 projector_color = vec4(0.0);
            LightProperties param_49 = light;
            vec4 param_50 = vFrustum;
            float param_51 = depth;
            float _3452 = volumetric_sample_shadow_spot(param_49, param_50, param_51, param_52, param_53);
            attenuation_2 = param_52;
            projector_color = param_53;
            float volume_attenuation = _3452;
            outColor.w += (light.diffuse.w * volume_attenuation);
        }
        vec4 _3464 = outColor;
        vec3 _3466 = _3464.xyz + light_color;
        outColor.x = _3466.x;
        outColor.y = _3466.y;
        outColor.z = _3466.z;
    }
    vec4 _3476 = outColor;
    vec3 _3478 = _3476.xyz + base_emissive;
    outColor.x = _3478.x;
    outColor.y = _3478.y;
    outColor.z = _3478.z;
    if (is_background == false)
    {
        vec3 param_54 = vNorm;
        vec3 param_55 = view;
        vec3 param_56 = materialInfo.diffuseColor;
        vec3 param_57 = specularEnvironmentR0;
        vec3 param_58 = specularEnvironmentR90;
        float param_59 = materials.material_properties[material].roughness;
        vec4 _3506 = outColor;
        vec3 _3508 = _3506.xyz + (ibl(param_54, param_55, param_56, param_57, param_58, param_59) * 0.100000001490116119384765625);
        outColor.x = _3508.x;
        outColor.y = _3508.y;
        outColor.z = _3508.z;
        vec4 _3527 = outColor;
        vec3 _3529 = _3527.xyz * (((materials.material_properties[material].emissive * materials.material_properties[material].emissive_factor) * base_color.xyz) + vec3(1.0));
        outColor.x = _3529.x;
        outColor.y = _3529.y;
        outColor.z = _3529.z;
        vec4 _3546 = outColor;
        vec3 _3548 = _3546.xyz + ((materials.material_properties[material].emissive * materials.material_properties[material].emissive_factor) * base_color.xyz);
        outColor.x = _3548.x;
        outColor.y = _3548.y;
        outColor.z = _3548.z;
    }
}

 