#version 460

#ifndef SPIRV_ENABLED
#extension GL_NV_gpu_shader5 : enable
#endif

#define LIGHT_PROPERTIES_BINDING 1
#ifndef MATERIAL_PROPERTIES_BINDING
#define MATERIAL_PROPERTIES_BINDING 2
#endif

#include <shaders/materials/commons.glsl >
#include <shaders/materials/commons_rand.glsl >
#include <shaders/commons_hlsl.glsl >
#include <shaders/particles/particles_commons.glsl >

uniform sampler2DArray s_BlueNoise;
#include <shaders/deferred/lighting_support.glsl>

layout(std140, row_major) uniform TransformParamsBuffer{
	EntityTransformParams transform_params;
};

layout(std430) buffer RibbonPositionsData {
	float position[];
} prt_position_snapshots;

struct ParticleRibbonState
{
	float generation_id;
	uint segments;
};

layout(std430) buffer RibbonStatesData {
	ParticleRibbonState states[];
} prt_states_snapshots;

struct RibbonRenderParams
{
	int   max_segments;
	int   base_idx;
	int   base_segment;
	int   particles_per_segment;
	float thickness;
	int   lights_num;
	int   material_index;
	float first_segment_time_factor;
	int   render_as_strips;
	int   _pad0;
	int   _pad1;
	int   _pad2;
};

layout(std140) uniform RibbonRenderParamsBuffer {
	RibbonRenderParams ribbon_render_params;
};

#if defined(DEFERRED_PASS) || defined(SHADOWMAP_PASS)
#ifndef SPIRV_ENABLED
out Vertex
{
	vec3 vCoords;
	f16vec3 vNorm;
	f16vec3 vWorldNorm;
	vec3 vLocalPos;
	vec3 vWorldPos;
	f16vec4 vColor;
	f16vec2 vUV0;
} vtx_output;

out uint instanceID;
#else

layout(location = 1) out struct
{
	vec3 vCoords;
	f16vec3 vNorm;
	f16vec3 vWorldNorm;
	vec3 vLocalPos;
	vec3 vWorldPos;
	f16vec4 vColor;
	f16vec2 vUV0;
} vtx_output;
layout(location = 0) out uint instanceID;

#endif
#endif

vec3 hsv2rgb(vec3 c) {
	vec4 K = vec4(1.0, 2.0 / 3.0, 1.0 / 3.0, 3.0);
	vec3 p = abs(fract(c.xxx + K.xyz) * 6.0 - K.www);
	return c.z * mix(K.xxx, clamp(p - K.xxx, 0.0, 1.0), c.y);
}

vec3 rgb2hsv(vec3 c)
{
	vec4 K = vec4(0.0, -1.0 / 3.0, 2.0 / 3.0, -1.0);
	vec4 p = mix(vec4(c.bg, K.wz), vec4(c.gb, K.xy), step(c.b, c.g));
	vec4 q = mix(vec4(p.xyw, c.r), vec4(c.r, p.yzx), step(p.x, c.r));

	float d = q.x - min(q.w, q.y);
	float e = 1.0e-10;
	return vec3(abs(q.z + (q.w - q.y) / (6.0 * d + e)), d / (q.x + e), q.x);
}

// NOTE: this is new version using triangle strips and not lines as lines with thickness
// stopped being supported:()

int calculate_vidx_for_ribbon_segment(int ribbon_id, int ribbon_segment)
{
	ribbon_segment = ribbon_render_params.base_segment - ribbon_segment;
	if (ribbon_segment < 0)
		ribbon_segment += ribbon_render_params.max_segments;
	if (ribbon_segment >= ribbon_render_params.max_segments)
		ribbon_segment -= ribbon_render_params.max_segments;
	int vidx = ribbon_segment * ribbon_render_params.particles_per_segment + ribbon_id + ribbon_render_params.base_idx;
	return vidx;
}

// should be enabled only for the transparent particles which don't land in the deferred buffer?
#ifdef DEFERRED_PASS
vec4 calculate_lighting(vec4 color, MaterialPropertiesGPU material)
{
	vec3 light_color = vec3(0.0);
	for (int light_idx = 0; light_idx < ribbon_render_params.lights_num; light_idx++)
	{
		LightProperties light = lights.light_properties[light_idx];
		f16vec4 projector_color = f16vec4(0.0);
		vec3 world = vtx_output.vWorldPos;// -transform_params.vCameraPosition.xyz;

		float shadow = 0.0;
		{
			bool calculate_shadows = (light.type & LightType_Shadowcasting) != 0;
			if ((material.flags & MaterialFlag_ShadowReceive) == 0)
				calculate_shadows = false;

			if ((light.type & (LightType_Spot | LightType_Directional)) != 0 && calculate_shadows)
			{
				float penumbra_scaling_factor = 0.5;

				vec4 vShadowCoords = light.mat_shadow_mvp[0] * vec4(world.xyz, 1.0);

				float in_frustum = 0.0;
				shadow = sampleShadowPCF(
					LightShadowmapCmpSamplers[light.shadowmap_sampler0],
					vShadowCoords,
					in_frustum,
					3,
					-0.001);	// was bias = 0.0001
			}
		}

		shadow = 1.0 - shadow;

		if ((light.type & (LightType_Spot | LightType_Attenuating)) == (LightType_Spot | LightType_Attenuating))
		{
			float attenuation = light_calculate_spot_attenuation(light, world.xyz);
			shadow *= attenuation;
		}

		light_color.rgb += light.diffuse.rgb * sqrt(light.intensity) * shadow;
	}

	color.rgb *= light_color;
	color.rgb *= material.diffuse.rgb;
	color.rgb += material.emissive.rgb * material.emissive_factor;

	return color;
}
#endif

void main() {

#ifdef DEFERRED_PASS	
	instanceID = SYS_InstanceIndex;
#endif

	// fetch input data
	int odd_even = (SYS_VertexIndex & 1) == 0 ? 1 : -1;
	int ribbon_id;
	int ribbon_segment;

	if (ribbon_render_params.render_as_strips != 0)
	{
		ribbon_id = SYS_InstanceIndex;
		ribbon_segment = int(SYS_VertexIndex) / 2;
	}
	else
	{
		uint max_segments_uint = uint(ribbon_render_params.max_segments);
		const int RIBBONS_PER_INSTANCE = 64;
		ribbon_segment = int(SYS_VertexIndex) / 2;
		ribbon_id = SYS_InstanceIndex * RIBBONS_PER_INSTANCE + int(uint(ribbon_segment) /max_segments_uint);
		ribbon_segment = int(uint(ribbon_segment) % max_segments_uint);
	}

	if (!prt_is_alive(ribbon_id))
	{
		gl_Position = vec4(0.0, 0.0, 0.0, -1.0);
		return;
	}

	int ribbon_segments = max(0, min(ribbon_render_params.max_segments, int(prt_states_snapshots.states[ribbon_id].segments)) - 1);
	
	ribbon_segment = min(ribbon_segment, ribbon_segments);
	int vidx = calculate_vidx_for_ribbon_segment(ribbon_id, ribbon_segment);

	vec3 vInstPosition = vec3(prt_position_snapshots.position[vidx * 3 + 0], prt_position_snapshots.position[vidx * 3 + 1], prt_position_snapshots.position[vidx * 3 + 2]);
	vec3 vInstNormal = vec3(1.0, 0.0, 0.0);

	vec4 vInstColor = prt_get_color(ribbon_id);
	ParticleState vState = prt_get_state(ribbon_id);

	bool is_last = ribbon_segment >= ribbon_segments;
	bool is_first = ribbon_segment == 0;

	if (ribbon_segment > 0 || true)
	{
		float shrink_factor = ribbon_render_params.first_segment_time_factor;
		vec3  pos_off = vec3(0.01);	// HACK!
		float min_shrink_modifier = 0.15;

		int next_vidx = calculate_vidx_for_ribbon_segment(ribbon_id, ribbon_segment - 1);
		vec3 nextPosition = vec3(prt_position_snapshots.position[next_vidx * 3 + 0], prt_position_snapshots.position[next_vidx * 3 + 1], prt_position_snapshots.position[next_vidx * 3 + 2]);
		//vec3 dir = normalize(cross(-normalize(vInstPosition - nextPosition + pos_off), vInstNormal));
		//vec3 dir = cross(-normalize(vInstPosition - nextPosition + pos_off), vInstNormal);
		vec3 dir = cross(-normalize(vInstPosition), vInstNormal);

		//if (dot(normalize(vInstPosition), normalize(nextPosition)) < 0.0)
		//	dir = vec3(0.0, 0.0, 1.0);
		

		// if we are last segment then interpolate to shrink it towards previous. smoother end of trail motion
		if (ribbon_segment == ribbon_segments && ribbon_segments > 1) {
			vInstPosition = mix(vInstPosition, nextPosition, ribbon_render_params.first_segment_time_factor);
			dir *= min_shrink_modifier;
			//	vInstColor.rgb *= 0.0;
		}

		if (is_first)
		{
			dir *= -min_shrink_modifier;
		}

		// if we are first after leading, also transition it from the pingle point towards full width instead of immediate smap into full width
		if (ribbon_segment == 1 && ribbon_segments > 2) {
			dir *= max(min_shrink_modifier, shrink_factor);
		}

		// if we are first before last, also transition it from the pingle point towards full width instead of immediate smap into full width
		if (ribbon_segment == (ribbon_segments - 1) && ribbon_segments > 2) {
			dir *= max(min_shrink_modifier, 1.0 - shrink_factor);
		}

		//if (ribbon_segment < ribbon_segments) {
			vInstPosition .xyz += dir * ribbon_render_params.thickness * float(odd_even);
		//}
	}
	//else
	{
	}

	vec3 pos = vInstPosition;
#ifdef DEFERRED_PASS	
	vtx_output.vLocalPos = pos;
#endif	
	vec3 vPos1 = pos;
	vec3 vPos = vector_transform_by_mat43(vPos1, transform_params.mModelView);

#ifdef DEFERRED_PASS
	MaterialPropertiesGPU material = materials.material_properties[ribbon_render_params.material_index];
	vtx_output.vWorldPos = vector_transform_by_mat43(vPos1, transform_params.mModel);

    // calculate displacement. this sucks as it will not be perpendicular to camera:()
    #if 0
    {
        int next_vidx = (vidx + 1) % RIBBON_LENGTH;
        vec3 nextPosition = vec3(indata.verts[next_vidx].px, indata.verts[next_vidx].py, indata.verts[next_vidx].pz);
        vec3 vNextPos;
        vNextPos.x = dot(transform_params.mModelView[0], vec4(nextPosition, 1.0));
	    vNextPos.y = dot(transform_params.mModelView[1], vec4(nextPosition, 1.0));
	    vNextPos.z = dot(transform_params.mModelView[2], vec4(nextPosition, 1.0));

        vec3 dir = cross(vec3(0.0, 1.0, 0.0), normalize(vPos.xyz - vNextPos));
        vPos.xyz += vec3(odd_even) * dir * 0.0075;
    }
    #endif

	// skew the normals so we dont get so much black lighting
	vtx_output.vWorldNorm.x = float16_t(vidx);
	vtx_output.vWorldNorm.y = float16_t(ribbon_segment);
	vtx_output.vWorldNorm.z = float16_t(ribbon_id);
	vtx_output.vWorldNorm.xyz = f16vec3(vInstPosition);

	vtx_output.vNorm = f16vec3(vInstNormal);
	vtx_output.vCoords = vPos.xyz;

	vInstColor = calculate_lighting(vInstColor, material);

	float life_time_factor = min(1.0, vState.life_time / vState.life_span);
	//if (life_time_factor < 0.1)
	life_time_factor = 1.0 - sin(life_time_factor * M_PI * 0.5);
	vInstColor.rgb *= 1.0 - pow(life_time_factor, 4.0);
	vtx_output.vColor = f16vec4(vInstColor);

#endif // DEFERRED_PASS

	gl_Position = transform_params.mProjection * vec4(vPos, 1.0);
}

