; **********************************************************************************************************************************
; *  "The Christmas Star Demo" - A C64 entry for the "Wild" category of Logiker's "Vintage Computing Christmas Challenge 2022"
; *
; *  Concept and code by Geir Straume
; *  SID arrangements by Roy Widding
; **********************************************************************************************************************************

; **********************************************************************************************************************************
; *  Constants, variables and addresses
; **********************************************************************************************************************************

; Constants
MAX_SPRITES             = 13               ; Max number of logical sprites
CSPC                    = 32               ; Space character
CBLOCK                  = 91               ; Block character
CSTAR                   = 92               ; Inverse asterisk character
CLINE                   = 93               ; Squiggly line character

; Zero-page variables
                        .virtual $60

Counter                 .byte ?            ; Variable for counting rows and columns
Timer                   .byte ?            ; Timer for controlling the demo flow
SynchWithIRQ            .byte ?            ; 0=Start new mainline cycle, 128=Mainline cycle complete
LogSprCount             .byte ?            ; Number of active logical sprites
LogSprIdx               .byte ?            ; Index to current logical sprite
ActSprIdx               .byte ?            ; Index to current actual sprite
IntroTextIdx            .byte ?            ; Index to current char in intro text
IntroTextScreenIdx      .byte ?            ; Index to current screen column for intro text
IntroTextDir            .byte ?            ; Current direction for displaying intro text (0=right, 128=left)
IntroTextFadeIdx        .byte ?            ; Index to current color when fading the intro text
StarCoordAdjustIdx      .byte ?            ; Index to star movement data
StarColorIdx            .byte ?            ; Index to star colors
RotationDelay           .byte ?            ; Delay for star rotation
RotationIdx             .byte ?            ; Index to star rotation frame
IRQTasks                .byte ?            ; 0=None, 1=Music, 2=Music+scroll+sprites
XScroll                 .byte ?            ; Horizontal smooth scroll value (0-7)
CharColorIdx            .byte ?            ; Index to table with scroll text color changes
CharColor               .byte ?            ; Current char color used in scroll text
ScreenAddr              .word ?            ; Pointer to screen address
ColorAddr               .word ?            ; Pointer to color RAM
TextAddr                .word ?            ; Pointer to current char in scroll text
LogSprXCoord            .fill MAX_SPRITES  ; X-coordinates of logical sprites
LogSprYCoord            .fill MAX_SPRITES  ; Y-coordinates of logical sprites
LogSprImage             .fill MAX_SPRITES  ; Sprite pointers for logical sprites
LogSprColor             .fill MAX_SPRITES  ; Colors of logical sprites
LogSprNumber            .fill MAX_SPRITES  ; Logical sprite numbers sorted by y-coordinate
StarAppearDelay         .fill MAX_SPRITES  ; Delay before stars should appear
StarMovementIdx         .fill MAX_SPRITES  ; Index to table of adjustment values for y-coordinate of star
LogSprXCoordLSB         .fill MAX_SPRITES  ; LSB of x-coordinates for stars
StarAnimDelay           .fill MAX_SPRITES  ; Delay for each star animation frame
StarXCoordAdjustLo      .fill MAX_SPRITES  ; Lo byte of adjustment value for moving stars

.cerror * > $100, "Zero page overflow"

                        .endvirtual

; Other variables
                        .virtual $200

StarXCoordAdjustHi      .fill MAX_SPRITES  ; Hi byte of adjustment value for moving star

                        .endvirtual

; Addresses
Screen                  = $0400            ; Screen RAM
SpritePointer           = Screen+1016      ; First sprite pointer
v                       = $d000            ; VIC-II
s                       = $d400            ; SID
ColorRAM                = $d800            ; What it says on the tin
NMIVectorHW             = $fffa            ; Hardware NMI vector
IRQVectorHW             = $fffe            ; Hardware IRQ vector

MusicBaseAddr           = $4000            ; Base address of music player
InitMusic               = MusicBaseAddr    ; Initiate music
PlayMusic               = MusicBaseAddr+3  ; Play music

; **********************************************************************************************************************************
; *  Start of program
; **********************************************************************************************************************************

                        * = $0801
; BASIC stub
                        .word (NextLine),1337
                        .null $9e,format("%4d",Start)
NextLine                .word 0

Start                   lda #$7f
                        sta $dc0d                   ; Disable timer interrupts
                        lda $dc0d                   ; Acknowledge CIA #1 interrupts

                        sei
                        cld
                        ldx #$ff
                        txs

                        ldx #90-1
ZpCodeLoop              lda ZeroPageCode,x          ; Copy some code to zero-page
                        sta $02,x
                        dex
                        bpl ZpCodeLoop

                        lda #<DoRti
                        ldx #>DoRti
                        sta NMIVectorHW             ; Set address of NMI routine
                        stx NMIVectorHW+1

                        lda #<DispatchIRQ
                        ldx #>DispatchIRQ
                        sta IRQVectorHW             ; Set address of IRQ routine
                        stx IRQVectorHW+1

                        lda #$35
                        sta 1                       ; Enable I/O and disable BASIC and KERNAL ROMs

                        lda #%00011011
                        sta $d011                   ; Set screen visible
                        lda #%00000000
                        sta $d016
                        lda #%00011000
                        sta $d018                   ; Screen RAM at $0400 and charset at $2000
                        lda #0
                        sta $d017                   ; No expanded sprites
                        sta $d01d
                        sta $d020                   ; Black border
                        sta $d021                   ; Black screen
                        lda #%11111111
                        sta $d015                   ; Enable all sprites

                        ldy #0
ClearLoop               lda #CBLOCK                 ; Block character
                        sta Screen,y                ; Put on entire screen
                        sta Screen+$100,y
                        sta Screen+$200,y
                        sta Screen+$300,y
                        lda #0
                        sta ColorRAM,y              ; Set char color to black
                        sta ColorRAM+$100,y
                        sta ColorRAM+$200,y
                        sta ColorRAM+$300,y
                        iny
                        bne ClearLoop

                        ldx #38
TextLineInit            lda #CSPC                   ; Space character
                        sta Screen+40,x
                        lda #1                      ; White
                        sta ColorRAM+40,x
                        dex
                        bpl TextLineInit

                        lda #<MusicData             ; Source address start
                        ldx #>MusicData
                        sta ScreenAddr
                        stx ScreenAddr+1
                        lda #<MusicBaseAddr         ; Destination address
                        ldx #>MusicBaseAddr
                        sta ColorAddr
                        stx ColorAddr+1
                        lda #<EndOfData-1           ; Source address end (inclusive)
                        ldx #>EndOfData-1
                        jsr CopyMemory              ; Copy music player to its final destination

                        ldx #MAX_SPRITES-1
LogSprInitLoop          txa
                        sta LogSprNumber,x          ; Set initial sort order for all logical sprites
                        dex
                        bpl LogSprInitLoop

                        lda #255
                        sta v+18                    ; Set line for initial raster interrupt
                        lda #%11111111
                        sta v+25                    ; Acknowledge all interrupts
                        lda #%00000001
                        sta v+26                    ; Interrupt Mask Register

                        ldx #24
                        lda #0
ResetSID                sta s,x
                        dex
                        bpl ResetSID

                        stx IntroTextIdx

                        sta IntroTextScreenIdx
                        sta IntroTextDir
                        sta IntroTextFadeIdx
                        sta IRQTasks
                        sta RotationDelay
                        sta StarCoordAdjustIdx
                        sta StarColorIdx
                        sta LogSprCount

                        ldx #7
                        stx XScroll
                        inx
                        stx RotationIdx

                        lda #<DemoFlow
                        ldx #>DemoFlow
                        sta DemoFlowAddr+1          ; Set initial demo state handler
                        stx DemoFlowAddr+2

                        ldx #MAX_SPRITES-1
StarInitLoop            lda StarAppearDelayData,x   ; Initialize delays for appearing star sprites
                        sta StarAppearDelay,x
                        dex
                        bpl StarInitLoop

                        jsr ResetScrollText

                        lda #0
                        jsr InitMusic

                        lda #128
                        sta SynchWithIRQ

                        cli

; **********************************************************************************************************************************
; *  Main program loop
; **********************************************************************************************************************************

MainLoop                bit SynchWithIRQ
                        bmi MainLoop

DemoFlowAddr            jsr DemoFlow                ; Address is modified by code

                        lda #128
                        sta SynchWithIRQ
                        bne MainLoop

; **********************************************************************************************************************************
; *  Demo state handlers
; **********************************************************************************************************************************

DemoFlow                lda #50
                        sta Timer
                        jsr NextDemoState

; State: Waiting before displaying initial star with asterisk characters
                        dec Timer
                        bne HandlerDone

                        lda #CSTAR                  ; Inverse asterisk character
                        jsr DisplayInitialStar
                        lda #80
                        sta Timer
                        jsr NextDemoState

; State: Waiting before starting the music
                        dec Timer
                        bne HandlerDone

                        lda #8
                        sta Timer
                        inc IRQTasks                ; 1=Music
                        jsr NextDemoState

; State: Displaying "When you wish upon a char" on the screen (synchronized with music)
                        jsr IntroText
                        bne HandlerDone

                        jsr DisplaySquigglyLine
                        lda #80
                        sta Timer
                        jsr NextDemoState

; State: Waiting before correcting text
                        dec Timer
                        bne HandlerDone

                        lda #1
                        sta Timer
                        jsr NextDemoState

; State: Correcting "char" to "star"
                        jsr IntroText
                        bne HandlerDone

                        lda #120
                        sta Timer
                        jsr NextDemoState

; State: Waiting after correcting text
                        dec Timer
                        bne HandlerDone

                        lda #33
                        sta Timer
                        jsr NextDemoState

; State: Fading text to black
                        jsr FadeIntroText
                        bne HandlerDone

                        lda #CSPC                   ; Space character (white star)
                        jsr DisplayInitialStar
                        lda #110
                        sta Timer
                        jsr NextDemoState

; State: Waiting before turning star into white block characters (for rotation animation)
                        dec Timer
                        bne HandlerDone

                        lda #CBLOCK                 ; Block character
                        jsr DisplayInitialStar
                        lda #100
                        sta Timer
                        jsr NextDemoState

; State: Displaying rotating star
                        dec Timer
                        bne DoRotation
                        inc IRQTasks                ; 2=Music+scroll+sprites
                        jsr NextDemoState

; State: Displaying rotating star, scrolling text and moving star sprites
DoRotation              jsr DisplayRotationFrame

HandlerDone             rts

; Set address of state handler
NextDemoState           pla                         ; Get lo byte of return address
                        clc
                        adc #1                      ; Adjust for next instruction
                        sta DemoFlowAddr+1          ; Set address in code
                        pla
                        adc #0
                        sta DemoFlowAddr+2
                        rts

; **********************************************************************************************************************************
; *  Display the intro text one character at a time
; **********************************************************************************************************************************

IntroText               dec Timer
                        bne DoRts

                        inc IntroTextIdx            ; Next char
                        ldx IntroTextIdx
                        lda IntroTextTiming,x       ; Get timer value for char
                        sta Timer
                        lda IntroTextData,x         ; Get intro text char
                        beq DoRts                   ; Branch if done
                        bpl CheckForLineRemoval     ; Branch if not toggling direction
                        lda IntroTextDir
                        eor #128                    ; Toggle direction for displaying text
                        sta IntroTextDir
                        lda #CSPC
                        bne DisplayIntroChar        ; Branch always
CheckForLineRemoval     cpx #27                     ; Remove the squiggly line now?
                        bne DisplayIntroChar        ; Branch if no
                        ldx #CSPC
                        stx Screen+40*2+29
                        stx Screen+40*2+30
                        stx Screen+40*2+31
                        stx Screen+40*2+32
DisplayIntroChar        ldy IntroTextScreenIdx
                        sta Screen+40*1+8,y         ; Put char on screen
                        bit IntroTextDir            ; Displaying text to the left?
                        bmi TextGoingLeft           ; Branch if yes
                        iny
                        .char $24                   ; Skip next byte using a bit instruction
TextGoingLeft           dey
                        sty IntroTextScreenIdx
DoRts                   rts

DisplaySquigglyLine     lda #2
                        sta ColorRAM+40*2+29
                        sta ColorRAM+40*2+30
                        sta ColorRAM+40*2+31
                        sta ColorRAM+40*2+32
                        lda #CLINE                  ; Squiggly line character
                        sta Screen+40*2+29
                        sta Screen+40*2+30
                        sta Screen+40*2+31
                        sta Screen+40*2+32
                        rts

; **********************************************************************************************************************************
; *  Fade intro text to black
; **********************************************************************************************************************************

FadeIntroText           dec Timer
                        beq Exit
                        lda Timer
                        and #7
                        bne Exit
                        ldx IntroTextFadeIdx
                        lda FadeColors,x
                        ldx #24
ColorLoop               sta ColorRAM+40*1+8,x
                        dex
                        bpl ColorLoop
                        inc IntroTextFadeIdx
Exit                    rts

FadeColors              .char 15,12,11,0

; **********************************************************************************************************************************
; *  Do rotation of star
; **********************************************************************************************************************************

DisplayRotationFrame    dec RotationDelay
                        bpl Exit
                        lda #2
                        sta RotationDelay

                        ldx RotationIdx
                        dex
                        bpl SkipIdxReset
                        ldx #8
SkipIdxReset            stx RotationIdx

                        lda StarDataLo,x
                        sta DataAddr1+1             ; Modify code in zero-page
                        sta DataAddr2+1
                        lda StarDataHi,x
                        sta DataAddr1+2
                        sta DataAddr2+2

                        lda #$d3                    ; Base address for displaying star by modifying color RAM
                        ldx #>ColorRAM
                        sta ColorAddr
                        stx ColorAddr+1

                        jmp DisplayStar            ; Jump to code in zero-page

; **********************************************************************************************************************************
; *  Top of screen IRQ
; **********************************************************************************************************************************

TopIRQ                  ldx #1
                        stx $d021                   ; Set screen color to white
                        dex
                        stx $d016                   ; Set static screen

                        lda LogSprCount             ; Are any logical sprites being displayed?
                        beq SetBottomIRQ            ; Branch if no
                        cmp LogSprIdx               ; Have they all been displayed already?
                        beq SetBottomIRQ            ; Branch if yes

                        lda #0
                        sta ActSprIdx

                        lda #<MultiplexerIRQ
                        ldx #>MultiplexerIRQ
                        sta IRQVector+1             ; Set interrupt routine address
                        stx IRQVector+2

                        ldy LogSprIdx               ; Get index to table of logical sprite no
                        ldx LogSprNumber,y          ; Get next logical sprite no to be displayed
SetNextMultiplexerIRQ   lda LogSprYCoord,x          ; Get y-coordinate of logical sprite
                        sec
                        sbc #8                      ; Calculate raster line for interrupt
                        tay
                        jmp SetLineAndReturnFromIRQ

; **********************************************************************************************************************************
; *  Sprite multiplexer IRQ
; **********************************************************************************************************************************

MultiplexerIRQ          ldy LogSprIdx               ; Get index to table of logical sprite no
                        ldx LogSprNumber,y          ; Get logical sprite no to be displayed
MultiplexLoop           ldy ActSprIdx               ; Get actual sprite no to be displayed
                        lda LogSprImage,x           ; Get image for logical sprite
                        sta SpritePointer,y         ; Set sprite pointer
                        lda LogSprColor,x           ; Get color of logical sprite
                        sta v+39,y                  ; Set sprite color
                        asl                         ; Shift x-coordinate MSB into carry flag
                        lda v+16                    ; Get x-coordinate MSB register value
                        and AndTable,y              ; Clear MSB
                        bcc SetXMSBReg
                        ora OraTable,y              ; Set MSB
SetXMSBReg              sta v+16
                        tya
                        asl                         ; Calculate index to VIC sprite coordinate registers
                        tay
                        lda LogSprYCoord,x          ; Get and set y-coordinate of sprite
                        sta v+1,y
                        lda LogSprXCoord,x          ; Get and set x-coordinate of sprite
                        sta v,y
                        ldx ActSprIdx               ; Get current actual sprite no
                        inx
                        cpx #8                      ; Was the last actual sprite just displayed?
                        bne SetActSprIdx            ; Branch if no
                        ldx #0                      ; Reset actual sprite no
SetActSprIdx            stx ActSprIdx               ; Store the new value
IncLogSprIdx            inc LogSprIdx
                        ldy LogSprIdx
                        cpy LogSprCount             ; Have all logical sprites been displayed?
                        beq SetBottomIRQ            ; Branch if yes
                        ldx LogSprNumber,y          ; Get next logical sprite no to be displayed
                        lda LogSprYCoord,x          ; Get its y-coordinate
                        sec
                        sbc v+18                    ; Subtract current raster line
                        bcc IncLogSprIdx            ; It's too late to display this sprite
                        cmp #12                     ; Should the sprite be displayed immediately?
                        bcc MultiplexLoop           ; Branch if yes
                        jmp SetNextMultiplexerIRQ   ; Set up the next multiplexer interrupt

SetBottomIRQ            lda #<BottomIRQ
                        ldx #>BottomIRQ
                        ldy #255                    ; Raster line
                        jmp SetNextAndReturnFromIRQ

; **********************************************************************************************************************************
; *  Bottom of screen IRQ
; **********************************************************************************************************************************

BottomIRQ               ldx #255                    ; Make sure any actual sprites not in use are outside of the visible screen
                        stx v+1
                        stx v+3
                        stx v+5
                        stx v+7
                        stx v+9
                        stx v+11
                        stx v+13
                        stx v+15

                        inx
                        stx $d021                   ; Set screen color to black

                        lda IRQTasks                ; Are there some tasks to be done?
                        beq SkipTasks               ; Branch if no

                        jsr PlayMusic

                        lda IRQTasks
                        cmp #2                      ; Should we move star sprites and scroll text too?
                        bcc SkipTasks               ; Branch if no

                        jsr MoveStarSprites
                        jsr ScrollText

SkipTasks               lda #0
                        sta SynchWithIRQ            ; Tell mainline that a frame has been completed

                        ldx LogSprCount             ; Are there any logical sprites to display?
                        beq SetTopIRQ               ; Branch if no

                        lda #255                    ; Make sure any inactive logical sprites are sorted last
YCoordLoop              cpx #MAX_SPRITES
                        beq DoSort
                        sta LogSprYCoord,x
                        inx
                        bne YCoordLoop

DoSort                  jsr SortSprites             ; Determine display order of all logical sprites

                        lda #0                      ; Reset the multiplexer
                        sta LogSprIdx
                        sta ActSprIdx

                        jsr DisplayInitialSprites   ; Display up to eight initial sprites

SetTopIRQ               lda #<TopIRQ
                        ldx #>TopIRQ
                        ldy #75                     ; Raster line
                        jmp SetNextAndReturnFromIRQ

; **********************************************************************************************************************************
; *  Display up to eight logical sprites
; **********************************************************************************************************************************

DisplayInitialSprites   ldy LogSprIdx               ; Get index to table of logical sprite no
                        ldx LogSprNumber,y          ; Get logical sprite no to be displayed
DisplayLoop             ldy ActSprIdx               ; Get actual sprite no to be displayed
                        lda LogSprImage,x           ; Get image for logical sprite
                        sta SpritePointer,y         ; Set sprite pointer
                        lda LogSprColor,x           ; Get color of logical sprite
                        sta v+39,y                  ; Set sprite color
                        asl                         ; Shift x-coordinate MSB into carry flag
                        lda v+16                    ; Get x-coordinate MSB register value
                        and AndTable,y
                        bcc SetXMSBReg2
                        ora OraTable,y
SetXMSBReg2             sta v+16
                        tya
                        asl                         ; Calculate index to VIC sprite coordinate registers
                        tay
                        lda LogSprYCoord,x          ; Get and set y-coordinate of sprite
                        sta v+1,y
                        lda LogSprXCoord,x          ; Get and set x-coordinate of sprite
                        sta v,y
                        inc LogSprIdx
                        ldx ActSprIdx               ; Get current actual sprite no
                        inx
                        cpx #8                      ; Was the last actual sprite just displayed?
                        beq EndDisplay              ; Branch if yes
                        stx ActSprIdx
                        ldy LogSprIdx
                        cpy LogSprCount             ; Have all logical sprites been displayed?
                        beq EndDisplay              ; Branch if yes
                        ldx LogSprNumber,y          ; Get next logical sprite no to be displayed
                        lda LogSprYCoord,x          ; Get its y-coordinate
                        bne DisplayLoop             ; Branch always
EndDisplay              rts

; **********************************************************************************************************************************
; *  Determine display order of all logical sprites based upon their y-coordinates
; **********************************************************************************************************************************

SortSprites             ldy LogSprNumber            ; Get 1st logical sprite no to compare
                        lda LogSprYCoord,y          ; Get the y-coordinate
                        ldx #1
SpriteSortLoop          ldy LogSprNumber,x          ; Get 2nd logical sprite no to compare
                        cmp LogSprYCoord,y          ; Compare the two y-coordinates
                        beq YCoordIsEqual
                        bcc YCoordIsLess
                        stx CurrentSortIndex+1      ; Save current index to sort order table (self-modifying code)
                        sty SpriteSwapDone+1        ; Save 2nd logical sprite no (self-modifying code)
                        lda LogSprYCoord,y          ; Get its y-coordinate
                        ldy LogSprNumber-1,x        ; Swap sprite numbers
                        sty LogSprNumber,x
SpriteSwapLoop          dex
                        beq SpriteSwapDone          ; Branch if start of table has been reached
                        ldy LogSprNumber-1,x        ; Swap sprite numbers
                        sty LogSprNumber,x
                        cmp LogSprYCoord,y          ; Done swapping?
                        bcc SpriteSwapLoop          ; Branch if no
SpriteSwapDone          ldy #0
                        sty LogSprNumber,x
CurrentSortIndex        ldx #0                      ; Get current index to sort order table again
                        ldy LogSprNumber,x          ; Get 1st logical sprite no to compare
YCoordIsLess            lda LogSprYCoord,y          ; Get the y-coordinate
YCoordIsEqual           inx
                        cpx #MAX_SPRITES            ; Have all logical sprites been processed?
                        bne SpriteSortLoop          ; Branch if no
                        rts

; **********************************************************************************************************************************
; *  Move all the star sprites
; **********************************************************************************************************************************

MoveStarSprites         ldx #MAX_SPRITES-1          ; Number of stars to move - 1
StarLoop                lda StarAppearDelay,x       ; Get current delay value for star
                        bmi AppearDelayDone         ; Branch if initial delay is done
                        bne DecrementDelay          ; Branch if delay is not zero yet
                        inc LogSprCount             ; Allocate logical sprite for star
                        dec StarAppearDelay,x       ; Decrement delay
                        jmp ResetStar               ; Set up star

DecrementDelay          dec StarAppearDelay,x       ; Decrement delay
                        jmp NextStar

AppearDelayDone         dec StarAnimDelay,x         ; Animation delay done?
                        bpl AdjustXCoord            ; Branch if no
                        lda #7
                        sta StarAnimDelay,x         ; Reset delay
                        ldy LogSprImage,x
                        cpy #145                    ; Last frame of animation?
                        beq AdjustXCoord            ; Branch if yes
                        inc LogSprImage,x           ; Next frame
AdjustXCoord            lda LogSprXCoordLSB,x       ; Get current LSB of x-coordinate
                        clc
                        adc StarXCoordAdjustLo,x    ; Adjust it
                        sta LogSprXCoordLSB,x
                        lda LogSprXCoord,x          ; Get current x-coordinate
                        adc StarXCoordAdjustHi,x    ; Adjust it
                        sta LogSprXCoord,x
                        cmp #$20                    ; Set carry flag if the star is in the left part of the screen
                        lda LogSprColor,x           ; Get color of star
                        and #%01111111              ; Default: Disable MSB of x-coordinate
                        bcs SkipSetXMSB             ; Branch if star is in the left part of the screen
                        ora #%10000000              ; Enable MSB
SkipSetXMSB             sta LogSprColor,x
                        lda LogSprYCoord,x          ; Get current y-coordinate of star
                        ldy StarMovementIdx,x       ; Get current index to adjustment values
                        cpy #42                     ; Has the star started moving downwards?
                        bcc AdjustYCoord            ; Branch if no
                        cmp #248                    ; Has the star reached the bottom of the screen?
                        bcs ResetStar               ; Branch if yes
AdjustYCoord            adc StarYCoordAdjust,y      ; Adjust y-coordinate according to table
                        sta LogSprYCoord,x          ; Store new y-coordinate
                        cpy #72                     ; Max index to movement table reached?
                        beq NextStar                ; Branch if yes
                        iny                         ; Increment index
                        bne StoreIndex              ; Branch always

ResetStar               lda #170
                        sta LogSprXCoord,x          ; Set initial x-coordinate
                        lda #0 
                        sta LogSprXCoordLSB,x       ; Reset LSB of x-coordinate
                        lda #254
                        sta LogSprYCoord,x          ; Set initial y-coordinate
                        lda #140                    ; Initial star (size 1)
                        sta LogSprImage,x           ; Set sprite image
                        lda #7 
                        sta StarAnimDelay,x         ; Set initial animation delay
                        ldy StarColorIdx            ; Get current index to star colors
                        dey
                        bpl StoreColorIndex         ; Branch if index should not be reset yet
                        ldy #11-1                   ; Reset index
StoreColorIndex         sty StarColorIdx
                        lda StarColors,y            ; Get which color to use
                        sta LogSprColor,x
                        ldy StarCoordAdjustIdx      ; Get current index to star movement data
                        dey 
                        bpl StoreMovementIndex      ; Branch if index should not be reset yet
                        ldy #8-1                    ; Reset index
StoreMovementIndex      sty StarCoordAdjustIdx
                        lda StarXAdjustLo,y         ; Get adjustment value for LSB of x-coordinate
                        sta StarXCoordAdjustLo,x
                        lda StarXAdjustHi,y         ; Get adjustment value for x-coordinate
                        sta StarXCoordAdjustHi,x
                        lda StarYCoordAdjustIdx,y   ; Get initial index to adjustment values for y-coordinate
                        tay
StoreIndex              sty StarMovementIdx,x

NextStar                dex
                        bmi AllStarsDone
                        jmp StarLoop
AllStarsDone            rts

; Delay before each star appears
StarAppearDelayData     .char   0,  5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55, 60

; Colors of stars
StarColors              .char   6, 13,  2,  3,  8,  2,  9, 14,  4,  5, 10

; Adjustment values for x-coordinate of stars
StarXAdjustLo           .char -12, 32,-20, 12,-32, 24,-24, 20
StarXAdjustHi           .char $fe,  1,$fe,  1,$fe,  1,$fe,  1

; Initial index to 'StarYCoordAdjust' table
StarYCoordAdjustIdx     .char   0,  2,  7,  4,  1,  0,  5,  9

; Adjustment values for y-coordinate of star
StarYCoordAdjust        .char -9,-9,-9,-9,-9,-9,-8,-8,-8,-8,-7,-7,-7,-7,-6,-6,-6,-5,-5,-5,-4,-4,-4,-3,-3,-2,-2,-2,-2,-1,-1,-1,-1
                        .char -1, 0,-1, 0, 0,-1, 0, 0, 0, 1, 0, 1, 0, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 5, 5
                        .char  5, 5, 6, 6, 6, 7, 7

; **********************************************************************************************************************************
; *  Scroll text at top of screen
; **********************************************************************************************************************************

ScrollText              ldx XScroll
                        dex
                        bpl SkipCharScroll

                        ldx #0
CharLoop                lda Screen+41,x             ; Scroll chars and their colors left
                        sta Screen+40,x
                        lda ColorRAM+41,x
                        sta ColorRAM+40,x
                        inx
                        cpx #38
                        bne CharLoop

                        ldy #0
                        lda (TextAddr),y            ; Get next char in scroll text
                        bne CheckColorChange        ; Branch if text not re-starting yet
                        jsr ResetScrollText
                        lda #CSPC                   ; First char is a space
                        bne DisplayChar             ; Branch always
CheckColorChange        cmp #96                     ; Change text color now?
                        bcc DisplayChar             ; Branch if no
                        ldx CharColorIdx
                        ldy CharColorData,x         ; Get and set next char color to use
                        sty CharColor
                        inc CharColorIdx
                        sec
                        sbc #32                     ; Convert char value to actual char
DisplayChar             sta Screen+40+38            ; Put it in rightmost column
                        lda CharColor
                        sta ColorRAM+40+38          ; Set current char color
                        inc TextAddr
                        bne ResetXScroll
                        inc TextAddr+1
ResetXScroll            ldx #7

SkipCharScroll          stx XScroll
                        stx $d016                   ; Set horizontal smooth scroll value
                        rts

ResetScrollText         lda #<TextData
                        ldx #>TextData
                        sta TextAddr
                        stx TextAddr+1
                        lda #0
                        sta CharColorIdx
                        lda #10                     ; Light red
                        sta CharColor
                        rts

; **********************************************************************************************************************************
; *  Display the initial star using a specific char
; **********************************************************************************************************************************

DisplayInitialStar      sta StarChar+1              ; Set which character to use (self-modifying code)
                        ldx #3                      ; Index to data for last triangle
TriangleLoop            lda AddrLo,x                ; Get lo byte of screen address
                        sta ScreenAddr
                        cmp #$dc                    ; Set carry flag if this is the 1st or 2nd triangle
                        ldy #12                     ; Row and column counter
                        tya
                        adc #-7                     ; Calculate hi byte of screen address
                        clc
RowLoop                 sta ScreenAddr+1
                        sty Counter
StarChar                lda #0                      ; Character to use
ColumnLoop              sta (ScreenAddr),y          ; Put it on screen
                        dey
                        bpl ColumnLoop              ; Loop for all columns
                        lda ScreenAddr
                        adc AddrAdjLo,x             ; Adjust lo byte of screen address
                        sta ScreenAddr
                        tya                         ; y is $ff here
                        adc ScreenAddr+1            ; Adjust hi byte of screen address
                        cpx #2                      ; Set carry flag if 3rd or 4th triangle
                        adc #0                      ; Adjust hi byte again if required
                        ldy Counter
                        dey
                        bpl RowLoop                 ; Loop for all rows
                        dex
                        bpl TriangleLoop            ; Loop for all triangles
                        rts

; Address adjustment for each row of each triangle
AddrAdjLo               .char -40,-39,40,41

; Lo byte of initial screen address of each triangle
AddrLo                  .char $e0,$dc,$a0,$9c

; **********************************************************************************************************************************
; *  Copy memory from source to destination
; **********************************************************************************************************************************

CopyMemory              sta cm4+1                   ; Store lo byte of source address end (self-modifying code)
                        ldy #0
cm0                     lda (ScreenAddr),y          ; Copy a byte
                        sta (ColorAddr),y
                        inc ColorAddr               ; Increment destination address
                        bne cm1
                        inc ColorAddr+1
cm1                     cpx ScreenAddr+1            ; Source address end reached?
                        beq cm3                     ; Maybe
cm2                     inc ScreenAddr              ; Increment source address
                        bne cm0
                        inc ScreenAddr+1
                        bne cm0                     ; Branch always
cm3                     lda ScreenAddr
cm4                     cmp #$ff                    ; Source address end reached?
                        bne cm2                     ; Branch if no
                        rts

; **********************************************************************************************************************************
; *  Code which is copied to zero-page
; **********************************************************************************************************************************

ZeroPageCode
                        .logical $02

; IRQ framework code
DispatchIRQ             sta AValue+1                ; Save registers
                        stx XValue+1
                        sty YValue+1
IRQVector               jmp BottomIRQ               ; Jump to actual IRQ routine

SetNextAndReturnFromIRQ sta IRQVector+1             ; Set interrupt routine address
                        stx IRQVector+2
SetLineAndReturnFromIRQ sty v+18                    ; Set raster line

ReturnFromIRQ           dec v+25                    ; Acknowledge current raster interrupt
AValue                  lda #0                      ; Restore registers
XValue                  ldx #0
YValue                  ldy #0
DoRti                   rti

; Code for displaying one frame of the rotating star
DisplayStar             ldx #0                      ; Index to current data byte
                        ldy #0                      ; Index to screen column
DataAddr1               lda Star00,x                ; Address is modified by code
                        lsr
                        lsr
                        lsr
                        lsr
                        sta (ColorAddr),y
                        iny
                        cpy #19                     ; New row?
                        bne SkipAdj1                ; Branch if no
                        lda ColorAddr
                        clc
                        adc #40
                        sta ColorAddr
                        bcc SkipHi1
                        inc ColorAddr+1
SkipHi1                 ldy #0
SkipAdj1                cpx #180                    ; All data read?
                        beq EndStar                 ; Branch if yes
DataAddr2               lda Star00,x                ; Address is modified by code
                        and #$0f
                        sta (ColorAddr),y
                        iny
                        cpy #19                     ; New row?
                        bne SkipAdj2                ; Branch if no
                        lda ColorAddr
                        clc
                        adc #40
                        sta ColorAddr
                        bcc SkipHi2
                        inc ColorAddr+1
SkipHi2                 ldy #0
SkipAdj2                inx
                        bne DataAddr1
EndStar                 rts

                        .endlogical

; **********************************************************************************************************************************
; *  Data, graphics, music
; **********************************************************************************************************************************

OraTable                .char $01,$02,$04,$08,$10,$20,$40,$80 ; Values for setting and clearing bits
AndTable                .char $fe,$fd,$fb,$f7,$ef,$ef,$bf,$7f

                        .include "data.asm" ; Data for rotating star

IntroTextData           .text "WHEN YOU WISH UPON A CHAR",0,128,"    ",128,"STAR",0 ; 0=End, 128=Toggle direction

;                             W H E N   Y O U   W I S H   U P  O N    A     C H A R . . _ _ _ _ . S T A R .
IntroTextTiming         .char 7,7,7,6,3,7,7,6,5,7,7,7,5,5,9,17,7,7,13,14,14,7,7,7,7,7,7,7,7,7,7,7,5,5,5,5,15

TextData                .text "          WELCOME TO THIS C64 ENTRY FOR THE wILD cATEGORY OF vCCC2022. "
                        .text "mUSIC BY rOY WIDDING. cONCEPT AND CODE BY gEIR STRAUME.     "
                        .text "yOU'RE LISTENING TO rUN SIDOLPH RUN, wHICH IS A SID ARRANGEMENT OF "
                        .text "wHEN YOU WISH UPON A STAR aND rUN RUDOLPH RUN.     "
                        .text "a HUGE THANKS TO lOGIKER fOR ORGANISING THE vINTAGE COMPUTING CHRISTMAS CHALLENGE! "
                        .text "hERE'S HOPING FOR ANOTHER CHALLENGE NEXT YEAR!                              ",0

CharColorData           .char 7,10,7,10,1,10,1,10,7,10,7,10,7,10,1,10,7,10 ; Color changes for each lower case letter above

                        .align $2100,$ff ; Align charset
                        .binary "charset.bin"

                        .align $2300,$ff ; Align sprites
                        .char $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00 ; Star sprite size 1 (140)
                        .char $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$1C,$00,$00,$1C
                        .char $00,$00,$1C,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
                        .char $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00

                        .char $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00 ; Star sprite size 2 (141)
                        .char $00,$00,$00,$00,$00,$00,$24,$00,$00,$7E,$00,$00,$3C,$00,$00,$3C
                        .char $00,$00,$7E,$00,$00,$24,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
                        .char $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00

                        .char $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00 ; Star sprite size 3 (142)
                        .char $00,$00,$00,$44,$00,$00,$6C,$00,$01,$FF,$00,$00,$FE,$00,$00,$7C
                        .char $00,$00,$FE,$00,$01,$FF,$00,$00,$6C,$00,$00,$44,$00,$00,$00,$00
                        .char $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00

                        .char $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$82,$00,$00 ; Star sprite size 4 (143)
                        .char $C6,$00,$00,$EE,$00,$07,$FF,$C0,$03,$FF,$80,$01,$FF,$00,$00,$FE
                        .char $00,$01,$FF,$00,$03,$FF,$80,$07,$FF,$C0,$00,$EE,$00,$00,$C6,$00
                        .char $00,$82,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00

                        .char $00,$00,$00,$00,$00,$00,$01,$01,$00,$01,$83,$00,$01,$C7,$00,$01 ; Star sprite size 5 (144)
                        .char $EF,$00,$1F,$FF,$F0,$0F,$FF,$E0,$07,$FF,$C0,$03,$FF,$80,$01,$FF
                        .char $00,$03,$FF,$80,$07,$FF,$C0,$0F,$FF,$E0,$1F,$FF,$F0,$01,$EF,$00
                        .char $01,$C7,$00,$01,$83,$00,$01,$01,$00,$00,$00,$00,$00,$00,$00,$00

                        .char $02,$00,$80,$03,$01,$80,$03,$83,$80,$03,$C7,$80,$03,$EF,$80,$7F ; Star sprite size 6 (145)
                        .char $FF,$FC,$3F,$FF,$F8,$1F,$FF,$F0,$0F,$FF,$E0,$07,$FF,$C0,$03,$FF
                        .char $80,$07,$FF,$C0,$0F,$FF,$E0,$1F,$FF,$F0,$3F,$FF,$F8,$7F,$FF,$FC
                        .char $03,$EF,$80,$03,$C7,$80,$03,$83,$80,$03,$01,$80,$02,$00,$80,$00

; "Run Sidolph Run": SID arrangements of "When You Wish Upon a Star" and "Run Rudolph Run" by Roy Widding
MusicData
                        .binary "music.bin"
EndOfData
