; This source code is for use with beebasm (https://github.com/stardot/beebasm).
; Labels are defined with a leading "." and indentation is not significant.
; Multiple instructions can be given on a single line of input, separated by
; ":".

; We use a simplistic compression technique which breaks the image down into
; alternating runs of " " and "*" characters, using the fact the cursor will
; wrap over automatically at the right hand edge of the screen to avoid any need
; to encode newlines. Because there are therefore only two states, we just need
; to record the length of each run as data. The reliance on wrapping at the
; right hand edge of the screen means this will only work correctly in a 40
; column text mode.

; We take advantage of the vertical symmetry of the star to roughly halve the
; size of the run length data; we work through the run length data at "data"
; backwards, then fudge the start offset and run through it again forwards until
; we hit the end.

; This program is small enough that we can run in the language zero page
; workspace at &00-&8f inclusive, which saves a few bytes by allowing the use of
; zero page instead of absolute addressing.
org &0000
guard &0090

; OSWRCH (OS WRite CHaracter) is the Acorn OS subroutine to output the ASCII
; character in A. It preserves A, X and Y.
oswrch = &ffee

; The opcode for the "iny" instruction.
iny_opcode = &c8

.start
    ; The BBC B and Electron both usually start up in a 40 column screen mode,
    ; but this isn't guaranteed so we burn 10 bytes entering mode 7 (which will
    ; select mode 6 on an Electron) in order to ensure the wrapping works
    ; correctly.
    lda #22:jsr oswrch:lda #7:jsr oswrch

    ; A is the character to use for the current run.
    lda #' '
    ; Y is the current offset into the run length table.
    ldy #(data_end-data)-1

.run_loop
    ; Get the run length for the current run in X. Because we spend too much
    ; time looking at the 6502 instruction set, we know ldx supports zp,y
    ; addressing so we can do this without corrupting A.
    ldx data,y
    ; If we hit the 0 byte at the end of the data, we're done and we just loop
    ; forever. It would take an extra byte of code to "rts", and we can't do
    ; that anyway because we've corrupted the language zero page workspace and
    ; the language we'd return to would probably crash if we did.
.hang
    beq hang

    ; Output X copies of the character in A.
.char_loop
    jsr oswrch
    dex:bne char_loop

    ; Toggle A between ' ' and '*'.
    eor #' ' eor '*'
    ; Move Y on to the next run. This is a decrement for the top half of the
    ; star and is patched to turn it into an increment for the bottom half.
.dey_address
    dey:bpl run_loop
    ; We've finished the top half of the star. Patch the "dey" to turn it into
    ; an "iny" and initialise Y ready for the bottom half of the star.
    ldy #iny_opcode:sty dey_address
    ldy #data_second_half-data:bne run_loop ; always branch

; Run length data. The row numbers in the comments are just rough guidance, given we
; rely on runs of spaces wrapping onto the next line of the screen.
.data
    equb 28, 11         ; Row 7
    equb 28             ; Row 6b
.data_second_half
    equb 13             ; Row 6a
    equb 26, 15         ; Row 5
    equb 24, 17         ; Row 4
    equb 27, 4, 1, 4    ; Row 3
    equb 31, 3, 3, 3    ; Row 2
    equb 31, 2, 5, 2    ; Row 1
    equb 31, 1, 7, 1, 4 ; Row 0
.data_end
    equb 0

.end

save "STAR", start, end
print end - start, "bytes"
