@tool
extends MeshInstance3D

#@export var replayTime:float = 0
@export var generatedFlyingDistance:float = 5.0

@export var meshMaterial:ShaderMaterial
@export var wireframeMaterial:ShaderMaterial

@export var readTextureData:bool = true
@export var additiveGeometryStorageNodePath:String = "/root/Main/AdditiveGeometryStorage"
#@export var lidarDataStorageNodePath:String = "/root/Main/LidarDataStorage"
@export var lidarDataStorageNodePath:NodePath
@export var forcedShaderBaseTime:float = 0

@export var overrideReplayTime_WireFrame_Local:bool = 0
@export var localReplayTimeOverride_WireFrame:float = 0

@export var overrideReplayTime_AdditiveGeometry_Local:bool = 0
@export var localReplayTimeOverride_AdditiveGeometry:float = 0

@export var generateOnReady:bool = false	# Force synchronous generating in _ready

signal loadingCompleted

var additiveGeometryStorage:AdditiveGeometryStorage
var lidarDataStorage:LidarDataStorage

var shaderBaseTimeInUse:float = 0

var generatedMesh:Mesh

var meshGenThread:Thread = Thread.new()

var generationFraction:float = 0
var fractionMutex:Mutex = Mutex.new()
var generatingAsynchronously:bool = false

class WireframeLine:
	var startPointIndex:int
	var endPointIndex:int
	var minShotTime:float
	var maxShotTime:float
	func _init(startPointIndex_p:int, endPointIndex_p:int, minShotTime_p:float, maxShotTime_p:float):
		self.startPointIndex = startPointIndex_p
		self.endPointIndex = endPointIndex_p
		self.minShotTime = minShotTime_p
		self.maxShotTime = maxShotTime_p

# 0...1, 1 = Fully generated
func getGenerationProgress() -> float:
	fractionMutex.lock()
	var fraction = generationFraction
	fractionMutex.unlock()

	if (fraction >= 1):
		if (meshGenThread.is_alive()):
			# This is to make sure mesh generating thread really has done it's work
			fraction = 0.99

	return fraction

# Called when the node enters the scene tree for the first time.
func _ready():
#	print_debug("_ready\t",Time.get_ticks_msec(),"\t",self.get_path())
	if (Global && Engine.is_editor_hint() && Global.cleanTempToolData):
		# @tool-scripts will generate changes that are saved into .tscn (scene)-files.
		# Clean them when requested
		
		print("Cleaning data generated by @tool, ", self.name)
		mesh = null
		meshMaterial.set_shader_parameter("customDataSampler", null)
		wireframeMaterial.set_shader_parameter("customDataSampler", null)
		return

	additiveGeometryStorage = get_node(additiveGeometryStorageNodePath)
	lidarDataStorage = get_node(lidarDataStorageNodePath)
	
	mesh = null

	if (generateOnReady):
		generateMeshSynchronously()

func generateMeshSynchronously():
	meshGenThread.start(Callable(self, "generateMeshThread"))
	meshGenThread.wait_to_finish()
	mesh = generatedMesh

func startAsyncMeshGenerating():
	mesh = null
	meshGenThread.start(Callable(self, "generateMeshThread"), Thread.PRIORITY_LOW)
	generatingAsynchronously = true

func generateMeshThread():
	var elapsedStartTime = Time.get_ticks_msec()
	
	fractionMutex.lock()
	generationFraction = 0
	fractionMutex.unlock()

	# Lidar's position when this vertex (of a face) was "shot":
	# Not actually centroid since face is zero-sized here
	#var faceCentroidStartOrigins = PackedColorArray()

	# Final centroid of the vertex's face:
	#var faceCentroidFinalOrigins = PackedColorArray()

	# Vertex's "flat-face"-normals:
	#var faceNormals = PackedColorArray()
	
	# Vertex (unmodified) normals, read from a file:
	var vertexNormals = PackedVector3Array()
	
	# Vertices (unmodified, non-indexed):
	var vertices = PackedVector3Array()
	
	# UV is used for texture (as expected?)
	var uv = PackedVector2Array()
	
	# UV2 is used for "shot time"
	#var uv2 = PackedVector2Array()

	# x here is used as times (unmodified) when faces/vertices were "shot", read from a file.
	# Other fields are used to draw edges.
	#var shotTimesAndEdgeIds = PackedColorArray()
	
	if ((!additiveGeometryStorage.faceSyncKeys.is_empty()) && (forcedShaderBaseTime == 0.0)):
		shaderBaseTimeInUse = additiveGeometryStorage.faceSyncKeys[0]
	else:
		shaderBaseTimeInUse = forcedShaderBaseTime
		
#	var numOfFaces:int = 0
#	var vertexIndex:int = 0

	# * 3 / 3 here comes from 3 values for each face (consisting of 3 points)
	var imageHeight_Mesh = nearest_po2(int(((additiveGeometryStorage.getNumOfVertices() * 3 / 3) / 4096) + 1))
	
	var customDataImage_Mesh:Image = Image.create(4096, imageHeight_Mesh, false, Image.FORMAT_RGBAF)
	var customDataImagePointer_Mesh:int = 0

	# Wireframe needs only one value (shot time) for each face (consisting of 3 points)
	var imageHeight_Wireframe = nearest_po2(int(((additiveGeometryStorage.getNumOfVertices() / 3) / 4096) + 1))
	
	var customDataImage_Wireframe:Image = Image.create(4096, imageHeight_Wireframe, false, Image.FORMAT_RF)
	var customDataImagePointer_Wireframe:int = 0

	for currentFaceSyncItemIndex in range(0, additiveGeometryStorage.faceSyncKeys.size()):
		var currentFraction = float(currentFaceSyncItemIndex) / additiveGeometryStorage.faceSyncKeys.size()
		
		if (abs(currentFraction - generationFraction) > 0.01):
			fractionMutex.lock()
			generationFraction = currentFraction
			fractionMutex.unlock()

		var currentFaceSyncItemTime:int = additiveGeometryStorage.faceSyncKeys[currentFaceSyncItemIndex]
		var originFromLidar:Vector3
		var originFromLidarKnown:bool = false
		
		# Try to find origin from lidar data
		
		if (lidarDataStorage.beamData.has(currentFaceSyncItemTime)):
			# Although there is more items in the array, we here rely that
			# the first one represents them all good enough
			# (they are all 1 ms apart at most anyway)
			
			originFromLidar = lidarDataStorage.beamData.get(currentFaceSyncItemTime)[0].origin
			originFromLidarKnown = true
		
		for subItem in additiveGeometryStorage.faceSync[currentFaceSyncItemTime]:
#			numOfFaces += 1
			
			var shotTime:float = currentFaceSyncItemTime - shaderBaseTimeInUse
			
			var shotTime_Color:Color = Color(shotTime, shotTime, shotTime, shotTime)
			customDataImage_Wireframe.set_pixel(customDataImagePointer_Wireframe % 4096, customDataImagePointer_Wireframe / 4096, shotTime_Color)
			customDataImagePointer_Wireframe += 1

			var faceVerts_0 = additiveGeometryStorage.fileVertices[additiveGeometryStorage.fileVertexIndexes[subItem * 3 + 0]]
			var faceVerts_1 = additiveGeometryStorage.fileVertices[additiveGeometryStorage.fileVertexIndexes[subItem * 3 + 1]]
			var faceVerts_2 = additiveGeometryStorage.fileVertices[additiveGeometryStorage.fileVertexIndexes[subItem * 3 + 2]]

			vertices.push_back(faceVerts_0)
			vertices.push_back(faceVerts_1)
			vertices.push_back(faceVerts_2)

			var faceNormal = ((faceVerts_2 - faceVerts_0).cross(faceVerts_1 - faceVerts_0)).normalized()
			var faceNormal_Color:Color = Color(faceNormal.x, faceNormal.y, faceNormal.z, shotTime)
			customDataImage_Mesh.set_pixel(customDataImagePointer_Mesh % 4096, customDataImagePointer_Mesh / 4096, faceNormal_Color)
			customDataImagePointer_Mesh += 1
#			faceNormals.push_back(faceNormal_Color)
#			faceNormals.push_back(faceNormal_Color)
#			faceNormals.push_back(faceNormal_Color)

			vertexNormals.push_back(additiveGeometryStorage.fileNormals[additiveGeometryStorage.fileVertexIndexes[subItem * 3 + 0]])
			vertexNormals.push_back(additiveGeometryStorage.fileNormals[additiveGeometryStorage.fileVertexIndexes[subItem * 3 + 1]])
			vertexNormals.push_back(additiveGeometryStorage.fileNormals[additiveGeometryStorage.fileVertexIndexes[subItem * 3 + 2]])
			
			if (readTextureData):
				uv.push_back(additiveGeometryStorage.fileTextureCoords[additiveGeometryStorage.fileTextureIndexes[subItem * 3 + 0]])
				uv.push_back(additiveGeometryStorage.fileTextureCoords[additiveGeometryStorage.fileTextureIndexes[subItem * 3 + 1]])
				uv.push_back(additiveGeometryStorage.fileTextureCoords[additiveGeometryStorage.fileTextureIndexes[subItem * 3 + 2]])
			else:
				# Wireframe shader uses uv as shotTime
				uv.push_back(Vector2(shotTime, shotTime))
				uv.push_back(Vector2(shotTime, shotTime))
				uv.push_back(Vector2(shotTime, shotTime))

			var medianOrigin = (faceVerts_0 + faceVerts_1 + faceVerts_2) / 3
			var medianOrigin_Color:Color = Color(medianOrigin.x, medianOrigin.y, medianOrigin.z, shotTime)
			customDataImage_Mesh.set_pixel(customDataImagePointer_Mesh % 4096, customDataImagePointer_Mesh / 4096, medianOrigin_Color)
			customDataImagePointer_Mesh += 1
#			faceCentroidFinalOrigins.push_back(medianOrigin_Color)
#			faceCentroidFinalOrigins.push_back(medianOrigin_Color)
#			faceCentroidFinalOrigins.push_back(medianOrigin_Color)

			var startOrigin:Vector3

			if originFromLidarKnown:
				startOrigin = originFromLidar
			else:
				startOrigin = medianOrigin + faceNormal * generatedFlyingDistance
			
			var startOrigin_Color:Color = Color(startOrigin.x, startOrigin.y, startOrigin.z, shotTime)
			
			customDataImage_Mesh.set_pixel(customDataImagePointer_Mesh % 4096, customDataImagePointer_Mesh / 4096, startOrigin_Color)
			customDataImagePointer_Mesh += 1
			
#			customDataImagePointer += 2

#			customDataImagePointer += 8

#			faceCentroidStartOrigins.push_back(startOrigin_Color)
#			faceCentroidStartOrigins.push_back(startOrigin_Color)
#			faceCentroidStartOrigins.push_back(startOrigin_Color)
			
			# UV2 used to feed the time and distance from one edge in y
			
#			shotTimesAndEdgeIds.push_back(Color(shotTime, 1, 0, 0))
#			shotTimesAndEdgeIds.push_back(Color(shotTime, 0, 1, 0))
#			shotTimesAndEdgeIds.push_back(Color(shotTime, 0, 0, 1))
			

#	for i in range(numOfFaces * 3):
#		faceCentroidStartOriginsImage.set_pixel(i % 4096, i / 4096, faceCentroidStartOrigins[i])
#		faceCentroidFinalOriginsImage.set_pixel(i % 4096, i / 4096, faceCentroidFinalOrigins[i])

	var vertexIndexes = PackedInt32Array()
	for i in range(vertices.size()):
		vertexIndexes.push_back(i)

	generatedMesh = ArrayMesh.new()

	var arrayMeshArrays_MainMesh = []
	arrayMeshArrays_MainMesh.resize(ArrayMesh.ARRAY_MAX)

	arrayMeshArrays_MainMesh[ArrayMesh.ARRAY_INDEX] = vertexIndexes
	arrayMeshArrays_MainMesh[ArrayMesh.ARRAY_VERTEX] = vertices
	arrayMeshArrays_MainMesh[ArrayMesh.ARRAY_NORMAL] = vertexNormals
	
	arrayMeshArrays_MainMesh[ArrayMesh.ARRAY_TEX_UV] = uv


	# Solid / flying:
	generatedMesh.add_surface_from_arrays(Mesh.PRIMITIVE_TRIANGLES, arrayMeshArrays_MainMesh)

	var arrayMeshArrays_Wireframe = []
	arrayMeshArrays_Wireframe.resize(ArrayMesh.ARRAY_MAX)
	arrayMeshArrays_Wireframe[ArrayMesh.ARRAY_INDEX] = vertexIndexes
	arrayMeshArrays_Wireframe[ArrayMesh.ARRAY_VERTEX] = vertices

	# Wireframe:
	generatedMesh.add_surface_from_arrays(Mesh.PRIMITIVE_TRIANGLES, arrayMeshArrays_Wireframe)

	generatedMesh.surface_set_material((generatedMesh.get_surface_count() - 1), wireframeMaterial)
	generatedMesh.surface_set_material((generatedMesh.get_surface_count() - 2), meshMaterial)

	var customDataImageTexture_Mesh = ImageTexture.create_from_image(customDataImage_Mesh)

	meshMaterial.set_shader_parameter("customDataSampler", customDataImageTexture_Mesh)

	var customDataImageTexture_Wireframe = ImageTexture.create_from_image(customDataImage_Wireframe)
	wireframeMaterial.set_shader_parameter("customDataSampler", customDataImageTexture_Wireframe)

	var elapsed = Time.get_ticks_msec() - elapsedStartTime
	print("Additive geometry mesh creation time: ", elapsed, " ms")

	fractionMutex.lock()
	generationFraction = 1
	fractionMutex.unlock()


# Called every frame. 'delta' is the elapsed time since the previous frame.
func _process(_delta):
	if (generatingAsynchronously):
		var progress = getGenerationProgress()
		if (progress >= 1):
			mesh = generatedMesh
			generatingAsynchronously = false
			loadingCompleted.emit()
	
	elif (mesh && Global):
		if (overrideReplayTime_AdditiveGeometry_Local):
			get_active_material(0).set_shader_parameter("replayTime", localReplayTimeOverride_AdditiveGeometry - shaderBaseTimeInUse)
		elif (Global.overrideReplayTime_AdditiveGeometries):
			get_active_material(0).set_shader_parameter("replayTime", Global.replayTimeOverride_AdditiveGeometries - shaderBaseTimeInUse)
		else:
			get_active_material(0).set_shader_parameter("replayTime", Global.replayTime_Lidar - shaderBaseTimeInUse)

		if (overrideReplayTime_WireFrame_Local):
			get_active_material(1).set_shader_parameter("replayTime", localReplayTimeOverride_WireFrame - shaderBaseTimeInUse)
		elif (Global.overrideReplayTime_WireFrames):
			get_active_material(1).set_shader_parameter("replayTime", Global.replayTimeOverride_WireFrames - shaderBaseTimeInUse)
		else:
			get_active_material(1).set_shader_parameter("replayTime", Global.replayTime_Lidar - shaderBaseTimeInUse)

class StashData:
	var mesh
	var customDataSampler_Mesh
	var customDataSampler_Wireframe

func stashToolData():
	var stashStorage:StashData = StashData.new()

	if (generatingAsynchronously):
		meshGenThread.wait_to_finish()
	
	stashStorage.mesh = mesh
	stashStorage.customDataSampler_Mesh = meshMaterial.get_shader_parameter("customDataSampler")
	stashStorage.customDataSampler_Wireframe = wireframeMaterial.get_shader_parameter("customDataSampler")

	mesh = null
	meshMaterial.set_shader_parameter("customDataSampler", null)
	wireframeMaterial.set_shader_parameter("customDataSampler", null)

	return stashStorage

func stashPullToolData(stashStorage:StashData):
	mesh = stashStorage.mesh
	meshMaterial.set_shader_parameter("customDataSampler", stashStorage.customDataSampler_Mesh)
	wireframeMaterial.set_shader_parameter("customDataSampler", stashStorage.customDataSampler_Wireframe)
	if (mesh):
		mesh.surface_set_material((mesh.get_surface_count() - 1), wireframeMaterial)
		mesh.surface_set_material((mesh.get_surface_count() - 2), meshMaterial)

func waitThreadExit(threadToWait:Thread):
	if (threadToWait.is_started()):
		threadToWait.wait_to_finish()

func _exit_tree():
	print("Exit tree (AdditiveGeometry)")
	waitThreadExit(meshGenThread)
