import math

import bpy
import bmesh
import random
from mathutils import Vector, noise, Matrix
from math import sin, cos, tau, pi, sqrt
from utils.interpolation import *
from utils.math import *
from utils.blender_utils import *

frame_start = 1
total_frames = 150
bpy.context.scene.render.fps = 30
bpy.context.scene.frame_start = frame_start
bpy.context.scene.frame_end = total_frames
bpy.context.scene.render.use_lock_interface = True
seed = 123

w = 300

N1 = 60
N2 = 2
K = 15

m1 = 30
m2 = 30

def softplus(q: float, p: float):
    qq = q + p
    if qq <= 0:
        return 0

    if qq >= 2 * p:
        return qq - p

    return 1/(4 * p) * qq * qq


def lerp(a, b, t):
    return (1 - t) * a + t * b


def clamp(x: float, low: float=0, high: float=1) -> float:
    if x <= low:
        return low
    if x >= high:
        return high
    return x


def remap(x, start1, end1, start2, end2, do_clamp=False):
    range1 = end1 - start1
    range2 = end2 - start2
    if range1 == 0:
        range1 = 1
    t = (x - start1) / range1
    return start2 + clamp(t) * range2 if do_clamp else start2 + t * range2


def polar(angle: float, radius: float, z: float = 0.0) -> Vector:
    x = radius * cos(angle)
    y = radius * sin(angle)
    return Vector((x, y, z))


def scale_matrix(x: float, y: float, z: float) -> Matrix:
    matx = Matrix.Scale(x, 4, Vector((1, 0, 0)))
    maty = Matrix.Scale(y, 4, Vector((0, 1, 0)))
    matz = Matrix.Scale(z, 4, Vector((0, 0, 1)))
    return matx @ maty @ matz


def ease(p, g):
    if p < 0.5:
        return 0.5 * pow(2 * p, g)
    else:
        return 1 - 0.5 * pow(2*(1 - p), g)


def surface(
        p: float,
        theta: float,
        offset: float = 0
) -> Vector:
    p2 = remap(p, 0, 1, -300, 800)
    z = softplus(-p2, w) - offset
    r = 600 + softplus(p2, w) - offset
    r *= pow(p, 2.0)
    # r += 100
    return polar(theta, r, z)


def get_transformation(t: float, i: int, j: int, k: int) -> Matrix:
    theta = remap(i, 0, N1, 0, tau)
    row = k + t + j / N2
    p = row / K

    center_pos = surface(p, theta, 0) / 600.0
    radius1 = (center_pos.xy).length

    center_pos2 = surface(p + 0.001, theta, 0) / 600.0
    radius2 = (center_pos2.xy).length
    delta = (center_pos2 - center_pos)

    slope = pi - math.atan2(delta.z, radius2 - radius1)
    slope_offset = (5 * p + t + 0.25 * sin(3 * theta))
    slope_offset_aux = slope_offset - math.floor(slope_offset)
    slope_offset_aux = ease(slope_offset_aux, 4)
    slope_offset = (math.floor(slope_offset) + slope_offset_aux) * pi

    y_rotation = Matrix.Rotation(slope + slope_offset, 4, 'Y')
    z_rotation = Matrix.Rotation(theta, 4, 'Z')

    l2 = 6.0 * pow(p, 2.5)
    scale = scale_matrix(1, l2, 1)

    return (
        Matrix.Translation(center_pos) @
        z_rotation @
        y_rotation @
        scale
    )


def update(t: float):
    for i in range(N1):
        for j in range(N2):
            for k in range(K):
                obj = bpy.data.objects[f'obj-{i}-{j}-{k}']
                obj.matrix_world = get_transformation(t, i, j, k)


def setup():
    col = get_or_create_collection('generated')

    mesh = get_or_create_mesh('mesh')

    # obj = get_or_create_object('points', mesh, col)
    for i in range(N1):
        for j in range(N2):
            for k in range(K):
                obj = get_or_create_object(f'obj-{i}-{j}-{k}', mesh, col)

    update(0.0)

    # offset = 15
    # verts = []
    # for i in range(m1):
    #     for j in range(m2):
    #         theta1 = remap(i, 0, m1, 0, tau)
    #         # theta2 = remap(i + 1, 0, m1, 0, tau)
    #         p = remap(j + 3 + 0.05, 0.0, float(m2), 0.0, 1.0)
    #         v1 = surface(p, theta1, offset) / 600.0
    #         # v2 = surface(p, theta2, offset)
    #         verts.append(v1)
    #
    # mesh.clear_geometry()
    # mesh.from_pydata(
    #     vertices=verts,
    #     edges=[],
    #     faces=[]
    # )


def frame_update(scene):
    frame = scene.frame_current
    t = frame / float(total_frames)
    update(t)


setup()
bpy.app.handlers.frame_change_pre.clear()
bpy.app.handlers.frame_change_pre.append(frame_update)
