/** Audio callback
 * consumes 44100 samples per second, each sample is 8 bit uint
 * Callback is called every 1024 samples, or 43 times per second.
 * Userdata is null, don't use.
 *
 *
 */

long long int totalSamples = 0;
const double pi = 3.1415926536;

// Math functions
double mod(double a, double b)
{
	return a - b * (long long int)(a / b);
}

double fract(double a)
{
	return a - floor(a);
}

double pitch(double p)
{
	return pow(1.059460646483, p) * 440.0;
}

double clamp(double a, double low, double high)
{
	if (a < low)
		return low;
	if (a > high)
		return high;
	return a;
}

double mix(double a, double b, double x)
{
	return a * x + b * (1.0 - x);
}

// Base wave functions
double saw(double freq, double s)
{
	double f = 1.0 / freq;
	return freq * mod(s, f);
}

double zaw(double freq, double s)
{
	return 1.0 - saw(freq, s);
}

double wave(double freq, double s)
{
	return 0.5 + 0.5 * sin(s * 2.0 * pi * freq);
}

double box(double freq, double s)
{
	return (saw(freq, s)  > 0.5) ? 1.0 : 0.0;
}

// Mixing
double fadein(double start, double length, double s)
{
	return clamp((s - start) / length, 0.0, 1.0);
}

double fadeout(double start, double length, double s)
{
	return clamp(1.0 - (s - start) / length, 0.0, 1.0);
}

double alternate(double a, double b, double freq, double s)
{
	return mix(a, b, box(freq, s));
}

double slide(double a, double b, double freq, double s)
{
	return mix(a, b, wave(freq, s));
}

// Notes
double aMinor(double s)
{
	return 0.33 * (wave(pitch(0), s) + wave(pitch(3), s) + wave(pitch(7), s));
}

// Actual sounds
double baseBeat(double s)
{
	double a = wave(441.0, s) + wave(441.0, s+0.001) + wave(441.0, s+0.002) + wave(441.0, s+0.003);
	double b = wave(330.0, s) + wave(330.0, s+0.001) + wave(330.0, s+0.002) + wave(330.0, s+0.003);
	return fadein(0.0, 3.0, s) * zaw(2.0, s) * 0.25 * alternate(a, 0.5 * b, 0.5, s);
}

double sideBeat(double s)
{
	double a = (saw(441.0, s) + saw(442.0, s)) * saw(44.1, s) * saw(4.41, s) * box(2.21, s);
	return 0.2 * a * fadein(2.0, 3.0, s);
}

double scene1(double s)
{
	double a = aMinor(s) * zaw(8.0, s) * box(4.0, s);
	return a * 0.5 * fadein(0.0, 5.0, s) * fadeout(9.0, 1.0, s);
}

double melody(double s)
{
	double a = 0.0;
	double base = -24.0;
	double semitones[4] = {0.0, 4.0, 7.0, 9.0};
	for(int i = 0; i < 4; ++i)
	{
		double f = pitch(base + semitones[i]);
		for(int j = 0; j < 4; ++j)
		{
			double fu = (double) j;
			double fr = f + fu*sin(fu);
			a += saw(fr, s) / 16.0;
		}
	}
	return a;
}

double melody2(double s)
{
	double a = 0.0;
	double base = -24.0;
	double semitones[4] = {0.0, 4.0, 7.0, 10.0};
	for(int i = 0; i < 4; ++i)
	{
		double f = pitch(base + semitones[i]);
		for(int j = 0; j < 4; ++j)
		{
			double fu = (double) j;
			double fr = f + fu*sin(fu);
			a += saw(fr, s) / 16.0;
		}
	}
	return a;
}

double scene3(double s)
{
	double a = melody(s);
	double b = melody2(s);

	return slide(a, b, 0.5, s);
}

double part(double s)
{
	double sample = 0.0;
	sample += melody(s);
	sample += baseBeat(s);
	if (s < 10.0)
		sample += sideBeat(s);
	if (s > 10.0 && s < 20.0)
		sample += scene1(s);
	if (s > 20.0 && s < 30.0)
		sample += melody2(s);
	if (s > 30.0)
		sample += scene3(s);
	return sample * 0.33f;
}

void syna(void* userdata, uint8_t* stream, int len)
{
	for(int i = 0; i < len; i+=2)
	{
		// Samples created so far
		double t = (double)(totalSamples);
		// Sample position in seconds
		double s = t / 44100.0;
		// Get sample, store it in both channels
		double sample = part(s);
		stream[i] = 250 * sample;
		stream[i+1] = 250 * sample;
		totalSamples++;
	}
}

