/*
 * This is an invitation to the Assembly Summer 2019 One Scene Compo
 * This invitation happens to also be an example of what one scene demo also means
 * See the actual compo rules here (COMPO SPECIFIC RULES -> Real Time compos -> One Scene competition): https://www.assembly.org/summer19/demoscene/rules
 */

// Import external functions here:

// Moonlander is a library for integrating Processing with GNU Rocket, https://github.com/anttihirvonen/moonlander
// GNURocket is a tool for synchronizing music and visuals in demoscene productions: https://github.com/rocket/rocket
import moonlander.library.*;
// Audio library for playing sound, http://code.compartmental.net/minim/
import ddf.minim.*;


// Some demo global variables 

// the time in seconds (taken from Moonlander / GNURocket)
float now = 0.0;

// These control how big the opened window is.
// Assembly has FullHD (a.k.a. 1080p, a.k.a. 1920x1080) maximum projector resolution
// ref. https://www.assembly.org/summer19/demoscene/rules
float CANVAS_WIDTH = 1920;
float CANVAS_HEIGHT = 1080;

Moonlander moonlander;

PImage overlay;

PShape gear;
PShape asmLogo;

String[] overlayFileNames = { "overlay2.png", "overlay3.png", "overlay4.png", "overlay5.png", "credits.png", "logo.png"};
PImage[] overlayImages = new PImage[overlayFileNames.length];

String[] fontNames = { "Inter-ExtraLight-BETA.ttf" };
PFont[] fonts = new PFont[fontNames.length];

String[] texts = {
  "We invite you to",
  "to compete in ONE SCENE COMPO",
  "60 seconds, one constant scene",
  "There will be workshop with assistants",
  "in the DEMOSCENE LOUNGE",
  "Greetings fly out to",
  "Bad Felix",
  "Experimental Graphics Research Group",
  "Peisik",
  "Substandard",
  "Funny Curly Cat Code Factory",
  "Jumalauta",
  "Bjakke",
  "Qma",
  "Sooda",
  "Dekadence",
  "Nalleperhe",
  "Trilobit",
  "Adapt",
  "Paraguay",
  "MFX",
  "Byterapers",
  "Wide Load",
  "Prismbeings",
  "Conspiracy",
  "The Black Lotus",
  "Darklite",
  "Extend",
  "Primitive",
  "Damones",
  "ISO",
  "pieni",
  "and all of you <3"
};

void settings() {
  // Set up the drawing area size and renderer (P2D / P3D).
  size((int)CANVAS_WIDTH,(int)CANVAS_HEIGHT, P3D);
  //Set the window fullscreen
  fullScreen();
  
  // if you want to list all the fonts available in the system
  //String[] fontList = PFont.list();
  //printArray(fontList);
}

void setup() {
  // Assembly has 60Hz maximum projector framerate, ref. https://www.assembly.org/summer19/demoscene/rules
  frameRate(60);

  // Loads 3D mesh. OBJ format can be created with, for example, Blender 3D program: https://www.blender.org/
  gear = loadShape("gear.obj");
  asmLogo = loadShape("assylogo.obj");
  
  for (int i=0; i < overlayImages.length; i++){
    overlayImages[i] = loadImage(overlayFileNames[i]);
  }

  for (int i=0; i < fonts.length; i++){
    fonts[i] = createFont(fontNames[i], 200);
  }
  
  overlay = loadImage("overlay1.png");

  //hide mouse cursor
  noCursor();

  // Init & start moonlander
  int bpm = 120;
  int rowsPerBeat = 4;
  moonlander = Moonlander.initWithSoundtrack(this, "20190608_graffathon_onescene.mp3", bpm, rowsPerBeat);
  moonlander.start();
}

//This function draw multiple gears / cogwheels
void drawGears() {
  //pushMatrix / popMatrix functions ensure that matrix operations like rotation/translation/scaling will only happen inside the push/pop
  pushMatrix();

  //global translation of the gears
  translate((float)moonlander.getValue("gear:x"), (float)moonlander.getValue("gear:y"), (float)moonlander.getValue("gear:z"));
  
  gear.setFill(color(255, 255, 255, (int)(moonlander.getValue("gear:fill_alpha") * 255)));

  for(int i = 0; i < 5; i++) {
    pushMatrix();
    // odd/even gears should be placed apart in Y axis and rotate to different directions
    float direction = 1.0;
    float y = (float)moonlander.getValue("gear:spacing_y1");
    if (i%2 == 0) {
      direction = -1.0;
      y = (float)moonlander.getValue("gear:spacing_y2");
    }
    
    // Note that matrix operations are in "reverse order" of the functions.
    // => So first scale, then rotateZ and then translate
    // => Things will look different if you change the function calling order, go ahead and try!

    //position gear in a "row"
    translate(i*(float)moonlander.getValue("gear:spacing"), y, 0);
    //rotate the gear
    rotateZ(now * direction);
    //scale the gear
    scale((float)moonlander.getValue("gear:scale"));
    //draw the gear
    shape(gear);
    popMatrix();
  }
  popMatrix();
}

void drawAsmLogo() {
  if((float)moonlander.getValue("asmlogo:scale") <= 0.0) return;
  pushMatrix();
  translate((float)moonlander.getValue("asmlogo:x"), (float)moonlander.getValue("asmlogo:y"), (float)moonlander.getValue("asmlogo:z"));
  rotateX(PI*(float)moonlander.getValue("asmlogo:rotatex")/180);
  rotateY(PI*(float)moonlander.getValue("asmlogo:rotatey")/180);
  rotateZ(PI*(float)moonlander.getValue("asmlogo:rotatez")/180);
  scale((float)moonlander.getValue("asmlogo:scale"));
  shape(asmLogo);
  popMatrix();
}

void drawCube() {
  if (moonlander.getValue("cube:fill_alpha") <= 0.0 && moonlander.getValue("cube:line_alpha") <= 0.0) {
    //cube's line and fill alphas are zero or less, so let's skip trying to draw the cube at all
    return;
  }
  pushMatrix();
  translate((float)moonlander.getValue("cube:x"), (float)moonlander.getValue("cube:y"), (float)moonlander.getValue("cube:z"));
  // Cube rotation is in degrees.
  // Note that usually rotation is done using radians but you can convert degrees to radians with the function radians()
  rotateX(radians((float)moonlander.getValue("cube:rotateX")));
  rotateY(radians((float)moonlander.getValue("cube:rotateY")));
  rotateZ(radians((float)moonlander.getValue("cube:rotateZ")));
  fill(0,0,0,(int)(moonlander.getValue("cube:fill_alpha") * 255));
  box((float)moonlander.getValue("cube:width"), (float)moonlander.getValue("cube:height"), (float)moonlander.getValue("cube:depth"));
  noFill();
  fill(255,255,255,(int)(moonlander.getValue("cube:line_alpha") * 255));
  box((float)moonlander.getValue("cube:width"), (float)moonlander.getValue("cube:height"), (float)moonlander.getValue("cube:depth"));
  popMatrix();
}

void drawOverlays() {
  pushMatrix();
  scale(width/CANVAS_WIDTH,height/CANVAS_HEIGHT);

  // Center the view
  translate(width/2, height/2, 0);

  tint(255,255,255,(int)(moonlander.getValue("overlay:alpha1") * 255));
  image(overlay, -width/2, -height/2);

  int overLayImageNumber = (int)moonlander.getValue("overlay:image") % overlayImages.length;
  if (overLayImageNumber >= 0) {
    tint(255,255,255,(int)(moonlander.getValue("overlay:alpha2") * 255));
    image(overlayImages[overLayImageNumber], -width/2 + (int)moonlander.getValue("overlay:x"), -height/2 + (int)moonlander.getValue("overlay:y"));
  }

  popMatrix();
}

void drawText() {
  if (moonlander.getValue("font:text") >= 0) {
    pushMatrix();

    // Center the view
    translate(width/2, height/2, 0);

    scale((float)moonlander.getValue("font:scale"));
    textFont(fonts[0]);
    fill((int)(moonlander.getValue("font:r") * 255),(int)(moonlander.getValue("font:g") * 255),(int)(moonlander.getValue("font:b") * 255),(int)(moonlander.getValue("font:a") * 255));
    text(texts[(int)moonlander.getValue("font:text")%texts.length], (float)moonlander.getValue("font:x"), (float)moonlander.getValue("font:y"));
    popMatrix();
  }
}

void draw() {
  moonlander.update();

  now = (float)moonlander.getCurrentTime();
  float end = 64.0; //end production after 60 secs
  if (now > end) {
    exit();
  }
  
  
  background((int)(moonlander.getValue("bg:r") * 255),(int)(moonlander.getValue("bg:g") * 255),(int)(moonlander.getValue("bg:b") * 255),(int)(moonlander.getValue("bg:a") * 255));
  
  // Enable lights and depth testing to ensure that 3D meshes are drawn in correct order
  lights();
  hint(ENABLE_DEPTH_TEST);

  pushMatrix();

  // Center the view
  translate(width/2, height/2, 0);

  scale(width/CANVAS_WIDTH,height/CANVAS_HEIGHT,width/CANVAS_WIDTH);

  drawGears();

  drawCube();
  
  drawAsmLogo();
  
  popMatrix();

  // disable lights and depth testing so that 2D overlays and text can be draw on top of 3D
  noLights();
  hint(DISABLE_DEPTH_TEST);

  pushMatrix();

  drawText();

  drawOverlays();
  
  popMatrix();
}
