using System;
using System.Collections.Generic;
using System.Linq;
using SharpDX;
using SharpDX.Direct3D;
using SharpDX.Direct3D11;
using Framefield.Core.Rendering;
using Buffer = SharpDX.Direct3D11.Buffer;

namespace Framefield.Core.ID398fe1da_995e_4361_b370_02be9592afd5
{

    public class Class_InstanciateMesh : FXSourceCodeFunction
    {
        public override void Dispose()
        {
            Utilities.DisposeObj(ref _mesh);
            base.Dispose();
        }

        //>>> _inputids
        private enum InputId
        {
            Code = 0,
            InstanceMesh = 1,
            SurfacePoints = 2
        }
        //<<< _inputids


        private bool _firstEval = true;
        public override OperatorPartContext Eval(OperatorPartContext context, List<OperatorPart> inputs, int outputIdx)
        {
            //>>> _params
            var Code = inputs[(int)InputId.Code].Eval(context).Text;
            var InstanceMesh = inputs[(int)InputId.InstanceMesh].Eval(context).Mesh;
            var SurfacePoints = inputs[(int)InputId.SurfacePoints].Eval(context).Dynamic;  // Needs to be checked for null!
            //<<< _params        

            if (InstanceMesh == null  || SurfacePoints == null)
            {
                return context;
            }
            
            if (_firstEval)
            {
                for (int i = 0; i < NumCodes(); ++i) 
                {
                    Compile(i);
                }
                _firstEval = false;
                Changed = true;
            }

            _mesh.AttributesSize = InstanceMesh.AttributesSize;
            _mesh.InputElements = InstanceMesh.InputElements;
            int instanceMeshSize = InstanceMesh.Vertices.Description.SizeInBytes;
            //Logger.Info(this, "instance mesh size: {0}", instanceMeshSize);
            Buffer pointBuffer = SurfacePoints.PosBuffer;
            const int instanceDataSize = 4*16; // float4x4
            int numInstances = pointBuffer.Description.SizeInBytes / instanceDataSize;
            _mesh.NumTriangles = InstanceMesh.NumTriangles*numInstances;
            int targetSize = instanceMeshSize * numInstances;
            //Logger.Info(this, "target size: {0}", targetSize);
            if (_mesh.Vertices == null || _mesh.Vertices.Description.SizeInBytes != targetSize)
            {
                Utilities.DisposeObj(ref _mesh.Vertices);
                _mesh.Vertices = new Buffer(D3DDevice.Device, new BufferDescription()
                                                                  {
                                                                      BindFlags = BindFlags.StreamOutput | BindFlags.VertexBuffer,
                                                                      CpuAccessFlags = CpuAccessFlags.None,
                                                                      OptionFlags = ResourceOptionFlags.None,
                                                                      SizeInBytes = targetSize,
                                                                      Usage = ResourceUsage.Default,                                                                      
                                                                  }) 
                                     {
                                         DebugName = "MeshReplicateOnSurfaceBuffer"
                                     };
            }

            var technique = _effect.GetTechniqueByIndex(0);
            var pass = technique.GetPassByIndex(0);
           
            context.D3DDevice.ImmediateContext.InputAssembler.InputLayout = new InputLayout(context.D3DDevice, pass.Description.Signature, _instanceDataInputElements);
            context.D3DDevice.ImmediateContext.InputAssembler.PrimitiveTopology = PrimitiveTopology.TriangleList;

            // Configure Pipeline for stream output
            int streamOutputOffset = 0;
            context.D3DDevice.ImmediateContext.StreamOutput.SetTargets(new [] { new StreamOutputBufferBinding(_mesh.Vertices, streamOutputOffset) });
            context.D3DDevice.ImmediateContext.InputAssembler.SetVertexBuffers(0, new [] { InstanceMesh.Vertices, pointBuffer }, new [] {76, instanceDataSize}, new [] {0,0} );

            // draw
            //Logger.Info(this, "Draw {0} instances", numInstances);
            pass.Apply(context.D3DDevice.ImmediateContext);
            context.D3DDevice.ImmediateContext.DrawInstanced(InstanceMesh.NumTriangles*3, numInstances, 0, 0);

            // clean up pipeline
            context.D3DDevice.ImmediateContext.InputAssembler.SetVertexBuffers(0, new VertexBufferBinding(null,0,0));
            context.D3DDevice.ImmediateContext.InputAssembler.SetVertexBuffers(1, new VertexBufferBinding(null,0,0));
            context.D3DDevice.ImmediateContext.StreamOutput.SetTargets(new [] { new StreamOutputBufferBinding(null, 0) });

            context.Mesh = _mesh;

            return context;
        }

        private Mesh _mesh = new Mesh();
        private InputElement[] _instanceDataInputElements = new []
                                                                {
                                                                    new InputElement("POSITION", 0, SharpDX.DXGI.Format.R32G32B32A32_Float, 0, 0),
                                                                    new InputElement("NORMAL", 0, SharpDX.DXGI.Format.R32G32B32_Float, 16, 0),
                                                                    new InputElement("COLOR", 0, SharpDX.DXGI.Format.R32G32B32A32_Float, 28, 0),
                                                                    new InputElement("TEXCOORD", 0, SharpDX.DXGI.Format.R32G32_Float, 44, 0),
                                                                    new InputElement("TANGENT", 0, SharpDX.DXGI.Format.R32G32B32_Float, 52, 0),
                                                                    new InputElement("BINORMAL", 0, SharpDX.DXGI.Format.R32G32B32_Float, 64, 0),
                                                                    new InputElement("INSTANCE_OBJ_TO_WORLD_ROW", 0, SharpDX.DXGI.Format.R32G32B32A32_Float, 0,
                                                                                     1, InputClassification.PerInstanceData, 1),
                                                                    new InputElement("INSTANCE_OBJ_TO_WORLD_ROW", 1, SharpDX.DXGI.Format.R32G32B32A32_Float, 16,
                                                                                     1, InputClassification.PerInstanceData, 1),
                                                                    new InputElement("INSTANCE_OBJ_TO_WORLD_ROW", 2, SharpDX.DXGI.Format.R32G32B32A32_Float, 32,
                                                                                     1, InputClassification.PerInstanceData, 1),
                                                                    new InputElement("INSTANCE_OBJ_TO_WORLD_ROW", 3, SharpDX.DXGI.Format.R32G32B32A32_Float, 48,
                                                                                     1, InputClassification.PerInstanceData, 1),
                                                                };
    }
}

