//>>> _using
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SharpDX;
using SharpDX.Direct3D11;
using SharpDX.Windows;
//<<< _using

using System.Runtime.InteropServices;
using Framefield.Core.Rendering;
using Framefield.Core.Profiling;
using SharpDX.Direct3D;
using SharpDX.DXGI;
using SharpDX.D3DCompiler;
using Buffer = SharpDX.Direct3D11.Buffer;
using System.CodeDom.Compiler;
using System.Diagnostics;
using System.Dynamic;

namespace Framefield.Core.ID72f1fe7a_9983_44ac_ba77_6f7b2d908ae4
{
    public class Class_EnvIS : FXSourceCodeFunction
    {

        //>>> _inputids
        private enum InputId
        {
            Code = 0,
            Intensity = 1,
            NumSamples = 2,
            MeshPositions = 3,
            MeshNormals = 4
        }
        //<<< _inputids

        //>>> _outputids
        private enum OutputId
        {
            Image = 0,
            PosBuffer = 1
        }
        //<<< _outputids

        public override void Dispose()
        {
            Utilities.DisposeObj(ref _sumTexture);
            Utilities.DisposeObj(ref _sumTextureUAV);
            Utilities.DisposeObj(ref _sumTextureSRV);
            Utilities.DisposeObj(ref _summedColumnTexture);
            Utilities.DisposeObj(ref _summedColumnTextureUAV);
            Utilities.DisposeObj(ref _summedColumnTextureSRV);
            Utilities.DisposeObj(ref _visuTexture);
            Utilities.DisposeObj(ref _visuTextureUAV);
            Utilities.DisposeObj(ref _sampleBuffer);
            Utilities.DisposeObj(ref _sampleBufferUAV);
            Utilities.DisposeObj(ref _sampleBufferSRV);
            Utilities.DisposeObj(ref _worldPosBuffer);
            Utilities.DisposeObj(ref _worldPosBufferUAV);
            Utilities.DisposeObj(ref _worldPosBufferSRV);
            Utilities.DisposeObj(ref _constantBuffer);
            Utilities.DisposeObj(ref _csHistoClear);
            Utilities.DisposeObj(ref _csSumRows);
            Utilities.DisposeObj(ref _csSumColumn);
            Utilities.DisposeObj(ref _csVisu);
            _queryTimeStampDisjoint.Dispose();
            _queryTimeStampFrameBegin.Dispose();
            _queryTimeStampFrameEnd.Dispose();
            base.Dispose();
        }

        protected bool BuildRenderTarget(OperatorPartContext context, Texture2D inputImage)
        {
            if (_sumTexture != null && _sumTexture.Description.Height == inputImage.Description.Height)
            {
                return false; // nothing to do, buffers have already the right size
            }

            // summed up rows texture, has an additional column with the row sum
            var sumTextureDesc = new Texture2DDescription
                                     {
                                         BindFlags = BindFlags.ShaderResource | BindFlags.UnorderedAccess,
                                         Format = Format.R32_Float,
                                         Width = inputImage.Description.Width + 2, // [N] = 1, [N + 1] = Sum of all elements
                                         Height = inputImage.Description.Height,
                                         MipLevels = 1,
                                         SampleDescription = new SampleDescription(1, 0),
                                         Usage = ResourceUsage.Default,
                                         OptionFlags = ResourceOptionFlags.None,
                                         CpuAccessFlags = CpuAccessFlags.None,
                                         ArraySize = 1
                                     };
            Utilities.DisposeObj(ref _sumTexture);
            _sumTexture = new Texture2D(context.D3DDevice, sumTextureDesc);
            Utilities.DisposeObj(ref _sumTextureUAV);
            _sumTextureUAV = new UnorderedAccessView(context.D3DDevice, _sumTexture);
            Utilities.DisposeObj(ref _sumTextureSRV);
            _sumTextureSRV = new ShaderResourceView(context.D3DDevice, _sumTexture);

            // summed up column texture, has an additional row with the column sum

            sumTextureDesc.Width = 1;
            sumTextureDesc.Height = inputImage.Description.Height + 2; // [N] = 1, [N + 1] = Sum of all elements

            Utilities.DisposeObj(ref _summedColumnTexture);
            _summedColumnTexture = new Texture2D(context.D3DDevice, sumTextureDesc);
            Utilities.DisposeObj(ref _summedColumnTextureUAV);
            _summedColumnTextureUAV = new UnorderedAccessView(context.D3DDevice, _summedColumnTexture);
            Utilities.DisposeObj(ref _summedColumnTextureSRV);
            _summedColumnTextureSRV = new ShaderResourceView(context.D3DDevice, _summedColumnTexture);
            
            // helper texture for visualization
            var uavVisuDesc = new Texture2DDescription
                                  {
                                      BindFlags = BindFlags.ShaderResource | BindFlags.UnorderedAccess,
                                      Format = Format.R8G8B8A8_UNorm,
                                      Width = inputImage.Description.Width,
                                      Height = inputImage.Description.Height,
                                      MipLevels = 1,
                                      SampleDescription = new SampleDescription(1, 0),
                                      Usage = ResourceUsage.Default,
                                      OptionFlags = ResourceOptionFlags.None,
                                      CpuAccessFlags = CpuAccessFlags.None,
                                      ArraySize = 1
                                  };
            Utilities.DisposeObj(ref _visuTexture);
            _visuTexture = new Texture2D(context.D3DDevice, uavVisuDesc);
            Utilities.DisposeObj(ref _visuTextureUAV);
            _visuTextureUAV = new UnorderedAccessView(context.D3DDevice, _visuTexture);

            Utilities.DisposeObj(ref _constantBuffer);
            _constantBuffer = Buffer.Create(context.D3DDevice, BindFlags.ConstantBuffer, _constantBufferData, 16);

            return true;
        }


        public override OperatorPartContext Eval(OperatorPartContext context, List<OperatorPart> inputs, int outputIdx) 
        {
            //>>> _params
            var Code = inputs[(int)InputId.Code].Eval(context).Text;
            var Intensity = inputs[(int)InputId.Intensity].Eval(context).Image; // Needs to be checked for null!
            var NumSamples = inputs[(int)InputId.NumSamples].Eval(context).Value;
            var MeshPositions = inputs[(int)InputId.MeshPositions].Eval(context).Image; // Needs to be checked for null!
            var MeshNormals = inputs[(int)InputId.MeshNormals].Eval(context).Image; // Needs to be checked for null!
            //<<< _params
            
            if (Intensity == null)
            {
                Logger.Warn(this, "No imput image connected");
                return context;    
            }
            
            int inputImageHeight = Intensity.Description.Height;
            //Logger.Info(this, "input size: {0}/{1}", Intensity.Description.Width, inputImageHeight);
            
            if (_csSumRows == null)
            {
                Utilities.DisposeObj(ref _csHistoClear);
                Utilities.DisposeObj(ref _csSumRows);
                Utilities.DisposeObj(ref _csVisu);
                var errors = new CompilerErrorCollection();
                try
                {
                    using (var bytecode = ShaderBytecode.Compile(GetCode(0), "ClearVisuBuffer", "cs_5_0", ShaderFlags.Debug))
                        _csHistoClear = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.Compile(GetCode(0), "SumRows", "cs_5_0", ShaderFlags.Debug))
                        _csSumRows = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.Compile(GetCode(0), "SumColumn", "cs_5_0", ShaderFlags.Debug))
                        _csSumColumn = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.Compile(GetCode(0), "GenerateUVPositions", "cs_5_0", ShaderFlags.Debug))
                        _csGenerateUVPositions = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.Compile(GetCode(0), "GenerateWorldPositions", "cs_5_0", ShaderFlags.Debug))
                        _csGenerateWorldPositions = new ComputeShader(D3DDevice.Device, bytecode);
                    using (var bytecode = ShaderBytecode.Compile(GetCode(0), "VisualizeSumBuffer", "cs_5_0", ShaderFlags.Debug))
                        _csVisu = new ComputeShader(D3DDevice.Device, bytecode);
                }
                catch (CompilationException ex)
                {
                    errors = ErrorsFromString(ex.Message);
                    Logger.Error(this, "CS compile error: {0}", ex.Message);
                }
            }

            BuildRenderTarget(context, Intensity);

            var targetSize = sizeof(float)*2*(int)NumSamples;
            if (_sampleBuffer == null || _sampleBuffer.Description.SizeInBytes != targetSize)
            {
                // create buffer for sample locations
                Utilities.DisposeObj(ref _sampleBuffer);
                _sampleBuffer = new Buffer(context.D3DDevice, new BufferDescription()
                                                                  {
                                                                      BindFlags = BindFlags.ShaderResource | BindFlags.UnorderedAccess | BindFlags.VertexBuffer,
                                                                      CpuAccessFlags = CpuAccessFlags.None,
                                                                      OptionFlags = ResourceOptionFlags.BufferAllowRawViews,
                                                                      SizeInBytes = targetSize,
                                                                      Usage = ResourceUsage.Default
                                                                  });
                
                {
                    // create its shader resource view                
                    Utilities.DisposeObj(ref _sampleBufferSRV);
                    var bufferResource = new ShaderResourceViewDescription.ExtendedBufferResource()
                                             {
                                                 ElementCount = _sampleBuffer.Description.SizeInBytes/4,
                                                 FirstElement = 0,
                                                 Flags = ShaderResourceViewExtendedBufferFlags.Raw
                                             };
                    var srvDesc = new ShaderResourceViewDescription()
                                      {
                                          Format = Format.R32_Typeless,
                                          Dimension = ShaderResourceViewDimension.ExtendedBuffer,
                                          BufferEx = bufferResource
                                      };
                    _sampleBufferSRV = new ShaderResourceView(context.D3DDevice, _sampleBuffer, srvDesc);
                }
                
                {
                    // create its unordered access view
                    Utilities.DisposeObj(ref _sampleBufferUAV);
                                      
                    var bufferResource = new UnorderedAccessViewDescription.BufferResource()
                                             {
                                                 ElementCount = _sampleBuffer.Description.SizeInBytes/4,
                                                 FirstElement = 0,
                                                 Flags = UnorderedAccessViewBufferFlags.Raw
                                             };
                    var uavDesc = new UnorderedAccessViewDescription()
                                      {
                                          Format = Format.R32_Typeless,
                                          Dimension = UnorderedAccessViewDimension.Buffer,
                                          Buffer = bufferResource
                                      };
                    _sampleBufferUAV = new UnorderedAccessView(context.D3DDevice, _sampleBuffer, uavDesc);                                  
                }
                
                
                // create buffer for world positions 
                Utilities.DisposeObj(ref _worldPosBuffer);
                _worldPosBuffer = new Buffer(context.D3DDevice, new BufferDescription()
                                                                    {
                                                                        BindFlags = BindFlags.ShaderResource | BindFlags.UnorderedAccess | BindFlags.VertexBuffer,
                                                                        CpuAccessFlags = CpuAccessFlags.None,
                                                                        OptionFlags = ResourceOptionFlags.BufferAllowRawViews,
                                                                        SizeInBytes = /*sizeof(float)*3*/64*(int)NumSamples,
                                                                        Usage = ResourceUsage.Default
                                                                    });

                {
                    // create its shader resource view                
                    Utilities.DisposeObj(ref _worldPosBufferSRV);
                    var bufferResource = new ShaderResourceViewDescription.ExtendedBufferResource()
                                             {
                                                 ElementCount = _worldPosBuffer.Description.SizeInBytes/4,
                                                 FirstElement = 0,
                                                 Flags = ShaderResourceViewExtendedBufferFlags.Raw
                                             };
                    var srvDesc = new ShaderResourceViewDescription()
                                      {
                                          Format = Format.R32_Typeless,
                                          Dimension = ShaderResourceViewDimension.ExtendedBuffer,
                                          BufferEx = bufferResource
                                      };
                    _worldPosBufferSRV = new ShaderResourceView(context.D3DDevice, _worldPosBuffer, srvDesc);
                }

                {
                    // create its unordered access view
                    Utilities.DisposeObj(ref _worldPosBufferUAV);
                                      
                    var bufferResource = new UnorderedAccessViewDescription.BufferResource()
                                             {
                                                 ElementCount = _worldPosBuffer.Description.SizeInBytes/4,
                                                 FirstElement = 0,
                                                 Flags = UnorderedAccessViewBufferFlags.Raw
                                             };
                    var uavDesc = new UnorderedAccessViewDescription()
                                      {
                                          Format = Format.R32_Typeless,
                                          Dimension = UnorderedAccessViewDimension.Buffer,
                                          Buffer = bufferResource
                                      };
                    _worldPosBufferUAV = new UnorderedAccessView(context.D3DDevice, _worldPosBuffer, uavDesc);                                  
                }

            }



            var deviceContext = context.D3DDevice.ImmediateContext;

            if (_readyToMeasure)
            {
                _queryTimeStampDisjoint.Begin(deviceContext);
                _queryTimeStampFrameBegin.End(deviceContext);
            }

            using (var intensityImageView = new ShaderResourceView(context.D3DDevice, Intensity))
            using (var meshPosSRV = new ShaderResourceView(context.D3DDevice, MeshPositions))
            using (var meshNormalsSRV = new ShaderResourceView(context.D3DDevice, MeshNormals))
            {
//                deviceContext.ComputeShader.Set(_csHistoClear);
//                deviceContext.ComputeShader.SetUnorderedAccessView(0, _sumTextureUAV);
//                deviceContext.Dispatch(1, Image.Description.Height/128, 1);

                deviceContext.ComputeShader.Set(_csSumRows);
                deviceContext.ComputeShader.SetUnorderedAccessView(0, _sumTextureUAV);
                deviceContext.ComputeShader.SetShaderResource(0, intensityImageView);
                deviceContext.Dispatch(1, Intensity.Description.Height/128, 1);
                deviceContext.ComputeShader.SetUnorderedAccessView(0, null);
 
                deviceContext.ComputeShader.Set(_csSumColumn);
                deviceContext.ComputeShader.SetUnorderedAccessView(1, _summedColumnTextureUAV);
                deviceContext.ComputeShader.SetShaderResource(1, _sumTextureSRV);
                deviceContext.Dispatch(1, 1, 1);
                deviceContext.ComputeShader.SetUnorderedAccessView(1, null);
                deviceContext.ComputeShader.SetShaderResource(1, null);

deviceContext.ComputeShader.SetConstantBuffer(0, _constantBuffer);
_constantBufferData[0] = (int)NumSamples;
deviceContext.UpdateSubresource(_constantBufferData, _constantBuffer);

                deviceContext.ComputeShader.SetUnorderedAccessView(2, _sampleBufferUAV);
                deviceContext.ComputeShader.SetShaderResource(2, _sumTextureSRV);
                deviceContext.ComputeShader.SetShaderResource(3, _summedColumnTextureSRV);
                deviceContext.ComputeShader.Set(_csGenerateUVPositions);
                deviceContext.Dispatch((int)NumSamples/256 + 1, 1, 1);
                deviceContext.ComputeShader.SetShaderResource(2, null);
                deviceContext.ComputeShader.SetShaderResource(3, null);
                deviceContext.ComputeShader.SetUnorderedAccessView(2, null);
                deviceContext.ComputeShader.SetShaderResource(0, null);

                deviceContext.ComputeShader.SetUnorderedAccessView(3, _worldPosBufferUAV);
                deviceContext.ComputeShader.SetShaderResource(4, _sampleBufferSRV);
                deviceContext.ComputeShader.SetShaderResource(5, meshPosSRV);
                deviceContext.ComputeShader.SetShaderResource(6, meshNormalsSRV);
                deviceContext.ComputeShader.Set(_csGenerateWorldPositions);
                deviceContext.Dispatch((int)NumSamples/256 + 1, 1, 1);
                deviceContext.ComputeShader.SetShaderResource(4, null);
                deviceContext.ComputeShader.SetShaderResource(5, null);
                deviceContext.ComputeShader.SetUnorderedAccessView(3, null);

                    deviceContext.ComputeShader.SetUnorderedAccessView(7, _visuTextureUAV);
                    deviceContext.ComputeShader.SetShaderResource(4, _sampleBufferSRV);
                    int width = _visuTexture.Description.Width;
                    int height = _visuTexture.Description.Height;
                    deviceContext.ComputeShader.Set(_csHistoClear);
                    deviceContext.Dispatch(width/32 + 1, height/32 + 1, 1);
                    deviceContext.ComputeShader.Set(_csVisu);
                    deviceContext.Dispatch((int)NumSamples/256 + 1, 1, 1);
                    deviceContext.ComputeShader.SetUnorderedAccessView(7, null);
                    deviceContext.ComputeShader.SetShaderResource(2, null);
                    deviceContext.ComputeShader.SetShaderResource(3, null);
                    deviceContext.ComputeShader.SetShaderResource(0, null);


            }
            
            if (_readyToMeasure)
            {
                _queryTimeStampFrameEnd.End(deviceContext);
                _queryTimeStampDisjoint.End(deviceContext);
                _readyToMeasure = false;
            }
            else
            {
                // check if last measurement is ready
                QueryDataTimestampDisjoint disjointData;
                long timeStampframeBegin;
                long timeStampframeEnd;
                bool dataFetched = true;
                dataFetched &= _queryTimeStampDisjoint.GetData(deviceContext, AsynchronousFlags.None, out disjointData);
                dataFetched &= _queryTimeStampFrameBegin.GetData(deviceContext, AsynchronousFlags.None, out timeStampframeBegin);
                dataFetched &= _queryTimeStampFrameEnd.GetData(deviceContext, AsynchronousFlags.None, out timeStampframeEnd);
                
                if (dataFetched && !disjointData.Disjoint)
                {
                    float duration = (float)(timeStampframeEnd - timeStampframeBegin)/disjointData.Frequency;
                    //Logger.Info(this, "tsfb: {0}", timeStampframeBegin);
                    //Logger.Info(this, "tsfe: {0}", timeStampframeEnd);
                    //Logger.Info(this, "freq: {0}", disjointData.Frequency);
                    //Logger.Info(this,"row calc took: {0}s", duration);
                    _readyToMeasure = true;
                }
            }
                        
            switch (outputIdx)
            {
                case (int)OutputId.Image:
                
                    context.Image = _visuTexture;
                    break;
                case (int)OutputId.PosBuffer:
                    dynamic obj = new ExpandoObject();
                    obj.PosBuffer = _worldPosBuffer;
                    context.Dynamic = obj;
                    break;
                default:
                    Logger.Warn(this, "Output index '{0}' requested, but not existing.", outputIdx);
                    break;
            }
            
            return context;
        }

        private Texture2D _sumTexture;
        private UnorderedAccessView _sumTextureUAV;
        private ShaderResourceView _sumTextureSRV;
        private Texture2D _summedColumnTexture;
        private UnorderedAccessView _summedColumnTextureUAV;
        private ShaderResourceView _summedColumnTextureSRV;
        private Texture2D _visuTexture;
        private UnorderedAccessView _visuTextureUAV;
        private Buffer _sampleBuffer;
        private ShaderResourceView _sampleBufferSRV;
        private UnorderedAccessView _sampleBufferUAV;
        private Buffer _worldPosBuffer;
        private ShaderResourceView _worldPosBufferSRV;
        private UnorderedAccessView _worldPosBufferUAV;
        private Buffer _constantBuffer;
        private int[] _constantBufferData = new int[4];
        private ComputeShader _csHistoClear;
        private ComputeShader _csSumRows;
        private ComputeShader _csSumColumn;
        private ComputeShader _csGenerateUVPositions;
        private ComputeShader _csGenerateWorldPositions;
        private ComputeShader _csVisu;
        private GPUQuery _queryTimeStampDisjoint = new GPUQuery(D3DDevice.Device, new QueryDescription() { Type = QueryType.TimestampDisjoint });
        private GPUQuery _queryTimeStampFrameBegin = new GPUQuery(D3DDevice.Device, new QueryDescription() { Type = QueryType.Timestamp });
        private GPUQuery _queryTimeStampFrameEnd = new GPUQuery(D3DDevice.Device, new QueryDescription() { Type = QueryType.Timestamp });
        private bool _readyToMeasure = true;
    }
}

