//--------------------------------------------------------------------------------------
// Constant Buffer Variables
//--------------------------------------------------------------------------------------
Texture2D texAmbient : register(t0);
Texture2D texDiffuse : register(t1);
Texture2D texEmissive: register(t2);
SamplerState sampLinearClamp : register(s0);
SamplerState sampNearestClamp : register(s1);
SamplerState sampLinearWrap : register(s2);
SamplerState sampNearestWrap : register(s3);

cbuffer GlobalRenderData : register(b3)
{
    matrix g_matWorld;
    matrix g_matView;
    matrix g_matProjection;
    float4 g_vecEyePoint;
};

struct Light
{
    float4 m_vecPosition;
    float4 m_vecDirection;
    float4 m_vecColour;
    float m_fMaxDistance;
    float m_fFalloff;
    int m_iType;
    float lpad1;
};
StructuredBuffer<Light> g_aLights : register(t5);
cbuffer GlobalLightData : register(b1)
{
    int g_iNumLights;
    float g_fVisibilityFallOffMin;
    float g_fVisibilityFallOffLength;
    float gbrpad1;
};

struct ModelData
{
    matrix m_matModel;
};
StructuredBuffer<ModelData> g_aModels : register(t6);

cbuffer MaterialData : register(b2)
{
    float4 g_vecAmbientColour;
    float4 g_vecDiffuseColour;
    float4 g_vecEmissiveColour;
    float4 g_vecSpecularColour;
    float g_fSpecularHardness;
    float mrdpad1, mrdpad2, mrpad3;
};


//--------------------------------------------------------------------------------------
struct VS_INPUT
{
    float3 vecPosition : POSITION;
    float3 vecNormal : NORMAL;
    float2 vecTexCoord : TEXCOORD0;
    uint iInstanceID : SV_InstanceID;
};

struct PS_INPUT
{
    float4 vecPosition : SV_POSITION;
    float4 vecWorldPosition : TEXCOORD0;
    float3 vecNormal : NORMAL;
    float2 vecTexCoord : TEXCOORD1;
};


//--------------------------------------------------------------------------------------
// Vertex Shader
//--------------------------------------------------------------------------------------
PS_INPUT VS( VS_INPUT input )
{
    PS_INPUT output;

    matrix matWorld = mul(g_matWorld, g_aModels[input.iInstanceID].m_matModel);
    matrix matWorldView = mul(matWorld, g_matView);
    matrix matWorldViewProjection = mul(matWorldView, g_matProjection);

    output.vecPosition = mul(float4(input.vecPosition, 1), matWorldViewProjection);
    output.vecWorldPosition = mul(float4(input.vecPosition, 1), matWorld);
    output.vecNormal = normalize(mul(input.vecNormal, matWorld));

    output.vecTexCoord = input.vecTexCoord;

    return output;
}


//--------------------------------------------------------------------------------------
// Pixel Shader
//--------------------------------------------------------------------------------------
float4 PS(PS_INPUT input) : SV_Target
{
    float4 vecAmbientTexture = texAmbient.Sample(sampLinearWrap, input.vecTexCoord);
    float4 vecDiffuseTexture = texDiffuse.Sample(sampLinearWrap, input.vecTexCoord);
    float4 vecEmissiveTexture = texEmissive.Sample(sampLinearWrap, input.vecTexCoord);

    float3 vecSurfaceNormal = normalize(input.vecNormal);       // need to normalize?
    float3 vecWorldPos = input.vecWorldPosition.xyz;

    float4 vecCumulativeColour = float4(0, 0, 0, 0);
    for (int i = 0; i < g_iNumLights; i++)
    {
        Light currentLight = g_aLights[i];
     
        switch (currentLight.m_iType)
        {
            // -- Ambient --
            case 0:
            {
                float4 vecLightColour = currentLight.m_vecColour;
                float4 vecAmbientColour = g_vecAmbientColour * vecAmbientTexture * vecDiffuseTexture * vecLightColour;
                vecCumulativeColour += vecAmbientColour;
            }
            break;

            // -- Directional --
            case 1:
            {
                float3 vecLightDirection = currentLight.m_vecDirection.xyz;
                float3 vecToLightNormal = -vecLightDirection.xyz;
                float fLightingValue = max(0, dot(vecToLightNormal, vecSurfaceNormal));
                float4 vecDiffuseColour = fLightingValue * g_vecDiffuseColour * vecDiffuseTexture * currentLight.m_vecColour;
                vecCumulativeColour += vecDiffuseColour;
            }
            break;

            // -- Point --
            case 2:
            {
                // Input rays
                float3 vecToLight = currentLight.m_vecPosition - vecWorldPos;
                float3 vecToLightNormal = normalize(vecToLight);
                float fDistanceToLight = length(vecToLight);
                float fLightIntensity = pow(1 - saturate(fDistanceToLight / currentLight.m_fMaxDistance), currentLight.m_fFalloff);

                // Diffuse colour
                float fDiff = max(0, dot(vecSurfaceNormal, vecToLightNormal));
                float4 vecDiffColour = g_vecDiffuseColour * vecDiffuseTexture * currentLight.m_vecColour * fDiff;
 
//                 // Specular colour
//                 float3 vecReflect = normalize(reflect(-vecToLightNormal, vecSurfaceNormal));
//                 float fCosAng = max(0, dot(vecToCameraNormal, vecReflect));
//                 float fSpec = pow(fCosAng, fShininess);
//                 float3 vecSpecColour = vecMaterialSpecularColour * currentLight.m_vecColour.rgb * fSpec * fDiff;

                // Sum to get final colour
                vecCumulativeColour += (vecDiffColour/* + vecSpecColour*/) * fLightIntensity;
            }
            break;
        }
    }

    // Emissive
    vecCumulativeColour += vecEmissiveTexture * g_vecEmissiveColour;

    // Max visible distance
    const float fDistance = length(vecWorldPos - g_vecEyePoint.xyz);
    float fFade = clamp(1 - (fDistance - g_fVisibilityFallOffMin) / g_fVisibilityFallOffLength, 0, 1);
    vecCumulativeColour.rgb *= fFade;

    return vecCumulativeColour;

    /*
//float3 g_vecLightDirection = normalize(float3(0.3, -1.0, -0.5));
float3 g_vecLightDirection = g_aLights[0].m_vecDirection;
    float3 vecToLightNormal = -g_vecLightDirection.xyz;
    float fLightingValue = max(0, dot(vecToLightNormal, vecSurfaceNormal));
    //     float3 vecToCameraNormal = normalize(g_vecEyePos.xyz - input.vecWorldPosition.xyz);
    //     float3 vecReflect = normalize(reflect(-vecToLightNormal, vecSurfaceNormal.xyz));
    //     float fCosAng = max(0, dot(vecToCameraNormal, vecReflect));
    //     float fSpec = pow(fCosAng, fShininess);
    //     float3 vecSpecColour = vecMaterialSpecularColour * fSpec;

    float4 vecAmbientColour = g_vecAmbientColour * vecAmbientTexture * vecDiffuseTexture;
    float4 vecDiffuseColour = fLightingValue * g_vecDiffuseColour * vecDiffuseTexture;
    float4 vecEmissiveColour = g_vecEmissiveColour * vecEmissiveTexture;
    float4 vecSpecularColour = float4(0, 0, 0, 0);//g_vecSpecularColour;

    float4 finalColour = vecAmbientColour + vecDiffuseColour + vecEmissiveColour + vecSpecularColour;
    return float4(finalColour.rgb, 1);
    */
}
