/**
 * Created by Hans on 25-10-2016.
 */
"use strict";
var express = require('express');
var app = express();

var http = require('http').createServer(app);
var fs = require('fs');
var io = require('socket.io').listen(http);

app.use(express.static('public'));

var port = 3001;

http.listen(port, function() {
    console.log('listening on port ' + port);
});

// SERVERSIDE GAME LOGIC

var namePrefix = 'noob';
var nameCounter = 0;

let tankBehavior = {
    rotationSpeed: 2,
    moveSpeed: 4,
    shootTimer: 3000
};

let worldBehavior = {
    updateSpeed: 1000 / 30,
    projectileSpeed: 10,
    projectileTravel: 600,
    projectileLifetime: 1500,
    outsideTimer: 5000,
    width: 2000,
    height: 1600,
    relativeX: 0,
    relativeY: 0,
    powerUps: []
};

worldBehavior.puddles = generatePuddles(10);
worldBehavior.trees = generateTrees(40);

let playerTank = {
    moveUp: false,
    moveDown: false,
    moveLeft: false,
    moveRight: false,
    x: 0,
    y: 0,
    rotation: 0,
    hasShot: false,
    lastShot: 0
};

var players = [];
var playerScores = [];

// SOCKET.IO SETTINGS

io.on('connection', function(client) {

    // HANDLING DISCONNECTS
    client.on('disconnect', function () {
        if (playerIsAlive(client.id)) {
            client.broadcast.emit('serverClientDied', {player: getPlayerById(client.id), reason: 'disconnected.'});
            removePlayer(client.id);
        }
    });

    client.on('clientReady', function(settings) {

        let newTank = JSON.parse(JSON.stringify(playerTank));

        // MANIPULATE TANK HERE

        newTank.x = getRandomInt(0, worldBehavior.width);
        newTank.y = getRandomInt(0, worldBehavior.height);

        //newTank.x = 0;
        //newTank.y = 0;

        var name = settings.name;

        if (name.trim() === '') {
            nameCounter++;
            name = namePrefix + ('00' + nameCounter).slice(-3);
        }

        let player = {
            id: client.id,
            name: name,
            tank: newTank,
            points: 0,
            bodyColour: settings.body,
            turretColour: settings.turret,
            barrelColour: settings.barrel
        };

        let playerScore = {
            id: client.id,
            name: name,
            score: 0
        };

        players.push(player);
        playerScores.push(playerScore);

        //console.log(players);

        client.emit('serverHandShake', {
            world: worldBehavior,
            behavior: tankBehavior,
            player: player,
            players: players
        });
        client.broadcast.emit('serverNewHandShake', {newPlayer: player, allPlayers: players});

        updateScoreBoard(client.id, true);

    });

    // GAME LOGIC

    client.on('clientChangedKeyState', function(player) {
        updateSinglePlayer(player);

        client.broadcast.emit('serverChangedKeyState', player);
    });

    client.on('clientProjectileFired', function(projectile) {
        client.broadcast.emit('serverNewProjectileFired', projectile);
    });

    client.on('clientProjectileHit', function(toFrom) {
        //console.log(players);
        client.broadcast.emit('serverProjectileHit', toFrom);
    });

    client.on('clientDied', function(death) {
        // emit getPlayerById(death.id).name + reason
        client.broadcast.emit('serverClientDied', {player: getPlayerById(death.id), reason: death.reason});
        removePlayer(death.id);
        updateScoreBoard(death.creditId, false);
    });

    /*
    client.on('clientHitPowerUp', function(powerUpId) {
        deadPowerUp(powerUpId.id);
    });
    */
});

// WORLD LOGIC

function generatePuddles(amount) {
    var puddleSizeMin = 100;
    var puddleSizeMax = 250;

    var puddles = [];

    for (var i = 0; i < amount; i++) {
        var puddleSize = getRandomInt(puddleSizeMin, puddleSizeMax);
        let puddle = {
            size: puddleSize,
            worldX: getRandomInt(0, worldBehavior.width - puddleSize),
            worldY: getRandomInt(0, worldBehavior.height - puddleSize)
        };

        puddles.push(puddle);
    }

    return puddles;
}

function generateTrees(amount) {
    var treeSizeMin = 50;
    var treeSizeMax = 100;

    var trees = [];

    for (var i = 0; i < amount; i++) {
        var treeSize = getRandomInt(treeSizeMin, treeSizeMax);
        let tree = {
            size: treeSize,
            markerSize: Math.round(treeSize * 0.5),
            worldX: getRandomInt(treeSize, worldBehavior.width - treeSize),
            worldY: getRandomInt(treeSize, worldBehavior.height - treeSize)
        };

        trees.push(tree);
    }

    return trees;
}

/*
var powerUpTimer = setInterval(function() {

    if (worldBehavior.powerUps.length > 5) {
        var removed = worldBehavior.powerUps[0];
        worldBehavior.powerUps.splice(0, 1);
        io.sockets.emit('removePowerUp', removed);
    }

    var pu = generatePowerUp();

    io.sockets.emit('newPowerUp', pu);

    worldBehavior.powerUps.push(pu);

    //console.log(worldBehavior.powerUps);
    //console.log(worldBehavior.powerUps.length);

}, 10000);
*/

function deadPowerUp(id) {
    var x = 0;
    for (var i = 0; i < worldBehavior.powerUps.length; i++) {
        if (worldBehavior.powerUps[i].id === id) {
            x = i;
        }
    }
    worldBehavior.powerUps.splice(x, 1);

    console.log('dead: ' + id);

    io.sockets.emit('removePowerUp', {id: id});
}

// SERVER LOGIC

function updateScoreBoard(playerId, newPlayer) {

    if (!newPlayer) {
        updateScore(playerId);
    }

    playerScores.sort(sortScores);

    io.sockets.emit('serverUpdateScoreboard', playerScores);
}

function updateScore(playerId) {
    for (let player of playerScores) {
        if (playerId === player.id) {
            player.score += 1;
            break;
        }
    }
}

function sortScores(a, b) {
    return b.score - a.score;
}

function updateSinglePlayer(playerData) {
    for (var i = 0; i < players.length; i++) {
        if (playerData.id === players[i].id) {
            players[i] = playerData;
            break;
        }
    }
}

function removePlayer(clientId) {
    let player = undefined;
    for (var i = 0; i < players.length; i++) {
        player = players[i];
        if (clientId === player.id) {
            players.splice(i, 1);
            break;
        }
    }

    let playerScore = undefined;
    for (var i = 0; i < playerScores.length; i++) {
        playerScore = playerScores[i];
        if (clientId === playerScore.id) {
            playerScores.splice(i, 1);
            break;
        }
    }
}

function getPlayerById(clientId) {
    for (let player of players) {
        if (player.id === clientId) {
            return player;
        }
    }
}

function getRandomInt(min, max) {
    return Math.floor(Math.random() * (max - min + 1)) + min;
}

function playerIsAlive(clientId) {
    for (let player of players) {
        if (clientId === player.id) {
            return true;
        }
    }
    return false;
}



function generatePowerUp() {
    /*
    switch(getRandomInt(1,5)) {
        case 1:

            break;
        case 2:

            break;
    }
    */

    var powerUpSize = 50;

    return {
        time: getRandomInt(1, 4) * 15,
        type: 'speed',
        multiplier: getRandomInt(50, 100) / 100,
        worldX: getRandomInt(0, worldBehavior.width - powerUpSize),
        worldY: getRandomInt(0, worldBehavior.height - powerUpSize),
        id: Math.random().toString(36).substr(2, 4)
    }
}


// EXPRESS SETTINGS

app.get('/', function(req, res) {

    res.sendFile(__dirname + '/views/index.html');

});
app.get('/*', function(req, res) {
    res.redirect('/');
});