import QtQuick 2.3
import AdaptDemoSystem 1.0

Group {
    property bool hasCamOverride: false
    onHasCamOverrideChanged: {
        updateCamOverrideDisplay();
    }
    property vector3d camOfs: Qt.vector3d(0.0, 0.0, 0.0)

    property vector3d camPos: Qt.vector3d(0.0, 0.0, 0.0)
    property vector3d camPosOrig: Qt.vector3d(0.0, 0.0, 0.0)
    property vector3d camLookAt: Qt.vector3d(0.0, 0.0, 0.0)
    property vector3d camUp: Qt.vector3d(0.0, 0.0, 0.0)
    property vector3d camUpOrig: Qt.vector3d(0.0, 0.0, 0.0)
    property vector3d camLookDir: Qt.vector3d(0.0, 0.0, 0.0)
    property vector3d camLookDirOrig: Qt.vector3d(0.0, 0.0, 0.0)
    property vector3d camRotAxis: Qt.vector3d(0.0, 0.0, 0.0)

    property vector3d camRightNorm: Qt.vector3d(0.0, 0.0, 0.0)
    property vector3d camUpNorm: Qt.vector3d(0.0, 0.0, 0.0)

    property bool smoothInited: false
    property vector3d camPosSmooth: Qt.vector3d(0.0, 0.0, 0.0)
    property vector3d camLookAtSmooth: Qt.vector3d(0.0, 0.0, 0.0)
    property vector3d camUpSmooth: Qt.vector3d(0.0, 0.0, 0.0)


    property real camMoveStep: 0.1
    property real camMoveStepBase: 0.1
    property real rotSpeed: 0.5
    property real rotSpeedBase: 0.5

    property bool camAutoUpdate: false

    property bool keepCamUp: true

    property bool bAltDown: false
    property bool bShiftDown: false
    property bool bIDown: false

    property bool bLeftPressed: false
    property bool bRightPressed: false
    property bool bUpPressed: false
    property bool bDownPressed: false

    property int mousePressX: 0
    property int mousePressY: 0
    property bool mousePressed: false


    property real resX: 0.0
    property real resY: 0.0


    property bool bDisplayInited: false

    function updateUpModeDisplay() {
        if (keepCamUp) {
            root.setCustomText("fly camera", "fly camera up", "<b>U</b>: Up vector force ON");
        } else {
            root.setCustomText("fly camera", "fly camera up", "<b>U</b>: Up vector force OFF");
        }
    }
    function updateCamOverrideDisplay() {
        if (hasCamOverride) {
            root.setCustomText("fly camera", "fly camera active", "<br>Camera override active<br><b>R</b>: Sync to rocket<br><b>C</b>: Clear modifications");
        } else {
            root.setCustomText("fly camera", "fly camera active", "");
        }
    }

    Connections {
        target: demo;
        onFrameRendered: {

            if (bLeftPressed) {
                activateCamOverride();
                if (bShiftDown) {
                    camUp = rotateAxis(camUp, camLookDir, 5.0*rotSpeed);
                } else {
                    camPos = camPos.minus(camRightNorm.times(camMoveStep));
                    camLookAt = camLookAt.minus(camRightNorm.times(camMoveStep));
                }
                updateCam();
            }
            if (bRightPressed) {
                activateCamOverride();
                if (bShiftDown) {
                    camUp = rotateAxis(camUp, camLookDir, -5.0*rotSpeed);
                } else {
                    camPos = camPos.plus(camRightNorm.times(camMoveStep));
                    camLookAt = camLookAt.plus(camRightNorm.times(camMoveStep));
                }
                updateCam();
            }
            if (bUpPressed) {
                activateCamOverride();
                if (bShiftDown) {
                    camPos = camPos.plus(camUpNorm.times(camMoveStep));
                    camLookAt = camLookAt.plus(camUpNorm.times(camMoveStep));
                } else {
                    camPos = camPos.plus(camLookDir.times(camMoveStep));
                    camLookAt = camLookAt.plus(camLookDir.times(camMoveStep));
                }
                updateCam();
            }
            if (bDownPressed) {
                activateCamOverride();
                if (bShiftDown) {
                    camPos = camPos.minus(camUpNorm.times(camMoveStep));
                    camLookAt = camLookAt.minus(camUpNorm.times(camMoveStep));
                } else {
                    camPos = camPos.minus(camLookDir.times(camMoveStep));
                    camLookAt = camLookAt.minus(camLookDir.times(camMoveStep));
                }
                updateCam();
            }


            if (!bDisplayInited) {
                root.setCustomText("fly camera", "fly camera title", "<br>------------------------------------<br>Fly camera settings (<b>F2</b> hide)");
                updateUpModeDisplay();
                updateCamOverrideDisplay();
                root.setCustomText("fly camera", "fly camera footer", "------------------------------------<br>");
                bDisplayInited = true;
            }

            var camMoveMult = rocket.timeStepReal/16.67;
            camMoveStep = camMoveStepBase*camMoveMult;
            rotSpeed = rotSpeedBase*camMoveMult;

            // var smoothMult = (1.0-Math.min(rocket.timeStepReal, 900)*0.001)*0.5;
            var smoothMult = Math.pow((1.0-0.01667)*0.5, 16.67/rocket.timeStepReal);
            camPosSmooth = camPosSmooth.plus(camPos.minus(camPosSmooth).times(smoothMult));
            camLookAtSmooth = camLookAtSmooth.plus(camLookAt.minus(camLookAtSmooth).times(smoothMult));
            camUpSmooth = camUpSmooth.plus(camUp.minus(camUpSmooth).times(smoothMult));
        }
    }


    function rotateXY(x, y, a) {
      resX = Math.cos(a)*x - Math.sin(a)*y;
      resY = Math.sin(a)*x + Math.cos(a)*y;
    }

    /// Rotate position p around given axis ax for given angle a degrees.
    /// The axis has to be normalized.
    function rotateAxis(p, ax, a) {
        var result = Qt.vector3d(0.0, 0.0, 0.0);
        a = a*Math.PI/180.0;

        var cosa = Math.cos(a);
        var sina = Math.sin(a);
        var mcosa = 1.0-cosa;

        var x = p.x; var y = p.y; var z = p.z;
        var u = ax.x; var v = ax.y; var w = ax.z;
        var dot = p.dotProduct(ax);

        result.x = u*dot*mcosa+x*cosa+(-w*y+v*z)*sina;
        result.y = v*dot*mcosa+y*cosa+(w*x-u*z)*sina;
        result.z = w*dot*mcosa+z*cosa+(-v*x+u*y)*sina;

        return result;
    }

    function updateCamVecs() {
        camLookDir = camLookAt.minus(camPos);
        camLookDir = camLookDir.normalized();
        camRightNorm = camLookDir.crossProduct(camUp);
        camRightNorm = camRightNorm.normalized();
        if (!keepCamUp) {
            camUp = camRightNorm.crossProduct(camLookDir);
        }
        camUpNorm = camRightNorm.crossProduct(camLookDir).normalized();
    }

    function activateCamOverride() {
        rocket. markUpdateFrequent();
        if (!hasCamOverride) {
            camPos = getCamPos();
            camLookAt = getCamLookAt();
            camUp = getCamUp();
            hasCamOverride = true;
        }
        updateCamVecs();
    }

    function updateCam() {
        if (camAutoUpdate) {
            saveCamToRocket();
        }
    }

    function saveCamToRocket() {
        rocket.setTrackValue(sceneName+".camera.x", camPos.x);
        rocket.setTrackValue(sceneName+".camera.y", camPos.y);
        rocket.setTrackValue(sceneName+".camera.z", camPos.z);
        rocket.setTrackValue(sceneName+".camera.lookAtX", camLookAt.x);
        rocket.setTrackValue(sceneName+".camera.lookAtY", camLookAt.y);
        rocket.setTrackValue(sceneName+".camera.lookAtZ", camLookAt.z);
        rocket.setTrackValue(sceneName+".camera.upX", camUp.x);
        rocket.setTrackValue(sceneName+".camera.upY", camUp.y);
        rocket.setTrackValue(sceneName+".camera.upZ", camUp.z);
    }

    Connections {
        target: root
        onKeyC: {
            rocket.markUpdateFrequent();
            hasCamOverride = false;
            smoothInited = false;
        }
        onKeyR: {
            rocket.markUpdateFrequent();
            if (hasCamOverride) {
                saveCamToRocket();
            }
            hasCamOverride = false;
        }

        onKeyPressed: {
            console.log(keyCode);
            if (keyCode === Qt.Key_Left) {
                bLeftPressed = true;
            } else if (keyCode === Qt.Key_Right) {
                bRightPressed = true;
            } else if (keyCode === Qt.Key_Up) {
                bUpPressed = true;
            } else if (keyCode === Qt.Key_Down) {
                bDownPressed = true;
            }

            if (keyCode === 16777251) {
                bAltDown = true;
            } else if (keyCode === 16777248) {
                bShiftDown = true;
            } else if (keyCode === 85) { // U
                keepCamUp = !keepCamUp;
                updateUpModeDisplay();
            } else if (keyCode === 73) { // I
                bIDown = true;
            } else if (keyCode === Qt.Key_F2) {
                console.log("toggleCustomTextGroupShow");
                root.toggleCustomTextGroupShow("fly camera");
            }

        }
        onKeyReleased: {
            if (keyCode === Qt.Key_Left) {
                bLeftPressed = false;
            } else if (keyCode === Qt.Key_Right) {
                bRightPressed = false;
            } else if (keyCode === Qt.Key_Up) {
                bUpPressed = false;
            } else if (keyCode === Qt.Key_Down) {
                bDownPressed = false;
            }

            if (keyCode === 16777251) {
                bAltDown = false;
            } else if (keyCode === 16777248) {
                bShiftDown = false;
            } else if (keyCode === 73) { // I
                bIDown = false;
            }
        }
        property real prevX: 0.0
        property real prevY: 0.0
        property bool bMousePressHandled: false

        onMouseMove: {
            if (mousePressed) {
                rocket.markUpdateFrequent();
                if (!bMousePressHandled) {
                    bMousePressHandled = true;
                    activateCamOverride();
                    camPosOrig.x = camPos.x;
                    camPosOrig.y = camPos.y;
                    camPosOrig.z = camPos.z;
                    camLookDirOrig = camLookDir;
                    if (!keepCamUp) {
                        camUpOrig = camUp;
                    }
                    camRotAxis = camLookDir.crossProduct(camUp);
                    camRotAxis = camRotAxis.normalized();
                    prevX = mousePressX;
                    prevY = mousePressY;
                }


                var dx = mouseX-prevX;
                var dy = mouseY-prevY;

                prevX = mouseX;
                prevY = mouseY;


                camRotAxis = rotateAxis(camRotAxis, camUp, -dx*rotSpeed);
                camLookDir = rotateAxis(camLookDir, camUp, -dx*rotSpeed);

                if (!keepCamUp) {
                    camUp = rotateAxis(camUp, camRotAxis, dy*rotSpeed);
                }
                camLookDir = rotateAxis(camLookDir, camRotAxis, dy*rotSpeed);

                var camDistFromOrigo = camPos.length();

                camLookAt = camPos.plus(camLookDir.times(camDistFromOrigo));
            }
        }

        onMousePress: {
            rocket.markUpdateFrequent();
            mousePressX = mouseX;
            mousePressY = mouseY;
            mousePressed = true;
            bMousePressHandled = false;
            console.log("mouse pressed at:"+mouseX, mouseY);
        }
        onMouseRelease: {
            mousePressed = false;
            console.log("mouse released");
        }
    }

    function getCamPos() {
        if (hasCamOverride) {
            return camPos;
        }
        return Qt.vector3d(sync("camera.x")+camOfs.x, sync("camera.y")+camOfs.y, sync("camera.z")+camOfs.x);
    }
    function getCamPosSmooth() {
        if (hasCamOverride) {
            if (!smoothInited) {
                camPosSmooth = camPos;
                camLookAtSmooth = camLookAt;
                camUpSmooth = camUp;
                smoothInited = true;
            }
            return camPosSmooth;
        }
        return getCamPos();
    }


    function getCamLookAt() {
        if (hasCamOverride) {
            return camLookAt;
        }
        return Qt.vector3d(sync("camera.lookAtX"), sync("camera.lookAtY"), sync("camera.lookAtZ"));
    }
    function getCamLookAtSmooth() {
        if (hasCamOverride) {
            return camLookAtSmooth;
        }
        return getCamLookAt();
    }

    function getCamUp() {
        if (hasCamOverride) {
            return camUp;
        }
        return Qt.vector3d(sync("camera.upX"), sync("camera.upY"), sync("camera.upZ"));
    }
    function getCamUpSmooth() {
        if (hasCamOverride) {
            return camUpSmooth;
        }
        return getCamUp();
    }

    Camera {
        name: "camera"
        property vector3d pos: getCamPosSmooth()
        property vector3d lookAt: getCamLookAtSmooth()
        property vector3d up: getCamUpSmooth()
        x: pos.x; y: pos.y; z: pos.z
        lookAtX: lookAt.x; lookAtY: lookAt.y; lookAtZ: lookAt.z
        upX: up.x; upY: up.y; upZ: up.z
    }
}
