package main

import (
	"encoding/json"
	"io"
	"io/ioutil"
	"log"
	"os"
	"path"
	"regexp"
	"strings"
)

type Config struct {
	QMLSearchPath     string
	QMLResourceRegexp string
	BasePaths         []string
	InitialFiles      []string
	TargetPath        string
}

func fileList(searchPath string, rx *regexp.Regexp) ([]string, error) {
	files, err := ioutil.ReadDir(searchPath)
	if err != nil {
		return nil, err
	}

	var list []string
	for _, fileinfo := range files {
		fullPath := path.Join(searchPath, fileinfo.Name())

		if fileinfo.IsDir() {
			foundFiles, _ := fileList(fullPath, rx)
			list = append(list, foundFiles...)
		} else {
			if path.Ext(fileinfo.Name()) == ".qml" {
				stripped := filterComments(fullPath)
				log.Printf("Reading %s", fileinfo.Name())
				list = append(list, gatherResources(searchPath, stripped, rx)...)
			}
		}
	}

	return list, nil
}

var commentFilter = regexp.MustCompile("(?s)//.*?\n|/\\*.*?\\*/")

func filterComments(filepath string) []byte {
	bs, _ := ioutil.ReadFile(filepath)
	return commentFilter.ReplaceAll(bs, nil)
}

func gatherResources(currentPath string, qmlSource []byte, rx *regexp.Regexp) []string {
	resources := rx.FindAll(qmlSource, -1)
	var reslist []string

	for _, res := range resources {
		str := strings.Replace(string(res), "\"", "", -1)
		cleaned := path.Clean(path.Join(currentPath, str))

		var filename string
		if strings.HasPrefix(str, "..") {
			filename = cleaned
		} else {
			filename = str
		}

		log.Printf("\tFound %s\n", filename)
		reslist = append(reslist, filename)
	}

	return reslist
}

func findFile(basePaths []string, filename string) (string, error) {
	file, err := os.Open(filename)
	if err == nil {
		file.Close()
		return filename, nil
	}

	for _, basePath := range basePaths {
		testPath := path.Join(basePath, filename)
		file, err := os.Open(testPath)
		if err == nil {
			file.Close()
			return testPath, nil
		}
	}

	log.Printf("Failed to read file %s", filename)
	return "", os.ErrNotExist
}

func copyFiles(basePaths []string, list []string, targetPath string) {
	for _, str := range list {
		filename, err := findFile(basePaths, str)

		if err == nil {
			err = os.MkdirAll(path.Join(targetPath, path.Dir(str)), 0755)
			if err != nil {
				log.Printf("Failed to mkdir %s\n", str)
			} else {
				copyFile(filename, path.Join(targetPath, filename))
			}
		}
	}
}

func copyFile(infile, outfile string) (written int64, err error) {
	out, err := os.Create(outfile)
	if err != nil {
		return
	}
	defer out.Close()

	in, err := os.Open(infile)
	if err != nil {
		return
	}
	defer in.Close()

	return io.Copy(out, in)
}

func main() {
	cfgdata, err := ioutil.ReadFile("fsconfig.json")
	if err != nil {
		log.Fatalf("Failed to find fsconfig.json")
	}

	var cfg Config
	if err = json.Unmarshal(cfgdata, &cfg); err != nil {
		log.Fatalf("Failed to parse config file %v", err)
	}

	resFilter := regexp.MustCompile(cfg.QMLResourceRegexp)

	list, err := fileList(cfg.QMLSearchPath, resFilter)
	if err != nil {
		log.Fatalf("Failed to read dir %v", err)
	}

	list = append(list, cfg.InitialFiles...)
	copyFiles(cfg.BasePaths, list, cfg.TargetPath)
}
