import processing.core.*; 
import processing.data.*; 
import processing.event.*; 
import processing.opengl.*; 

import ddf.minim.*; 
import moonlander.library.*; 
import ddf.minim.analysis.*; 
import java.util.ArrayDeque; 

import java.util.HashMap; 
import java.util.ArrayList; 
import java.io.File; 
import java.io.BufferedReader; 
import java.io.PrintWriter; 
import java.io.InputStream; 
import java.io.OutputStream; 
import java.io.IOException; 

public class Graffathon extends PApplet {




Moonlander ml;
TerrainManager terrain;
ShipRowManager shipRows;
Sky sky;

AudioController ac;

CameraController cam;

City c = new City(0.45f, 5);
Ship s;
ArrayList<PVector> shipRoute = new ArrayList();

private Scene currentScene;

public void settings() {
  fullScreen(P3D);
  //size(1280, 720, P3D);
}

public void setup()
{
  frameRate(60);
  
  colorMode(RGB, 255);
  sky = new Sky();
  
  ml = new Moonlander(this, new TimeController(10));
  ml.start();
  
  ac = new AudioController(new Minim(this));
  
  cam = new CameraController(new PVector(320.0f, -240.0f, 640.0f), new PVector(0.0f, 0.0f, 0.0f), ml);
  s = new Ship(new PVector(320.0f, -120.0f, 320.0f), ml);
  
  terrain = new TerrainManager();
  cam.setViewTarget(s.pos);

  shipRows = new ShipRowManager();
  currentScene = new StartScene(cam, ac, s, c, terrain, shipRows);
  
  int SEED = 0;
  randomSeed(SEED);
}

public void draw()
{
  ml.update();
  ac.update();
  
  background(0);
  
  currentScene = currentScene.update();
  cam.update();

  directionalLight(153, 192, 255, 0.5f, 1, 0.5f);
  
  float f = (ac.getBassIntensity() > 0.3f ? pow(ac.getBassIntensity(), 0.5f) : 0.1f); // * ac.getBassIntensity();
  float ambient = f * 255;
  ambientLight(ambient, ambient, ambient);

  float skyIntensity = ac.getBassIntensity() > 0.3f ? constrain((ac.getBassIntensity() - 0.3f) / 0.7f, 0, 1)  : 0;
  sky.draw(cam.pos, skyIntensity);

  terrain.draw(cam);
  c.draw(cam.pos);
  s.draw();
  shipRows.draw();
}


class AudioController {
  
  Minim minim;
  public AudioPlayer ap;
  FFT fft;
  
  private float bass, mid, treble;
  
  AudioController(Minim minim) {
    this.minim = minim;
    this.ap = minim.loadFile("Exit the Premises.mp3");
    this.ap.play();
    this.fft = new FFT(ap.bufferSize(), ap.sampleRate());
  }
  
  public void update() {
    
    
    bass = mid = treble = 0;
    int thirdOfBand = (fft.specSize() - 120) / 3;
    
    fft.forward(ap.mix);
    for (int i = 20; i < fft.specSize() - 100; i++) {
      if (i < thirdOfBand) {
        bass += fft.getBand(i);
      } else if (i - thirdOfBand < thirdOfBand) {
        mid += fft.getBand(i);
      } else {
        treble += fft.getBand(i);
      }
    }
    bass =  (bass / thirdOfBand) / 8;
    mid = (mid / thirdOfBand) / 3.75f;
    treble = (treble / thirdOfBand) / 2.85f;
  }
  
  public float getBassIntensity() {
    return bass;
  }
  
  public float getMidIntensity() {
    return mid;
  }
  
  public float getTrebleIntensity() {
    return treble;
  }
}
class CameraController
{
  public PVector pos;
  private PVector lookAt;
  
  private MovementTracker movement;
  private Moonlander ml;
 
  private ArrayList<PVector> waypoints;
  private PVector viewTarget;
   
  public CameraController(Moonlander _ml)
  {
    pos = new PVector(0.0f, 0.0f, 0.0f);
    lookAt = new PVector(0.0f, 0.0f, 0.0f);
    movement = new MovementTracker(pos, lookAt);
    ml = _ml;
    waypoints = new ArrayList<PVector>();
    movement.route = waypoints;
  }
  
  public CameraController(PVector _pos, PVector _lookAt, Moonlander _ml)
  {
    pos = _pos;
    lookAt = _lookAt;
    movement = new MovementTracker(pos, lookAt);
    ml = _ml;
    waypoints = new ArrayList<PVector>();
    movement.route = waypoints;
  }
  
  public void addWayPoints(ArrayList<PVector> p)
  {
    if (movement.route == null)
    {
      movement.route = waypoints;
    }
    waypoints.addAll(p);
  }
  
  public void setViewTarget(PVector _target)
  {
    viewTarget = _target;
    movement.fixedTarget = viewTarget;
  }
  
  public void update()
  {    
    float time = (float)ml.getValue("camera");
    movement.update(time);
    
    camera(pos.x, pos.y, pos.z, lookAt.x, lookAt.y, lookAt.z, 0, 1, 0);
  }
}
class City
{
  private final static float w = 25;
  
  private final static float h_avg = 150;
  private final static float h_stdev = 100;
  
  private final static int drawAmount = 35;
  
  private float density;
  private int blockSize;
  
  private boolean active = false;
  
  private HashMap<Integer, HashMap<Integer, House>> houses;
  
  City(float _density, int _block)
  {
    density = _density;
    blockSize = _block;
    houses = new HashMap<Integer, HashMap<Integer, House>>();
  }
  
  public void draw(PVector pos)
  {
    float _x = pos.x;
    float _z = pos.z;
    
    genHouses(_x, _z);
    
    int x_index = floor((float)_x / (w/density));
    int z_index = floor((float)_z / (w/density));
    x_index -= floor(x_index/(blockSize+1)) * (x_index < 0 ? -1 : 1);
    z_index -= floor(z_index/(blockSize+1)) * (x_index < 0 ? -1 : 1);
    
    for (int x = x_index - drawAmount; x < x_index + drawAmount; x++)
    {
      pushMatrix();
      translate((x + floor(x/(blockSize + 1))) * (w/density), 0.0f, 0.0f);
      for (int z = z_index - drawAmount; z < z_index + drawAmount; z++)
      {
        pushMatrix();
        translate(0.0f, 0.0f, (z + floor(z/(blockSize + 1))) * (w/density));
        houses.get(x).get(z).update_height(active);
        houses.get(x).get(z).draw();
        popMatrix();
      }
      popMatrix();
    }
  }
  
  public float getBlockDist(int n)
  {
    return w/density * ((blockSize+1) * n - 0.5f) + w;
  }
  
  public void setActive(boolean value)
  {
    this.active = value;
  }
  
  public void genHouses(float _x, float _z)
  {
    int x_index = floor(_x / (w/density));
    int z_index = floor(_z / (w/density));
    x_index -= floor(x_index/(blockSize+1)) * (x_index < 0 ? -1 : 1);
    z_index -= floor(z_index/(blockSize+1)) * (x_index < 0 ? -1 : 1);
    
    for (int x = x_index - drawAmount; x < x_index + drawAmount; x++)
    {
      if (houses.get(x) == null)
      {
        houses.put(x, new HashMap<Integer, House>());
      }
      for (int z = z_index - drawAmount; z < z_index + drawAmount; z++)
      {
        if (houses.get(x).get(z) == null)
        {
          float h = abs(randomGaussian() * h_stdev + h_avg);
          houses.get(x).put(z, new House(w, w, h));
        }
      }
    }
  }
}

class House
{
  private float w, d, h;
  private float current_h = 0;
  
  private final static float growSpeed = 5.0f;
  private float prevMillis = 0;
  
  House(float _w, float _d, float _h)
  {
    w = _w;
    d = _d;
    h = _h;
    prevMillis = millis();
  }
  public void draw()
  {
    if (abs(current_h) < 0.01f)
    {
      return;
    }
    pushMatrix();
    translate(0.0f, -current_h/2.0f, 0.0f);
    fill(color(24, 24, 64));
    stroke(128, 64, 64);
    shapeMode(CORNER);
    box(w,current_h,d);
    popMatrix();
  }
  public void update_height(boolean active)
  {
    if (this.current_h >= this.h)
    {
      return;
    }
    
    float coeff = active ? 1 : -1;
    current_h += coeff * growSpeed * (millis()-prevMillis)/1000.0f;
    current_h = constrain(current_h, 0, h);
  }
}
class CityScene extends Scene {
  CityScene(CameraController camera, AudioController audio, Ship ship, City city, TerrainManager terrain) {
    super(camera, audio, ship, city, terrain);
    
      city.setActive(true);
    ArrayList<PVector> route = new ArrayList();
    float radius = 1200;
    
    route.add(new PVector(0.0f, -1200.0f, radius));
    route.add(new PVector(radius/4, -1150.0f, 3*radius/4));
    route.add(new PVector(radius/2, -1100.0f, radius/2));
    route.add(new PVector(3*radius/4, -1050.0f, radius/4));
    
    route.add(new PVector(radius, -1000.0f, 0));
    route.add(new PVector(3*radius/4, -950.0f, -radius/4));
    route.add(new PVector(radius/2, -900.0f, -radius/2));
    route.add(new PVector(radius/4, -850.0f, -3*radius/4));
    
    route.add(new PVector(0.0f, -800.0f, -radius));
    route.add(new PVector(-radius/4, -750.0f, -3*radius/4));
    route.add(new PVector(-radius/2, -700.0f, -radius/2));
    route.add(new PVector(-3*radius/4, -650.0f, -radius/4));
    
    route.add(new PVector(-radius, -600.0f, 0));
    route.add(new PVector(-3*radius/4, -550.0f, radius/4));
    route.add(new PVector(-radius/2, -500.0f, radius/2));
    route.add(new PVector(-radius/4, -450.0f, 3*radius/4));
    
    route.add(new PVector(0.0f, -400.0f, radius)); // 16
    route.add(new PVector(100, -400, 600));
    route.add(new PVector(400.0f, -400, -1200));
    
    route.add(new PVector(1200, -500, 15)); // 19
    route.add(new PVector(0, -500, 15));
    route.add(new PVector(320, -400, 320)); // 21
    route.add(new PVector(-320, -100, -320));
    route.add(new PVector(0, -1200, 0));
    
    this.camera.addWayPoints(route);
    camera.setViewTarget(new PVector(0, 50, 0));
  
    ArrayList<PVector> shipRoute = new ArrayList();
    shipRoute.add(new PVector(320, -400, 320));
    this.ship.addWayPoints(shipRoute);
  }
  
  public Scene update() {
    if (this.camera.movement.isVisited(16)) {
      camera.setViewTarget(new PVector(100, -200, 0));
    }
    
    /*
    if (this.camera.movement.isVisited(18)) {
      camera.setViewTarget(new PVector(400, -350, -10000));
    }*/
    
    if (this.camera.movement.isVisited(18)) {
      camera.setViewTarget(new PVector(1200, -350, 0));
    }
    
    if (this.camera.movement.isVisited(19)) {
      camera.setViewTarget(new PVector(-1000, -400, 15));
      //camera.setViewTarget(new PVector(0,0,0));
    }
      
    if (this.camera.movement.allVisited())
    {
      return new FollowScene(camera, audio, ship, city, terrain);
    }
    else
    {
      return this;
    }
  }
}
class FollowScene extends Scene
{

  FollowScene(CameraController camera, AudioController audio, Ship ship, City city, TerrainManager terrain)
  {
    super(camera, audio, ship, city, terrain);
    
    terrain.setMountainsStatic(true);
    
    this.camera.setViewTarget(this.ship.pos);
    //this.ship.direction = new PVector(305.0, -400.0, 6150.0);
    
    ArrayList<PVector> route = new ArrayList();
    route.add(new PVector(320.0f, -500.0f, 640.0f));
    
    route.add(new PVector(320.0f, -500.0f, 3000.0f));
    route.add(new PVector(640.0f,
                          -500.0f,
                          5900));
    route.add(new PVector(320,
                          -500.0f,
                          15100));
    route.add(new PVector(5100,
                          -500.0f,
                          15100));
    route.add(new PVector(3000,
                          -1000.0f,
                          13000));
    this.camera.addWayPoints(route);
  
    ArrayList<PVector> shipRoute = new ArrayList();
    shipRoute.add(new PVector(320.0f, -400.0f, 320.0f));
    
    shipRoute.add(new PVector(320.0f, -400.0f, 3000.0f));
    shipRoute.add(new PVector(320,
                              -200.0f,
                              15100));
    shipRoute.add(new PVector(5200,
                              -200.0f,
                              15100));
    shipRoute.add(new PVector(8000,
                              -1000.0f,
                              18050));
    this.ship.addWayPoints(shipRoute);
    
    city.setActive(true);
  }
  
  public Scene update()
  {
    if (this.ship.movement.allVisited())
    {
      float endTime = (float)ml.getValue("time");
      println(endTime);
      camera();
      fill(230, 230, 230);
        
      if (endTime >= 0.5f)
      {
        beginShape(QUAD);
        vertex(-100, -100, 0);
        vertex(-100, height+100, 0);
        vertex(width/3.0f, height+100, 0);
        vertex(width/4.0f, -100, 0);
        endShape();
      }
      if (endTime >= 1.0f)
      {
        beginShape(QUAD);
        vertex(width+100, -100, 0);
        vertex(width+100, height+100, 0);
        vertex(width-width/3.0f, height+100, 0);
        vertex(width-width/4.0f, -100, 0);
        endShape();
      }
      
      if (endTime >= 1.5f)
      {
        beginShape(QUAD);
        vertex(width+100, -100, 0);
        vertex(width+100, height+100, 0);
        vertex(-100, height+100, 0);
        vertex(-100, -100, 0);
        endShape();
      }
      if (endTime >= 2.5f)
      {
        fill(0, 0, 0);
        textSize(72);
        text("Heart of Neon", 138, 96);
        text("GNU/Konala", 240, 280);
      }
      if (endTime >= 3.5f)
      {
        textSize(48);
        text("impliedfeline", width-360, height-64);
      }
      if (endTime >= 3.7f)
      {
        text("ruuben", width-360, height-192);
      }
      if (endTime >= 3.9f)
      {
        text("flai", width-360, height-320);
      }
      if (endTime >= 5.0f)
      {
        text("Music: Kevin MacLeod - Exit the Premises", 22, height-96);
      }
      if (endTime >= 10.0f)
      {
        exit();
      }
      
      ac.ap.setGain(-32 * endTime / 10);
      
      return this;
    }
    else
    {
      return this;
    }
  }
}
class LineEffect {
  
  final int maxX = 1000;
  final int maxY = -3000;
  final int maxZ = 1000;
  ArrayList<PVector> lines;
  float millisCounter = 0;
  float lastTime = 0;
  
  LineEffect() {
    lines = new ArrayList<PVector>();
  }
  
  public void update() {
    float elapsedTime = millis() - lastTime;
    lastTime = millis();
    if (millisCounter > 0) {
      millisCounter -= elapsedTime; 
      return;
    }
    millisCounter = (60 / 128) * 1000;
    lines.clear();
    //randomSeed(0);
    for (int i = 0; i < 50; i++) {
      lines.add(new PVector(random(-maxX, maxX), 0, random(-maxZ, maxZ)));
    }
  }
  
  public void draw() {
    //randomSeed(0);
    for(PVector line : lines) {
      int colour = (int)random(0, 3);
      if (colour == 0) { stroke(255, 0, 0); }
      else if (colour == 1) { stroke(0, 255, 0); }
      else { stroke(0, 0, 255); }
      
      float xOffSet = random(-1, 1);
      float zOffSet = random(-1, 1);
      strokeWeight(4);
      line(line.x + xOffSet * maxX / 2, 0, line.z + zOffSet * maxZ / 2,
           line.x - xOffSet * maxX / 2, maxY, line.z - zOffSet * maxZ / 2);
      strokeWeight(1);
    }
  }
}


class MovementTracker
{
  private PVector pos;
  private PVector lookAt;
  
  private final static int prevSize = 15;
  private ArrayDeque<PVector> prevPos;
  
  public int lastVisited = -1;
  
  private final static float steeringPower = 0.02f;
  public float maxVel = 5;
  
  public ArrayList<PVector> route;
  public PVector fixedTarget;
  
  public MovementTracker()
  {
    pos = new PVector(0.0f, 0.0f, 0.0f);
    prevPos = new ArrayDeque<PVector>(prevSize);
    prevPos.addLast(new PVector(0.0f, 0.0f, 0.0f));
    lookAt = new PVector(0.0f, 0.0f, 0.0f);
  }
  
  public MovementTracker(PVector _pos, PVector _lookAt)
  {
    pos = _pos;
    prevPos = new ArrayDeque<PVector>(prevSize);
    prevPos.add(pos.copy());
    lookAt = _lookAt;
  }
  
  public boolean isVisited(int index)
  {
    return index <= lastVisited;
  }
  
  public boolean allVisited()
  {
    return route.size() - 1 <= lastVisited;
  }
  
  public void update(float lerpValue)
  {
    int next = ceil(lerpValue);
    int prev = floor(lerpValue);
    lastVisited = prev;
    
    if (route.size() <= next)
    {
      return;
    }
    
    PVector goal = route.get(next);
    PVector start = route.get(prev);
    PVector heading = PVector.lerp(start, goal, lerpValue - prev);
    
    PVector vel = pos.copy().sub(prevPos.peek()).div(prevPos.size());
    PVector direction = goal.copy().sub(pos);
    float deg = PVector.angleBetween(direction,goal);
    PVector steering = direction.copy().limit(steeringPower).mult(abs(deg/PI));
    PVector newVel = vel.copy().add(steering).limit(maxVel);
    
    PVector steered = pos.copy().add(newVel);
    
    PVector newPos = PVector.lerp(steered, heading, lerpValue - prev);
    
    if (prevPos.size() >= 15)
    {
      prevPos.removeFirst();
    }
    
    prevPos.add(pos.copy());
    pos.set(newPos);
    
    if (fixedTarget == null)
    {
      PVector lookDirection = pos.copy().add(vel);    
      lookAt.set(lookDirection);
    }
    else
    {
      lookAt.set(fixedTarget);
    }
  }
}
abstract class Scene
{
  protected CameraController camera;
  protected AudioController audio;
  
  protected Ship ship;
  protected City city;
  protected TerrainManager terrain;
  
  Scene(CameraController camera, AudioController audio, Ship ship, City city, TerrainManager terrain)
  {
    this.camera = camera;
    this.audio = audio;
    this.ship = ship;
    this.city = city;
    this.terrain = terrain;
  }
  public abstract Scene update();
}
class Ship
{
  private PShape model;
  
  public PVector pos;
  public PVector direction;
  
  public MovementTracker movement;
  private Moonlander ml;
  
  private ArrayList<PVector> waypoints;
  
  Ship(PVector _pos, Moonlander _ml)
  {
    pos = _pos;
    direction = new PVector(0.0f, 0.0f, 0.0f);
    movement = new MovementTracker(pos, direction);
    model = loadShape("ship.obj");
    ml = _ml;
    waypoints = new ArrayList<PVector>();
    movement.route = waypoints;
  }
  
  public void addWayPoints(ArrayList<PVector> p)
  {
    if (movement.route == null)
    {
      movement.route = waypoints;
    }
    waypoints.addAll(p);
  }
  
  public void draw()
  {
    float time = (float)ml.getValue("ship");
    movement.update(time);
    
    pushMatrix();
    translate(pos.x, pos.y, pos.z);
    scale(0.75f);
    PVector diff = direction.copy().sub(pos);
    float rotY = atan2(diff.x, diff.z);
    rotateY(rotY);
    shape(model);
    popMatrix();
  }  
}

class DumbShip {
  public final PVector size;
  public final float speedMultiplier;
  public final PVector offset;
  
  public DumbShip(PVector size, float speedMultiplier, PVector offset) {
     this.size = size;
     this.speedMultiplier = speedMultiplier;
     this.offset = offset;
  }
  
  public void draw(PVector pos, float rotationX) {
    pushMatrix();
    translate(pos.x, pos.y, pos.z);
    rotateY(rotationX);
    box(size.x, size.y, size.z);
    popMatrix();
  }
}

class ShipRow {
  public static final float BaseSpeed = 40;
  
  public final PVector from, to;
  public final float dist;
  public final int count;
  
  private ArrayList<DumbShip> ships = new ArrayList<DumbShip>();
  
  public ShipRow(PVector from, PVector to) {
    this.from = from;
    this.to = to;
    
    dist = from.dist(to);
    count = (int)(dist / 10);
    
    this.generateShips();
  }
  
  public void draw() {
    float seconds = ((float)millis() / 1000.0f);
    float speed = BaseSpeed / dist;
    float spacing = 1.0f / count;
   // fill(255);
    fill(24, 24,40);
    stroke(96, 96, 160);
    strokeWeight(1);
    
    PVector diff = to.copy().sub(from);
    float rotationX = atan2(diff.x, diff.z);
    // println(rotationX + " " + to.copy().sub(from));
    for(int i = 0; i < ships.size(); i++) {
     // ship.progress += BaseSpeed * ship.speedMultiplier;
      ships.get(i).draw(
        PVector.lerp(
          from, 
          to, 
          (spacing * i + seconds * speed * ships.get(i).speedMultiplier) % 1).add(ships.get(i).offset), rotationX);
    }
  }  
  
  public void generateShips() {
     for(int i = 0; i < count; i++) {
        float width = random(2.5f, 4);
        float height = random(1.2f, 2);
        float depth = random(5, 10);
        
        PVector offset = new PVector(random(-4, 4), random(-4, 4), 0);
        ships.add(new DumbShip(new PVector(width, height, depth), random(0.5f, 1.5f), offset));
     }
  }
}
class ShipRowManager {
  ArrayList<ShipRow> shipRows = new ArrayList<ShipRow>();
  float y = 0;
  boolean visible = true;
  
  ShipRowManager() {
      shipRows.add(new ShipRow(new PVector(-1550, -360, -40), new PVector(1000, -360, -40)));
      shipRows.add(new ShipRow(new PVector(-40, -400, -1500), new PVector(-40, -400, 1000)));
      shipRows.add(new ShipRow(new PVector(-1000, -380, -1000), new PVector(1000, -380, 1000)));
      
      for (int i = 1; i < 7; i++) {
        int axis = (int)random(-10,5);
        int start = (int)random(-20,20);
        int goal = (int)random(-20,20);
        shipRows.add(new ShipRow(new PVector(c.getBlockDist(axis), -70 * i, c.getBlockDist((start))),
                                 new PVector(c.getBlockDist(axis), -70 * i, c.getBlockDist((goal)))));
        shipRows.add(new ShipRow(new PVector(c.getBlockDist(start), -70 * i, c.getBlockDist(axis)),
                                 new PVector(c.getBlockDist(goal), -70 * i, c.getBlockDist(axis))));
      }
  }
  
  public void setIsVisible(boolean val) {
    if(!val) y = 640;
    visible = val;
    
  }
  
  public void draw() {
    if(visible && y > 0) y -= 1 / (float)frameRate * 600; //(float)millis();
    pushMatrix();
    translate(0, y, 0);
    for(ShipRow shipRow : shipRows) {
      shipRow.draw();
    }
    
    popMatrix();
  }
}
class Sky {
  private PShader skyShader;
  // TODO: sun?
  
  public Sky() {
   skyShader = loadShader("sky-frag.glsl", "sky-vert.glsl");
  }

  public void draw(PVector cameraPos, float intensity) {
    skyShader.set("intensity", intensity);
    shader(skyShader);
    
    pushMatrix();
    translate(cameraPos.x, cameraPos.y, cameraPos.z); // render the sphere to be in center of the camera
    noStroke();
    sphere(5000);
    popMatrix();
    
    resetShader();
  }
}
class StartScene extends Scene
{
  private static final float StartZ = -8000;
  private ShipRowManager shipRows;
  StartScene(CameraController camera, AudioController audio, Ship ship, City city, TerrainManager terrain, ShipRowManager shipRow)
  {
    super(camera, audio, ship, city, terrain);
    shipRows = shipRow;
    
    /* ArrayList<PVector> shipRoute = new ArrayList();
     shipRoute.add(new PVector(0, -120.0, StartZ));
     shipRoute.add(new PVector(0, -120.0, StartZ));
     this.ship.addWayPoints(shipRoute);
     
     ArrayList<PVector> cameraRoute = new ArrayList();
     cameraRoute.add(new PVector(0, -120.0, StartZ));
     cameraRoute.add(new PVector(0, -120.0, StartZ));
     this.camera.addWayPoints(cameraRoute);*/
     
     terrain.setMountainsStatic(false);
     terrain.setRenderingOutside(true);
     shipRows.setIsVisible(false);
     city.setActive(false);
  }
  
  public Scene update()
  {
    float time = constrain((float)ml.getValue("time"), 0, 1);
    
    float y = time < 0.48f ? -520 : lerp(-520, -50, slerp(0, 1, constrain((time - 0.48f) * 4.5f, 0, 1)));
    PVector cameraPos = PVector.lerp(new PVector(30, y, StartZ), new PVector(30, y, 0), slerp(0, 1, time));
    camera.pos.set(cameraPos);
    ship.pos.set(PVector.lerp(new PVector(0, -120, StartZ), new PVector(0, -120, 0), slerp(0, 1, time)));
    camera.lookAt.set(new PVector(0, -120, +10000));
    
    if(time > 0.75f) {
       terrain.setRenderingOutside(false); 
    }
    if(time > 0.93f) {
      shipRows.setIsVisible(true);
    }
    
    if(time > 0.95f) {
      city.setActive(true);
    }
    if(time > 0.999f) {
      terrain.setMountainsStatic(true);
      return new CityScene(camera, audio, ship, city, terrain);
    }
    
    return this;
  }
  
  public PVector slerp(PVector from, PVector to, float t) {
     return new PVector(
       slerp(from.x, to.x, t), 
       slerp(from.y, to.y, t),
       slerp(from.z, to.z, t));
  }
  
  public float slerp(float edge0, float edge1, float x)
  {
    // Scale, bias and saturate x to 0..1 range
    x = clamp((x - edge0)/(edge1 - edge0), 0.0f, 1.0f); 
    // Evaluate polynomial
    return x*x*(3 - 2*x);
  }

public float clamp(float x, float lowerlimit, float upperlimit)
{
    if (x < lowerlimit) x = lowerlimit;
    if (x > upperlimit) x = upperlimit;
    return x;
}
}

class Terrain {
  public static final int GridCellSize = 50;
  
  public final float[] heightMap;
  public final boolean[] isMountain;
  public final int _sizeX, _sizeY;
  public final int hmapSizeX, hmapSizeY;
  public final int hmapOffsetX, hmapOffsetY;
  public final boolean isMountains;
  
  public final ArrayList<PVector> vertices = new ArrayList<PVector>();
  public final PShape shape;
  
  public Terrain(int sizeX, int sizeY, int hmapOffsetX, int hmapOffsetY, boolean isMountains) {
    this._sizeX = sizeX;
    this._sizeY = sizeY;
    
    this.hmapSizeX = sizeX / GridCellSize;
    this.hmapSizeY = sizeY / GridCellSize;
    this.hmapOffsetX = hmapOffsetX;
    this.hmapOffsetY = hmapOffsetY;
    
    this.isMountains = isMountains;
    isMountain = new boolean[(this.hmapSizeX + 1) * (this.hmapSizeY + 1)];
    heightMap = this.generateHeightMap(this.hmapSizeX, this.hmapSizeY);
    
    shape = createShape();
    genVertices();
  }
  public void draw(PVector cameraPos) { this.draw(cameraPos, false); }
  public void draw(PVector cameraPos, boolean forceDrawWithCamera) {
    pushMatrix();
    PVector t = new PVector(cameraPos.x - (cameraPos.x) % (GridCellSize), cameraPos.z - (cameraPos.z ) % (GridCellSize));
    //println(t);
    
    PVector baseOffset = new PVector(-this._sizeX / 2 + (hmapOffsetX / (float)GridCellSize) * _sizeX, -this._sizeY / 2+ (hmapOffsetY / (float)GridCellSize) * _sizeY);
    if(forceDrawWithCamera) { translate(baseOffset.x, 0, baseOffset.y); }
    else if(!this.isMountains) translate(baseOffset.x + t.x, 0, baseOffset.y + t.y);
    else translate(baseOffset.x + cameraPos.x, 0, baseOffset.y + cameraPos.z);

    shape.setFill(color(24, 24, 64));
    shape.setStroke(color(255, 0, 0));
    shape.setStrokeWeight(2);
   /*beginShape(QUADS);
    
    for(int i = 0; i < vertices.size(); i++) {
      PVector v = vertices.get(i);
      vertex(v.x, v.y, v.z);
    }
     endShape(); */
    
    shape(shape);
    /*
    // TODO: one performance improvement would be to calculate all vertices to a
    // single array and then just loop the vertices in here and just call vertex(v)
    for(int hx = 0; hx < this.hmapSizeX; hx++) {
        for(int hy = 0; hy < this.hmapSizeY; hy++) {
          if(this.isMountains != this.isMountain[hx + hy * this.hmapSizeX]) continue;
           float x = hx * GridCellSize;
           float y = hy * GridCellSize;
           
           vertex(x, getZ(hx, hy), y);
           vertex(x + GridCellSize, getZ(hx + 1, hy), y);
           vertex(x + GridCellSize, getZ(hx + 1, hy + 1), y + GridCellSize);
           vertex(x, getZ(hx, hy + 1), y + GridCellSize); 
         }
      }
      */
      popMatrix();
    }
    
    private void genVertices() {
      // TODO: one performance improvement would be to calculate all vertices to a
    // single array and then just loop the vertices in here and just call vertex(v)
     // stroke(255, 0, 0); // red stroke
   // strokeWeight(2); // 2px 
    shape.setFill(color(24, 24, 64));
  //  shape.setStroke(color(255, 0, 0));
    shape.beginShape(QUADS);
    
    for(int hx = 0; hx < this.hmapSizeX; hx++) {
        for(int hy = 0; hy < this.hmapSizeY; hy++) {
          if(this.isMountains != this.isMountain[hx + hy * this.hmapSizeX]) continue;
           float x = hx * GridCellSize;
           float y = hy * GridCellSize;
           
    stroke(255, 0, 0); // red stroke
    strokeWeight(2); // 2px 
           shape.vertex(x, getZ(hx, hy), y);
           shape.vertex(x + GridCellSize, getZ(hx + 1, hy), y);
           shape.vertex(x + GridCellSize, getZ(hx + 1, hy + 1), y + GridCellSize);
           shape.vertex(x, getZ(hx, hy + 1), y + GridCellSize);
           
           vertices.add(new PVector(x, getZ(hx, hy), y));
           vertices.add(new PVector(x + GridCellSize, getZ(hx + 1, hy), y));
           vertices.add(new PVector(x + GridCellSize, getZ(hx + 1, hy + 1), y + GridCellSize));
           vertices.add(new PVector(x, getZ(hx, hy + 1), y + GridCellSize));
         }
      }
      
      shape.endShape();
    }
    
    private float getZ(int x, int y) {
      x = constrain(x, 0, this.hmapSizeX - 1);
      y = constrain(y, 0, this.hmapSizeY - 1);
     
      return this.heightMap[x + y * this.hmapSizeX] + 0.1f;
    }
    
    private final float HeightMapBias = 100;
    private float[] generateHeightMap(int sizeX, int sizeY) {
    //  sizeX++; sizeY++;
      
      float[] hmap = new float[sizeX * sizeY];
      PVector center = new PVector(sizeX, sizeY).div(2);
      for(int y = 0; y < sizeY; y++) {
        for(int x = 0; x < sizeX; x++) {
          PVector cur = new PVector(x, y);
          float distFromCenter = center.dist(cur) / (sizeX / 2);
          if(hmapOffsetX != 0 || hmapOffsetY != 0) {
            distFromCenter = 1;
          }
          
          float multiplier = 10;
          float baseAdd = 0;
          isMountain[x + y * sizeX] = false;
          if(distFromCenter > 0.6f) {
            float lerpVal = (distFromCenter - 0.6f) * 5;
             multiplier = lerp(10, 400,  lerpVal);
             baseAdd += lerp(0, -500, constrain(lerpVal, 0, 1));
             isMountain[x + y * sizeX] = true;
          }
          
          if(hmapOffsetX != 0 || hmapOffsetY != 0) {
            multiplier *= 0.2f;
          }
          
          int testX = x + hmapOffsetX;
          int testY = y + hmapOffsetY;
          hmap[x + y * sizeX] = noise(testX * GridCellSize / HeightMapBias, testY * GridCellSize / HeightMapBias) * multiplier + baseAdd;
        }
      }
      
      return hmap;
    }
    
}
class TerrainManager {
  static final int GridSize = 8000;
  Terrain grid;
  Terrain mountains;
  Terrain outsideMountains;
  
  boolean areMountainsStatic = false;
  boolean renderingOutside = false;
  
  public TerrainManager() {
    noiseSeed(0);
    grid = new Terrain(GridSize, GridSize, 0, 0, false);
    mountains = new Terrain(GridSize, GridSize, 0, 0, true);
    outsideMountains = new Terrain(GridSize, GridSize, 0, -Terrain.GridCellSize,  true);
  }
  
  public void setMountainsStatic(boolean value) {
    areMountainsStatic = value;
  }
  
  public void setRenderingOutside(boolean val) {
    renderingOutside = val;
  }
  
  public void draw(CameraController camera) {
    boolean forceDrawWithCamera = !areMountainsStatic;
    grid.draw(camera.pos, forceDrawWithCamera);
    mountains.draw(camera.pos, forceDrawWithCamera);
    
    if(renderingOutside)
      outsideMountains.draw(camera.pos, forceDrawWithCamera);
  }
}
  static public void main(String[] passedArgs) {
    String[] appletArgs = new String[] { "Graffathon" };
    if (passedArgs != null) {
      PApplet.main(concat(appletArgs, passedArgs));
    } else {
      PApplet.main(appletArgs);
    }
  }
}
