#version 330 core

// inputz
in vec2 UV;

uniform vec2 iResolution;
uniform float iGlobalTime;
uniform vec2 iMouse; //const vec2 iMouse = vec2(0.5);

// outputz
out vec4 fragColor;

#define MAX_BOUNCES 2
#define SAMPLES 50
float seed;
float gamma = 2.2;
vec3 spos;

float stime;

struct Material
{
    vec3 c;     // diffuse color
    float f0;   // specular color (monochrome)
    float r;
};

// ---8<----------------------------------------------------------------------
// Geometry

#define PI 3.14159265358979
float hash(float x) { return fract(sin(x) * 43758.5453); }

struct Ray
{
    vec3 o;     // origin
    vec3 d;     // direction
};

struct Hit
{
    float t;    // solution to p=o+t*d
    vec3 n;     // normal
    Material m; // material
};
const Hit noHit = Hit(1e10, vec3(0.), Material(vec3(-1.), -1., -1.));

struct Plane
{
    float d;    // solution to dot(n,p)+d=0
    vec3 n;     // normal
    Material m; // material
};

struct Sphere
{
    float r;    // radius
    vec3 p;     // center position
    Material m; // material
};

struct Cone
{
    float cosa; // half cone angle
    float h;    // height
    vec3 c;     // tip position
    vec3 v;     // axis
    Material m; // material
};

Hit intersectPlane(Plane p, Ray r)
{
    float dotnd = dot(p.n, r.d);
    if (dotnd > 0.) return noHit;

    float t = -(dot(r.o, p.n) + p.d) / dotnd;
    return Hit(t, p.n, p.m);
}

Hit intersectSphere(Sphere s, Ray r)
{
    vec3 op = s.p - r.o;
    float b = dot(op, r.d);
    float det = b * b - dot(op, op) + s.r * s.r;
    if (det < 0.) return noHit;

    det = sqrt(det);
    float t = b - det;
    if (t < 0.) t = b + det;
    if (t < 0.) return noHit;

    return Hit(t, (r.o + t*r.d - s.p) / s.r, s.m);
}

vec3 randomVector(float seed)
{
    float r2 = hash(seed);
    float phi = 2. * PI * hash(seed + r2);
    float sina = sqrt(r2);
    float cosa = sqrt(1. - r2);

    return vec3(cos(phi) * sina, cosa, sin(phi) * sina);
}

float rr;
float ff;
Hit intersectScene(Ray r)
{
    /*
    vec3 axis1 = randomVector(floor(stime));
    vec3 axis2 = randomVector(floor(stime+1.));
    vec3 axis = normalize(mix(axis1, axis2, fract(stime)));
    float translation = 4.*abs(2.*fract(stime/8.)-1.) - 2.;
*/
    //float rr = 0.75;
    //float ff = 0.03;
    
    Plane pb  = Plane(5.0, vec3(0,1,0),  Material(vec3(0),     ff, rr));
    Plane pl  = Plane(5.0, vec3(1,0,0),  Material(vec3(1),     ff, rr));
    //Plane pf  = Plane(5.0, vec3(0,0,1),  Material(vec3(0),     ff, rr));
    Plane pu  = Plane(5.0, vec3(0,-1,0), Material(vec3(0),  ff, rr));
    Plane pr  = Plane(5.0, vec3(-1,0,0), Material(vec3(1),     ff, rr));
    //Plane pt  = Plane(5.0, vec3(0,0,-1), Material(vec3(0),     ff, rr));
    
    //Sphere s = Sphere(1., spos, Material(vec3(0.0,0,0), 0.0, 0.1));
    //Cone c = Cone(0.95, 2., vec3(translation, 2., 1.), -axis, Material(vec3(1., 0., 0.), 0.02));

    Hit hit = Hit(1e5, vec3(0.), Material(vec3(-1.), -1., -1.));
    Hit hitb = intersectPlane(pb, r);  if (hitb.m.f0 >= 0. && hitb.t < hit.t) { hit = hitb; }
    Hit hitl = intersectPlane(pl, r);  if (hitl.m.f0 >= 0. && hitl.t < hit.t) { hit = hitl; }
    //Hit hitf = intersectPlane(pf, r);  if (hitf.m.f0 >= 0. && hitf.t < hit.t) { hit = hitf; }
    Hit hitu = intersectPlane(pu, r);  if (hitu.m.f0 >= 0. && hitu.t < hit.t) { hit = hitu; }
    Hit hitr = intersectPlane(pr, r);  if (hitr.m.f0 >= 0. && hitr.t < hit.t) { hit = hitr; }
    //Hit hitt = intersectPlane(pt, r);  if (hitt.m.f0 >= 0. && hitt.t < hit.t) { hit = hitt; }
    
    //Hit hits = intersectSphere(s, r); if (hits.m.f0 >= 0. && hits.t < hit.t) { hit = hits; }
    //Hit hitc = intersectCone(c, r);   if (hitc.m.f0 >= 0. && hitc.t < hit.t) { hit = hitc; }

    return hit;
}

// ---8<----------------------------------------------------------------------
// Light

struct DirectionalLight
{
    vec3 d;     // Direction
    vec3 c;     // Color
};

DirectionalLight sunLight = DirectionalLight(normalize(vec3(0.0000001, 1.0, .0)), vec3(1e3));
vec3 pointLight;
vec3 skyColor(vec3 d)
{
    float transition = pow(smoothstep(0.02, .5, d.y), 0.4);

    vec3 sky = 2e2*mix(vec3(0.52, 0.77, 1), vec3(0.12, 0.43, 1), transition);
    vec3 sun = sunLight.c * pow(abs(dot(d, sunLight.d)), 5000.);
    return sky + sun;
}

float pow5(float x) { return x * x * x * x * x; }
float fresnel(vec3 h, vec3 v, float f0)
{
    return pow5(1. - clamp(dot(h, v), 0., 1.)) * (1. - f0) + f0;
}

float epsilon = 4e-4;
/*
vec3 accountForDirectionalLight(vec3 p, vec3 n, DirectionalLight l)
{
    if (intersectScene(Ray(p + epsilon * l.d, l.d)).m.f0 < 0.)
    {
        return clamp(dot(n, l.d), 0., 1.) * l.c;
    }
    return vec3(0.);
}
*/
vec3 accountPointLight(vec3 p, vec3 n, vec3 l)
{
    vec3 v = normalize(l-p);
    float e = max(0.0, dot(n, v));
    return 10000.0*vec3(e)/max(0.0, pow(distance(l, p), 2.0));
}

float hash1(float p){
    vec3 p3  = fract(vec3(p) * 0.1031);
    p3 += dot(p3, p3.yzx + 19.19);
    return fract((p3.x + p3.y) * p3.z);
}


vec2 hash2(float seed)
{
    return fract(sin(vec2(seed))*vec2(43758.5453123,22578.1459123));
}

vec3 cosw(const vec3 n, float seed)
{
    vec2 rv2 = hash2(seed);
    vec3  uu = normalize( cross( n, vec3(0.0,1.0,1.0) ) );
    vec3  vv = normalize( cross( uu, n ) ); 
    float ra = sqrt(rv2.y);
    float rx = ra*cos(6.2831*rv2.x); 
    float ry = ra*sin(6.2831*rv2.x);
    float rz = sqrt( 1.0-rv2.y );
    vec3  rr = vec3( rx*uu + ry*vv + rz*n );
    return normalize( rr );
}

//vec3 getLight(Hit h)
vec3 getLight(vec3 p)
{
    vec3 ap = vec3(p.y,
                   int(p.z*0.1) ^ int(0.1*p.x+stime*5.0),
                   int(0.3*p.x+stime*5.0) ^ int( 0.01*float(int(p.z)^int(mod(stime,5.0))) *p.z));
    return vec3(pow(min(1.0, 0.1+mod(distance(vec3(0), ap), 1.0)),9999.0))/pow(distance(vec3(0),p),2.0);
}

vec3 radiance(Ray r)
{
    vec3 accum = vec3(0.);
    vec3 filte = vec3(1.);

        for (int i = 0; i <= MAX_BOUNCES; i++)
        {
            Hit hit = intersectScene(r);
            
            vec3 pos = r.o+r.d*hit.t;
            //pos = cross(pos, hit.n.yyx);
            
            float f = fresnel(hit.n, -r.d, hit.m.f0);

            vec3 hitPos = r.o + hit.t * r.d;

            // Diffuse
            vec3 incoming = vec3(1000.0)*getLight(pos);
            //incoming += accountPointLight(hitPos, hit.n, pointLight);

            accum += hit.m.c * incoming;

            // Specular: next bounce
            //filte *= f;
            
            vec3 d = reflect(r.d, hit.n);
            d = mix(d, cosw(d, seed), f);
            r = Ray(r.o + hit.t * r.d + epsilon * d, d);
        }
    //}
    return accum;
}

vec3 Uncharted2ToneMapping(vec3 color)
{
    float A = 0.15;
    float B = 0.50;
    float C = 0.10;
    float D = 0.20;
    float E = 0.02;
    float F = 0.30;
    float W = 11.2;
    float exposure = 0.015;
    color *= exposure;
    color = ((color * (A * color + C * B) + D * E) / (color * (A * color + B) + D * F)) - E / F;
    float white = ((W * (A * W + C * B) + D * E) / (W * (A * W + B) + D * F)) - E / F;
    color /= white;
    color = pow(color, vec3(1. / gamma));
    return color;
}

void main()
{
    vec2 uv = 2. * UV - 1.;
    rr = iMouse.y;
    ff = pow(iMouse.x, 4.);
    
    float o1 = 0.25;
    float o2 = 0.75;
    //vec2 msaa[4];
    //msaa[0] = vec2( o1,  o2);
    //msaa[1] = vec2( o2, -o1);
    //msaa[2] = vec2(-o1, -o2);
    //msaa[3] = vec2(-o2,  o1);

    vec3 color = vec3(0.);
    float shutter = 0.5;
    
    vec3 p0 = vec3(0., 1.0, 4.);
    float rtime = mod(iGlobalTime, 12.3);
    for (int i = 0; i < SAMPLES; ++i)
    {
        seed = hash1(dot(uv.x,uv.y)+rtime+float(i));
        stime = (iGlobalTime + shutter*(1.0/24.0)*seed)*2.0;
        pointLight = 3.9*vec3(sin(stime*0.91), cos(stime*0.77), sin(0.5-stime*1.23));
        spos = 3.0*vec3(sin(stime*1.11), cos(stime*0.87), sin(0.5-stime*1.43));
        
        vec3 p = p0;
        vec3 offset = vec3(2.0*hash2(dot(uv.x,uv.y)+rtime+float(i)) / iResolution.y, 0.);
        vec3 d = normalize(vec3(iResolution.x/iResolution.y * uv.x, uv.y, -1.5) + offset);
        Ray r = Ray(p, d);
        color += radiance(r);
    }
    
    color /= float(SAMPLES);
    color *= color;
    fragColor = max(vec4(0), vec4(Uncharted2ToneMapping(color),1.0));
}
