#version 330 core

in vec2 UV;
out vec4 fragColor;

uniform float iGlobalTime;
#define iTime iGlobalTime
#define iFrame (60.0*iGlobalTime)
uniform vec2 iResolution;

uniform sampler2D iChannel0;
uniform sampler2D iChannel1;

// original created by florian berger (flockaroo) - 2017 ("mud planet")
// modified by noby
// License Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.

#define MAX_RADIUS 400.
#define RotNum 5
const float ang = 2.0*3.1415926535/float(RotNum);
mat2 m = mat2(cos(ang),sin(ang),-sin(ang),cos(ang));
mat2 mh = mat2(cos(ang*0.5),sin(ang*0.5),-sin(ang*0.5),cos(ang*0.5));

vec4 randS(vec2 uv)
{
    return texture(iChannel1,uv)-vec4(0.5);
}

vec2 sph2frag(vec3 p)
{
	float r = min(iResolution.x/4.,MAX_RADIUS);
    vec2 center = vec2(r,r);
    p=normalize(p);
    if(p.z<.0) center.x += 2.*r;
    return center+sqrt(1.-abs(p.z))*normalize(p.xy)*r;
}

vec3 frag2sph(vec2 f)
{
	float r = min(iResolution.x/4.,MAX_RADIUS);
    vec2 center = vec2(r,r);
    float sz=1.0;
    if(f.x>2.*r)  { center.x += 2.*r; sz=-1.; }
    vec2 R = (f-center)/r/sqrt(2.);
    if(dot(R,R)>.51) return vec3(100);
    //R/=sqrt(2.);
    vec3 n = vec3(R.xy,sqrt(1.-dot(R,R))*sz);
    vec3 p = reflect(vec3(0,0,-1.*sz),n);
    //p=p.yzx;
    return p;
}

vec3 frag2sphDiff(vec2 p, vec2 dp)
{
    float eps=.001;
    return (frag2sph(p+normalize(dp)*eps)-frag2sph(p))/eps*length(dp);
}

vec4 inverseQuat(vec4 q)
{
    return vec4(-q.xyz,q.w);
}

vec4 multQuat(vec4 a, vec4 b)
{
    return vec4(cross(a.xyz,b.xyz) + a.xyz*b.w + b.xyz*a.w, a.w*b.w - dot(a.xyz,b.xyz));
}

vec4 rotateQuatbyAngle(vec4 quat, vec3 angle)
{
    float angleScalar=length(angle);
    if (angleScalar<0.00001) return quat;
    return multQuat(quat,vec4(angle*(sin(angleScalar*0.5)/angleScalar),cos(angleScalar*0.5)));
}

vec3 transformVecByQuat( vec3 v, vec4 q )
{
    return v + 2.0 * cross( q.xyz, cross( q.xyz, v ) + q.w*v );
}

vec4 transformVecByQuat( vec4 v, vec4 q )
{
    return vec4( transformVecByQuat( v.xyz, q ), v.w );
}

vec3 rotAx(vec3 p, vec3 ang)
{
	return transformVecByQuat( p, rotateQuatbyAngle(vec4(0,0,0,1), ang) );
}

vec3 getRot(vec3 pos, vec3 b)
{
    vec3 n=normalize(pos);
    vec3 p = normalize(b-dot(b,n)*n);
    vec3 rot=vec3(0);
    for(int i=0;i<RotNum;i++)
    {
        vec2 fr = sph2frag(pos+p);
        vec2 v2d = texture(iChannel0,fr/iResolution.xy).xy-vec2(0.5);
        vec3 v = texture(iChannel0,fr/iResolution.xy).xyz-.5;
        //vec3 v = frag2sphDiff(fr,v2d);
        rot+=cross(v,p);
        p = rotAx(p,n*ang);
    }
    return rot/float(RotNum)/dot(b,b);
}

vec3 getSun()
{
    float phi=.2*iGlobalTime;
    float phi2=.7*cos(phi);
    vec3 sun=vec3(vec2(cos(phi),sin(phi))*cos(phi2),sin(phi2));
    return sun;
}

void main()
{
    vec3 pos = frag2sph( floor(UV * iResolution) );
    if(pos.x>1.0) discard;
    vec3 n = normalize(pos);
    
    vec3 b = vec3(1);
    b=normalize(b-dot(b,n)*n)*0.1;
    
    vec3 v=vec3(0);
    float bbMax=5.; bbMax*=bbMax;
    for(int l=0;l<6;l++)
    {
        if ( dot(b,b) > bbMax ) break;
        vec3 p = b;
        for(int i=0;i<RotNum;i++)
        {
            float str=.08;
            str*=pow(dot(b,b),1.0)*1.25;
            v+=cross(getRot(pos+p,b),p)*str;
            p = rotAx(p,n*ang);
        }
        b*=2.;
    }
    
    fragColor=texture(iChannel0,sph2frag(pos+v)/iResolution.xy);
    fragColor.xyz=mix(fragColor.xyz,v*5.0+.5,.01);
    vec3 sun=getSun();
    vec3 vel = mix(vec3(0), 8.0*vec3(sun.yx*vec2(1,-1),0)*.5, pow(0.5+0.5*sin(iGlobalTime), 250.0));
    fragColor.xyz = mix(fragColor.xyz,vel+.5,0.01/(dot(pos-sun,pos-sun)/.2+.2));
}