import ctypes
import math


class Shader(object):
    def __init__(self, max_time_step, width, height, color_depth=256, scale=1):
        self.__timer = 0.0
        self.__max_time_step = max_time_step
        self.__color_depth = color_depth
        self._offset = 0, 0
        self._width = width
        self._height = height
        self._scale = scale

    def set_position(self, x, y):
        self._offset = x, y

    def pre_sample(self, alive):
        """
        Called once per frame

        :type alive: float, Local time this effect is running
        :type width: int, Number of horizontal pixels
        :type height: int, Number of vertical pixels
        :rtype: tuple, The returned arguments are the same for each pixel and passed as extra *args to sample()
        """
        return tuple()

    def sample(self, alive, u, v, *args):
        """
        Called for each pixel every frame

        :type alive: float, Local time this effect is running
        :type u: float, U coord
        :type v: float, V coord
        :type args: tuple, the return value of pre_sample
        :rtype: (float, float, float), the resulting color
        """
        raise NotImplementedError()

    def render(self, alive, delta_time, buffer):
        self.__timer += min(delta_time, self.__max_time_step)

        ih = 1.0 / float(self._height)
        iw = 1.0 / float(self._width)
        args = self.pre_sample(self.__timer)

        ou = self._offset[0] * iw if self._offset[0] < 0.0 else 0.0
        ov = self._offset[1] * ih if self._offset[1] < 0.0 else 0.0

        for y in xrange(self._height):
            v = y * ih - ov
            for x in xrange(self._width):
                u = x * iw - ou
                rgb = self.sample(self.__timer, u, v, *args)
                # quantify the colors to improve smart buffer usage
                inv_steps = 1.0 / float(self.__color_depth)

                def round_to_int(value):
                    return int(round(value))

                rgb = (round_to_int(rgb[0] * self.__color_depth) * inv_steps,
                       round_to_int(rgb[1] * self.__color_depth) * inv_steps,
                       round_to_int(rgb[2] * self.__color_depth) * inv_steps)

                buffer.setPixelF(self._offset[0] + x, self._offset[1] + y, *rgb)

        #if not self._scale == 1:
        #    buffer.resizeRectangle(self._offset[0], self._offset[1], self._width, self._height,
        #                           self._offset[0], self._offset[1], int(self._width * self._scale), int(self._height * self._scale))
