import ctypes
import os

_drawing_utils = None


def DrawingUtils():  # We should pretend this is a singleton class.
    global _drawing_utils
    if _drawing_utils is None:
        dllPath = os.path.abspath('DrawingUtils/DrawingUtils.dll')
        _drawing_utils = ctypes.cdll.LoadLibrary(dllPath)

        _drawing_utils.sbBufferCreate.argtypes = (ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbBufferCreate.restype = ctypes.c_void_p
        _drawing_utils.sbBufferDelete.argtypes = (ctypes.c_void_p,)
        _drawing_utils.sbBufferFlush.argtypes = (ctypes.c_void_p,)
        _drawing_utils.sbBufferPixels.argtypes = (ctypes.c_void_p,)
        _drawing_utils.sbBufferPixels.restype = ctypes.POINTER(ctypes.c_int)
        _drawing_utils.sbBufferDirties.argtypes = (ctypes.c_void_p,)
        _drawing_utils.sbBufferDirties.restype = ctypes.POINTER(ctypes.c_bool)
        _drawing_utils.sbBufferSetPixel.argtypes = (ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbBufferSetPixelF.argtypes = (ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_float, ctypes.c_float, ctypes.c_float)
        _drawing_utils.sbBufferSetPixelA.argtypes = (ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_float)
        _drawing_utils.sbBufferSetPixelFA.argtypes = (
        ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_float, ctypes.c_float, ctypes.c_float, ctypes.c_float)
        _drawing_utils.sbBrushCreateColor.argtypes = (ctypes.c_int,)
        _drawing_utils.sbBrushCreateColor.restype = ctypes.c_void_p
        _drawing_utils.sbBrushCreateColorF.argtypes = (ctypes.c_float, ctypes.c_float, ctypes.c_float)
        _drawing_utils.sbBrushCreateColorF.restype = ctypes.c_void_p

        _drawing_utils.sbBrushCreateLinearGradient.restype = ctypes.c_void_p
        _drawing_utils.sbBrushCreateLinearGradientF.restype = ctypes.c_void_p
        _drawing_utils.sbBrushCreateCircularGradient.restype = ctypes.c_void_p
        _drawing_utils.sbBrushCreateCircularGradientF.restype = ctypes.c_void_p

        _drawing_utils.sbBrushCreateLinearGradient.argtypes = (ctypes.c_int, ctypes.c_int, ctypes.c_float, ctypes.c_float, ctypes.c_int)
        _drawing_utils.sbBrushCreateLinearGradientF.argtypes = (ctypes.c_float, ctypes.c_float, ctypes.c_float, ctypes.c_float, ctypes.c_float, ctypes.c_float,
                                                                ctypes.c_float, ctypes.c_float, ctypes.c_int)
        _drawing_utils.sbBrushCreateCircularGradient.argtypes = (ctypes.c_int, ctypes.c_int, ctypes.c_float, ctypes.c_float, ctypes.c_float, ctypes.c_int)
        _drawing_utils.sbBrushCreateCircularGradientF.argtypes = (ctypes.c_float, ctypes.c_float, ctypes.c_float, ctypes.c_float, ctypes.c_float,
                                                                  ctypes.c_float, ctypes.c_float, ctypes.c_float, ctypes.c_float, ctypes.c_int)
        _drawing_utils.sbBrushDelete.argtypes = (ctypes.c_void_p,)

        _drawing_utils.sbDrawPointI.argtypes = (ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbDrawLineI.argtypes = (ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbDrawRectangleI.argtypes = (ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbDrawEllipseI.argtypes = (ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbDrawCircleI.argtypes = (ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbFillRectangleI.argtypes = (ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbFillEllipseI.argtypes = (ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbFillCircleI.argtypes = (ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)

        _drawing_utils.sbDrawPoint.argtypes = (ctypes.c_void_p, ctypes.c_void_p, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbDrawLine.argtypes = (ctypes.c_void_p, ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbDrawRectangle.argtypes = (ctypes.c_void_p, ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbDrawEllipse.argtypes = (ctypes.c_void_p, ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbDrawCircle.argtypes = (ctypes.c_void_p, ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbFillRectangle.argtypes = (ctypes.c_void_p, ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbFillEllipse.argtypes = (ctypes.c_void_p, ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbFillCircle.argtypes = (ctypes.c_void_p, ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbDrawImage.argtypes = (ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int,
                                               ctypes.POINTER(ctypes.c_int))

        _drawing_utils.sbDrawTriangleI.argtypes = (ctypes.c_void_p, ctypes.c_int,
                                                   ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbDrawTriangle.argtypes = (ctypes.c_void_p, ctypes.c_void_p, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int,
                                                  ctypes.c_int, ctypes.c_float, ctypes.c_float, ctypes.c_float, ctypes.c_float, ctypes.c_float, ctypes.c_float)

        _drawing_utils.sbDrawSushi.argtypes = (ctypes.c_void_p, ctypes.c_float, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbDrawPlasma.argtypes = (ctypes.c_void_p, ctypes.c_float, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)
        _drawing_utils.sbDrawShaderBall.argtypes = (ctypes.c_void_p, ctypes.c_float, ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)

        _drawing_utils.sbResizeRectangle.argtypes = (ctypes.c_void_p,  # buffer ptr
                                                     ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int,  # source x,y,w,h rect
                                                     ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_int)  # dest x,y,w,h rect

    return _drawing_utils


class WrapMode(object):
    Clamp = 0
    Repeat = 1
    Reflect = 2

    @staticmethod
    def test(wrapMode):
        return wrapMode >= 0 or wrapMode <= 2

    @staticmethod
    def repr(wrapMode):
        if wrapMode == 0:
            return 'Clamp'
        if wrapMode == 1:
            return 'Repeat'
        if wrapMode == 2:
            return 'Reflect'


class Brush(object):
    def brush(self):
        return ctypes.c_void_p(self._brush)

    def __del__(self):
        lib = DrawingUtils()
        lib.sbBrushDelete(self._brush)


class ColorBrush(Brush):
    def __init__(self, r, g=None, b=None):
        super(ColorBrush, self).__init__()
        lib = DrawingUtils()
        if g is None:
            # assume r is a packed integer color
            self._brush = lib.sbBrushCreateColor(r)
            return
        self._brush = lib.sbBrushCreateColorF(r, g, b)


class LinearGradientBrush(Brush):
    def __init__(self, r, g, b, r1, g1, b1=None, inAngle=None, inScale=None, inWrapMode=None):
        super(LinearGradientBrush, self).__init__()
        assert WrapMode.test(inWrapMode), 'WrapMode "%s" out of bounds (0, 2), see static members of WrapMode for valid values.' % inWrapMode
        lib = DrawingUtils()
        if g1 is None:
            # assume r and g are two packed integer colors and all other arguments are shifted
            self._brush = lib.sbBrushCreateLinearGradient(r, g, b, r1, g1)
            return
        self._brush = lib.sbBrushCreateLinearGradientF(r, g, b, r1, g1, b1, inAngle, inScale, inWrapMode)


class CircularGradientBrush(Brush):
    def __init__(self, r, g, b, r1, g1, b1, inCenterX=None, inCenterY=None, inScale=None, inWrapMode=None):
        super(CircularGradientBrush, self).__init__()
        lib = DrawingUtils()
        if inCenterX is None:
            # assume r and g are two packed integer colors and all other arguments are shifted
            self._brush = lib.sbBrushCreateCircularGradient(r, g, b, r1, g1, b1)
            return
        self._brush = lib.sbBrushCreateCircularGradientF(r, g, b, r1, g1, b1, inCenterX, inCenterY, inScale, inWrapMode)


class TextureBrush(Brush):
    def __init__(self, pixels, width, height):
        super(TextureBrush, self).__init__()
        lib = DrawingUtils()
        self._brush = lib.sbBrushCreateTexture(width, height, pixels)


def unpack2rgb(intcol):
    a = intcol >> 24
    b = (intcol >> 16) & 0xff
    g = (intcol >> 8) & 0xff
    r = intcol & 0xff
    f = 1.0 / 255.0
    return r * f, g * f, b * f, a * f


def rgb2packedInt(r, g, b, a=1, short = False ):
    if short:
        return  int('%0.2X%0.2X%0.2X' % (b *255, g *255, r*255 ), 16)
    r = int(r * 255.999)
    g = int(g * 255.999)
    b = int(b * 255.999)
    a = int(a * 255.999)
    #"%0.2x%0.2x%0.2x%0.2x"%(b,g,r,a)
    #print "%0.2x%0.2x%0.2x%0.2x"%(b,g,r,a)
    #return int("%0.2x%0.2x%0.2x%0.2x"%(b,g,r,a))\
    color = 0;
    color |= (a & 255) << 24;
    color |= (b & 255) << 16;
    color |= (g & 255) << 8;
    color |= (r & 255);
    return color


class SmartBuffer(object):
    def __init__(self, width, height, buffer_id):
        lib = DrawingUtils()
        self.__buffer_id = buffer_id
        self.__buffer = lib.sbBufferCreate(width, height)

    def index(self):
        return self.__buffer_id

    def __del__(self):
        lib = DrawingUtils()
        lib.sbBufferDelete(self.__buffer)

    def buffer(self):
        return self.__buffer

    def flush(self):
        lib = DrawingUtils()
        lib.sbBufferFlush(self.__buffer)

    def pixels(self):
        lib = DrawingUtils()
        return lib.sbBufferPixels(self.__buffer)

    def dirties(self):
        lib = DrawingUtils()
        return lib.sbBufferDirties(self.__buffer)

    def setPixel(self, x, y, color, transparency=0):
        """
        :type x: int
        :type y: int
        :type color: int

        :param color: Packed integer color, also see setPixelF or the colorFromFloats() utility function.
        """
        if unpack2rgb(color)[3] == transparency:
            return
        lib = DrawingUtils()
        lib.sbBufferSetPixel(self.__buffer, x, y, color)

    def setPixelF(self, x, y, r, g, b):
        """
        :type r: float
        :type g: float
        :type b: float

        :param r: (0.0, 1.0) range
        :param g: (0.0, 1.0) range
        :param b: (0.0, 1.0) range
        """
        lib = DrawingUtils()
        lib.sbBufferSetPixelF(self.__buffer, x, y, r, g, b)

    def setPixelA(self, x, y, color, a):
        DrawingUtils().sbBufferSetPixelA(self.__buffer, x, y, color, a)

    def setPixelFA(self, x, y, r, g, b, a):
        DrawingUtils().sbBufferSetPixelFA(self.__buffer, x, y, r, g, b, a)

    def getPixel(self, x, y):
        lib = DrawingUtils()
        return lib.sbBufferGetPixel(self.__buffer, x, y)

    def drawPointI(self, colorInt, x, y):
        DrawingUtils().sbDrawPointI(self.__buffer, colorInt, x, y)

    def drawLineI(self, colorInt, x1, y1, x2, y2):
        DrawingUtils().sbDrawLineI(self.__buffer, colorInt, x1, y1, x2, y2)

    def drawRectangleI(self, colorInt, x, y, w, h):
        DrawingUtils().sbDrawRectangleI(self.__buffer, colorInt, x, y, w, h)

    def drawEllipseI(self, colorInt, x, y, w, h):
        DrawingUtils().sbDrawEllipseI(self.__buffer, colorInt, x, y, w, h)

    def drawCircleI(self, colorInt, x, y, r):
        DrawingUtils().sbDrawCircleI(self.__buffer, colorInt, x, y, r)

    def fillRectangleI(self, colorInt, x, y, w, h):
        DrawingUtils().sbFillRectangleI(self.__buffer, colorInt, x, y, w, h)

    def fillEllipseI(self, colorInt, x, y, w, h):
        DrawingUtils().sbFillEllipseI(self.__buffer, colorInt, x, y, w, h)

    def fillCircleI(self, colorInt, x, y, r):
        DrawingUtils().sbFillCircleI(self.__buffer, colorInt, x, y, r)

    def drawPoint(self, brush, x, y):
        DrawingUtils().sbDrawPoint(self.__buffer, brush.brush(), x, y)

    def drawLine(self, brush, x1, y1, x2, y2):
        DrawingUtils().sbDrawLine(self.__buffer, brush.brush(), x1, y1, x2, y2)

    def drawRectangle(self, brush, x, y, w, h):
        DrawingUtils().sbDrawRectangle(self.__buffer, brush.brush(), x, y, w, h)

    def drawEllipse(self, brush, x, y, w, h):
        DrawingUtils().sbDrawEllipse(self.__buffer, brush.brush(), x, y, w, h)

    def drawCircle(self, brush, x, y, r):
        DrawingUtils().sbDrawCircle(self.__buffer, brush.brush(), x, y, r)

    def fillRectangle(self, brush, x, y, w, h):
        DrawingUtils().sbFillRectangle(self.__buffer, brush.brush(), x, y, w, h)

    def fillEllipse(self, brush, x, y, w, h):
        DrawingUtils().sbFillEllipse(self.__buffer, brush.brush(), x, y, w, h)

    def fillCircle(self, brush, x, y, r):
        DrawingUtils().sbFillCircle(self.__buffer, brush.brush(), x, y, r)

    def drawImage(self, x, y, w, h, transparentColorInt, colorIntArray):
        DrawingUtils().sbDrawImage(self.__buffer, x, y, w, h, transparentColorInt, colorIntArray)

    def drawTriangleI(self, colorInt, x1, y1, x2, y2, x3, y3):
        DrawingUtils().sbDrawTriangleI(self.__buffer, colorInt, x1, y1, x2, y2, x3, y3)

    def drawTriangle(self, brush, x1, y1, x2, y2, x3, y3, u1, v1, u2, v2, u3, v3):
        DrawingUtils().sbDrawTriangle(self.__buffer, brush.brush(), x1, y1, x2, y2, x3, y3, u1, v1, u2, v2, u3, v3)

    def resizeRectangle(self, x, y, w, h, outX, outY, outW, outH):
        DrawingUtils().sbResizeRectangle(self.__buffer, x, y, w, h, outX, outY, outW, outH)
