
varying vec3 vNormal;
varying vec3 vWorldPos;
varying vec2 vTexCoord;

struct BaseLight
{
	vec3 AmbientColor;
	vec3 DiffuseColor;
	vec3 SpecularColor;
};

struct Atten
{
	float Constant;
	float Linear;
	float Exp;
};

struct PointLight
{
	BaseLight BaseLight;
	Atten Atten;
	vec3 Position;
};

struct DirLight
{
	BaseLight BaseLight;
	vec3 Direction;
};

uniform float specularPower;

uniform PointLight pointLights[4];
uniform int numPointLights;

uniform DirLight directionalLight;

uniform vec3 colorAmbient;
uniform vec3 colorDiffuse;
uniform vec3 colorSpecular;
uniform vec3 colorEmissive;
uniform vec3 cameraPos;
uniform int hasNormalMap;
uniform sampler2D normalTex;
uniform vec2 normalScale;

uniform int hasDiffuseTex;
uniform sampler2D diffuseTex;
uniform vec2 diffuseTexScale;
uniform float emissivity;

vec3 calcBaseLight(in BaseLight l, in vec3 lightDir, in vec3 normal)
{
	float diffuseFactor = dot(normal, -lightDir);

	vec3 diffuseColor = vec3(0);
	vec3 specularColor = vec3(0);

	if(diffuseFactor > 0)
	{
		diffuseColor = l.DiffuseColor * max(0, diffuseFactor);

		vec3 vertToEye = normalize(cameraPos - vWorldPos);
		vec3 lightReflect = normalize(reflect(lightDir, normal));
		float specularFactor = dot(vertToEye, lightReflect);
		specularFactor = pow(specularFactor, 10);
		if(specularFactor > 0)
		{
			specularColor = l.SpecularColor * specularFactor;
		}
	}

	vec3 diff = colorDiffuse;
	if(hasDiffuseTex == 1)
	{
		diff *= texture2D(diffuseTex, vTexCoord*diffuseTexScale).rgb;
	}

	return l.AmbientColor * colorAmbient + diffuseColor * diff + specularColor * colorSpecular;
}

vec4 calcPointLight(in PointLight l, in vec3 normal)
{
	vec4 color = vec4(0);
	vec3 toLight = vWorldPos - l.Position;
	float dist = length(toLight);
	if(dist > 0)
	{
		toLight /= dist;
		color = vec4(calcBaseLight(l.BaseLight, toLight, normal), 1);
		color /= l.Atten.Constant + l.Atten.Linear * dist + l.Atten.Exp * dist * dist;
	}

	return color;
}

vec4 calcFinalColor(in vec3 normal)
{
	vec4 color = vec4(0);

	for(int i = 0;i < numPointLights;++i)
		color += calcPointLight(pointLights[i], normal);

	return color + vec4(calcBaseLight(directionalLight.BaseLight, directionalLight.Direction, normal), 0);
}

vec3 perturbNormal2Arb( vec3 eye_pos, vec3 surf_norm )
{
	vec3 q0 = dFdx( eye_pos.xyz );
	vec3 q1 = dFdy( eye_pos.xyz );
	vec2 st0 = dFdx( vTexCoord.st );
	vec2 st1 = dFdy( vTexCoord.st );

	vec3 S = normalize( q0 * st1.t - q1 * st0.t );
	vec3 T = normalize( -q0 * st1.s + q1 * st0.s );
	vec3 N = normalize( surf_norm );

	vec3 mapN = texture2D(normalTex, vTexCoord).xyz * 2.0 - 1.0;
	mapN.xy = normalScale * mapN.xy;
	mat3 tsn = mat3( S, T, N );
	return normalize( tsn * mapN );
}

void main()
{
	vec3 normal;
	if(hasNormalMap == 1)
		normal = perturbNormal2Arb(vWorldPos, vNormal);
	else
		normal = normalize(vNormal);

	gl_FragColor = mix(calcFinalColor(normal), vec4(colorDiffuse, 1.0), emissivity);//vec4(greyFactor*4., greyFactor * 2., greyFactor,1.);
}
