#define _USE_MATH_DEFINES
#include <stdio.h>
#include <string>
#include <vector>
#include <Python.h>
#include "structmember.h"

#include "parameters.h"
#include "player.h"
#include "missile.h"
#include "simulator.h"

Simulator *g_simulator = nullptr;

std::vector<Missile *> *  parseMissiles(PyObject* py_missles)
{
    std::vector<Missile *> *missiles = new std::vector<Missile *>();
    Missile::Type missile_type = Missile::Normal;

    PyObject *key, *value, *next;
    Py_ssize_t pos = 0;

    int energy = 0;
    int owner = 0;

    qreal rotation_real = 0.0;
    qreal velocityX = 0.0;
    qreal velocityY = 0.0;
    qreal x = 0.0;
    qreal y = 0.0;

    while (PyDict_Next(py_missles, &pos, &key, &value)) {
        PyObject *iter = PyObject_GetIter(value);
        if (iter)
        {
          for (int i = 1; (next = PyIter_Next(iter)) != NULL; ++i)
          {
            switch (i) {
              case 1:
                owner = (int)PyInt_AsLong(next);
                break;
              case 2:
                energy = (int)PyInt_AsLong(next);
                break;
              case 3:
                rotation_real = PyFloat_AsDouble(next);
                break;
              case 4:
                missile_type = (Missile::Type)PyInt_AsLong(next);
                break;
              case 5:
                velocityX = PyFloat_AsDouble(next);
                break;
              case 6:
                velocityY = PyFloat_AsDouble(next);
                break;
              case 7:
                x = PyFloat_AsDouble(next);
                break;
              case 8:
                y = PyFloat_AsDouble(next);
                break;
            }
          }
          missiles->push_back(new Missile(missile_type, QPointF(x, y), rotation_real, velocityX, velocityY, energy, owner));
        }
    }

    return missiles;
}

std::vector<Player *> * parsePlayers(PyObject* py_enemies) {
    PyObject *key, *value, *next;
    Py_ssize_t pos = 0;

    int energy = 0;
    int id = 0;
    int rotation = 0;

    qreal velocityX = 0.0;
    qreal velocityY = 0.0;
    qreal x = 0.0;
    qreal y = 0.0;

    std::vector<Player *> *enemies = new std::vector<Player *>();

    while (PyDict_Next(py_enemies, &pos, &key, &value)) {
        PyObject *iter = PyObject_GetIter(value);
        if (iter)
        {
          for (int i = 1; (next = PyIter_Next(iter)) != NULL; ++i)
          {
            switch (i) {
              case 1:
                id = (int)PyInt_AsLong(next);
                break;
              case 2:
                energy = (int)PyInt_AsLong(next);
                break;
              case 3:
                rotation = (int)PyInt_AsLong(next);
                break;
              case 4:
                velocityX = PyFloat_AsDouble(next);
                break;
              case 5:
                velocityY = PyFloat_AsDouble(next);
                break;
              case 6:
                x = PyFloat_AsDouble(next);
                break;
              case 7:
                y = PyFloat_AsDouble(next);
                break;
            }
          }
          enemies->push_back(new Player(id, energy, rotation, QPointF(x, y), velocityX, velocityY));
        }
    }

    return enemies;
}

static PyObject * simulateAggressive(PyObject *self, PyObject *args)
{
  int my_id = 0;
  int num_ticks = 20;
  int max_missile_age = 20;
  bool hit[2*19] { 0 };

  Missile::Type my_missile = Missile::Normal;

  if (!PyArg_ParseTuple(args, "ii|ii", &my_id, &my_missile, &num_ticks, &max_missile_age)) {
    return NULL;
  }

  g_simulator->simulateMeShooting(my_id, my_missile, num_ticks, hit, max_missile_age);

  if (hit[0])
  {
    return Py_BuildValue("z", "MISSILE");
  }

  for (int i = 0; i < 19; i++)
  {
    if (hit[LEFT*19+i])
    {
      return Py_BuildValue("z", "LEFT");
    }
    if (hit[RIGHT*19+i])
    {
      return Py_BuildValue("z", "RIGHT");
    }
  }
  return Py_BuildValue("z", NULL);
}

static PyObject * simulateFake(PyObject *self, PyObject *args)
{
  int my_id = 0;
  int num_ticks = 20;

  Missile::Type my_missile = Missile::Normal;

  if (!PyArg_ParseTuple(args, "ii|i", &my_id, &my_missile, &num_ticks)) {
    return NULL;
  }

  bool hit = g_simulator->fakeSimulateMeShooting(my_id, my_missile, num_ticks);

  return Py_BuildValue("b", hit);
}

static PyObject * update(PyObject *self, PyObject *args)
{
  long tick = 0;

  std::vector<Player *> *players;
  std::vector<Missile *> *missiles;

  PyObject *py_players;
  PyObject *py_missles;
  PyObject *round_id;

  if (!PyArg_ParseTuple(args, "O!O!Oi", &PyDict_Type, &py_players, &PyDict_Type, &py_missles, &round_id, &tick)) {
      return NULL;
  }

  players = parsePlayers(py_players);
  missiles = parseMissiles(py_missles);
  const char *round_identifier = PyString_AsString(round_id);

  if (!g_simulator)
    g_simulator = new Simulator(round_identifier);
  g_simulator->update(*players, *missiles, tick);

  delete players;
  delete missiles;
  return Py_BuildValue("z", NULL);
}

static PyObject * endRound(PyObject *self, PyObject *args)
{
  delete g_simulator;
  g_simulator = nullptr;
  return Py_BuildValue("z", NULL);
}

static PyObject * avoid(PyObject *self, PyObject *args)
{
  int my_id = 0;
  int num_ticks = 20;
  if (!PyArg_ParseTuple(args, "i|i", &my_id, &num_ticks)) {
    return NULL;
  }

  std::vector<std::string> commands;
  for(int i = 0; i < 1000; i++){
    commands.push_back("NONE");
  }
  int nothing = g_simulator->simulatePlayerMoves(my_id,commands);
  for(int i = 0; i < 10; i++){
    commands[i] = "ACCELERATE";
    if(g_simulator->simulatePlayerMoves(my_id,commands) > nothing){
      return Py_BuildValue("z", "ACCELERATE");
    }
  }
  return Py_BuildValue("z", "RIGHT");
}

static PyMethodDef Simuli_methods[] = 
{
    { "Update", (PyCFunction) update, METH_VARARGS, "Update simulation with new game state" },
    { "SimulateAggressive", (PyCFunction) simulateAggressive, METH_VARARGS, "Run simulation with aggressive behaviour" },
    { "EndRound", (PyCFunction) endRound, METH_VARARGS, "Signal that the round has come to an end" },
    { "Avoid", (PyCFunction) avoid, METH_VARARGS, "What move should I do to avoid getting shot"},
    { "FakeMissile", (PyCFunction) simulateFake, METH_VARARGS, "Run simulation with faulty aggressive behaviour" },
    { NULL }
};


PyMODINIT_FUNC initsimuli(void)
{
  Py_InitModule("simuli", Simuli_methods);
}
