#pragma once

#include <QtTest>

#include "State.h"

class TestDeserialization : public QObject
{
	Q_OBJECT

	private slots:
		void testDeserializeGameState()
		{
			const auto& gameStateObject = QJsonDocument::fromJson(exampleGameState);
			QVERIFY(gameStateObject.isObject());

			State state;
			state.parse(gameStateObject.object());

			// Check that the amount of objects deserialized is correct
			QCOMPARE(state.ships().count(), 2);
			QCOMPARE(state.missiles().count(), 3);

			// Check each field of one of the missiles
			const auto& missile = state.missiles().last();
			QCOMPARE(missile.id(), 60);
			QCOMPARE(missile.energy(), Energy(350));
			QCOMPARE(missile.ownerId(), 0);
			QCOMPARE(missile.rotation(), Angle::fromDegrees(300));
			QCOMPARE(missile.type(), Missile::Type::Seeking);
			QCOMPARE(missile.velocity(), Velocity(0.032627412854647569, -0.03970474002645958));
			QCOMPARE(missile.position(), Position(-0.086567348713245326, 0.69192650021900126));

			// Check each field of one of the ships (our own)
			const auto& ship = state.myShip();
			QCOMPARE(ship.energy(), Energy(918));
			QCOMPARE(ship.id(), 1);
			QCOMPARE(ship.rotation(), Angle::fromDegrees(270));
			QCOMPARE(ship.velocity(), Velocity(0.0076515503238599285, 0.025595966187665863));
			QCOMPARE(ship.position(), Position(0.43292444925334561, -0.41881639474014176));
		}

		void benchmarkDeserializeGameState()
		{
			const auto& gameStateObject = QJsonDocument::fromJson(massiveGameState);
			QVERIFY(gameStateObject.isObject());

			State state; // Keep state outside loop to re-use allocated memory
			QBENCHMARK {
				state.parse(gameStateObject.object());
			}
		}

	private:
		// Example from https://github.com/sandsmark/aicompo-tg16/blob/master/README.md
		static constexpr auto exampleGameState = R"({"missiles":[{"id":0,"energy":0,"owner":0,"rotation":180,"type":"NORMAL","velocityX":0.002812616921765813,"velocityY":0.048244483355828163,"x":-0.1065170720879374,"y":0.057297497784823674},{"id":50,"energy":4981,"owner":0,"rotation":45,"type":"MINE","velocityX":0.00042820233654171418,"velocityY":-0.0015615237322931463,"x":-0.22436009460522804,"y":0.81817305140153029},{"id":60,"energy":350,"owner":0,"rotation":300,"type":"SEEKING","velocityX":0.032627412854647569,"velocityY":-0.03970474002645958,"x":-0.086567348713245326,"y":0.69192650021900126}],"others":[{"energy":858,"id":0,"rotation":90,"velocityX":-0.007226001197449654,"velocityY":-0.026401470594673109,"x":-0.42907664254625305,"y":0.40928029816990924}],"you":{"energy":918,"id":1,"rotation":270,"velocityX":0.0076515503238599285,"velocityY":0.025595966187665863,"x":0.43292444925334561,"y":-0.41881639474014176}})";

		// Approx. 46kB, 258 missiles, 3 ships
		static constexpr auto massiveGameState = R"({"missiles":[{"energy":0,"id":0,"owner":2,"rotation":274,"type":"NORMAL","velocityX":0.0023678051162366221,"velocityY":-0.027763497369049185,"x":0.31031571488272502,"y":0.5912376796404013},{"energy":0,"id":1,"owner":0,"rotation":94,"type":"NORMAL","velocityX":-0.002367805116236613,"velocityY":0.027763497369049189,"x":-0.31031571488272497,"y":-0.5912376796404013},{"energy":0,"id":2,"owner":1,"rotation":184,"type":"NORMAL","velocityX":-0.027763497369049192,"velocityY":-0.0023678051162366294,"x":0.59123767964040119,"y":-0.31031571488272502},{"energy":0,"id":3,"owner":2,"rotation":275,"type":"NORMAL","velocityX":0.00268220190540775,"velocityY":-0.028784321687225854,"x":0.30754845231748384,"y":0.54704186405486332},{"energy":0,"id":4,"owner":1,"rotation":185,"type":"NORMAL","velocityX":-0.028784321687225854,"velocityY":-0.0026822019054077492,"x":0.54704186405486332,"y":-0.30754845231748368},{"energy":0,"id":5,"owner":0,"rotation":95,"type":"NORMAL","velocityX":-0.0026822019054077483,"velocityY":0.028784321687225854,"x":-0.30754845231748373,"y":-0.54704186405486332},{"energy":0,"id":6,"owner":3,"rotation":5,"type":"NORMAL","velocityX":0.028784322437353804,"velocityY":0.0026822292725200581,"x":-0.54704184732616778,"y":0.30755156760572677},{"energy":0,"id":7,"owner":0,"rotation":95,"type":"NORMAL","velocityX":-0.0029998273469927918,"velocityY":0.029753839642051617,"x":-0.30360435262973923,"y":-0.5005670944094085},{"energy":0,"id":8,"owner":1,"rotation":185,"type":"NORMAL","velocityX":-0.029753839642051614,"velocityY":-0.0029998273469928113,"x":0.50056709440940861,"y":-0.30360435262973834},{"energy":0,"id":9,"owner":3,"rotation":5,"type":"NORMAL","velocityX":0.029753834540774817,"velocityY":0.0029999250528511164,"x":-0.50056721708442931,"y":0.30361327338195915},{"energy":0,"id":10,"owner":2,"rotation":275,"type":"NORMAL","velocityX":0.0029998273469927965,"velocityY":-0.029753839642051621,"x":0.30360435262973895,"y":0.50056709440940828},{"energy":0,"id":11,"owner":3,"rotation":6,"type":"NORMAL","velocityX":0.029552706260076177,"velocityY":0.0033000298002421754,"x":-0.5299060252991552,"y":0.30018365339425057},{"energy":0,"id":12,"owner":1,"rotation":186,"type":"NORMAL","velocityX":-0.029552702002996631,"velocityY":-0.0032998246918811125,"x":0.52990612023680883,"y":-0.30016503249326149},{"energy":0,"id":13,"owner":2,"rotation":276,"type":"NORMAL","velocityX":0.0032998246918811212,"velocityY":-0.029552702002996631,"x":0.30016503249326126,"y":0.52990612023680883},{"energy":0,"id":14,"owner":0,"rotation":96,"type":"NORMAL","velocityX":-0.0032998246918811177,"velocityY":0.029552702002996631,"x":-0.30016503249326132,"y":-0.52990612023680883},{"energy":0,"id":15,"owner":0,"rotation":96,"type":"NORMAL","velocityX":-0.0036623971529427066,"velocityY":0.030470620505143146,"x":-0.29112311404702035,"y":-0.4833300161402091},{"energy":0,"id":16,"owner":3,"rotation":6,"type":"NORMAL","velocityX":0.030470611684061288,"velocityY":0.0036627972617070234,"x":-0.48333022826874961,"y":0.29115244269221113},{"energy":0,"id":17,"owner":2,"rotation":276,"type":"NORMAL","velocityX":0.0036623971529427131,"velocityY":-0.030470620505143153,"x":0.29112311404702018,"y":0.4833300161402091},{"energy":0,"id":18,"owner":1,"rotation":186,"type":"NORMAL","velocityX":-0.030470620505143153,"velocityY":-0.0036623971529427353,"x":0.4833300161402091,"y":-0.29112311404701963},{"energy":0,"id":19,"owner":0,"rotation":97,"type":"NORMAL","velocityX":-0.0039359540797131746,"velocityY":0.030290179090420609,"x":-0.28699618248582576,"y":-0.51345955311059999},{"energy":0,"id":20,"owner":1,"rotation":187,"type":"NORMAL","velocityX":-0.030290179090420609,"velocityY":-0.003935954079713159,"x":0.51345955311059999,"y":-0.28699618248582637},{"energy":0,"id":21,"owner":2,"rotation":277,"type":"NORMAL","velocityX":0.0039359540797131824,"velocityY":-0.030290179090420609,"x":0.28699618248582559,"y":0.51345955311059999},{"energy":0,"id":22,"owner":3,"rotation":7,"type":"NORMAL","velocityX":0.030290163010913172,"velocityY":0.0039366009806527848,"x":-0.5134599117016706,"y":0.28704007845273632},{"energy":0,"id":23,"owner":0,"rotation":98,"type":"NORMAL","velocityX":-0.0042986410948563647,"velocityY":0.030204678147848409,"x":-0.27643905052753165,"y":-0.51536632013475214},{"energy":0,"id":24,"owner":1,"rotation":188,"type":"NORMAL","velocityX":-0.030204678147848409,"velocityY":-0.0042986410948563682,"x":0.51536632013475214,"y":-0.27643905052753143},{"energy":0,"id":25,"owner":3,"rotation":8,"type":"NORMAL","velocityX":0.030204670201638285,"velocityY":0.0042996369627886031,"x":-0.51536649734416029,"y":0.27649932324831777},{"energy":0,"id":26,"owner":2,"rotation":278,"type":"NORMAL","velocityX":0.0042986410948563708,"velocityY":-0.030204678147848409,"x":0.27643905052753165,"y":0.51536632013475214},{"energy":0,"id":27,"owner":2,"rotation":278,"type":"NORMAL","velocityX":0.0046521935469129678,"velocityY":-0.030167126446680046,"x":0.26591076117436258,"y":0.51620376523575995},{"energy":0,"id":28,"owner":0,"rotation":98,"type":"NORMAL","velocityX":-0.0046521935469129661,"velocityY":0.030167126446680046,"x":-0.2659107611743628,"y":-0.51620376523575995},{"energy":0,"id":29,"owner":1,"rotation":188,"type":"NORMAL","velocityX":-0.030167126446680046,"velocityY":-0.004652193546912967,"x":0.51620376523575995,"y":-0.26591076117436291},{"energy":0,"id":30,"owner":3,"rotation":8,"type":"NORMAL","velocityX":0.030167119573879159,"velocityY":0.0046536402363698704,"x":-0.5162039185069367,"y":0.26598944291666243},{"energy":0,"id":31,"owner":0,"rotation":99,"type":"NORMAL","velocityX":-0.0048900760038868753,"velocityY":0.029933981502126644,"x":-0.25973919042320248,"y":-0.54642778716786466},{"energy":0,"id":32,"owner":1,"rotation":189,"type":"NORMAL","velocityX":-0.029933981502126648,"velocityY":-0.0048900760038868589,"x":0.54642778716786455,"y":-0.25973919042320354},{"energy":0,"id":33,"owner":3,"rotation":9,"type":"NORMAL","velocityX":0.029933974171240565,"velocityY":0.0048920511598733426,"x":-0.54642793851484417,"y":0.25983951926292459},{"energy":0,"id":34,"owner":2,"rotation":279,"type":"NORMAL","velocityX":0.0048900760038868797,"velocityY":-0.029933981502126644,"x":0.2597391904232027,"y":0.54642778716786478},{"energy":0,"id":35,"owner":2,"rotation":279,"type":"NORMAL","velocityX":0.0052098662883446863,"velocityY":-0.029899725883006045,"x":0.24833398940299845,"y":0.54713499833148593},{"energy":0,"id":36,"owner":1,"rotation":189,"type":"NORMAL","velocityX":-0.029899725883006045,"velocityY":-0.005209866288344695,"x":0.54713499833148593,"y":-0.24833398940299778},{"energy":0,"id":37,"owner":3,"rotation":9,"type":"NORMAL","velocityX":0.029899715359975623,"velocityY":0.0052125246419741202,"x":-0.54713521558064915,"y":0.24845615998759707},{"energy":0,"id":38,"owner":0,"rotation":99,"type":"NORMAL","velocityX":-0.0052098662883446828,"velocityY":0.029899725883006045,"x":-0.24833398940299825,"y":-0.54713499833148593},{"energy":0,"id":39,"owner":3,"rotation":10,"type":"NORMAL","velocityX":0.029633104519057898,"velocityY":0.0054235948911473932,"x":-0.57711031485494402,"y":0.24122567375659326},{"energy":0,"id":40,"owner":2,"rotation":280,"type":"NORMAL","velocityX":0.0054201698402686054,"velocityY":-0.02963311378182687,"x":0.24107815519009348,"y":0.57711013821418777},{"energy":0,"id":41,"owner":0,"rotation":100,"type":"NORMAL","velocityX":-0.0054201698402685993,"velocityY":0.029633113781826877,"x":-0.24107815519009393,"y":-0.57711013821418777},{"energy":0,"id":42,"owner":1,"rotation":190,"type":"NORMAL","velocityX":-0.029633113781826873,"velocityY":-0.0054201698402686037,"x":0.57711013821418777,"y":-0.24107815519009312},{"energy":0,"id":43,"owner":0,"rotation":100,"type":"NORMAL","velocityX":-0.0056999129557947226,"velocityY":0.030758779616304549,"x":-0.2290100460279817,"y":-0.52939973603189128},{"energy":0,"id":44,"owner":1,"rotation":190,"type":"NORMAL","velocityX":-0.030758779616304549,"velocityY":-0.0056999129557947234,"x":0.52939973603189128,"y":-0.22901004602798167},{"energy":0,"id":45,"owner":2,"rotation":280,"type":"NORMAL","velocityX":0.0056999129557947234,"velocityY":-0.030758779616304549,"x":0.22901004602798156,"y":0.52939973603189128},{"energy":0,"id":46,"owner":3,"rotation":10,"type":"NORMAL","velocityX":0.030758736891394946,"velocityY":0.005704297360962265,"x":-0.52940061809252326,"y":0.22918163671765579},{"energy":0,"id":47,"owner":2,"rotation":280,"type":"NORMAL","velocityX":0.0058771072813490676,"velocityY":-0.030513534585828894,"x":0.22076735644060694,"y":0.5603205370600991},{"energy":0,"id":48,"owner":0,"rotation":100,"type":"NORMAL","velocityX":-0.0058771072813490642,"velocityY":0.030513534585828894,"x":-0.2207673564406068,"y":-0.5603205370600991},{"energy":0,"id":49,"owner":1,"rotation":190,"type":"NORMAL","velocityX":-0.030513534585828894,"velocityY":-0.0058771072813490685,"x":0.5603205370600991,"y":-0.22076735644060558},{"energy":0,"id":50,"owner":3,"rotation":10,"type":"NORMAL","velocityX":0.030513485598491295,"velocityY":0.0058825365281259942,"x":-0.56032147124731502,"y":0.22096666501401913},{"energy":0,"id":51,"owner":2,"rotation":281,"type":"NORMAL","velocityX":0.006112094289092861,"velocityY":-0.03049457132589074,"x":0.20819322104768445,"y":0.56068216591927167},{"energy":0,"id":52,"owner":0,"rotation":101,"type":"NORMAL","velocityX":-0.0061120942890928584,"velocityY":0.03049457132589074,"x":-0.20819322104768456,"y":-0.56068216591927167},{"energy":0,"id":53,"owner":3,"rotation":11,"type":"NORMAL","velocityX":0.030494514959218778,"velocityY":0.0061187969126491985,"x":-0.56068324083019649,"y":0.20841743673237934},{"energy":0,"id":54,"owner":1,"rotation":191,"type":"NORMAL","velocityX":-0.030494571325890747,"velocityY":-0.0061120942890928584,"x":0.56068216591927145,"y":-0.20819322104768423},{"energy":0,"id":55,"owner":1,"rotation":191,"type":"NORMAL","velocityX":-0.030215792038616969,"velocityY":-0.0062513921384542964,"x":0.59134542098876064,"y":-0.19917509657288934},{"energy":0,"id":56,"owner":0,"rotation":101,"type":"NORMAL","velocityX":-0.0062513921384543085,"velocityY":0.030215792038616969,"x":-0.19917509657288834,"y":-0.59134542098876064},{"energy":0,"id":57,"owner":3,"rotation":11,"type":"NORMAL","velocityX":0.030215756254246114,"velocityY":0.0062594495812065077,"x":-0.59134604961432347,"y":0.19942865791709746},{"energy":0,"id":58,"owner":2,"rotation":281,"type":"NORMAL","velocityX":0.0062513921384543172,"velocityY":-0.030215792038616969,"x":0.19917509657288818,"y":0.59134542098876064},{"energy":0,"id":59,"owner":3,"rotation":12,"type":"NORMAL","velocityX":0.030197447159614456,"velocityY":0.0064457871740518039,"x":-0.5916676862574497,"y":0.18654736228091082},{"energy":0,"id":60,"owner":2,"rotation":282,"type":"NORMAL","velocityX":0.006436108352358175,"velocityY":-0.030197491309149487,"x":0.18626996529008663,"y":0.59166691068069233},{"energy":0,"id":61,"owner":1,"rotation":192,"type":"NORMAL","velocityX":-0.030197491309149487,"velocityY":-0.0064361083523581794,"x":0.59166691068069233,"y":-0.1862699652900861},{"energy":0,"id":62,"owner":0,"rotation":102,"type":"NORMAL","velocityX":-0.0064361083523581646,"velocityY":0.030197491309149487,"x":-0.18626996529008716,"y":-0.59166691068069233},{"energy":0,"id":63,"owner":2,"rotation":282,"type":"NORMAL","velocityX":0.0065907518520656417,"velocityY":-0.03018260614940374,"x":0.17353045151366941,"y":0.59192839891423576},{"energy":0,"id":64,"owner":1,"rotation":192,"type":"NORMAL","velocityX":-0.03018260614940374,"velocityY":-0.0065907518520656374,"x":0.59192839891423576,"y":-0.17353045151366933},{"energy":0,"id":65,"owner":3,"rotation":12,"type":"NORMAL","velocityX":0.030182540411880763,"velocityY":0.0066022369567114089,"x":-0.59192955372809464,"y":0.17383041740169175},{"energy":0,"id":66,"owner":0,"rotation":102,"type":"NORMAL","velocityX":-0.0065907518520656261,"velocityY":0.03018260614940374,"x":-0.17353045151367016,"y":-0.59192839891423576},{"energy":0,"id":67,"owner":2,"rotation":282,"type":"NORMAL","velocityX":0.0066640342366021321,"velocityY":-0.029866734423083147,"x":0.16360092552682903,"y":0.62233678355140287},{"energy":0,"id":68,"owner":1,"rotation":192,"type":"NORMAL","velocityX":-0.029866734423083151,"velocityY":-0.0066640342366021555,"x":0.62233678355140265,"y":-0.1636009255268277},{"energy":0,"id":69,"owner":3,"rotation":12,"type":"NORMAL","velocityX":0.029866687148036503,"velocityY":0.0066774050208519434,"x":-0.62233754603699565,"y":0.16392759849312158},{"energy":0,"id":70,"owner":0,"rotation":102,"type":"NORMAL","velocityX":-0.006664034236602126,"velocityY":0.029866734423083151,"x":-0.16360092552682923,"y":-0.62233678355140265},{"energy":0,"id":71,"owner":1,"rotation":192,"type":"NORMAL","velocityX":-0.029519643349077068,"velocityY":-0.0067151252564750579,"x":0.65240272652585085,"y":-0.15356092542634422},{"energy":0,"id":72,"owner":0,"rotation":102,"type":"NORMAL","velocityX":-0.0067151252564750753,"velocityY":0.029519643349077068,"x":-0.15356092542634353,"y":-0.65240272652585085},{"energy":0,"id":73,"owner":3,"rotation":12,"type":"NORMAL","velocityX":0.029519597179897623,"velocityY":0.0067305622262152798,"x":-0.65240340744065728,"y":0.15391297440659218},{"energy":0,"id":74,"owner":2,"rotation":282,"type":"NORMAL","velocityX":0.006715125256475077,"velocityY":-0.029519643349077068,"x":0.15356092542634292,"y":0.65240272652585085},{"energy":0,"id":75,"owner":2,"rotation":282,"type":"NORMAL","velocityX":0.0067889636189822564,"velocityY":-0.029508079438887787,"x":0.14067968079709819,"y":0.65257327401574516},{"energy":0,"id":76,"owner":1,"rotation":192,"type":"NORMAL","velocityX":-0.029508079438887787,"velocityY":-0.0067889636189822616,"x":0.65257327401574516,"y":-0.14067968079709756},{"energy":0,"id":77,"owner":0,"rotation":102,"type":"NORMAL","velocityX":-0.0067889636189822529,"velocityY":0.029508079438887787,"x":-0.14067968079709828,"y":-0.65257327401574516},{"energy":0,"id":78,"owner":3,"rotation":12,"type":"NORMAL","velocityX":0.029507993786919744,"velocityY":0.0068067776482763276,"x":-0.65257453723276371,"y":0.14104789529266032},{"energy":0,"id":79,"owner":3,"rotation":13,"type":"NORMAL","velocityX":0.02912421700769861,"velocityY":0.0068112248474460476,"x":-0.68233036163808591,"y":0.13081920228901867},{"energy":0,"id":80,"owner":1,"rotation":193,"type":"NORMAL","velocityX":-0.029124307352881296,"velocityY":-0.0067909684324517113,"x":0.68232914923334997,"y":-0.13043041960842511},{"energy":0,"id":81,"owner":2,"rotation":283,"type":"NORMAL","velocityX":0.0067909684324517312,"velocityY":-0.029124307352881296,"x":0.13043041960842472,"y":0.68232914923334997},{"energy":0,"id":82,"owner":0,"rotation":103,"type":"NORMAL","velocityX":-0.006790968432451713,"velocityY":0.029124307352881296,"x":-0.13043041960842477,"y":-0.68232914923334997},{"energy":0,"id":83,"owner":3,"rotation":13,"type":"NORMAL","velocityX":0.029117464939330449,"velocityY":0.0068289162262177181,"x":-0.68242097233168264,"y":0.11830674054685285},{"energy":0,"id":84,"owner":0,"rotation":103,"type":"NORMAL","velocityX":-0.0068058976082285352,"velocityY":0.029117559941433824,"x":-0.11790779289805932,"y":-0.68241969743249453},{"energy":0,"id":85,"owner":2,"rotation":283,"type":"NORMAL","velocityX":0.006805897608228543,"velocityY":-0.029117559941433824,"x":0.11790779289805907,"y":0.68241969743249453},{"energy":0,"id":86,"owner":1,"rotation":193,"type":"NORMAL","velocityX":-0.029117559941433824,"velocityY":-0.0068058976082285396,"x":0.68241969743249453,"y":-0.11790779289806015},{"energy":0,"id":87,"owner":0,"rotation":103,"type":"NORMAL","velocityX":-0.0067565609420874055,"velocityY":0.028700430792353272,"x":-0.1077186430166214,"y":-0.71178566357518869},{"energy":0,"id":88,"owner":2,"rotation":283,"type":"NORMAL","velocityX":0.0067565609420874063,"velocityY":-0.028700430792353272,"x":0.10771864301662136,"y":0.71178566357518869},{"energy":0,"id":89,"owner":1,"rotation":193,"type":"NORMAL","velocityX":-0.028700430792353272,"velocityY":-0.0067565609420874063,"x":0.71178566357518869,"y":-0.10771864301662144},{"energy":0,"id":90,"owner":3,"rotation":13,"type":"NORMAL","velocityX":0.028700333341080957,"velocityY":0.0067823986116579754,"x":-0.71178684640875156,"y":0.10813144620152583},{"energy":0,"id":91,"owner":3,"rotation":13,"type":"NORMAL","velocityX":0.028699961316282774,"velocityY":0.0067403616260772146,"x":-0.71179136193130033,"y":0.096257023215794393},{"energy":0,"id":92,"owner":0,"rotation":103,"type":"NORMAL","velocityX":-0.0067113809173800906,"velocityY":0.028700131383945383,"x":-0.095843484148631433,"y":-0.71178929770233201},{"energy":0,"id":93,"owner":2,"rotation":283,"type":"NORMAL","velocityX":0.0067113809173800915,"velocityY":-0.028700131383945383,"x":0.095843484148631544,"y":0.71178929770233201},{"energy":0,"id":94,"owner":1,"rotation":193,"type":"NORMAL","velocityX":-0.028700131383945386,"velocityY":-0.0067113809173801027,"x":0.7117892977023319,"y":-0.095843484148630531},{"energy":0,"id":95,"owner":2,"rotation":283,"type":"NORMAL","velocityX":0.0066097085043353657,"velocityY":-0.028249890627729979,"x":0.085936988090865432,"y":0.74073791486980034},{"energy":0,"id":96,"owner":0,"rotation":103,"type":"NORMAL","velocityX":-0.0066097085043353588,"velocityY":0.028249890627729979,"x":-0.085936988090865613,"y":-0.74073791486980034},{"energy":0,"id":97,"owner":1,"rotation":193,"type":"NORMAL","velocityX":-0.028249890627729979,"velocityY":-0.0066097085043353562,"x":0.74073791486980034,"y":-0.085936988090865626},{"energy":0,"id":98,"owner":3,"rotation":13,"type":"NORMAL","velocityX":0.028249714954316441,"velocityY":0.0066418727353948614,"x":-0.74073982926279469,"y":0.086355504887036083},{"energy":0,"id":99,"owner":1,"rotation":193,"type":"NORMAL","velocityX":-0.027766647732797932,"velocityY":-0.0064826663708255142,"x":0.76923609679206506,"y":-0.07626364457159697},{"energy":0,"id":100,"owner":2,"rotation":283,"type":"NORMAL","velocityX":0.006482666370825528,"velocityY":-0.027766647732797932,"x":0.076263644571596192,"y":0.76923609679206506},{"energy":0,"id":101,"owner":0,"rotation":103,"type":"NORMAL","velocityX":-0.0064826663708255133,"velocityY":0.027766647732797932,"x":-0.076263644571596692,"y":-0.76923609679206506},{"energy":0,"id":102,"owner":3,"rotation":13,"type":"NORMAL","velocityX":0.027766468667746156,"velocityY":0.0065181850603723687,"x":-0.76923783276548763,"y":0.076682542516399874},{"energy":0,"id":103,"owner":2,"rotation":282,"type":"NORMAL","velocityX":0.00635092515477792,"velocityY":-0.029323188048885184,"x":0.065606586217182211,"y":0.72904170528013701},{"energy":0,"id":104,"owner":0,"rotation":102,"type":"NORMAL","velocityX":-0.0063509251547779174,"velocityY":0.029323188048885184,"x":-0.065606586217182294,"y":-0.72904170528013701},{"energy":0,"id":105,"owner":3,"rotation":12,"type":"NORMAL","velocityX":0.029322859919083042,"velocityY":0.0063901021419649874,"x":-0.72904528105965805,"y":0.066011551675703287},{"energy":0,"id":106,"owner":1,"rotation":192,"type":"NORMAL","velocityX":-0.029323188048885177,"velocityY":-0.0063509251547779165,"x":0.72904170528013701,"y":-0.065606586217183155},{"energy":0,"id":107,"owner":2,"rotation":282,"type":"NORMAL","velocityX":0.0061715355990508546,"velocityY":-0.028862195175582566,"x":0.056570554114506011,"y":0.75861514521492979},{"energy":0,"id":108,"owner":0,"rotation":102,"type":"NORMAL","velocityX":-0.0061715355990508407,"velocityY":0.028862195175582566,"x":-0.056570554114506511,"y":-0.75861514521492979},{"energy":0,"id":109,"owner":1,"rotation":192,"type":"NORMAL","velocityX":-0.028862195175582573,"velocityY":-0.0061715355990508407,"x":0.75861514521492968,"y":-0.056570554114505914},{"energy":0,"id":110,"owner":3,"rotation":12,"type":"NORMAL","velocityX":0.028861861402700646,"velocityY":0.0062143891279666744,"x":-0.75861838102697976,"y":0.056964315213819984},{"energy":0,"id":111,"owner":0,"rotation":101,"type":"NORMAL","velocityX":-0.0059672129437249649,"velocityY":0.028367429440265709,"x":-0.04792947586347699,"y":-0.78772863597392917},{"energy":0,"id":112,"owner":3,"rotation":11,"type":"NORMAL","velocityX":0.028367094342699233,"velocityY":0.006013886216001697,"x":-0.78773149280097277,"y":0.048306038770027239},{"energy":0,"id":113,"owner":2,"rotation":281,"type":"NORMAL","velocityX":0.0059672129437249709,"velocityY":-0.028367429440265709,"x":0.047929475863476748,"y":0.78772863597392917},{"energy":0,"id":114,"owner":1,"rotation":191,"type":"NORMAL","velocityX":-0.028367429440265709,"velocityY":-0.0059672129437249562,"x":0.78772863597392917,"y":-0.04792947586347733},{"energy":0,"id":115,"owner":0,"rotation":101,"type":"NORMAL","velocityX":-0.005752731922372102,"velocityY":0.028406237653821383,"x":-0.038939103290073364,"y":-0.78739778202241362},{"energy":0,"id":116,"owner":1,"rotation":191,"type":"NORMAL","velocityX":-0.028406237653821383,"velocityY":-0.0057527319223721063,"x":0.78739778202241362,"y":-0.038939103290073572},{"energy":0,"id":117,"owner":3,"rotation":11,"type":"NORMAL","velocityX":0.028405795704391959,"velocityY":0.0058034966461915048,"x":-0.78740154979994559,"y":0.039282689490400396},{"energy":0,"id":118,"owner":2,"rotation":281,"type":"NORMAL","velocityX":0.0057527319223721089,"velocityY":-0.028406237653821383,"x":0.038939103290073447,"y":0.78739778202241362},{"energy":0,"id":119,"owner":0,"rotation":101,"type":"NORMAL","velocityX":-0.0054985008886971184,"velocityY":0.027878571490099837,"x":-0.031253257543995647,"y":-0.81605588251767336},{"energy":0,"id":120,"owner":3,"rotation":11,"type":"NORMAL","velocityX":0.027878131837770839,"velocityY":0.0055533333980111567,"x":-0.81605912978568251,"y":0.031565522809857187},{"energy":0,"id":121,"owner":1,"rotation":191,"type":"NORMAL","velocityX":-0.02787857149009983,"velocityY":-0.0054985008886971254,"x":0.81605588251767336,"y":-0.031253257543995161},{"energy":0,"id":122,"owner":2,"rotation":281,"type":"NORMAL","velocityX":0.0054985008886971245,"velocityY":-0.02787857149009983,"x":0.031253257543995411,"y":0.81605588251767336},{"energy":0,"id":123,"owner":0,"rotation":100,"type":"NORMAL","velocityX":-0.0052209123799688908,"velocityY":0.027317371601633222,"x":-0.024119110641833294,"y":-0.84418671866128414},{"energy":0,"id":124,"owner":2,"rotation":280,"type":"NORMAL","velocityX":0.0052209123799689004,"velocityY":-0.027317371601633222,"x":0.024119110641833384,"y":0.84418671866128414},{"energy":0,"id":125,"owner":1,"rotation":190,"type":"NORMAL","velocityX":-0.027317371601633222,"velocityY":-0.00522091237996889,"x":0.84418671866128414,"y":-0.024119110641833676},{"energy":0,"id":126,"owner":3,"rotation":10,"type":"NORMAL","velocityX":0.027316940531276788,"velocityY":0.0052799109082180672,"x":-0.84418942287764209,"y":0.024392901717911607},{"energy":0,"id":127,"owner":0,"rotation":100,"type":"NORMAL","velocityX":-0.004920765292406596,"velocityY":0.026722830279735637,"x":-0.017584033890375571,"y":-0.87175620485842853},{"energy":0,"id":128,"owner":3,"rotation":10,"type":"NORMAL","velocityX":0.026722414661941137,"velocityY":0.0049840095804166117,"x":-0.87175835947638391,"y":0.017811894003980405},{"energy":0,"id":129,"owner":1,"rotation":190,"type":"NORMAL","velocityX":-0.026722830279735637,"velocityY":-0.0049207652924065986,"x":0.87175620485842853,"y":-0.017584033890375845},{"energy":0,"id":130,"owner":2,"rotation":280,"type":"NORMAL","velocityX":0.0049207652924066047,"velocityY":-0.026722830279735637,"x":0.017584033890375411,"y":0.87175620485842853},{"energy":0,"id":131,"owner":3,"rotation":9,"type":"NORMAL","velocityX":0.026786352360922524,"velocityY":0.0046756500927259605,"x":-0.87142689793630068,"y":0.011475408074194022},{"energy":0,"id":132,"owner":0,"rotation":99,"type":"NORMAL","velocityX":-0.0046079627333446664,"velocityY":0.026786829234024936,"x":-0.011308443652630304,"y":-0.87142442576267753},{"energy":0,"id":133,"owner":2,"rotation":279,"type":"NORMAL","velocityX":0.004607962733344669,"velocityY":-0.026786829234024936,"x":0.011308443652630021,"y":0.87142442576267753},{"energy":0,"id":134,"owner":1,"rotation":189,"type":"NORMAL","velocityX":-0.026786829234024936,"velocityY":-0.0046079627333446777,"x":0.87142442576267753,"y":-0.011308443652630593},{"energy":0,"id":135,"owner":2,"rotation":279,"type":"NORMAL","velocityX":0.0042645811335507364,"velocityY":-0.026160282160520243,"x":0.0061255839489776226,"y":0.89846239769547209},{"energy":0,"id":136,"owner":0,"rotation":99,"type":"NORMAL","velocityX":-0.004264581133550726,"velocityY":0.026160282160520236,"x":-0.0061255839489779253,"y":-0.8984623976954722},{"energy":0,"id":137,"owner":1,"rotation":189,"type":"NORMAL","velocityX":-0.026160282160520243,"velocityY":-0.0042645811335507303,"x":0.89846239769547209,"y":-0.0061255839489771412},{"energy":0,"id":138,"owner":3,"rotation":9,"type":"NORMAL","velocityX":0.026159836908672313,"velocityY":0.0043366112024038912,"x":-0.89846423026091116,"y":0.0062304352794197507},{"energy":0,"id":139,"owner":0,"rotation":98,"type":"NORMAL","velocityX":-0.0039018230442638565,"velocityY":0.025500777505939046,"x":-0.0016762950293225595,"y":-0.92487270443206615},{"energy":0,"id":140,"owner":2,"rotation":278,"type":"NORMAL","velocityX":0.0039018230442638617,"velocityY":-0.025500777505939046,"x":0.0016762950293223011,"y":0.92487270443206615},{"energy":0,"id":141,"owner":3,"rotation":8,"type":"NORMAL","velocityX":0.025500373232966625,"velocityY":0.0039782071567567171,"x":-0.92487394377852117,"y":0.0017109879276729139},{"energy":0,"id":142,"owner":1,"rotation":188,"type":"NORMAL","velocityX":-0.025500777505939046,"velocityY":-0.0039018230442638556,"x":0.92487270443206615,"y":-0.0016762950293231407},{"energy":0,"id":143,"owner":2,"rotation":278,"type":"NORMAL","velocityX":0.0035210100586988121,"velocityY":-0.024808610485328755,"x":-0.0020010977285073684,"y":0.95062147480684511},{"energy":0,"id":144,"owner":1,"rotation":188,"type":"NORMAL","velocityX":-0.024808610485328755,"velocityY":-0.0035210100586988086,"x":0.95062147480684511,"y":0.0020010977285079348},{"energy":0,"id":145,"owner":3,"rotation":8,"type":"NORMAL","velocityX":0.024808256944097558,"velocityY":0.0036017331089247952,"x":-0.95062219289552063,"y":-0.0020446772857945377},{"energy":0,"id":209,"owner":3,"rotation":354,"type":"NORMAL","velocityX":0.042805079889692094,"velocityY":-0.004148870428880385,"x":0.36756750523706327,"y":0.078424456793315583},{"energy":0,"id":210,"owner":0,"rotation":83,"type":"NORMAL","velocityX":0.0046171678108031791,"velocityY":0.04354031602897622,"x":-0.093072366485699123,"y":0.32400425108654346},{"energy":0,"id":211,"owner":2,"rotation":263,"type":"NORMAL","velocityX":-0.0046171678108031808,"velocityY":-0.043540316028976213,"x":0.093072366485698943,"y":-0.32400425108654329},{"energy":0,"id":212,"owner":3,"rotation":354,"type":"NORMAL","velocityX":0.043544836366601711,"velocityY":-0.0044876942211102824,"x":0.32409333641318278,"y":0.090410448725522535},{"energy":0,"id":214,"owner":3,"rotation":353,"type":"NORMAL","velocityX":0.044884842980551135,"velocityY":-0.0048207400930103785,"x":0.23400723530485792,"y":0.10183467444454031},{"energy":0,"id":215,"owner":2,"rotation":263,"type":"NORMAL","velocityX":-0.0049556465391438903,"velocityY":-0.044879478063945308,"x":0.10475276984482772,"y":-0.23390888469694784},{"energy":0,"id":216,"owner":1,"rotation":173,"type":"NORMAL","velocityX":-0.044879098212208528,"velocityY":0.0049714357638200252,"x":-0.2338760492911407,"y":-0.1050572694564514},{"energy":0,"id":217,"owner":0,"rotation":83,"type":"NORMAL","velocityX":0.0049556465391438877,"velocityY":0.044879478063945308,"x":-0.10475276984482781,"y":0.23390888469694784},{"energy":0,"id":218,"owner":3,"rotation":353,"type":"NORMAL","velocityX":0.045507408556852876,"velocityY":-0.0051051595702333048,"x":0.18847158444095791,"y":0.11465638930913201},{"energy":0,"id":219,"owner":0,"rotation":83,"type":"NORMAL","velocityX":0.0052814612159139841,"velocityY":0.046093182205206903,"x":-0.11660619311115021,"y":0.14162921241571128},{"energy":0,"id":220,"owner":1,"rotation":173,"type":"NORMAL","velocityX":-0.045500573689550017,"velocityY":0.0052690808837987458,"x":-0.18831005725462541,"y":-0.11836221633309521},{"energy":0,"id":221,"owner":2,"rotation":263,"type":"NORMAL","velocityX":-0.0052814612159139962,"velocityY":-0.046093182205206903,"x":0.11660619311114996,"y":-0.14162921241571128},{"energy":0,"id":222,"owner":0,"rotation":83,"type":"NORMAL","velocityX":0.0055460128806422487,"velocityY":0.046630946739958545,"x":-0.13010683862394726,"y":0.094854912735794328},{"energy":0,"id":223,"owner":2,"rotation":263,"type":"NORMAL","velocityX":-0.0055460128806422522,"velocityY":-0.046630946739958545,"x":0.1301068386239472,"y":-0.094854912735794328},{"energy":0,"id":224,"owner":1,"rotation":173,"type":"NORMAL","velocityX":-0.046629629380287169,"velocityY":0.0055758913521509771,"x":-0.094791975899496977,"y":-0.13074219651090055},{"energy":0,"id":225,"owner":3,"rotation":353,"type":"NORMAL","velocityX":0.046638968452964821,"velocityY":-0.0053941529413075737,"x":0.095002987392409102,"y":0.12647894921233835},{"energy":0,"id":226,"owner":1,"rotation":172,"type":"NORMAL","velocityX":-0.047124315400663995,"velocityY":0.0058177183883245717,"x":-0.047469285757325771,"y":-0.14476895630390793},{"energy":0,"id":227,"owner":3,"rotation":353,"type":"NORMAL","velocityX":0.047599001675651495,"velocityY":-0.0056647800194986418,"x":-0.0010388659599986499,"y":0.13832248904740921},{"energy":0,"id":228,"owner":0,"rotation":83,"type":"NORMAL","velocityX":0.0058293232858275534,"velocityY":0.0475883066526053,"x":-0.14240364727045196,"y":-0.0012056561336942426},{"energy":0,"id":229,"owner":2,"rotation":263,"type":"NORMAL","velocityX":-0.0058293232858275638,"velocityY":-0.047588306652605293,"x":0.14240364727045157,"y":0.0012056561336942564},{"energy":0,"id":230,"owner":3,"rotation":353,"type":"NORMAL","velocityX":0.048004895563241397,"velocityY":-0.0058513527172919973,"x":-0.049045735650599047,"y":0.15170610210857563},{"energy":0,"id":231,"owner":0,"rotation":82,"type":"NORMAL","velocityX":0.0060302149301121183,"velocityY":0.047992413202184406,"x":-0.1563786115969151,"y":-0.049256193526644565},{"energy":0,"id":232,"owner":1,"rotation":172,"type":"NORMAL","velocityX":-0.047989510602206284,"velocityY":0.0060807635565222531,"x":0.049420681959703963,"y":-0.15754750291741931},{"energy":0,"id":233,"owner":2,"rotation":262,"type":"NORMAL","velocityX":-0.0060302149301121235,"velocityY":-0.047992413202184413,"x":0.15637861159691499,"y":0.049256193526644641},{"energy":0,"id":234,"owner":0,"rotation":82,"type":"NORMAL","velocityX":0.0062650155800046238,"velocityY":0.048673817614199269,"x":-0.16885248379863579,"y":-0.14810160339142733},{"energy":0,"id":235,"owner":3,"rotation":352,"type":"NORMAL","velocityX":0.048690320848728694,"velocityY":-0.0060669437708323571,"x":-0.14783219269073544,"y":0.16351693474219492},{"energy":0,"id":236,"owner":2,"rotation":262,"type":"NORMAL","velocityX":-0.0062650155800046282,"velocityY":-0.048673817614199269,"x":0.16885248379863566,"y":0.14810160339142733},{"energy":0,"id":237,"owner":1,"rotation":172,"type":"NORMAL","velocityX":-0.048668903208006459,"velocityY":0.0063291213723131933,"x":0.1483054089029697,"y":-0.17037621201317976},{"energy":0,"id":238,"owner":0,"rotation":82,"type":"NORMAL","velocityX":0.006482582918189205,"velocityY":0.049160615577098311,"x":-0.18127616466769783,"y":-0.24829453051555331},{"energy":0,"id":239,"owner":1,"rotation":172,"type":"NORMAL","velocityX":-0.048932976275204342,"velocityY":0.0064740802546052537,"x":0.19740209232782441,"y":-0.18488975639078056},{"energy":0,"id":240,"owner":2,"rotation":262,"type":"NORMAL","velocityX":-0.0064825829181892293,"velocityY":-0.049160615577098311,"x":0.18127616466769739,"y":0.24829453051555331},{"energy":0,"id":241,"owner":2,"rotation":262,"type":"NORMAL","velocityX":-0.0065759674344608703,"velocityY":-0.049329940880027623,"x":0.19524654203993869,"y":0.29772560895181877},{"energy":0,"id":242,"owner":1,"rotation":172,"type":"NORMAL","velocityX":-0.049319712952569331,"velocityY":0.00667274187684314,"x":0.2980375750095296,"y":-0.19775196847978144},{"energy":0,"id":243,"owner":0,"rotation":82,"type":"NORMAL","velocityX":0.006575967434460866,"velocityY":0.049329940880027623,"x":-0.1952465420399388,"y":-0.29772560895181877},{"energy":0,"id":244,"owner":3,"rotation":352,"type":"NORMAL","velocityX":0.049355923098456399,"velocityY":-0.0063313171874752874,"x":-0.2973256075733971,"y":0.18812671700120828},{"energy":0,"id":245,"owner":3,"rotation":352,"type":"NORMAL","velocityX":0.049551890369196445,"velocityY":-0.0064625051500421362,"x":-0.3985722681556701,"y":0.19920523025358483},{"energy":0,"id":246,"owner":1,"rotation":172,"type":"NORMAL","velocityX":-0.049502693983707866,"velocityY":0.0068556066071116415,"x":0.39956095524161478,"y":-0.21055394156427029},{"energy":0,"id":247,"owner":0,"rotation":82,"type":"NORMAL","velocityX":0.0067375864760245941,"velocityY":0.049517653926949505,"x":-0.20741444523901745,"y":-0.39909998273155634},{"energy":0,"id":248,"owner":2,"rotation":262,"type":"NORMAL","velocityX":-0.0067375864760246098,"velocityY":-0.049517653926949505,"x":0.20741444523901745,"y":0.39909998273155634},{"energy":0,"id":249,"owner":3,"rotation":352,"type":"NORMAL","velocityX":0.04963888225667467,"velocityY":-0.0065829108971445071,"x":-0.50058694180324448,"y":0.20994436793783094},{"energy":0,"id":250,"owner":0,"rotation":82,"type":"NORMAL","velocityX":0.0068950733254925321,"velocityY":0.049599289694068134,"x":-0.21942267613536942,"y":-0.50122458010320792},{"energy":0,"id":251,"owner":1,"rotation":172,"type":"NORMAL","velocityX":-0.049548697669495947,"velocityY":0.0068923929556497874,"x":0.44909284279165096,"y":-0.22470056561169932},{"energy":0,"id":252,"owner":2,"rotation":262,"type":"NORMAL","velocityX":-0.0068950733254925485,"velocityY":-0.049599289694068134,"x":0.21942267613536884,"y":0.50122458010320803},{"energy":0,"id":253,"owner":0,"rotation":82,"type":"NORMAL","velocityX":0.0068835323469006379,"velocityY":0.04965149019243606,"x":-0.23235783935828799,"y":-0.55055787433525594},{"energy":0,"id":254,"owner":3,"rotation":352,"type":"NORMAL","velocityX":0.049725268232852851,"velocityY":-0.0067075746531258147,"x":-0.60285963919717467,"y":0.22030051047927818},{"energy":0,"id":255,"owner":1,"rotation":171,"type":"NORMAL","velocityX":-0.049629892652134328,"velocityY":0.0070501647019275233,"x":0.55116786625748027,"y":-0.23717123436406759},{"energy":0,"id":256,"owner":2,"rotation":262,"type":"NORMAL","velocityX":-0.0068835323469006535,"velocityY":-0.04965149019243606,"x":0.23235783935828791,"y":0.55055787433525594},{"energy":0,"id":257,"owner":1,"rotation":171,"type":"NORMAL","velocityX":-0.04970862243176731,"velocityY":0.0072242366166785296,"x":0.65354746009493148,"y":-0.24955162362406666},{"energy":0,"id":258,"owner":2,"rotation":261,"type":"NORMAL","velocityX":-0.0070247449559515682,"velocityY":-0.049735060255032063,"x":0.24368110132938275,"y":0.65282651196174679},{"energy":0,"id":259,"owner":0,"rotation":81,"type":"NORMAL","velocityX":0.0070247449559515639,"velocityY":0.049735060255032063,"x":-0.24368110132938275,"y":-0.65282651196174679},{"energy":0,"id":260,"owner":3,"rotation":352,"type":"NORMAL","velocityX":0.049811306110864789,"velocityY":-0.0068398691151162413,"x":-0.70612954210133505,"y":0.23036619129299454},{"energy":0,"id":261,"owner":2,"rotation":261,"type":"NORMAL","velocityX":-0.0071615309174091427,"velocityY":-0.04969102940804105,"x":0.25484844084489078,"y":0.75583790694082686},{"energy":0,"id":262,"owner":0,"rotation":81,"type":"NORMAL","velocityX":0.0071615309174091271,"velocityY":0.049691029408041043,"x":-0.25484844084489111,"y":-0.75583790694082686},{"energy":0,"id":263,"owner":3,"rotation":352,"type":"NORMAL","velocityX":0.048172063861808778,"velocityY":-0.0067354343578795222,"x":-0.80934071657666862,"y":0.24001365051412746},{"energy":0,"id":264,"owner":1,"rotation":171,"type":"NORMAL","velocityX":-0.049467359532504813,"velocityY":0.0073729404653146997,"x":0.75698329121905528,"y":-0.26202021395606051},{"energy":0,"id":265,"owner":0,"rotation":81,"type":"NORMAL","velocityX":0.007053531971063294,"velocityY":0.047881921185813352,"x":-0.2657864269208931,"y":-0.8577330228366179},{"energy":0,"id":266,"owner":2,"rotation":261,"type":"NORMAL","velocityX":-0.0070564762118301193,"velocityY":-0.047881133896858531,"x":0.26589620252900853,"y":0.85773541188537594},{"energy":0,"id":267,"owner":1,"rotation":171,"type":"NORMAL","velocityX":-0.049472917195448315,"velocityY":0.0072888671239767925,"x":0.8056567452047847,"y":-0.27421998682055454},{"energy":0,"id":268,"owner":2,"rotation":261,"type":"NORMAL","velocityX":-0.0069123658337275759,"velocityY":-0.045856956651327149,"x":0.27606106892462595,"y":0.95414304334867284},{"energy":0,"id":269,"owner":1,"rotation":171,"type":"NORMAL","velocityX":-0.047529147506560815,"velocityY":0.0071668096674743923,"x":0.90594464296908417,"y":-0.28569372719499586},{"energy":0,"id":270,"owner":0,"rotation":81,"type":"NORMAL","velocityX":0.0069023968224006796,"velocityY":0.045859752117687051,"x":-0.27566672344233206,"y":-0.954140247882313},{"energy":0,"id":271,"owner":3,"rotation":352,"type":"NORMAL","velocityX":0.044275746922938425,"velocityY":-0.0062099270214655902,"x":-1,"y":0.24862228488746527},{"energy":50,"id":272,"owner":2,"rotation":261,"type":"NORMAL","velocityX":-0.0066615805010968214,"velocityY":-0.045241560358216686,"x":0.28512949160736029,"y":-0.99800703828559378},{"energy":50,"id":273,"owner":0,"rotation":81,"type":"NORMAL","velocityX":0.0066228904174070197,"velocityY":0.045250186973400795,"x":-0.2834870021149401,"y":0.99726548939613213},{"energy":50,"id":274,"owner":3,"rotation":352,"type":"NORMAL","velocityX":0.045689959110709513,"velocityY":-0.0058188052223232334,"x":0.95362054501959825,"y":0.24990765794425884},{"energy":50,"id":275,"owner":1,"rotation":171,"type":"NORMAL","velocityX":-0.044923959102384357,"velocityY":0.0069355315952245884,"x":1,"y":-0.2962275844528407},{"energy":100,"id":276,"owner":0,"rotation":82,"type":"NORMAL","velocityX":0.0062795724239578564,"velocityY":0.04647713551196056,"x":-0.28894086448555978,"y":0.91288526077201859},{"energy":100,"id":277,"owner":1,"rotation":171,"type":"NORMAL","velocityX":-0.046179625974733907,"velocityY":0.006661765228149822,"x":-0.94687825760032251,"y":-0.30569771485268038},{"energy":100,"id":279,"owner":2,"rotation":262,"type":"NORMAL","velocityX":-0.0063679406956107633,"velocityY":-0.046460718041815646,"x":0.29298003163757769,"y":-0.91439084486355793},{"energy":150,"id":281,"owner":2,"rotation":262,"type":"NORMAL","velocityX":-0.0060375353241365198,"velocityY":-0.04750848849029065,"x":0.2994550911398759,"y":-0.82918824090864018},{"energy":350,"id":292,"owner":0,"rotation":85,"type":"NORMAL","velocityX":0.0038382231136699376,"velocityY":0.049796165691075298,"x":-0.26830894391853327,"y":0.47682940213517455},{"energy":350,"id":294,"owner":3,"rotation":356,"type":"NORMAL","velocityX":0.04992053241376225,"velocityY":-0.0030269623594529078,"x":0.40054002592339122,"y":0.21357121323067482},{"energy":400,"id":296,"owner":2,"rotation":265,"type":"NORMAL","velocityX":-0.0039779012535188932,"velocityY":-0.049923880645005546,"x":0.30698095471101899,"y":-0.39205864509595395},{"energy":400,"id":297,"owner":0,"rotation":86,"type":"NORMAL","velocityX":0.0032866811985763876,"velocityY":0.049967243702684176,"x":-0.25352478221662561,"y":0.39116168999102596},{"energy":400,"id":298,"owner":3,"rotation":357,"type":"NORMAL","velocityX":0.050083445397401102,"velocityY":-0.0025497371230175387,"x":0.31147557864318715,"y":0.19843279553908555},{"energy":450,"id":300,"owner":0,"rotation":86,"type":"NORMAL","velocityX":0.0027641876013088269,"velocityY":0.050129844536839399,"x":-0.23633328343026924,"y":0.30659733970474379},{"energy":450,"id":302,"owner":2,"rotation":265,"type":"NORMAL","velocityX":-0.0035337050939556235,"velocityY":-0.050090481798925973,"x":0.30230949603338386,"y":-0.30578097349133604},{"energy":500,"id":303,"owner":0,"rotation":87,"type":"NORMAL","velocityX":0.0022781329127172308,"velocityY":0.050284724110704462,"x":-0.21717178964972889,"y":0.22307551072717741},{"energy":500,"id":304,"owner":2,"rotation":266,"type":"NORMAL","velocityX":-0.0030824653647607966,"velocityY":-0.050251106730858294,"x":0.29401698202912019,"y":-0.2217612767868346},{"energy":500,"id":305,"owner":3,"rotation":358,"type":"NORMAL","velocityX":0.050382588333575486,"velocityY":-0.0016428531499613165,"x":0.14549698851389464,"y":0.15771321511368658},{"energy":500,"id":306,"owner":1,"rotation":176,"type":"NORMAL","velocityX":-0.050187602664805014,"velocityY":0.003489939689192912,"x":-0.26343487652418607,"y":-0.33167434420527797},{"energy":550,"id":307,"owner":2,"rotation":267,"type":"NORMAL","velocityX":-0.0025907047628598609,"velocityY":-0.050402332402002058,"x":0.27751693077180378,"y":-0.14467522494055615},{"energy":550,"id":308,"owner":1,"rotation":176,"type":"NORMAL","velocityX":-0.050347048081838659,"velocityY":0.0030660738259381849,"x":-0.17867131031165553,"y":-0.32760510753878508},{"energy":550,"id":309,"owner":3,"rotation":358,"type":"NORMAL","velocityX":0.050517545209855221,"velocityY":-0.0012281134503645931,"x":0.0705679043754011,"y":0.13229309463507136},{"energy":550,"id":310,"owner":0,"rotation":87,"type":"NORMAL","velocityX":0.0018331620105240614,"velocityY":0.050432876952011797,"x":-0.19636892807177236,"y":0.14055421153943687},{"energy":600,"id":311,"owner":2,"rotation":267,"type":"NORMAL","velocityX":-0.0021126005146007163,"velocityY":-0.050544621291774644,"x":0.25674655873182234,"y":-0.071347445746689747},{"energy":600,"id":312,"owner":0,"rotation":88,"type":"NORMAL","velocityX":0.0014256236174674957,"velocityY":0.050574207655245972,"x":-0.17335385660106847,"y":0.060299794470369178},{"energy":600,"id":313,"owner":1,"rotation":176,"type":"NORMAL","velocityX":-0.050501655137892079,"velocityY":0.0026529646237901387,"x":-0.094635458631537522,"y":-0.32195970108452221},{"energy":650,"id":314,"owner":2,"rotation":268,"type":"NORMAL","velocityX":-0.001655883745830127,"velocityY":-0.050677246779749906,"x":0.23144210366264675,"y":-0.0027859897412825102},{"energy":650,"id":315,"owner":3,"rotation":359,"type":"NORMAL","velocityX":0.050757935698320746,"velocityY":-0.00051940173834711344,"x":-0.05979234559459469,"y":0.072857602296205928},{"energy":650,"id":316,"owner":0,"rotation":88,"type":"NORMAL","velocityX":0.0010515911568273453,"velocityY":0.050707024379515631,"x":-0.14711813392295212,"y":-0.015027824149485974},{"energy":650,"id":317,"owner":1,"rotation":177,"type":"NORMAL","velocityX":-0.050651511112262987,"velocityY":0.0022525027404302128,"x":-0.011482139048412214,"y":-0.31473198419872467},{"energy":700,"id":318,"owner":2,"rotation":268,"type":"NORMAL","velocityX":-0.0012411819136081538,"velocityY":-0.050802110567526149,"x":0.20328931272334252,"y":0.062613412245978453},{"energy":700,"id":319,"owner":1,"rotation":177,"type":"NORMAL","velocityX":-0.050796521506563198,"velocityY":0.0018653321609953788,"x":0.070375320642015404,"y":-0.30568838667134673},{"energy":700,"id":320,"owner":0,"rotation":89,"type":"NORMAL","velocityX":0.00071770694806234643,"velocityY":0.050830547829740863,"x":-0.1176676865429976,"y":-0.083961597254809311},{"energy":750,"id":321,"owner":3,"rotation":359,"type":"NORMAL","velocityX":0.050972185771863167,"velocityY":-4.0404367469378909e-05,"x":-0.17197392940112372,"y":0.0079770972286763095},{"energy":750,"id":322,"owner":1,"rotation":178,"type":"NORMAL","velocityX":-0.050936749387863273,"velocityY":0.0014943093579937964,"x":0.15071702207697979,"y":-0.29482268392997729},{"energy":750,"id":323,"owner":2,"rotation":269,"type":"NORMAL","velocityX":-0.00087852382142266261,"velocityY":-0.050920884931804473,"x":0.17313458910071367,"y":0.12560867267034659},{"energy":800,"id":324,"owner":1,"rotation":178,"type":"NORMAL","velocityX":-0.051072300905343836,"velocityY":0.0011425448505191348,"x":0.22930835895897647,"y":-0.28213954076958236},{"energy":800,"id":325,"owner":3,"rotation":0,"type":"NORMAL","velocityX":0.051070209058752564,"velocityY":0.00010296223103063226,"x":-0.21999797877487057,"y":-0.025416073436254873},{"energy":800,"id":326,"owner":2,"rotation":269,"type":"NORMAL","velocityX":-0.00057244781850611056,"velocityY":-0.051034290877741799,"x":0.14116523753136703,"y":0.18588558918075798},{"energy":850,"id":327,"owner":3,"rotation":0,"type":"NORMAL","velocityX":0.051161569639703232,"velocityY":0.00017849388076209488,"x":-0.26139936685013765,"y":-0.058678638144885095},{"energy":850,"id":328,"owner":2,"rotation":269,"type":"NORMAL","velocityX":-0.00032745802106757258,"velocityY":-0.051142916220962162,"x":0.10761016328616749,"y":0.24309902988902579},{"energy":850,"id":329,"owner":1,"rotation":179,"type":"NORMAL","velocityX":-0.05120332045718453,"velocityY":0.00081342120955150217,"x":0.30589888525746201,"y":-0.26765581219316953},{"energy":900,"id":330,"owner":3,"rotation":0,"type":"NORMAL","velocityX":0.05124517393960458,"velocityY":0.00018495179348137212,"x":-0.29491659482906318,"y":-0.090982755526222384},{"energy":900,"id":331,"owner":2,"rotation":269,"type":"NORMAL","velocityX":-0.00014788041140234451,"velocityY":-0.051247157438095131,"x":0.072747780703739648,"y":0.29686642754571879},{"energy":900,"id":332,"owner":1,"rotation":179,"type":"NORMAL","velocityX":-0.051329982725061465,"velocityY":0.00051060543823088487,"x":0.38022097505128483,"y":-0.25140202836559017},{"energy":950,"id":333,"owner":3,"rotation":0,"type":"NORMAL","velocityX":0.051319176464310605,"velocityY":0.00012354191560894897,"x":-0.3188100378071701,"y":-0.12113370890814464},{"energy":950,"id":334,"owner":2,"rotation":269,"type":"NORMAL","velocityX":-3.7650218153377382e-05,"velocityY":-0.051347155686460026,"x":0.036916678757809136,"y":0.34675878590333059},{"energy":950,"id":335,"owner":1,"rotation":179,"type":"NORMAL","velocityX":-0.051452480537952074,"velocityY":0.00023805302452229872,"x":0.45198821491570323,"y":-0.23342408883211777},{"energy":1000,"id":336,"owner":1,"rotation":180,"type":"NORMAL","velocityX":-0.050000000000000003,"velocityY":6.1232339957367663e-18,"x":0.52156522814443973,"y":-0.21409757516468914},{"energy":1000,"id":337,"owner":2,"rotation":270,"type":"NORMAL","velocityX":-9.1848509936051487e-18,"velocityY":-0.050000000000000003,"x":0.000530638777252887,"y":0.39228783996125938}],"others":[{"energy":209,"id":1,"rotation":180,"velocityX":0.019074522523115138,"velocityY":0.01956017141400735,"x":0.52156522814443973,"y":-0.21409757516468914},{"energy":79,"id":2,"rotation":270,"velocityX":-0.036422993612946458,"velocityY":-0.0048681018839163929,"x":0.000530638777252887,"y":0.39228783996125938}],"you":{"energy":0,"id":3,"rotation":0,"velocityX":0.026478215038179608,"velocityY":-0.029999249056152873,"x":-0.36917921702419432,"y":-0.12125496387201665}})";
};
