#include <QCommandLineParser>
#include <QJsonDocument>
#include <QJsonObject>
#include <QTcpSocket>

#include "Action.h"
#include "Bot.h"
#include "State.h"

int main(int argc, char *argv[])
{
	QCoreApplication app(argc, argv);
	QCommandLineParser commandParser;
	commandParser.setApplicationDescription("A bot for the AI competition at The Gathering 2016. Written by Kwarf.");
	commandParser.addHelpOption();
	commandParser.addOptions({
		{"host", "Host to connect to", "ip/hostname", "127.0.0.1"},
		{"port", "TCP port", "port", "54321"}
	});
	commandParser.process(app);

	QTcpSocket socket;
	socket.connectToHost(commandParser.value("host"), commandParser.value("port").toInt());
	if (!socket.waitForConnected())
	{
		qCritical() << "Failed to connect to server:" << socket.errorString();
		return -1;
	}

	// Announce our name
	socket.write("NAME Allie\n");

	// Keep stuff here that we don't want to go out of scope between ticks for preformance reasons or whatever
	Bot bot;
	State currentState; // Better to .clear() between ticks and keep our pre-allocated vectors etc. than having a local that will get destructed

	while (socket.waitForReadyRead(-1))
	{
		if (!socket.canReadLine())
		{
			continue;
		}

		const auto& line = socket.readLine();
		const auto& json = QJsonDocument::fromJson(line);
		if (!json.isObject())
		{
			qWarning() << "Received something that wasn't a valid JSON object, skipping...";
			continue;
		}

		const auto& rootObject = json.object();
		const auto& messageTypeField = rootObject.find("messagetype");
		if (messageTypeField == rootObject.constEnd())
		{
			qWarning() << "Received JSON object with missing \"messagetype\" field, skipping...";
			continue;
		}

		const auto& messageType = messageTypeField.value();
		if (messageType == "dead" || messageType == "endofround")
		{
			// TODO: Reset bot state and wait for new round
		}
		else if (messageType == "stateupdate")
		{
			const auto& gameStateField = rootObject.find("gamestate");
			if (gameStateField == rootObject.constEnd())
			{
				qWarning() << "Received \"stateupdate\" message without \"gamestate\" field, skipping...";
				continue;
			}

			const auto& gameStateObject = gameStateField.value().toObject();
			currentState.parse(gameStateObject);

			QTime calculationTime;
			calculationTime.start();
			const auto& desiredAction = bot.handle(currentState);
			socket << desiredAction;
		}
		else
		{
			qWarning() << "Received unknown message type:" << messageType;
		}
	}

	return 0;
}
